% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/link_records.R
\name{link_records}
\alias{link_records}
\alias{links_wf_probabilistic}
\alias{prob_score_range}
\title{Record linkage}
\usage{
link_records(
  attribute,
  blocking_attribute = NULL,
  cmp_func = diyar::exact_match,
  attr_threshold = 1,
  probabilistic = TRUE,
  m_probability = 0.95,
  u_probability = NULL,
  score_threshold = 1,
  repeats_allowed = FALSE,
  permutations_allowed = FALSE,
  data_source = NULL,
  ignore_same_source = TRUE,
  display = "none"
)

links_wf_probabilistic(
  attribute,
  blocking_attribute = NULL,
  cmp_func = diyar::exact_match,
  attr_threshold = 1,
  probabilistic = TRUE,
  m_probability = 0.95,
  u_probability = NULL,
  score_threshold = 1,
  id_1 = NULL,
  id_2 = NULL,
  ...
)

prob_score_range(attribute, m_probability = 0.95, u_probability = NULL)
}
\arguments{
\item{attribute}{\code{[atomic|list|data.frame|matrix|d_attribute]}. Attributes to compare.}

\item{blocking_attribute}{\code{[atomic]}. Subsets of the dataset.}

\item{cmp_func}{\code{[list|function]}. String comparators for each \code{attribute}. See \code{Details}.}

\item{attr_threshold}{\code{[list|numeric|\link{number_line}]}. Weight-thresholds for each \code{cmp_func}. See \code{Details}.}

\item{probabilistic}{\code{[logical]}. If \code{TRUE}, scores are assigned base on Fellegi-Sunter model for probabilistic record linkage. See \code{Details}.}

\item{m_probability}{\code{[list|numeric]}. The probability that a matching records are the same entity.}

\item{u_probability}{\code{[list|numeric]}. The probability that a matching records are not the same entity.}

\item{score_threshold}{\code{[numeric|\link{number_line}]}. Score-threshold for linked records. See \code{Details}.}

\item{repeats_allowed}{\code{[logical]} If \code{TRUE}, repetition are included.}

\item{permutations_allowed}{\code{[logical]} If \code{TRUE}, permutations are included.}

\item{data_source}{\code{[character]}. Data source identifier. Adds the list of data sources in each record-group to the \code{\link[=pid-class]{pid}}. Useful when the data is from multiple sources.}

\item{ignore_same_source}{\code{[logical]} If \code{TRUE}, only records from different \bold{\code{data_source}} are compared.}

\item{display}{\code{[character]}. Display or produce a status update. Options are; \code{"none"} (default), \code{"progress"}, \code{"stats"}, \code{"none_with_report"}, \code{"progress_with_report"} or \code{"stats_with_report"}.}

\item{id_1}{\code{[list|numeric]}. Record id or index of one half of a record pair.}

\item{id_2}{\code{[list|numeric]}. Record id or index of one half of a record pair.}

\item{...}{Arguments passed to \bold{\code{links}}}
}
\value{
\code{\link[=pid-class]{pid}}; \code{list}
}
\description{
Deterministic and probabilistic record linkage with partial or evaluated matches.
}
\details{
\code{link_records()} and \code{links_wf_probabilistic()} are functions to implement deterministic, fuzzy or probabilistic record linkage.
\code{link_records()} compares every record-pair in one instance,
while \code{links_wf_probabilistic()} is a wrapper function of \code{\link{links}} and so compares batches of record-pairs in iterations.

\code{link_records()} is more thorough in the sense that it compares every combination of record-pairs.
This makes it faster but is memory intensive, particularly if there's no \code{blocking_attribute}.
In contrast, \code{links_wf_probabilistic()} is less memory intensive but takes longer since it does it's checks in batches.

The implementation of probabilistic record linkage is based on Fellegi and Sunter (1969) model for deciding if two records belong to the same entity.

In summary, record-pairs are created and categorised as matches and non-matches (\code{attr_threshold}) with user-defined functions (\code{cmp_func}).
Two probabilities (\code{m} and \code{u}) are then estimated for each record-pair to score the matches and non-matches.
The \code{m}-probability is the probability that matched records are actually from the same entity i.e. a true match,
while \code{u}-probability is the probability that matched records are not from the same entity i.e. a false match.
By default, \code{u}-probabilities are calculated as the frequency of each value of an \code{attribute} however,
they can also be supplied along with \code{m}-probabilities.
Record-pairs whose total score are above a certain threshold (\code{score_threshold}) are assumed to belong to the same entity.

Agreement (match) and disagreement (non-match) scores are calculated as described by Asher et al. (2020).

For each record pair, an agreement for attribute \eqn{i} is calculated as;

\deqn{\log_{2}(m_{i}/u_{i})}{log_2 (m_i / u_i)}

For each record pair, a disagreement score for attribute \eqn{i} is calculated as;

\deqn{\log_{2}((1-m_{i})/(1-u_{i}))}{log_2 ((1-m_i) / (1-u_i))}

where \eqn{m_{i}}{m_i} and \eqn{u_{i}}{u_i} are the \code{m} and \code{u}-probabilities for each value of attribute \eqn{i}.

Note that each probability is calculated as a combined probability for the record pair.
For example, if the values of the record-pair have \code{u}-probabilities of \code{0.1} and \code{0.2} respectively,
then the \code{u}-probability for the pair will be \code{0.02}.

Missing data (\code{NA}) are considered non-matches and assigned a \code{u}-probability of \code{0}.

By default, matches and non-matches for each \code{attribute} are determined as an \code{\link{exact_match}} with a binary outcome.
Alternatively, user-defined functions (\code{cmp_func}) are used to create similarity scores.
Pairs with similarity scores within (\code{attr_threshold}) are then considered matches for the corresponding \code{attribute}.

If \code{probabilistic} is \code{FALSE},
the sum of all similarity scores is used as the \code{score_threshold} instead of deriving one from the \code{m} and \code{u}-probabilities.

A \code{blocking_attribute} can be used to reduce the processing time by restricting comparisons to subsets of the dataset.

In \code{link_records()}, \code{score_threshold} is a convenience argument because every combination of record-pairs are returned
therefore, a new \code{score_threshold} can be selected after reviewing the final scores.
However, in \code{links_wf_probabilistic()}, the \code{score_threshold} is more important
because a final selection is made at each iteration.

As a result, \code{links_wf_probabilistic()} requires an acceptable \code{score_threshold} in advance.
To help with this, \code{prob_score_range()} can be used to return the range of scores attainable for a given set of \code{attribute}, \code{m} and \code{u}-probabilities.
Additionally, \code{id_1} and \code{id_2} can be used to link specific records pairs, aiding the review of potential scores.
}
\examples{
# Deterministic linkage
dfr <- missing_staff_id[c(2, 4, 5, 6)]

link_records(dfr, attr_threshold = 1, probabilistic = FALSE, score_threshold = 2)
links_wf_probabilistic(dfr, attr_threshold = 1, probabilistic = FALSE,
                       score_threshold = 2, recursive = TRUE)

# Probabilistic linkage
prob_score_range(dfr)
link_records(dfr, attr_threshold = 1, probabilistic = TRUE, score_threshold = -16)
links_wf_probabilistic(dfr, attr_threshold = 1, probabilistic = TRUE,
                       score_threshold = -16, recursive = TRUE)

# Using string comparators
# For example, matching last word in `hair_colour` and `branch_office`
last_word_wf <- function(x) tolower(gsub("^.* ", "", x))
last_word_cmp <- function(x, y) last_word_wf(x) == last_word_wf(y)

link_records(dfr, attr_threshold = 1,
             cmp_func = c(diyar::exact_match,
                          diyar::exact_match,
                          last_word_cmp,
                          last_word_cmp),
             score_threshold = -4)
links_wf_probabilistic(dfr, attr_threshold = 1,
                    cmp_func = c(diyar::exact_match,
                                 diyar::exact_match,
                                 last_word_cmp,
                                 last_word_cmp),
                    score_threshold = -4,
                    recursive = TRUE)

}
\references{
Fellegi, I. P., & Sunter, A. B. (1969). A Theory for Record Linkage. \emph{Journal of the Statistical Association}, 64(328), 1183–1210. https://doi.org/10.1080/01621459.1969.10501049

Asher, J., Resnick, D., Brite, J., Brackbill, R., & Cone, J. (2020). An Introduction to Probabilistic Record Linkage with a Focus on Linkage Processing for WTC Registries. \emph{International journal of environmental research and public health}, 17(18), 6937. https://doi.org/10.3390/ijerph17186937.
}
\seealso{
\code{\link{links}}
}
