#' Dot-and-Whisker Plots of Regression Results
#'
#' \code{dwplot} is a function for quickly and easily generating dot-and-whisker plots of regression models saved in tidy data frames.
#'
#' @param x Either a tidy data.frame (see 'Details'), a model object to be tidied with \code{\link[broom]{tidy}}, or a list of such model objects.
#' @param alpha A number setting the criterion of the confidence intervals. The default value is .05, corresponding to 95-percent confidence intervals.
#' @param dodge_size A number (typically between 0 and 0.3) indicating how much vertical separation should be between different models' coefficients when multiple models are graphed in a single plot.  Lower values tend to look better when the number of independent variables is small, while a higher value may be helpful when many models appear on the same plot.
#' @param order_vars A vector of variable names that specifies the order in which the variables are to appear along the y-axis of the plot.
#' @param show_intercept A logical constant indicating whether the coefficient of the intercept term should be plotted.
#' @param model_name The name of a variable that distinguishes separate models within a tidy data.frame.
#' @param dot_args A list of arguments specifying the appearance of the dots representing mean estimates.  For supported arguments, see \code{\link{geom_point}}.
#' @param whisker_args A list of arguments specifying the appearance of the whiskers representing confidence intervals.  For supported arguments, see \code{\link{geom_segment}}.
#' @param \dots Extra arguments to pass to \code{\link[broom]{tidy}}.
#'
#' @details \code{dwplot} visualizes regression results saved in tidy data.frames by, e.g., \code{\link[broom]{tidy}} as dot-and-whisker plots generated by \code{\link[ggplot2]{ggplot}}.
#'
#' Tidy data.frames to be plotted should include the variables \code{term} (names of predictors), \code{estimate} (corresponding estimates of coefficients or other quantities of interest), \code{std.error} (corresponding standard errors), and optionally \code{model} (when multiple models are desired on a single plot; a different name for this last variable may be specified using the model_name argument).
#' In place of \code{std.error} one may substitute \code{lb} (the lower bounds of the confidence intervals of each estimate) and \code{ub} (the corresponding upper bounds).
#'
#' For convenience, \code{dwplot} also accepts as input those model objects that can be tidied by \code{\link[broom]{tidy}}, or a list of such model objects.
#'
#' Because the function takes a data.frame as input, it is easily employed for a wide range of models, including those not supported by \code{\link[broom]{tidy}}.
#' And because the output is a \code{ggplot} object, it can easily be further customized with any additional arguments and layers supported by \code{ggplot2}.
#' Together, these two features make \code{dwplot} extremely flexible.
#'
#' @references
#' Kastellec, Jonathan P. and Leoni, Eduardo L. 2007. "Using Graphs Instead of Tables in Political Science." Perspectives on Politics, 5(4):755-771.
#'
#' @return The function returns a \code{ggplot} object.
#'
#' @import ggplot2
#' @import dplyr
#' @importFrom stats qnorm
#' @importFrom broom tidy
#' @importFrom plyr ldply
#'
#' @examples
#' library(broom)
#' library(dplyr)
#'
#' # Plot regression coefficients from a single model object
#' data(mtcars)
#' m1 <- lm(mpg ~ wt + cyl + disp, data = mtcars)
#'
#' dwplot(m1) +
#'     scale_y_continuous(breaks = 3:1, labels=c("Weight", "Cylinders", "Displacement")) +
#'     labs(x = "Coefficient", y = "") +
#'     geom_vline(xintercept = 0, colour = "grey50", linetype = 2) +
#'     theme(legend.position="none")
#'
#' # Plot regression coefficients from multiple models on the fly
#' m2 <- update(m1, . ~ . - disp)
#' dwplot(list(full = m1, nodisp = m2))
#'
#' # Change the appearance of dots and whiskers
#' dwplot(m1, dot_args = list(size = 6, pch = 21, fill = "white"),
#'      whisker_args = list(lwd = 2))
#'
#' # Plot regression coefficients from multiple models in a tidy data.frame
#' by_trans <- mtcars %>% group_by(am) %>%
#'     do(tidy(lm(mpg ~ wt + cyl + disp, data = .))) %>% rename(model=am)
#'
#' dwplot(by_trans, dodge_size = .05) +
#'     scale_y_continuous(breaks = 3:1, labels=c("Weight", "Cylinders", "Displacement")) +
#'     theme_bw() + labs(x = "Coefficient Estimate", y = "") +
#'     geom_vline(xintercept = 0, colour = "grey60", linetype = 2) +
#'     ggtitle("Predicting Gas Mileage, OLS Estimates") +
#'     theme(plot.title = element_text(face = "bold"),
#'           legend.justification=c(0, 0), legend.position=c(0, 0),
#'           legend.background = element_rect(colour="grey80"),
#'           legend.title.align = .5) +
#'     scale_colour_grey(start = .4, end = .8,
#'                       name = "Transmission",
#'                       breaks = c(0, 1),
#'                       labels = c("Automatic", "Manual"))
#'
#' @export

dwplot <- function(x, alpha = .05, dodge_size = .15, order_vars = NULL,
                   show_intercept = FALSE, model_name = "model",
                   dot_args = NULL, whisker_args = NULL, ...) {
    # If x is model object(s), convert to a tidy data.frame
    df <- dw_tidy(x,...)

    if (!show_intercept) df <- df %>% filter(term!="(Intercept)")

    # Set variables that will appear in pipelines to NULL to make R CMD check happy
    estimate <- model <- lb <- ub <- term <- std.error <- NULL

    n_vars <- length(unique(df$term))
    dodge_size <- dodge_size

    # Confirm number of models, get model names
    if (model_name %in% names(df)) {
        dfmod <- df[[model_name]]
        n_models <- length(unique(dfmod))
        ## re-order/restore levels by order in data set
        df[[model_name]] <- factor(dfmod, levels = unique(dfmod))
    } else {
        if (length(df$term) == n_vars) {
            df[[model_name]] <- factor("one")
            n_models <- 1
        } else {
            stop("Please add a variable named '",
                 model_name,"' to distinguish different models")
        }
    }
    mod_names <- unique(df[[model_name]])

    # Specify order of variables if an order is provided
    if (!is.null(order_vars)) {
        df$term <- factor(df$term, levels = order_vars)
    }

    # Add rows of NAs for variables not included in a particular model
    if (n_models > 1) {
        df <- add_NAs(df, n_models, mod_names)
    }

    # Prep arguments to ggplot
    var_names <- df$term

    y_ind <- rep(seq(n_vars, 1), n_models)
    df$y_ind <- y_ind

    # Confirm alpha within bounds
    if (alpha < 0 | alpha > 1) {
        stop("Value of alpha for the confidence intervals should be between 0 and 1.")
    }

    # Generate lower and upper bound if not included in results
    if ((!"lb" %in% names(df)) || (!"ub" %in% names(df))) {
        if ("std.error" %in% names(df)) {
            ci <- 1 - alpha/2
            df <- transform(df,
                            lb = estimate - stats::qnorm(ci) * std.error,
                            ub = estimate + stats::qnorm(ci) * df$std.error)
        } else {
            df <- transform(df, lb=NA, ub=NA)
        }
    }

    # Calculate y-axis shift for plotting multiple models
    if (n_models == 1) {
        shift <- 0
    } else {
        shift <- seq(dodge_size, -dodge_size, length.out = n_models)
    }
    shift_index <- data.frame(model = as.character(mod_names), shift, stringsAsFactors = F)
    df <- left_join(df, shift_index, by="model")

    # Catch difference between single and multiple models
    if (length(y_ind) != length(var_names)) {
        var_names <- unique(var_names)
    }

    # Generate arguments to geom_segment and geom_point
    seg_args0 <- list(aes(x = lb, xend = ub,
                          y = y_ind + shift, yend = y_ind + shift),
                      na.rm = TRUE)
    segment_args <- c(seg_args0, whisker_args)

    point_args0 <- list(na.rm = TRUE)
    point_args <- c(point_args0, dot_args)


    # Make the plot
    p <- ggplot(transform(df, model = factor(model)),
                          aes(x = estimate, y = y_ind + shift, colour = model)) +
        do.call(geom_segment, segment_args) +  # Draw segments first ...
        do.call(geom_point, point_args) +
        scale_y_continuous(breaks = y_ind, labels = var_names) +
        coord_cartesian(ylim = c(.5, n_vars+.5)) +
        ylab("") + xlab("")

    # Omit the legend if there is only one model
    if (!"model" %in% names(df) | length(mod_names) == 1){
        p <- p + theme(legend.position="none")
    }

    return(p)
}

dw_tidy <- function(x,...) {
    if (!is.data.frame(x)) {
        if (class(x)=="list") {
            ind <- seq(length(x))
            nm <- paste("Model", ind)
            if (!is.null(nm_orig <- names(x))) {
                setNm <- nchar(nm)>0
                nm[setNm] <- nm_orig[setNm]
            }
            names(x) <- nm

            df <- do.call(plyr::ldply,
                          c(list(.data=x,.fun=broom::tidy,.id="model"),
                            list(...)))

        } else if (class(x) == "lmerMod"){
            group <- vector() # only for avoiding the NOTE in check
            df <- broom::tidy(x) %>% filter(group == "fixed")
        } else {
            df <- broom::tidy(x,...)
        }
    } else {
        df <- x
    }
    return(df)
}

add_NAs <- function(df = df, n_models = n_models, mod_names = mod_names,
                    model_name = "model") {
    # Set variables that will appear in pipelines to NULL to make R CMD check happy
    term <- model <- NULL

    if (!is.factor(df$term)) {
        df$term <- factor(df$term, levels = unique(df$term))
    }
    if (!is.factor(dfmod <- df[[model_name]])) {
        df[[model_name]] <- factor(dfmod, levels = unique(dfmod))
    }
    for (i in seq(n_models)) {
        m <- df[dfmod==mod_names[[i]], ] %>% as.data.frame()
        for(k in seq(m)){
            if(is.factor(m[,k])) m[,k] <- as.character(m[,k])
        }
        not_in <- setdiff(unique(df$term), m$term)
        for (j in seq(not_in)) {
            t <- data.frame(term = not_in[j],
                            model = mod_names[[i]],
                            stringsAsFactors = FALSE)
            if ("submodel" %in% names(m)) {
                t$submodel <- m$submodel[1]
            }

            for(l in seq(t)){
              if(is.factor(t[,l])) t[,l] <- as.character(t[,l])
            }

            m <- full_join(m, t)
        }
        if (i==1) dft <- m else dft <- rbind(dft, m)
        if (i==1) order_var <- dft$term
    }

    order_df <- data.frame(term = order_var, stringsAsFactors = F)
    order_df$order <- seq(order_df$term)
    df <-  left_join(order_df, group_by(dft,model)) %>% ungroup %>% arrange(model)

    df$estimate <- as.numeric(df$estimate)
    if ("std.error" %in% names(df)) {
        df$std.error <- as.numeric(df$std.error)
    }
    if ("ub" %in% names(df)) {
        df$ub <- as.numeric(df$ub)
    }
    if ("lb" %in% names(df)) {
        df$lb <- as.numeric(df$lb)
    }


    return(df)
}
