#' @title Show an interactive visual network representation
#'   of your drake project.
#' @description To save time for repeated plotting,
#' this function is divided into
#' [drake_graph_info()] and [render_drake_graph()].
#' @export
#' @seealso [render_drake_graph()], [sankey_drake_graph()],
#'   [drake_ggraph()]
#' @return A `visNetwork` graph.
#' @inheritParams drake_graph_info
#' @inheritParams render_drake_graph
#' @examples
#' \dontrun{
#' test_with_dir("Quarantine side effects.", {
#' load_mtcars_example() # Get the code with drake_example("mtcars").
#' config <- drake_config(my_plan)
#' # Plot the network graph representation of the workflow.
#' vis_drake_graph(config, width = '100%') # The width is passed to visNetwork
#' make(my_plan) # Run the project, build the targets.
#' vis_drake_graph(config) # The red nodes from before are now green.
#' # Plot a subgraph of the workflow.
#' vis_drake_graph(
#'   config,
#'   from = c("small", "reg2"),
#'   to = "summ_regression2_small"
#' )
#' # Optionally visualize clusters.
#' config$plan$large_data <- grepl("large", config$plan$target)
#' vis_drake_graph(
#'   config, group = "large_data", clusters = c(TRUE, FALSE))
#' # You can even use clusters given to you for free in the `graph$nodes`
#' # data frame of `drake_graph_info()`.
#' vis_drake_graph(
#'   config, group = "status", clusters = "imported")
#' })
#' }
vis_drake_graph <- function(
  config = drake::read_drake_config(),
  file = character(0), selfcontained = FALSE,
  build_times = "build", digits = 3, targets_only = FALSE,
  split_columns = NULL, font_size = 20,
  layout = "layout_with_sugiyama", main = NULL,
  direction = "LR", hover = TRUE,
  navigationButtons = TRUE, # nolint
  from = NULL, mode = c("out", "in", "all"), order = NULL,
  subset = NULL,
  ncol_legend = 1,
  full_legend = FALSE,
  make_imports = TRUE,
  from_scratch = FALSE,
  group = NULL,
  clusters = NULL,
  show_output_files = TRUE,
  ...
){
  assert_pkgs("visNetwork")
  graph_info <- drake_graph_info(
    config = config,
    from = from,
    mode = mode,
    order = order,
    subset = subset,
    build_times = build_times,
    digits = digits,
    targets_only = targets_only,
    split_columns = split_columns,
    font_size = font_size,
    make_imports = make_imports,
    from_scratch = from_scratch,
    full_legend = full_legend,
    group = group,
    clusters = clusters,
    show_output_files = show_output_files
  )
  if (is.null(main)){
    main <- graph_info$default_title
  }
  render_drake_graph(graph_info, file = file, selfcontained = selfcontained,
    layout = layout, direction = direction,
    navigationButtons = navigationButtons, # nolint
    hover = hover, main = main,
    ncol_legend = ncol_legend, full_legend = full_legend, ...)
}

#' @title Render a visualization using the data frames
#'   generated by [drake_graph_info()].
#' @description This function is called inside
#' [vis_drake_graph()], which typical users
#' call more often.
#' @export
#' @seealso [vis_drake_graph()], [sankey_drake_graph()],
#'   [drake_ggraph()]
#' @return A `visNetwork` graph.
#'
#' @inheritParams drake_graph_info
#'
#' @param graph_info list of data frames generated by
#'   [drake_graph_info()].
#'   There should be 3 data frames: `nodes`, `edges`,
#'   and `legend_nodes`.
#'
#' @param file Name of a file to save the graph.
#'   If `NULL` or `character(0)`, no file is saved and
#'   the graph is rendered and displayed within R.
#'   If the file ends in a `.png`, `.jpg`, `.jpeg`, or `.pdf` extension,
#'   then a static image will be saved. In this case,
#'   the webshot package and PhantomJS are required:
#'   `install.packages("webshot"); webshot::install_phantomjs()`.
#'   If the file does not end in a `.png`, `.jpg`, `.jpeg`, or `.pdf`
#'   extension, an HTML file will be saved, and you can open the
#'   interactive graph using a web browser.
#'
#' @param layout name of an igraph layout to use,
#'   such as 'layout_with_sugiyama'
#'   or 'layout_as_tree'.
#'   Be careful with 'layout_as_tree': the graph is a directed
#'   acyclic graph, but not necessarily a tree.
#'
#' @param selfcontained logical, whether
#'   to save the `file` as a self-contained
#'   HTML file (with external resources base64 encoded) or a file with
#'   external resources placed in an adjacent directory. If `TRUE`,
#'   pandoc is required. The `selfcontained` argument only applies
#'   to HTML files. In other words, if `file` is a
#'   PNG, PDF, or JPEG file, for instance,
#'   the point is moot.
#'
#' @param direction an argument to `visNetwork::visHierarchicalLayout()`
#'   indicating the direction of the graph.
#'   Options include 'LR', 'RL', 'DU', and 'UD'.
#'   At the time of writing this, the letters must be capitalized,
#'   but this may not always be the case ;) in the future.
#'
#' @param navigationButtons logical, whether to add navigation buttons with
#'   `visNetwork::visInteraction(navigationButtons = TRUE)`
#'
#' @param hover logical, whether to show the command that generated the target
#'   when you hover over a node with the mouse. For imports, the label does not
#'   change with hovering.
#'
#' @param main character string, title of the graph
#'
#' @param ncol_legend number of columns in the legend nodes.
#'   To remove the legend entirely, set `ncol_legend` to `NULL` or `0`.
#'
#' @param ... arguments passed to `visNetwork()`.
#'
#' @examples
#' \dontrun{
#' test_with_dir("Quarantine side effects.", {
#' load_mtcars_example() # Get the code with drake_example("mtcars").
#' # Instead of jumpting right to vis_drake_graph(), get the data frames
#' # of nodes, edges, and legend nodes.
#' config <- drake_config(my_plan) # Internal configuration list
#' vis_drake_graph(config) # Jump straight to the interactive graph.
#' # Get the node and edge info that vis_drake_graph() just plotted:
#' graph <- drake_graph_info(config)
#' # You can pass the data frames right to render_drake_graph()
#' # (as in vis_drake_graph()) or you can create
#' # your own custom visNewtork graph.
#' render_drake_graph(graph, width = '100%') # Width is passed to visNetwork.
#' # Optionally visualize clusters.
#' config$plan$large_data <- grepl("large", config$plan$target)
#' graph <- drake_graph_info(
#'   config, group = "large_data", clusters = c(TRUE, FALSE))
#' render_drake_graph(graph)
#' # You can even use clusters given to you for free in the `graph$nodes`
#' # data frame.
#' graph <- drake_graph_info(
#'   config, group = "status", clusters = "imported")
#' render_drake_graph(graph)
#' })
#' }
render_drake_graph <- function(
  graph_info, file = character(0),
  layout = "layout_with_sugiyama", direction = "LR", hover = TRUE,
  main = graph_info$default_title, selfcontained = FALSE,
  navigationButtons = TRUE, # nolint
  ncol_legend = 1,
  ...
){
  assert_pkgs("visNetwork")
  out <- visNetwork::visNetwork(
    nodes = graph_info$nodes,
    edges = graph_info$edges,
    main = main,
    ...
  ) %>%
    visNetwork::visHierarchicalLayout(direction = direction)
  if (length(ncol_legend) && ncol_legend > 0){
    out <- visNetwork::visLegend(
      graph = out,
      useGroups = FALSE,
      addNodes = graph_info$legend_nodes,
      ncol = ncol_legend
    )
  }
  if (nrow(graph_info$edges)){
    out <- visNetwork::visIgraphLayout(
      graph = out,
      physics = FALSE,
      randomSeed = 2017,
      layout = layout
    )
  }
  if (navigationButtons) # nolint
    out <- visNetwork::visInteraction(out,
      navigationButtons = TRUE) # nolint
  if (hover)
    out <- with_hover(out)
  if (length(file)) {
    if (is_image_filename(file)){
      assert_pkgs("webshot")
      url <- file.path(fs::dir_create(tempfile()), "tmp.html")
      visNetwork::visSave(graph = out, file = url, selfcontained = FALSE)
      webshot::webshot(url = url, file = file)
    } else {
      visNetwork::visSave(graph = out, file = file,
        selfcontained = selfcontained)
    }
    return(invisible())
  }
  out
}

with_hover <- function(x) {
  visNetwork::visInteraction(x, hover = TRUE) %>%
    visNetwork::visEvents(hoverNode =
      "function(e){
        var label_info = this.body.data.nodes.get({
          fields: ['label', 'hover_label'],
          filter: function (item) {
            return item.id === e.node
          },
          returnType :'Array'
        });
        this.body.data.nodes.update({
          id: e.node,
          label : label_info[0].hover_label,
          hover_label : label_info[0].label
        });
      }"
    ) %>%
    visNetwork::visEvents(blurNode =
      "function(e){
        var label_info = this.body.data.nodes.get({
          fields: ['label', 'hover_label'],
          filter: function (item) {
            return item.id === e.node
          },
          returnType :'Array'
        });
        this.body.data.nodes.update({
          id: e.node,
          label : label_info[0].hover_label,
          hover_label : label_info[0].label
        });
      }"
    )
}
