% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dscore.R
\name{dscore}
\alias{dscore}
\alias{dscore_posterior}
\title{D-score estimation}
\usage{
dscore(
  data,
  items = names(data),
  key = NULL,
  population = NULL,
  xname = "age",
  xunit = c("decimal", "days", "months"),
  prepend = NULL,
  itembank = dscore::builtin_itembank,
  metric = c("dscore", "logit"),
  prior_mean = NULL,
  prior_sd = NULL,
  transform = NULL,
  qp = NULL,
  dec = c(2L, 3L),
  relevance = c(-Inf, Inf),
  algorithm = c("current", "1.8.7"),
  verbose = FALSE
)

dscore_posterior(
  data,
  items = names(data),
  key = NULL,
  population = NULL,
  xname = "age",
  xunit = c("decimal", "days", "months"),
  prepend = NULL,
  itembank = dscore::builtin_itembank,
  metric = c("dscore", "logit"),
  prior_mean = NULL,
  prior_sd = NULL,
  transform = NULL,
  qp = NULL,
  dec = c(2L, 3L),
  relevance = c(-Inf, Inf),
  algorithm = c("current", "1.8.7"),
  verbose = FALSE
)
}
\arguments{
\item{data}{A \code{data.frame} with the data.
A row collects all observations made on a child on a set of
milestones administered at a given age. The function calculates
a D-score for each row. Different rows can correspond to different
children or ages.}

\item{items}{A character vector containing names of items to be
included into the D-score calculation. Milestone scores are coded
numerically as \code{1} (pass) and \code{0} (fail). By default,
D-score calculation is done on all items found in the data
that have a difficulty parameter under the specified \code{key}.}

\item{key}{String. Name of the key that bundles the difficulty estimates
pertaining one the same Rasch model. View \code{builtin_keys} for an overview
of the available keys.}

\item{population}{String. The name of the reference population to calculate DAZ.
Use \code{unique(builtin_references$population)} to obtain the set of currently
available reference populations.}

\item{xname}{A string with the name of the age variable in
\code{data}. The default is \code{"age"}. Do not round age.}

\item{xunit}{A string specifying the unit in which age is measured
(either \code{"decimal"}, \code{"days"} or \code{"months"}).
The default \code{"decimal"} corresponds to decimal age in years.}

\item{prepend}{Character vector with column names in \code{data} that will
be prepended to the returned data frame. This is useful for copying
columns from data into the result, e.g., for matching.}

\item{itembank}{A \code{data.frame} with at least two columns
named \code{item} and \code{tau}. By default, the function uses
\code{dscore::builtin_itembank}. If you specify your own \code{itembank},
then you should also provide the relevant \code{transform} and \code{qp} arguments.}

\item{metric}{A string, either \code{"dscore"} (default) or
\code{"logit"}, signalling the metric in which ability is estimated.
\code{daz} is not calculated for the logit scale.}

\item{prior_mean}{A string or numeric scalar. If a string, it should
refer to a column name in \code{data} with user-supplied values of the prior mean
for each observation. If a numeric scalar, it is used as the prior mean
for all observations. The default (\code{NULL}) will consult the
\code{base_population} field in \code{builtin_keys}, and use the corresponding
median of that reference as prior mean for the D-score.}

\item{prior_sd}{A string or a numeric scalar. If a string, it should
refer to a column name in \code{data} with user-supplied values of the prior
sd for all observations. If a numeric scalar, it is used as the prior sd
for all observations. The default (\code{NULL}) uses a values of 5.}

\item{transform}{Numeric vector, length 2, containing the intercept
and slope of the linear transform from the logit scale into the
the D-score scale. The default (\code{NULL}) searches \code{builtin_keys}
for intercept and slope values.}

\item{qp}{Numeric vector of equally spaced quadrature points.
This vector should span the range of all D-score or logit values.
The default (\code{NULL}) creates \code{seq(from, to, by)} searching the
arguments from \code{builtin_keys}.}

\item{dec}{A vector of two integers specifying the number of
decimals for rounding the D-score and DAZ, respectively.
The default is \code{dec = c(2L, 3L)}.}

\item{relevance}{A numeric vector of length with the lower and
upper bounds of the relevance interval. The procedure calculates
a dynamic EAP for each item. If the difficulty level (tau) of the
next item is outside the relevance interval around EAP, the procedure
ignore the score on the item. The default is \code{c(-Inf, +Inf)} does not
ignore scores.}

\item{algorithm}{Computational method, for backward compatibility.
Either \code{"current"} (default) or \code{"1.8.7"} (deprecated).}

\item{verbose}{Logical. Print settings.}
}
\value{
The \code{dscore()} function returns a \code{data.frame} with \code{nrow(data)} rows.
Optionally, the first block of columns can be copied to the
result by using \code{prepend}. The second block consists of the
following columns:\tabular{ll}{
   Name \tab Label \cr
   \code{a} \tab Decimal age \cr
   \code{n} \tab Number of items with valid (0/1) data \cr
   \code{p} \tab Percentage of passed milestones \cr
   \code{d} \tab Ability estimate, mean of posterior \cr
   \code{sem} \tab Standard error of measurement, standard deviation of the posterior \cr
   \code{daz} \tab D-score corrected for age, calculated in Z-scale (for metric \code{"dscore"}) \cr
}


For more detail, the \code{dscore_posterior()} function returns a data frame with
\code{nrow(data)} rows and \code{length(qp)} plus prepended columns with the
full posterior density of the D-score at each quadrature point.
If no valid responses are found, \code{dscore_posterior()} returns the
prior density. Versions prior to 1.8.5 returned a \code{matrix} (instead of
a \code{data.frame}). Code that depends on the result being a \code{matrix} may break
and may need adaptation.
}
\description{
The \code{dscore()} function estimates the following quantities: \emph{D-score},
a numeric score that quantifies child development by one number,
\emph{Development-for-Age Z-score (DAZ)} that corrects the D-score for age,
\emph{standard error of measurement (SEM)} of the D-score.
}
\details{
The scoring algorithm is based on the method by Bock and Mislevy (1982).
The method uses Bayes rule to update a prior ability into a posterior
ability.

The item names should correspond to the \code{"gsed"} lexicon.

A key is defined by the set of estimated item difficulties.\tabular{lrrrcl}{
   Key \tab Model \tab Quadrature \tab Instruments \tab Direct/Caregiver \tab Reference \cr
   \code{"dutch"} \tab \verb{75_0} \tab \code{-10:80} \tab 1 \tab direct \tab Van Buuren, 2014/2020 \cr
   \code{"gcdg"} \tab \verb{565_18} \tab \code{-10:100} \tab 13 \tab direct \tab Weber, 2019 \cr
   \code{"gsed1912"} \tab \verb{807_17} \tab \code{-10:100} \tab 21 \tab mixed \tab GSED Team, 2019 \cr
   \code{"293_0"} \tab \verb{293_0} \tab \code{-10:100} \tab 2 \tab mixed \tab GSED Team, 2022 \cr
   \code{"gsed2212"} \tab \verb{818_6} \tab \code{-10:100} \tab 27 \tab mixed \tab GSED Team, 2022 \cr
   \code{"gsed2406"} \tab \verb{818_6} \tab \code{-10:100} \tab 27 \tab mixed \tab GSED Team, 2024 \cr
}


As a general rule, one should only compare D-scores
that are calculated using the same key and the same
set of quadrature points. For calculating D-scores on new data,
the advice is to use the default, which currently is \code{"gsed2406"}.

The default starting prior is a mean calculated from a so-called
"Count model" that describes mean D-score as a function of age. The
The Count models are implemented in the function \verb{[count_mu()]}.
By default, the spread of the starting prior
is 5 D-score points around the mean D-score, which corresponds to
approximately 1.5 to 2 times the normal spread of child of a given age. The
starting prior is informative for very short test (say <5 items), but has
little impact on the posterior for larger tests.
}
\examples{
# using all defaults and properly formatted data
ds <- dscore(milestones)
head(ds)

# step-by-step example
data <- data.frame(
  id = c(
    "Jane", "Martin", "ID-3", "No. 4", "Five", "6",
    NA_character_, as.character(8:10)
  ),
  age = rep(round(21 / 365.25, 4), 10),
  ddifmd001 = c(NA, NA, 0, 0, 0, 1, 0, 1, 1, 1),
  ddicmm029 = c(NA, NA, NA, 0, 1, 0, 1, 0, 1, 1),
  ddigmd053 = c(NA, 0, 0, 1, 0, 0, 1, 1, 0, 1)
)
items <- names(data)[3:5]

# third item is not part of the default key
get_tau(items, verbose = TRUE)

# calculate D-score
dscore(data)

# prepend id variable to output
dscore(data, prepend = "id")

# or prepend all data
# dscore(data, prepend = colnames(data))

# calculate full posterior
p <- dscore_posterior(data)

# check that rows sum to 1
rowSums(p)

# plot full posterior for measurement 7
barplot(as.matrix(p[7, 12:36]),
  names = 1:25,
  xlab = "D-score", ylab = "Density", col = "grey",
  main = "Full D-score posterior for measurement in row 7",
  sub = "D-score (EAP) = 11.58, SEM = 3.99")

# plot P10, P50 and P90 of D-score references
g <- expand.grid(age = seq(0.1, 4, 0.1), p = c(0.1, 0.5, 0.9))
d <- zad(z = qnorm(g$p), x = g$age, verbose = TRUE)
matplot(
  x = matrix(g$age, ncol = 3), y = matrix(d, ncol = 3), type = "l",
  lty = 1, col = "blue", xlab = "Age (years)", ylab = "D-score",
  main = "D-score preliminary standards: P10, P50 and P90")
abline(h = seq(10, 80, 10), v = seq(0, 4, 0.5), col = "gray", lty = 2)

# add measurements made on very preterms, ga < 32 weeks
ds <- dscore(milestones)
points(x = ds$a, y = ds$d, pch = 19, col = "red")
}
\references{
Bock DD, Mislevy RJ (1982).
Adaptive EAP Estimation of Ability in a Microcomputer Environment.
Applied Psychological Measurement, 6(4), 431-444.

Van Buuren S (2014). Growth charts of human development.
Stat Methods Med Res, 23(4), 346-368.
\url{https://stefvanbuuren.name/publication/van-buuren-2014-gc/}

Weber AM, Rubio-Codina M, Walker SP, van Buuren S, Eekhout I,
Grantham-McGregor S, Caridad Araujo M, Chang SM, Fernald LCH,
Hamadani JD, Hanlon A, Karam SM, Lozoff B, Ratsifandrihamanana L,
Richter L, Black MM (2019). The D-score: a metric for interpreting
the early development of infants and toddlers across global settings.
BMJ Global Health, BMJ Global Health 4: e001724.
\url{https://gh.bmj.com/content/bmjgh/4/6/e001724.full.pdf}
}
\seealso{
\code{\link[=builtin_keys]{builtin_keys()}}, \code{\link[=builtin_itembank]{builtin_itembank()}}, \code{\link[=builtin_itemtable]{builtin_itemtable()}},
\code{\link[=builtin_references]{builtin_references()}}, \code{\link[=get_tau]{get_tau()}}, \code{\link[=posterior]{posterior()}}, \code{\link[=milestones]{milestones()}}
}
\author{
Stef van Buuren, Iris Eekhout, Arjan Huizing (2022)
}
