% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/histo.R
\name{histo}
\alias{histo}
\title{Histogram with Added Options}
\usage{
histo(x, dis = "none", dis.shift = NULL, integer.breaks = NULL,
  points.list = NULL, axis.list = NULL, ...)
}
\arguments{
\item{x}{Numeric vector of values.}

\item{dis}{Character vector indicating which distribution should be used to 
add fitted PDF/PMF to the histogram. Possible values are \code{"none"}, 
\code{"beta"}, \code{"binom"} (must specify \code{size}), \code{"cauchy"}, 
\code{"chisq"}, \code{"exp"}, \code{"f"}, \code{"gamma"}, \code{"geom"}, 
\code{"hyper"} (must specify total number of balls in urn, \code{N}, and 
number of balls drawn each time, \code{k}), \code{"lnorm"}, \code{"nbinom"} 
(must specify \code{size}), \code{"norm"}, \code{"pois"}, \code{"t"}, 
\code{"unif"}, and \code{"weibull"}.}

\item{dis.shift}{Numeric value for shifting the fitted PDF/PMF along the 
x-axis of the histogram.}

\item{integer.breaks}{If \code{TRUE}, integers covering the range of \code{x} 
are used for breaks, so there is one bin for each integer. Useful for 
discrete distributions that don't take on too many unique values.}

\item{points.list}{Optional list of inputs to pass to 
\code{\link[graphics]{points}} function, which is used to add the fitted 
PDF/PMF.}

\item{axis.list}{Optional list of inputs to pass to 
\code{\link[graphics]{axis}} function.}

\item{...}{May include arguments to pass to \code{\link[graphics]{hist}} 
and/or parameter values needed for certain distributions (\code{size} if 
\code{dis = "binom"} or \code{dis = "nbinom"}, \code{N} and \code{k} if 
\code{dis = "hyper"}).}
}
\value{
Histogram with fitted PDF/PMF if requested.
}
\description{
Similar to base R function \code{\link[graphics]{hist}}, but with two added 
features: (1) Can overlay a fitted probability density/mass function 
(PDF/PMF) for any univariate distribution supported in R (see 
\code{\link[stats]{Distributions}}); and (2) Can generate more of a barplot 
type histogram, where each possible value gets its own bin centered over its 
value (useful for discrete variables with not too many possible values).
}
\details{
When \code{x} takes on whole numbers, you typically want to set 
\code{dis.shift = -0.5} if \code{right = TRUE} 
(\code{\link[graphics]{hist}}'s default) and \code{dis.shift = 0.5} if 
\code{right = FALSE}. The function will do this internally by default.

To illustrate, suppose a particular bin represents \code{(7, 10]}. Its 
midpoint will be at \code{x = 8.5} on the graph. But if input values are 
whole numbers, this bin really only includes values of 8, 9, and 10, which 
have a mean of 9. So you really want \code{f(9)} to appear at \code{x = 8.5}. 
This requires shifting the curve to the left 0.5 units, i.e. setting 
\code{dis.shift = -0.5}.

When \code{x} takes on whole numbers with not too many unique values, you may 
want the histogram to show one bin for each integer. You can do this by 
setting \code{integer.breaks = TRUE}. By default, the function sets 
\code{integer.breaks = TRUE} if \code{x} contains whole numbers with 10 or 
fewer unique values.
}
\examples{
# Generate 10,000 Poisson(2) values. Compare default histograms from hist vs. 
# histo.
set.seed(123)
x <- rpois(n = 10000, lambda = 2)
par(mfrow = c(1, 2))
hist(x)
histo(x)

# Generate 10,000 lognormal(0, 0.35) values. Create histograms with curves
# showing fitted log-normal and normal PDFs.
set.seed(123)
x <- rlnorm(n = 10000, meanlog = 0, sdlog = 0.35)
par(mfrow = c(1, 2))
histo(x, "lnorm", main = "Log-normal curve")
histo(x, "norm", main = "Normal curve")

# Generate 10,000 Binomial(8, 0.25) values. Create histogram, specifying 
# size = 5, with blue line/points showing fitted PMF.
set.seed(123)
x <- rbinom(n = 10000, size = 5, prob = 0.25)
par(mfrow = c(1, 1))
histo(x, "binom", size = 5, points.list = list(type = "b", col = "blue"))

}
