\name{eNetXplorer}
\alias{eNetXplorer}
\title{generates family of elastic net models for different alphas}
\description{
Elastic net uses a mixing parameter \code{alpha} to tune the penalty term continuously from ridge (\code{alpha=0}) to lasso (\code{alpha=1}). \code{eNetXplorer} generates a family of elastic net models over different values of \code{alpha} for the quantitative exploration of the effects of shrinkage. For each \code{alpha}, the regularization parameter \code{lambda} is chosen by optimizing a quality (objective) function based on out-of-bag cross-validation predictions. Statistical significance of each model, as well as that of individual features within a model, 
is assigned by comparison to a set of null models generated by random permutations of the response. \code{eNetXplorer} fits linear (gaussian), logistic (binomial), multinomial and Cox regression models.
}
\usage{
eNetXplorer(x, y, family=c("gaussian","binomial","multinomial","cox"),
alpha=seq(0,1,by=0.2), nlambda=100, nlambda.ext=NULL, seed=NULL, scaled=TRUE, 
n_fold=5, n_run=100, n_perm_null=25, save_obj=FALSE, dest_dir=getwd(), 
dest_dir_create=TRUE, dest_obj="eNet.Robj", QF.FUN=NULL, QF_label=NULL, 
cor_method=c("pearson","kendall","spearman"), 
binom_method=c("accuracy","precision","recall","Fscore","specificity","auc"),
multinom_method=c("avg accuracy","avg precision","avg recall","avg Fscore"), 
binom_pos=NULL, fscore_beta=NULL, fold_distrib_fail.max=100, 
cox_index=c("concordance","D-index"), logrank=FALSE, survAUC=FALSE, 
survAUC_time=NULL, survAUC_method=c("KM","NNE"), survAUC_lambda=NULL, 
survAUC_span=NULL, \dots)
}
\arguments{
  \item{x}{Input numerical matrix with instances as rows and features as columns. Instance and feature labels should be provided as row and column names, respectively. Can be in sparse matrix format (inherit from class \code{"sparseMatrix"} as in package \code{Matrix}). Cannot handle missing values.}
  \item{y}{Response variable. For \code{family="gaussian"}, numerical vector. For \code{family=}
  \code{"binomial"}, factor with two levels. For \code{family="multinomial"}, factor with two or more levels. For categorical families, if a vector is supplied, it will be coerced into a factor. 
  For \code{family="cox"}, matrix with columns named "time" and "status", where the latter is a binary indicator of event (1) or right-censoring (0).}
\item{family}{Response type: \code{"gaussian"} (numerical), \code{"binomial"} (2-level factor), \code{"multinomial"} (factor with >=2 levels) or \code{"cox"} (survival time and censoring status).}
\item{alpha}{Sequence of values for the mixing parameter penalty term in the elastic net family. Default is \code{seq(0,1,by=0.2)}.}
\item{nlambda}{Number of values for 
the regularization parameter \code{lambda}. Default is 100. Irrespective of \code{nlambda}, the range of \code{lambda} values is assigned by \code{glmnet}.}
\item{nlambda.ext}{If set to a value larger than \code{nlambda}, this will be the number of values for \code{lambda} obtained by extending the range assigned by \code{glmnet} symmetrically while keeping the \code{lambda} density uniform in log scale. Default is \code{NULL}, which will not extend the range of \code{lambda} assigned by \code{glmnet}.}
\item{seed}{Sets the pseudo-random number seed to enforce  reproducibility. Default is \code{NULL}.}
\item{scaled}{Z-score transformation of individual features across all instances. Default is \code{TRUE}.}
\item{n_fold}{Number of cross-validation folds per run. \code{lambda} is chosen based on the maximization of a quality function on out-of-bag-instances averaged over all runs. Default is 5.}
\item{n_run}{Number of runs (i.e. cross-validated model iterations); for each run, instances are randomly assigned to cross-validation folds. Default is 100.}
\item{n_perm_null}{Number of random null-model permutations of the response per run. Default is 25.}
\item{save_obj}{Logical to save the \code{eNetXplorer} object. Default is \code{FALSE}.}
\item{dest_dir}{Destination directory. Default is the working directory.}
\item{dest_dir_create}{Creates destination directory if it does not exist already. Default is \code{TRUE}.}
\item{dest_obj}{Name for output \code{eNetXplorer} object.}
\item{QF.FUN}{User-defined quality (objective) function as maximization criterion to select \code{lambda} based on response vs out-of-bag predictions (see example below). If not set, family-specific default quality functions are used, as follows: for \code{family="gaussian"}, default is correlation; for \code{family="binomial"}, it is accuracy; for \code{family=}\code{"multinomial"}, it is average accuracy; for \code{family=}\code{"cox"}, it is the concordance index (default) or D-index (set by \code{cox_index}).}
\item{QF_label}{Label for user-defined quality function, if \code{QF.FUN} is provided.}
\item{cor_method}{For \code{family="gaussian"}, correlation method to be used in the default quality function \code{cor.test}. Default is \code{"pearson"}.}
\item{binom_method}{For \code{family="binomial"}, method to be used in the quality function. Default is \code{"accuracy"}.}
\item{multinom_method}{For \code{family="multinomial"}, method to be used in the quality function. Default is \code{"avg accuracy"}.}
\item{binom_pos}{For \code{family="binomial"} and quality function methods other than the default (\code{"accuracy"}), this is the class to be considered positive. Default is the first level of the response factor.}
\item{fscore_beta}{For \code{family="binomial"} and quality function method \code{"Fscore"}, or for \code{family=} \code{"multinomial"} and quality function method \code{"avg Fscore"}, this is the beta factor to balance precision and recall. Default is 1.}
\item{fold_distrib_fail.max}{For categorical models, maximum number of failed attempts per run to have all classes represented in each in-bag fold. If this number is exceeded, the execution is halted; try again with larger \code{n_fold}, by removing/reasigning classes of small size, and/or with larger \code{fold_distrib_fail.max}. Default is 100.}
\item{cox_index}{For \code{family="cox"}, index method to be used in the default quality function.  Default is \code{"concordance"}, alternative choice is \code{"D-index"}.}
\item{logrank}{For \code{family="cox"}, logical to generate cross-validated log-rank test p-values of low- vs high-risk groups, defined by the median of out-of-bag predicted risk. Default is \code{FALSE}.}
\item{survAUC}{For \code{family="cox"}, logical to calculate area-under-curve (AUC) from cross-validated time-dependent ROC curves based on out-of-bag predicted risk. Default is \code{FALSE}.}
\item{survAUC_time}{For \code{family="cox"} (if \code{survAUC=T}), numerical vector with timepoints of interest; time must be in the same units as the response
variable \code{y}.}
\item{survAUC_method}{For \code{family="cox"} (if \code{survAUC=T}), Kaplan-Meier (\code{"KM"}) or Nearest-Neighbor Estimation (\code{"NNE"}) method. Default is \code{"KM"}.}
\item{survAUC_lambda}{For \code{family="cox"} (if \code{survAUC=T}), smoothing parameter for the \code{"NNE"} method. Default is \code{NULL}.}
\item{survAUC_span}{For \code{family="cox"} (if \code{survAUC=T}), span for the \code{"NNE"} method. Default is \code{NULL}.}
\item{\dots}{Accepts parameters from \code{glmnet.control(\dots)} to allow changes of factory default parameters in \code{glmnet}. If not explicitly set, it will use factory defaults.}
}
\details{
For each \code{alpha}, a set of \code{nlambda} values is 
obtained using the full data; if provided, \code{nlambda.ext} 
allows to extend the range of \code{lambda} values symmetrically while keeping its density uniform in log scale. Using these 
values of \code{lambda}, elastic net cross-validation models are generated for \code{n_run} random assignments of instances among \code{n_fold} folds; the best \code{lambda} is determined
by the maximization of a quality (objective) function that compares out-of-bag predictions against the response. 
A variety of quality functions are implemented for each response type, namely: for gaussian models, correlation (different correlation methods available); for binomial models, accuracy, precision, recall, F-score, specificity, area-under-curve; for multinomial models, average accuracy, precision, recall, F-score; for Cox regression models, concordance, D-index (Schroeder et al). Some of these choices require additional parameters: binomial measures that are not invariant under class permutation (see Sokolova & Lapalme) require to specify which class is to be considered positive; F-score requires to specify the value of the beta factor to balance precision and recall (F-score equals precision for beta=0 and tends to recall in the large beta limit). Besides these built-in options, user-defined quality functions can be provided via \code{QF.FUN}.
For each run, using the same assignment of instances into folds, \code{n_perm_null} null models are generated by shuffling the response. By using the quality function to compare the out-of-bag performance of the model to that of the null models, 
an empirical significance p-value is assigned to the model.
Similar procedures allow to obtain p-values for individual features based on absolute coefficient magnitude and on the frequency of non-zero coefficients. 
A family of elastic net models is thus generated for multiple 
values of \code{alpha} spanning the range from   
ridge (\code{alpha=0}) to lasso (\code{alpha=1}). This function 
returns an \code{eNetXplorer} object on which summary, plotting 
and export functions in this package can be applied for further 
analysis. 
For details about the underlying elastic net models (Friedman et al; Zhou & Hastie), refer to the \code{glmnet} package and references therein. 
For more details about \code{eNetXplorer}, see Candia & Tsang.

For Cox regression models, setting \code{logrank=T} generates cross-validated log-rank test p-values of low- vs high-risk groups, which are defined by the median of out-of-bag predicted risk (Simon et al). Moreover, setting \code{survAUC=T} and providing a numerical vector \code{survAUC_time} with timepoints of interest generates the AUC from cross-validated time-dependent ROC curves based on out-of-bag predicted risk (Simon et al) using the \code{survivalROC} package, which implements Kaplan-Meier and Nearest-Neighbor Estimation methods (Heagerty et al).
}
\value{
An object with S3 class \code{"eNetXplorer"}.
\item{predictor}{Predictor matrix used for regression (in sparse matrix format).}  
\item{response}{Response variable used for regression.}
\item{family}{Input parameter.}
\item{alpha}{Input parameter.}
\item{nlambda}{Input parameter.}
\item{nlambda.ext}{Input parameter.}
\item{seed}{Input parameter.}
\item{scaled}{Input parameter.}
\item{n_fold}{Input parameter.}
\item{n_run}{Input parameter.}
\item{n_perm_null}{Input parameter.}
\item{QF_label}{Input parameter.}
\item{cor_method}{Input parameter.}
\item{binom_method}{Input parameter.}
\item{multinom_method}{Input parameter.}
\item{binom_pos}{Input parameter.}
\item{fscore_beta}{Input parameter.}
\item{fold_distrib_fail.max}{Input parameter.}
\item{cox_index}{Input parameter.}
\item{logrank}{Input parameter.}
\item{survAUC}{Input parameter.}
\item{survAUC_time}{Input parameter.}
\item{survAUC_method}{Input parameter.}
\item{survAUC_lambda}{Input parameter.}
\item{survAUC_span}{Input parameter.}
\item{instance}{Instance labels.}
\item{feature}{Feature labels.}
\item{glmnet_params}{\code{glmnet} parameters used for regression.}
\item{best_lambda}{\code{lambda} values chosen by cross-validation.}
\item{model_QF_est}{Quality function values obtained by cross-validation.}
\item{QF_model_vs_null_pval}{P-value from model vs null comparison to assess statistical significance.}
\item{lambda_values}{List of \code{lambda} values used for each \code{alpha}.}
\item{lambda_QF_est}{List of quality function values obtained for each \code{alpha}.}
\item{predicted_values}{List of out-of-bag predicted values for each \code{alpha}; rows are instances and columns are median/mad predictions (for linear and Cox regression) or class predictions (for binomial and multinomial regression).}
\item{feature_coef_wmean}{Mean of feature coefficients (over runs) weighted by non-zero frequency (over folds) in sparse matrix format, with features as rows and \code{alpha} values as columns. For multinomial regression, it is a list of matrices (one matrix for each class).}
\item{feature_coef_wsd}{Standard deviation of feature coefficients (over runs) weighted by non-zero frequency (over folds) in sparse matrix format, with features as rows and \code{alpha} values as columns. For multinomial regression, it is a list of matrices (one matrix for each class).}
\item{feature_freq_mean}{Mean of non-zero frequency in sparse matrix format, with features as rows and \code{alpha} values as columns. For multinomial regression, it is a list of matrices (one matrix for each class).}
\item{feature_freq_sd}{Standard deviation of non-zero frequency in sparse matrix format, with features as rows and \code{alpha} values as columns. For multinomial regression, it is a list of matrices (one matrix for each class).}
\item{null_feature_coef_wmean}{Analogous to \code{feature_coef_wmean} for null model permutations.}
\item{null_feature_coef_wsd}{Analogous to \code{feature_coef_wsd} for null model permutations.}
\item{null_feature_freq_mean}{Analogous to \code{feature_freq_mean} for null model permutations.}
\item{null_feature_freq_sd}{Analogous to \code{feature_freq_sd} for null model permutations.}
\item{feature_coef_model_vs_null_pval}{P-value from model vs null comparison to assess statistical significance of mean non-zero feature coefficients in sparse matrix format, with features as rows and \code{alpha} values as columns. For multinomial regression, it is a list of matrices (one matrix for each class).}
\item{feature_freq_model_vs_null_pval}{P-value from model vs null comparison to assess statistical significance of mean non-zero feature frequencies in sparse matrix format, with features as rows and \code{alpha} values as columns. For multinomial regression, it is a list of matrices (one matrix for each class).}
\item{logrank_pval}{For Cox regression (if \code{logrank=T}), cross-validated log-rank test p-value of low- vs high-risk groups, defined by the median of out-of-bag predicted risk.}
\item{AUC_mean}{For Cox regression (if \code{survAUC=T}), mean AUC from cross-validated time-dependent ROC curves based on out-of-bag predicted risk, with timepoints (given by \code{survAUC_time}) as rows and \code{alpha} values as columns.}
\item{AUC_sd}{For Cox regression (if \code{survAUC=T}), standard deviation of AUC.}
\item{AUC_perc05}{For Cox regression (if \code{survAUC=T}), 5th percentile of AUC.}
\item{AUC_perc50}{For Cox regression (if \code{survAUC=T}), 50th percentile (median) of AUC.}
\item{AUC_perc95}{For Cox regression (if \code{survAUC=T}), 95th percentile of AUC.}
\item{AUC_pval}{For Cox regression (if \code{survAUC=T}), p-value of AUC from model vs null comparison to assess statistical significance.}
}
\author{Julian Candia and John S. Tsang\cr 
Maintainer: Julian Candia \email{julian.candia@nih.gov}}
\references{Candia J and Tsang JS. 
  \emph{eNetXplorer: an R package for the quantitative exploration of elastic net families 
  for generalized linear models}, BMC Bioinformatics (2019) 20:189.

Friedman J, Hastie T and Tibshirani R.
\emph{Regularization paths for generalized linear models via coordinate descent}, Journal of Statistical Software (2010) 33:1-22.

Heagerty PJ, Lumley T and Pepe MS.
\emph{Time-dependent ROC Curves for Censored Survival Data and a Diagnostic Marker}, Biometrics (2000) 56, 337-344.

Schroeder MS, Culhane AC, Quackenbush J, Haibe-Kains B.
\emph{survcomp: an R/Bioconductor package for performance assessment and comparison of survival models}, Bioinformatics (2011) 27:3206-8.

  Simon RM, Subramanian J, Li M-C and Menezes S.
\emph{Using cross-validation to evaluate predictive accuracy of survival risk classifiers based on high-dimensional data}, Briefings in Bioinformatics (2011) 12:203-14.

 Sokolova M and Lapalme G.
  \emph{A systematic analysis of performance measures for classification tasks}, Information Processing and Management (2009) 45, 427-437.
  
Zou H and Hastie T.
\emph{Regularization and variable selection via the elastic net}, Journal of the Royal Statistical Society Series B (2005) 67:301-20.
}
\seealso{\code{\link{summary}}, \code{\link{plot}}, \code{\link{summaryPDF}}, \code{\link{export}}
}
\examples{
\dontshow{
set.seed(123)
fit = eNetXplorer(x=matrix(rnorm(75),ncol=3),y=rnorm(25),family="gaussian",
n_run=5,n_perm_null=4,alpha=c(0.5,1))
}
\donttest{
# Linear models (synthetic dataset comprised of 20 features and 75 instances):
data(QuickStartEx)
fit = eNetXplorer(x=QuickStartEx$predictor, y=QuickStartEx$response,
family="gaussian", n_run=20, n_perm_null=10, seed=111)
}
\donttest{
# Custom QF provided (negative mean squared error)
data(QuickStartEx)
customQF = function(predicted,response){
     -mean((predicted-response)**2)
}
fit = eNetXplorer(x=QuickStartEx$predictor, y=QuickStartEx$response,
family="gaussian", n_run=20, n_perm_null=10, seed=111, QF.FUN=customQF, QF_label="MSE")
}
\donttest{
# Linear models to predict numerical day-70 H1N1 serum titers based on 
# day-7 cell population frequencies:
data(H1N1_Flow)
fit = eNetXplorer(x=H1N1_Flow$predictor_day7, y=H1N1_Flow$response_numer[rownames(
H1N1_Flow$predictor_day7)], family="gaussian", n_run=25, n_perm_null=15, seed=111)
}
\donttest{
# Binomial models to predict acute myeloid (AML) vs acute lymphoblastic (ALL) 
# leukemias: 
data(Leukemia_miR)
fit = eNetXplorer(x=Leuk_miR_filt$predictor, y=Leuk_miR_filt$response_binomial, 
family="binomial", n_run=25, n_perm_null=15, seed=111)
}
\donttest{
# Multinomial models to predict acute myeloid (AML), acute B-cell lymphoblastic 
# (B-ALL) and acute T-cell lymphoblastic (T-ALL) leukemias:
data(Leukemia_miR)
fit = eNetXplorer(x=Leuk_miR_filt$predictor, y=Leuk_miR_filt$response_multinomial,
family="multinomial", n_run=25, n_perm_null=15, seed=111)
}
\donttest{
# Binomial models to predict B-ALL vs T-ALL:
data(Leukemia_miR)
fit = eNetXplorer(x=Leuk_miR_filt$predictor[Leuk_miR_filt$response_multinomial!="AML",],
y=Leuk_miR_filt$response_multinomial[Leuk_miR_filt$response_multinomial!="AML"], 
family="binomial", n_run=25, n_perm_null=15, seed=111)
}
\donttest{
# Cox regression models to predict survival based on 7-gene signature:
data(breastCancerSurv)
fit = eNetXplorer(x=breastCancerSurv$predictor, y=breastCancerSurv$response, family="cox", 
n_run=25, n_perm_null=15, seed=111)
}
}

