% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eta_squared.R, R/eta_squared_posterior.R
\name{eta_squared}
\alias{eta_squared}
\alias{omega_squared}
\alias{epsilon_squared}
\alias{cohens_f}
\alias{cohens_f_squared}
\alias{eta_squared_posterior}
\title{Effect size for ANOVA}
\usage{
eta_squared(
  model,
  partial = TRUE,
  generalized = FALSE,
  ci = 0.9,
  verbose = TRUE,
  ...
)

omega_squared(model, partial = TRUE, ci = 0.9, verbose = TRUE, ...)

epsilon_squared(model, partial = TRUE, ci = 0.9, verbose = TRUE, ...)

cohens_f(
  model,
  partial = TRUE,
  ci = 0.9,
  squared = FALSE,
  verbose = TRUE,
  model2 = NULL,
  ...
)

cohens_f_squared(
  model,
  partial = TRUE,
  ci = 0.9,
  squared = TRUE,
  verbose = TRUE,
  model2 = NULL,
  ...
)

eta_squared_posterior(
  model,
  partial = TRUE,
  generalized = FALSE,
  ss_function = stats::anova,
  draws = 500,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{model}{A model, ANOVA object, or the result of \code{parameters::model_parameters}.}

\item{partial}{If \code{TRUE}, return partial indices.}

\item{generalized}{If TRUE, returns generalized Eta Squared, assuming all
variables are manipulated. Can also be a character vector of observed
(non-manipulated) variables, in which case generalized Eta Squared is
calculated taking these observed variables into account. For \code{afex_aov}
model, when \code{generalized = TRUE}, the observed variables are extracted
automatically from the fitted model, if they were provided then.}

\item{ci}{Confidence Interval (CI) level}

\item{verbose}{Toggle warnings and messages on or off.}

\item{...}{Arguments passed to or from other methods.
\itemize{
\item Can be \code{include_intercept = TRUE} to include the effect size for the intercept.
\item For Bayesian models, arguments passed to \code{ss_function}.
}}

\item{squared}{Return Cohen's \emph{f} or Cohen's \emph{f}-squared?}

\item{model2}{Optional second model for Cohen's f (/squared). If specified,
returns the effect size for R-squared-change between the two models.}

\item{ss_function}{For Bayesian models, the function used to extract
sum-of-squares. Uses \code{\link[=anova]{anova()}} by default, but can also be \code{car::Anova()}
for simple linear models.}

\item{draws}{For Bayesian models, an integer indicating the number of draws
from the posterior predictive distribution to return. Larger numbers take
longer to run, but provide estimates that are more stable.}
}
\value{
A data frame with the effect size(s) between 0-1 (\code{Eta2}, \code{Epsilon2},
\code{Omega2}, \code{Cohens_f} or \code{Cohens_f2}, possibly with the \code{partial} or
\code{generalized} suffix), and their CIs (\code{CI_low} and \code{CI_high}).
\cr\cr
For \code{eta_squared_posterior()}, a data frame containing the ppd of the Eta
squared for each fixed effect, which can then be passed to
\code{\link[bayestestR:describe_posterior]{bayestestR::describe_posterior()}} for summary stats.

A data frame containing the effect size values and their confidence
intervals.
}
\description{
Functions to compute effect size measures for ANOVAs, such as Eta, Omega and
Epsilon squared, and Cohen's f (or their partialled versions) for \code{aov},
\code{aovlist} and \code{anova} models. These indices represent an estimate of how much
variance in the response variables is accounted for by the explanatory
variable(s).
\cr\cr
Effect sizes are computed using the sums of squares obtained from
\code{anova(model)} which might not always be appropriate (\strong{\emph{Yeah... ANOVAs are
hard...}}). It is suggested that ANOVA models be fit with \code{afex} package.
See details.
}
\details{
For \code{aov} and \code{aovlist} models, the effect sizes are computed directly with
Sums-of-Squares (for \code{mlm} / \code{maov} models, effect sizes are computed for
each response separately). For all other model, the model is passed to
\code{anova()}, and effect sizes are approximated via test statistic conversion
(see \code{\link[=F_to_eta2]{F_to_eta2()}} for more details.)
\subsection{Type of Sums of Squares}{

The sums of squares (or F statistics) used for the computation of the effect
sizes is based on those returned by \code{anova(model)} (whatever those may be -
for \code{aov} and \code{aovlist} these are \emph{type-1} sums of squares; for \code{merMod}
these are \emph{type-3} sums of squares). Make sure these are the sums of squares
you are interested in; You might want to pass the result of \code{car::Anova(mode, type = 3)}, or use the \code{afex} package to fit ANOVA models.
\cr\cr
It is generally recommended to fit models with \emph{\code{contr.sum} factor weights}
and \emph{centered covariates}, for sensible results. See examples and the \code{afex}
package.
}

\subsection{Un-Biased Estimate of Eta}{

Both \emph{\strong{Omega}} and \emph{\strong{Epsilon}} are unbiased estimators of the
population's \emph{\strong{Eta}}, which is especially important is small samples. But
which to choose?
\cr\cr
Though Omega is the more popular choice (Albers \& Lakens, 2018), Epsilon is
analogous to adjusted R2 (Allen, 2017, p. 382), and has been found to be less
biased (Carroll & Nordholm, 1975).
\cr\cr
(Note that for \eqn{\omega_p^2} and \eqn{\epsilon_p^2} it is possible to
compute a negative number; even though this doesn't make any practical sense,
it is recommended to report the negative number and not a 0.)
}

\subsection{Cohen's f}{

Cohen's f can take on values between zero, when the population means are all
equal, and an indefinitely large number as standard deviation of means
increases relative to the average standard deviation within each group.
\cr\cr
When comparing two models in a sequential regression analysis, Cohen's f for
R-square change is the ratio between the increase in R-square
and the percent of unexplained variance.
\cr\cr
Cohen has suggested that the values of 0.10, 0.25, and 0.40 represent small,
medium, and large effect sizes, respectively.
}

\subsection{Eta Squared from Posterior Predictive Distribution}{

For Bayesian models (fit with \code{brms} or \code{rstanarm}),
\code{eta_squared_posterior()} simulates data from the posterior predictive
distribution (ppd) and for each simulation the Eta Squared is computed for
the model's fixed effects. This means that the returned values are the
population level effect size as implied by the posterior model (and not the
effect size in the sample data). See \code{\link[rstantools:posterior_predict]{rstantools::posterior_predict()}} for
more info.
}
}
\section{Confidence Intervals}{

Unless stated otherwise, confidence intervals are estimated using the
Noncentrality parameter method; These methods searches for a the best
non-central parameters (\code{ncp}s) of the noncentral t-, F- or Chi-squared
distribution for the desired tail-probabilities, and then convert these
\code{ncp}s to the corresponding effect sizes. (See full \link{effectsize-CIs} for
more.)
}

\section{CI Contains Zero}{

For positive only effect sizes (Eta squared, Cramer's V, etc.; Effect sizes
associated with Chi-squared and F distributions), special care should be
taken when interpreting CIs with a lower bound equal to 0, and even more care
should be taken when the \emph{upper} bound is equal to 0 (Steiger, 2004; Morey et
al., 2016). For example:\if{html}{\out{<div class="r">}}\preformatted{eta_squared(aov(mpg ~ factor(gear) + factor(cyl), mtcars[1:7, ]))
}\if{html}{\out{</div>}}\preformatted{## Parameter    | Eta2 (partial) |       90\% CI
## --------------------------------------------
## factor(gear) |           0.58 | [0.00, 0.84]
## factor(cyl)  |           0.46 | [0.00, 0.78]
}
}

\examples{
\donttest{
library(effectsize)
mtcars$am_f <- factor(mtcars$am)
mtcars$cyl_f <- factor(mtcars$cyl)

model <- aov(mpg ~ am_f * cyl_f, data = mtcars)

eta_squared(model)
eta_squared(model, generalized = "cyl_f")
omega_squared(model)
epsilon_squared(model)
cohens_f(model)
(etas <- eta_squared(model, partial = FALSE))

if (require(see)) plot(etas)

model0 <- aov(mpg ~ am_f + cyl_f, data = mtcars) # no interaction
cohens_f_squared(model0, model2 = model)

## Interpretation of effect sizes
## -------------------------------------

interpret_omega_squared(0.15, rules = "field2013")
interpret_eta_squared(0.15, rules = "cohen1992")
interpret_epsilon_squared(0.15, rules = "cohen1992")

# Recommended: Type-3 effect sizes + effects coding
# -------------------------------------------------
if (require(car, quietly = TRUE)) {
  contrasts(mtcars$am_f) <- contr.sum
  contrasts(mtcars$cyl_f) <- contr.sum

  model <- aov(mpg ~ am_f * cyl_f, data = mtcars)
  model_anova <- car::Anova(model, type = 3)

  eta_squared(model_anova)
}

# afex takes care of both type-3 effects and effects coding:
if (require(afex)) {
  data(obk.long, package = "afex")
  model <- aov_car(value ~ treatment * gender + Error(id / (phase)),
    data = obk.long, observed = "gender"
  )
  eta_squared(model)
  epsilon_squared(model)
  omega_squared(model)
  eta_squared(model, partial = FALSE)
  epsilon_squared(model, partial = FALSE)
  omega_squared(model, partial = FALSE)
  eta_squared(model, generalized = TRUE) # observed vars are pulled from the afex model.
}



## Approx. effect sizes for mixed models
## -------------------------------------
if (require(lmerTest, quietly = TRUE)) {
  model <- lmer(mpg ~ am_f * cyl_f + (1 | vs), data = mtcars)
  omega_squared(model)
}




## Bayesian Models (PPD)
## ---------------------
\dontrun{
if (require(rstanarm) && require(bayestestR) && require(car)) {
  fit_bayes <- stan_glm(mpg ~ factor(cyl) * wt + qsec,
    data = mtcars,
    family = gaussian(),
    refresh = 0
  )

  es <- eta_squared_posterior(fit_bayes,
    ss_function = car::Anova, type = 3
  )
  bayestestR::describe_posterior(es)


  # compare to:
  fit_freq <- lm(mpg ~ factor(cyl) * wt + qsec,
    data = mtcars
  )
  aov_table <- car::Anova(fit_freq, type = 3)
  eta_squared(aov_table)
}
}
}

}
\references{
\itemize{
\item Albers, C., \& Lakens, D. (2018). When power analyses based on pilot data
are biased: Inaccurate effect size estimators and follow-up bias. Journal of
experimental social psychology, 74, 187-195.
\item Allen, R. (2017). Statistics and Experimental Design for Psychologists: A
Model Comparison Approach. World Scientific Publishing Company.
\item Carroll, R. M., & Nordholm, L. A. (1975). Sampling Characteristics of
Kelley's epsilon and Hays' omega. Educational and Psychological Measurement,
35(3), 541-554.
\item Kelley, T. (1935) An unbiased correlation ratio measure. Proceedings of the
National Academy of Sciences. 21(9). 554-559.
\item Olejnik, S., & Algina, J. (2003). Generalized eta and omega squared
statistics: measures of effect size for some common research designs.
Psychological methods, 8(4), 434.
\item Steiger, J. H. (2004). Beyond the F test: Effect size confidence intervals
and tests of close fit in the analysis of variance and contrast analysis.
Psychological Methods, 9, 164-182.
}
}
\seealso{
\code{\link[=F_to_eta2]{F_to_eta2()}}

Other effect size indices: 
\code{\link{cohens_d}()},
\code{\link{effectsize}()},
\code{\link{phi}()},
\code{\link{rank_biserial}()},
\code{\link{standardize_parameters}()}
}
\concept{effect size indices}
