% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/woe.R
\name{step_woe}
\alias{step_woe}
\alias{tidy.step_woe}
\title{Weight of evidence transformation}
\usage{
step_woe(
  recipe,
  ...,
  role = "predictor",
  outcome,
  trained = FALSE,
  dictionary = NULL,
  Laplace = 1e-06,
  prefix = "woe",
  keep_original_cols = FALSE,
  skip = FALSE,
  id = rand_id("woe")
)
}
\arguments{
\item{recipe}{A recipe object. The step will be added to the sequence of
operations for this recipe.}

\item{...}{One or more selector functions to choose which variables will be
used to compute the components. See \link[recipes:selections]{recipes::selections} for more details. For
the \code{tidy} method, these are not currently used.}

\item{role}{For model terms created by this step, what analysis role should
they be assigned?. By default, the function assumes that the new woe
components columns created by the original variables will be used as
predictors in a model.}

\item{outcome}{The bare name of the binary outcome encased in \code{vars()}.}

\item{trained}{A logical to indicate if the quantities for preprocessing have
been estimated.}

\item{dictionary}{A tbl. A map of levels and woe values. It must have the
same layout than the output returned from \code{\link[=dictionary]{dictionary()}}. If \code{NULL} the
function will build a dictionary with those variables passed to \code{...}.
See \code{\link[=dictionary]{dictionary()}} for details.}

\item{Laplace}{The Laplace smoothing parameter. A value usually applied to
avoid -Inf/Inf from predictor category with only one outcome class. Set to
0 to allow Inf/-Inf. The default is 1e-6. Also known as 'pseudocount'
parameter of the Laplace smoothing technique.}

\item{prefix}{A character string that will be the prefix to the resulting new
variables. See notes below.}

\item{keep_original_cols}{A logical to keep the original variables in the
output. Defaults to \code{FALSE}.}

\item{skip}{A logical. Should the step be skipped when the recipe is baked by
\code{\link[recipes:bake]{recipes::bake()}}? While all operations are baked when \code{\link[recipes:prep]{recipes::prep()}} is
run, some operations may not be able to be conducted on new data (e.g.
processing the outcome variable(s)). Care should be taken when using \code{skip = TRUE} as it may affect the computations for subsequent operations}

\item{id}{A character string that is unique to this step to identify it.}
}
\value{
An updated version of \code{recipe} with the new step added to the
sequence of existing steps (if any). For the \code{tidy} method, a tibble with
the woe dictionary used to map categories with woe values.
}
\description{
\code{step_woe()} creates a \emph{specification} of a recipe step that will transform
nominal data into its numerical transformation based on weights of evidence
against a binary outcome.
}
\details{
WoE is a transformation of a group of variables that produces a new set of
features. The formula is

\deqn{woe_c = log((P(X = c|Y = 1))/(P(X = c|Y = 0)))}

where \eqn{c} goes from 1 to \eqn{C} levels of a given nominal predictor
variable \eqn{X}.

These components are designed to transform nominal variables into numerical
ones with the property that the order and magnitude reflects the association
with a binary outcome.  To apply it on numerical predictors, it is advisable
to discretize the variables prior to running WoE. Here, each variable will be
binarized to have woe associated later. This can achieved by using
\code{\link[recipes:step_discretize]{recipes::step_discretize()}}.

The argument \code{Laplace} is an small quantity added to the proportions of 1's
and 0's with the goal to avoid log(p/0) or log(0/p) results. The numerical
woe versions will have names that begin with \code{woe_} followed by the
respective original name of the variables. See Good (1985).

One can pass a custom \code{dictionary} tibble to \code{step_woe()}. It must have
the same structure of the output from \code{dictionary()} (see examples). If
not provided it will be created automatically. The role of this tibble is to
store the map between the levels of nominal predictor to its woe values. You
may want to tweak this object with the goal to fix the orders between the
levels of one given predictor. One easy way to do this is by tweaking an
output returned from \code{dictionary()}.
}
\section{Tidying}{
When you \code{\link[recipes:tidy.recipe]{tidy()}} this step, a tibble with columns \code{terms}
(the selectors or variables selected), \code{value}, \code{n_tot}, \code{n_bad}, \code{n_good},
\code{p_bad}, \code{p_good}, \code{woe} and \code{outcome} is returned.. See \code{\link[=dictionary]{dictionary()}} for
more information.

When you \code{\link[recipes:tidy.recipe]{tidy()}} this step, a tibble is returned with
columns \code{terms} \code{value}, \code{n_tot}, \code{n_bad}, \code{n_good}, \code{p_bad}, \code{p_good}, \code{woe}
and \code{outcome} and \code{id}:

\describe{
\item{terms}{character, the selectors or variables selected}
\item{value}{character, level of the outcome}
\item{n_tot}{integer, total number}
\item{n_bad}{integer, number of bad examples}
\item{n_good}{integer, number of good examples}
\item{p_bad}{numeric, p of bad examples}
\item{p_good}{numeric, p of good examples}
\item{woe}{numeric, weight of evidence}
\item{outcome}{character, name of outcome variable}
\item{id}{character, id of this step}
}
}

\section{Tuning Parameters}{
This step has 1 tuning parameters:
\itemize{
\item \code{Laplace}: Laplace Correction (type: double, default: 1e-06)
}
}

\section{Case weights}{


The underlying operation does not allow for case weights.
}

\examples{
\dontshow{if (rlang::is_installed("modeldata")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(modeldata)
data("credit_data")

set.seed(111)
in_training <- sample(1:nrow(credit_data), 2000)

credit_tr <- credit_data[in_training, ]
credit_te <- credit_data[-in_training, ]

rec <- recipe(Status ~ ., data = credit_tr) \%>\%
  step_woe(Job, Home, outcome = vars(Status))

woe_models <- prep(rec, training = credit_tr)

# the encoding:
bake(woe_models, new_data = credit_te \%>\% slice(1:5), starts_with("woe"))
# the original data
credit_te \%>\%
  slice(1:5) \%>\%
  dplyr::select(Job, Home)
# the details:
tidy(woe_models, number = 1)

# Example of custom dictionary + tweaking
# custom dictionary
woe_dict_custom <- credit_tr \%>\% dictionary(Job, Home, outcome = "Status")
woe_dict_custom[4, "woe"] <- 1.23 # tweak

# passing custom dict to step_woe()
rec_custom <- recipe(Status ~ ., data = credit_tr) \%>\%
  step_woe(
    Job, Home,
    outcome = vars(Status), dictionary = woe_dict_custom
  ) \%>\%
  prep()

rec_custom_baked <- bake(rec_custom, new_data = credit_te)
rec_custom_baked \%>\%
  dplyr::filter(woe_Job == 1.23) \%>\%
  head()
\dontshow{\}) # examplesIf}
}
\references{
Kullback, S. (1959). \emph{Information Theory and Statistics.} Wiley, New York.

Hastie, T., Tibshirani, R. and Friedman, J. (1986). \emph{Elements of Statistical
Learning}, Second Edition, Springer, 2009.

Good, I. J. (1985), "Weight of evidence: A brief survey", \emph{Bayesian
Statistics}, 2, pp.249-270.
}
\concept{preprocessing woe transformation_methods}
\keyword{datagen}
