#' plot.NestsResult Plot the information about a nest
#' @title Plot the information about a nest
#' @author Marc Girondot
#' @return A list with informations about statistics for the series of nests analyzed ($summary) and exact times series of length changes for each nest ($traces)
#' @param x A result file generated by searchR
#' @param ... Parameters for plot()
#' @param OutPlot A previous output of a plot.NestsResult() or info.nests()
#' @param parameters A set of parameters if result is not provided.
#' @param SexualisationTRN A set of parameters used to model thermal reaction norm during TSP
#' @param fixed.parameters Another set of parameters if result is not provided.
#' @param SE Standard error for each parameter if result is not provided.
#' @param temperatures Timeseries of temperatures. Will replace the one in result.
#' @param derivate Function used to fit embryo growth: dydt.Gompertz, dydt.exponential or dydt.linear
#' @param test Mean and SD of size of hatchlings ie test=c(Mean=xx, SD=yy)
#' @param stopattest True or False, does the plot stops when proxi of size reaches the mean test size.
#' @param M0 Measure of hatchling size proxi at laying date
#' @param series The name or number of the series to be displayed. Only one series can be displayed at a time.
#' @param col.stages The color of the stages
#' @param col.temperatures The color of the temperatures
#' @param col.TSP The color of the TSP
#' @param col.PT The color of the pivotal temperature
#' @param col.S The color of the size or mass. Can be a vector (useful when series="all" option).
#' @param lty.temperatures Type of line for temperatures
#' @param lwd.temperatures Width of line for temperatures
#' @param ylimT Range of temperatures to be displayed
#' @param ylimS Range of size to be displayed
#' @param xlim Range of incubation days to be displayed
#' @param TSP.borders The limits of TSP
#' @param embryo.stages The embryo stages. At least TSP.borders stages must be provided if show.TSP is TRUE
#' @param ref.stage If TRUE, the reference size for stage is the observed one in modelling. If FALSE, it is the field observed size.
#' @param show.stages True or False, does the embryo stages should be displayed
#' @param show.TSP True or False, does the TSP boders should be displayed
#' @param show.third True or False, does the first and second third boders should be displayed
#' @param show.CI True or False, do the confidence intervals should be displayed
#' @param replicate.CI Number of randomizations to estimate CI
#' @param show.temperatures True or False, does the temperatures should be displayed
#' @param show.PT TRUE or FALSE, does the pivotal temperature should be displayed
#' @param show.fioritures If FALSE, set show.PT, show.temperatures, show.stages, show.TSP, show.third, show.CI to FALSE
#' @param PT Value for pivotal temperature
#' @param show.test True or False, does the hatchling size should be displayed
#' @param lab.third Label for 2nd third of incubation
#' @param at.lab.third Position of Label for 2nd third of incubation [default=4]
#' @param lab.PT Label for Pivotal Temperature
#' @param lab.stages Label for Stages
#' @param xlab Label for axis
#' @param ylabT Label for temperature axis
#' @param ylabS Label for size axis
#' @param mar Parameter mar used for plot
#' @param show.plot If FALSE, only the text output is shown
#' @param progress If FALSE, the progress bar is not shown (useful for use with sweave or knitr)
#' @description Plot the information about a nest\cr
#' The embryo.stages is a list with stage numbers and relative size as compared to final size at the beginning of the stage.\cr
#' For example for Caretta caretta, embryo.stages=list(number=21:30, size=c(8.4, 9.4, 13.6, 13.8, 18.9, 23.5, 32.2, 35.2, 35.5, 38.5)/39.33) indicates that the stages 21 begins at the relative size of 8.4/39.33.\cr
#' The default is for the turtle "Caretta caretta".\cr
#' Series can be indicated as the name of the series, its number or succesion of TRUE or FALSE. "all" indicates that all series must be printed.\cr
#' @description The function returns an object with two components:\cr
#' \itemize{
#'   \item \code{summary} is a named list with summary statistics for each nest
#'   \item \code{traces} is a named list with detailled traces of embryo growth and 95% confidence interval
#' }
#' The summary objects are composed of these elements:
#' \itemize{
#'   \item \code{tsp.length.mean} Average number of days for the thermosensitive period
#'   \item \code{tsp.length.SE} Standard error for number of days for the thermosensitive period
#'   \item \code{tsp.begin.mean} Average number of days for the beginning of the thermosensitive period
#'   \item \code{tsp.begin.SE} Standard error for number of days for the beginning of the thermosensitive period
#'   \item \code{tsp.end.mean} Average number of days for the end of the thermosensitive period
#'   \item \code{tsp.end.SE} Standard error for number of days for the end of the thermosensitive period
#'   \item \code{incubation.length.mean} Average number of days for the incubation length
#'   \item \code{incubation.length.SE} Standard error for number of days for the incubation length
#'   \item \code{incubation.first.third} Day at which the middle third incubation begins
#'   \item \code{incubation.second.third} Day at which the middle third incubation ends
#'   \item \code{SSQ.borders.TSP} Squarred difference between incubation.first.third and tsp.begin.mean, and incubation.second.third and tsp.end.mean. Should be minimized at constant temperatures.
#'   \item \code{temperature.TSP.mean} Average temperature during the TSP
#'   \item \code{temperature.TSP.SE} Standard error for temperature during the TSP
#'   \item \code{temperature.mean} Average temperature during the entire incubation
#'   \item \code{end.size.mean} Average size of embryos at the end of incubation
#'   \item \code{end.size.SE} Standard error for size of embryos at the end of incubation
#'   \item \code{temperature.middle.third.mean} Average temperature during the middle third of incubation
#'   \item \code{weighted.TSP.temperature.mean} Averaged temperature during the TSP weighted by embryo growth
#'   \item \code{weighted.TSP.temperature.SE} Standard error for temperature during the TSP weighted by embryo growth
#'   \item \code{TSD.temperature.mean} Averaged temperature during the TSP weighted by sexualisation thermal reaction norm
#'   \item \code{TSD.temperature.SE} Standard error for temperature during the TSP weighted by sexualisation thermal reaction norm
#'   \item \code{weighted.TSD.temperature.mean} Averaged temperature during the TSP weighted by sexualisation thermal reaction norm and embryo growth
#'   \item \code{weighted.TSD.temperature.SE} Standard error for temperature during the TSP weighted by sexualisation thermal reaction norm and embryo growth
#'   \item \code{LnL} -Log Likelihood of observed embryo metric as compared to the model
#' }
#' show.fioritures parameter does not affect show.test option.\cr
#' @examples
#' \dontrun{
#' library(embryogrowth)
#' data(resultNest_4p)
#' plot(resultNest_4p, xlim=c(0,70), ylimT=c(22, 32), ylimS=c(0,45), series=1,  
#' 	SE=c(DHA=1.396525, DHH=4.101217, T12H=0.04330405, Rho25=1.00479))
#' # to plot all the nest at the same time, use
#' plot(resultNest_4p, xlim=c(0,70), ylimT=c(22, 32), ylimS=c(0,45),  
#' 	series="all", show.fioritures=FALSE)
#' # to use color different for series
#' plot(resultNest_4p, xlim=c(0,70), ylimT=c(22, 32), ylimS=c(0,45),  
#' 	series="all", show.fioritures=FALSE, col.S=c(rep("black", 5), rep("red", 6)))
#' }
#' @method plot NestsResult
#' @export


plot.NestsResult <-
function(x, ..., OutPlot=NULL, parameters=NULL, SexualisationTRN=NULL, fixed.parameters=NULL, SE=NULL, temperatures=NULL, derivate=NULL, 
	test=NULL, stopattest=FALSE, M0=NULL, series=1, col.stages="blue", col.PT="red", col.TSP="gray", 
		col.temperatures="green", col.S="black", 
	lty.temperatures=1, lwd.temperatures=2, ylimT=c(25, 35), ylimS=NULL, xlim=NULL, TSP.borders=c(21, 26), 
			embryo.stages="Caretta caretta", 
				show.stages=TRUE, show.TSP=TRUE, show.third=TRUE, show.CI=TRUE, replicate.CI=100, ref.stage=TRUE,
					show.fioritures=TRUE, progress=TRUE,
					show.temperatures=TRUE, show.PT=TRUE, PT=c(NA, NA), show.test=TRUE, 
            lab.third="2nd third of incubation", at.lab.third=4, lab.PT="PT", lab.stages="Stages", 
	mar = c(4, 5, 4, 5) + 0.3, 
  xlab="Days of incubation", ylabT=expression("Temperatures in " * degree * "C"), ylabS= "Embryo metric", show.plot=TRUE) {

# OutPlot=NULL;parameters=NULL;SexualisationTRN=NULL; fixed.parameters=NULL;SE=NULL;temperatures=NULL;derivate=NULL; test=NULL;stopattest=FALSE;M0=NULL;series=1;col.stages="blue";col.PT="red";col.TSP="gray"; col.temperatures="green";col.S="black";ylimT=c(25, 35);ylimS=NULL;xlim=NULL;TSP.borders=c(21, 26); embryo.stages="Caretta caretta"; show.stages=TRUE;show.TSP=TRUE;show.third=TRUE;show.CI=TRUE;replicate.CI=100;ref.stage=TRUE; show.fioritures=TRUE;progress=TRUE; show.temperatures=TRUE;show.PT=TRUE;PT=c(NA, NA);show.test=TRUE; lab.third="2nd third of incubation";at.lab.third=4;lab.PT="PT";lab.stages="Stages"; mar=c(4,5,4,5)+0.3; xlab="Days of incubation";ylabT=expression("Temperatures in"*degree*"C");ylabS="Embryo metric";show.plot=TRUE
# x=result.Georges; stopattest=TRUE; progress = FALSE

# out <- plot(x, parameters=parameters, fixed.parameters=fixed.parameters, SE=SE, temperatures=temperatures, derivate=derivate, 
#  test=test, stopattest=stopattest, M0=M0, series=series, TSP.borders=TSP.borders, progress=progress,
#  embryo.stages=embryo.stages, replicate.CI=replicate.CI, ref.stage=ref.stage, show.plot=FALSE)

#  x <- resultNest_4p; parameters=fitnewp; fixed.parameters = pfixed; temperatures=formated.Georges; stopattest = TRUE; progress=TRUE
  
#  parameters=parameters; fixed.parameters=fixed.parameters; SE=SE; temperatures=temperatures; derivate=derivate; test=test; stopattest=stopattest; M0=M0; series=series; TSP.borders=TSP.borders; progress=progress; embryo.stages=embryo.stages; replicate.CI=replicate.CI; ref.stage=ref.stage; show.plot=FALSE

  if (!replicate.CI | replicate.CI<2) replicate.CI <- 1
  
  result <- x

if (embryo.stages[1]=="Caretta caretta") 
	embryo.stages <- list(number=21:30, 
			size=c(8.4, 9.4, 13.6, 13.8, 18.9, 23.5, 32.2, 35.2, 35.5, 38.5)/39.33)

if (replicate.CI<=1) show.CI <- FALSE

# 27/1/2013
if (!show.fioritures) {
	show.PT <- FALSE
	show.temperatures <- FALSE
	show.stages <- FALSE
	show.TSP <- FALSE
	show.CI <- FALSE
	show.third <- FALSE
}


error=FALSE

# maintenant il n'est plus possible qu'il n'y ait pas de result

if (is.null(temperatures)) {
	temperatures <- result$data
	
	if (!is.null(test)) {
		if (is.numeric(test)) {
			testuse<-data.frame(Mean=rep(test["Mean"], result$data[["IndiceT"]][3]), SD=rep(test["SD"], result$data[["IndiceT"]][3]), row.names=names(result$data[1:result$data$IndiceT["NbTS"]]))
		} else {
			testuse<-test
		}
				
		test <- testuse

	} else {

		test <- result$test
	}

	
	
} else {
	
	if (!is.null(test)) {

# si j'ai une nouvelle série de température, je n'ai plus la correspondance des tailles de fin - 20/7/2012
		if (is.numeric(test)) {
			testuse<-data.frame(Mean=rep(test["Mean"], temperatures[["IndiceT"]][3]), SD=rep(test["SD"], temperatures[["IndiceT"]][3]), row.names=names(temperatures[1:temperatures$IndiceT["NbTS"]]))
		} else {
			testuse<-test
		}

		test <- testuse
	} else {
# si tous sont pareils, je reprends les mêmes
# Correction d'un bug, rajout de [[1]] dans result$test["Mean"][[1]][1] 30/7/2012
		if (all(result$test["Mean"]==result$test["Mean"][[1]][1]) & all(result$test["SD"]==result$test["SD"][[1]][1])) {
			test <- c(Mean=result$test["Mean"][[1]][1], SD=result$test["SD"][[1]][1])
			testuse <- data.frame(Mean=rep(test["Mean"], temperatures[["IndiceT"]][3]), SD=rep(test["SD"], temperatures[["IndiceT"]][3]), row.names=names(temperatures[1:temperatures$IndiceT["NbTS"]]))
			test <- testuse
		} else {	
			print("Error: the size at hatching must be provided because temperatures have been changed.")
			error <- TRUE
		}
	}
}


if (is.null(derivate)) {
	derivate <- result$derivate
}

if (is.null(M0)) {
	M0 <- result$M0
}

if (is.null(fixed.parameters)) {
	fixed.parameters <- result$fixed.parameters
}

if (is.null(parameters)) {parssm <- c(result$par, fixed.parameters)} else {parssm <- c(parameters, fixed.parameters)}

# je peux indiquer des SE en plus de ceux de result
if (is.null(SE)) {
	res <- result$SE
} else {
	res <- SE
}

if (is.null(SexualisationTRN)) SexualisationTRN <- parssm


if (show.stages & (length(embryo.stages$number) != length(embryo.stages$size))) {
	print("Size must be provided for all the embryological stages !")
	error=TRUE
}

if (show.TSP & (!any(embryo.stages$number==TSP.borders[1]) | !any(embryo.stages$number==TSP.borders[2]))) {
	print(paste("Information for at least embryological stages ", TSP.borders[1], " and ", TSP.borders[2], " must be provided !", sep=""))
	error=TRUE
}


###############################
#### je compte le nombre de séries à faire
###############################


NbTS <- temperatures[["IndiceT"]][3]
if (series[[1]]=="all") {
	series<-rep(TRUE, NbTS)
} else {
	if (any(!is.logical(series))) {
		if (is.numeric(series)) {
			seriesx <- rep(FALSE, NbTS)
			seriesx[series] <- TRUE
		} else {
			seriesx <- (names(temperatures[1:NbTS])==series)
		}
		series <- seriesx
	} else {
# c'est des valeurs logiques, je verifie si le bon nombre, sinon je complète
		if (length(series)!=NbTS) {
			series <- rep(series, NbTS)
			series <- series[1:NbTS]
		}
	}
}

seriesx <- series

nbseries <- sum(seriesx)
if (nbseries ==0) {
	error <- TRUE
	print("No series has been selected")
}

cptcol.S <- 0
col.S <- rep(col.S, nbseries)[1:nbseries]




if (!error) {



#####################
# Affiche les données de TSP

if (all(is.na(res))) {
	show.CI <- FALSE
#	nbpb <- nbseries
	replicate.CI <- 1
}

nbpb <- nbseries*replicate.CI
if (progress) pb<-txtProgressBar(min=0, max=nbpb, style=3)


premierprint <- TRUE
rtT <- NULL
listdf <- NULL
cptseriesfaites<-0

########################
#### Je teste les séries
########################

for(series in 1:NbTS) {

SDmass <- NA

# Tell if error occurs
errorcptduree <- FALSE

if (seriesx[series]) {
cptcol.S <- cptcol.S +1
cptseriesfaites<-cptseriesfaites+1

if (progress) setTxtProgressBar(pb, (cptseriesfaites-1)*replicate.CI)

namets<-names(temperatures[series])
mean.ts <- test[namets, "Mean"]
sd.ts <- test[namets, "SD"]

# 11/11/2012 La délimitation des stades est changée pour pouvoir être dépendante de la taille observée
# la taille du stade est dépendante de la taille finale. OK
stages <- (embryo.stages$size)*mean.ts
#convert stage to the individuals
nustage <- embryo.stages$number

nids <- temperatures[[series]]
colnames(nids) <- c("Time", "Temperature", "Temperature2", "r", "Mass", "IndiceK")

# 9/12/2014
# nids.sr <- temperatures[[series]]
# colnames(nids.sr) <- c("Time", "Temperature", "Temperature2", "r", "Mass")


# Je dois recalculer toutes les informations heure par heure pour avoir la TSP en heure
# la durée d'incubation est= (0:(nids[,1][length(nids[,1])]%/%60)-1)*60
# et ensuite j'intercale les températures

# Je créé un tableau avec les données heures par heure
tl1 <- seq(from=0, to=tail(nids[,1], n=1), by=60)
# je prends les vraies données
tl2 <- nids[,1]


it <- findInterval(tl1, tl2, all.inside=TRUE)
dftl1 <- data.frame(Time=tl1, Temperature=rep(NA, length(it)), 
                    Temperature2=rep(NA, length(it)), r=rep(NA, length(it)), 
                    Mass=rep(NA, length(it)), IndiceK=rep(NA, length(it)))

# en Celsius
dftl1[,"Temperature"] <- nids[it,"Temperature"]
# En Kelvin
dftl1[,"Temperature2"] <- nids[it,"Temperature2"]

tl <- rbind(nids, dftl1)

tls <- tl[do.call(order, c(tl, na.last=TRUE)),]


# je rajoute un NA et début sur une série et un NA à la fin sur une autre
tls1<-c(NA, tls[,1])
tls2<-c(tls[,1], NA)

# TRUE si le temps est identique
partial <- (tls1==tls2)

# FALSE sur le premier
partial[1] <- FALSE

# je retire le dernier 
partial <- partial[-length(partial)]

# je ne garde que les FALSE, donc ceux qui sont différents
tls <- tls[!partial,]

# je retire la colonne rang
#tls <- tls[,1:5]

tempaf<- as.factor(nids[,3])
templevels <- levels(tempaf)

nids <- tls

# valeur.r.list <- embryogrowth:::.SSM(as.numeric(templevels), parssm)
valeur.r.list <- .SSM(as.numeric(templevels), parssm)

valeur.r <- valeur.r.list[[1]]
valeur.r_L <- valeur.r.list[[2]]

names(valeur.r)<-templevels
names(valeur.r_L)<-templevels

# 12/2/2014
numerictemplevels <- as.numeric(templevels)

	anK <- NULL

	y <- M0
	nids[1,5] <- y
# 9/12/2014
#  nids.sr[1,5] <- y

	k <- dim(nids)[1]-1
	
	if (!is.na(parssm["rK"])) {
		anK <- parssm[["rK"]]*mean.ts
	}
	if (!is.na(parssm["K"])) {
		anK <- parssm[["K"]]
	}

meanduree <- NULL

transition_P <- parssm["transition_P"][[1]]
transition_S <- parssm["transition_S"][[1]]

	
if (stopattest) {
	sizefin <- mean.ts
	
	for (i in 1:k) {
		timesunique <- c(nids[i,"Time"], nids[i+1,"Time"])
		if (is.na(transition_S) | is.na(transition_P)) {
			transition <- 1
		} else {
			transition <- 1/(1+exp(transition_S*(y-transition_P)))
		}
		
#		parms <- c(alpha=as.numeric(valeur.r[names(valeur.r)==nids[i,3]]), K=anK)
		
    # dans pos j'ai la position de la température la plus proche
    # 12/2/2014 remplace as.numeric(names(valeur.r)) par templevels
		pos <- which.min(abs(numerictemplevels-nids[i,3]))
		a <- as.numeric(valeur.r[pos])*transition+as.numeric(valeur.r_L[pos])*(1-transition)

		parms <- c(alpha=a, K=anK)
		out1 <- lsoda(y, timesunique, derivate, parms)

    # parms.sr <- c(alpha=valeur.r.sr[pos], K=anK)
    # out1.sr <- lsoda(y.sr, timesunique, derivate, parms.sr)

#if (!is.finite(out1[2,2])) {
#  print(paste("numerictemplevels=", numerictemplevels))
#  print(paste("pos=", pos))
#  print(paste("valeur.r[pos]", valeur.r[pos]))
#  valeur.r <<- valeur.r
#  print(paste("parms=", parms))
#  print(paste("timesunique=", timesunique))
#  print(paste("y=", y))
#}

    y <- as.numeric(out1[2,2])
		nids[i+1,5] <- y
		if (y>=sizefin) {
			meanduree <- nids[i+1,"Time"]
			break
		}
	}
	
	LnL <- NA

	
} else {
	for (i in 1:k) {
		timesunique <- c(nids[i,"Time"], nids[i+1,"Time"])
		if (is.na(transition_S) | is.na(transition_P)) {
			transition <- 1
		} else {
			transition <- 1/(1+exp(transition_S*(y-transition_P)))
		}
    # 13/2/2014
    pos <- which.min(abs(numerictemplevels-nids[i,3]))
		a <- as.numeric(valeur.r[pos]*transition+valeur.r_L[pos]*(1-transition))
		parms <- c(alpha=a, K=anK)
		out1 <- lsoda(y, timesunique, derivate, parms)
		y <- as.numeric(out1[2,2])
		nids[i+1,5] <- y
	}
	LnL <- -dnorm(y, mean=mean.ts, sd=sd.ts, log=TRUE)

}

taillefin <- y
# je recalcule la position des stades en fonction de cette taille
if (ref.stage) stages <- (embryo.stages$size)*taillefin


# parpre <- par(mar = c(4, 5, 4, 5) + 0.3)

#GOTO 100
if (is.null(meanduree) & stopattest) {
	cat("\n")
	print(paste("The timeseries", namets, "of temperatures is too short to reach the end of incubation"))
	print(paste("The final size is ", y, sep=" "))
	print(paste("Whereas it should reach ", sizefin, sep=" "))

  
	rt <- c(tsp.length.mean=NA, tsp.length.SE=NA)
	rt <- c(rt, tsp.begin.mean=NA, tsp.begin.SE=NA)
	rt <- c(rt, tsp.end.mean=NA, tsp.end.SE=NA)
	rt <- c(rt, incubation.length.mean=NA, incubation.length.SE=NA, incubation.first.third=NA, incubation.second.third=NA )
	rt <- c(rt, SSQ.borders.TSP=NA)
	rt <- c(rt, temperature.TSP.mean=NA)
	rt <- c(rt, temperature.TSP.SE=NA)
	rt <- c(rt, temperature.mean=NA)
	rt <- c(rt, end.size.mean=NA, end.size.SE=NA)
	# 21/1/2014
	rt <- c(rt, temperature.middle.third.mean=NA)
	rt <- c(rt, weighted.TSP.temperature.mean=NA)
	rt <- c(rt, weighted.TSP.temperature.SE=NA)
  # 12/12/2014
	rt <- c(rt, TSD.temperature.mean=NA)
	rt <- c(rt, TSD.temperature.SE=NA)
	rt <- c(rt, LnL=LnL)

#ELSE 100
} else {

	if (show.plot) {
		# dummy graph
		if (premierprint) {
	
			x <- nids[,1]/1440
			y <- nids[,5]
      
# 25/3/2013			
			if (is.null(ylimS)) {
				ylimS <- c(0, max(y))
			}

			if (is.null(xlim)) {
			  xlim <- c(0, max(x))
			}
			
			if (show.test) {
				L <- modifyList(list(type = "n", axes = TRUE, bty = "n", xlab = xlab, ylab = ylabS, lwd=2, yaxs="r", xaxs="r", ylim=ylimS, xlim=xlim, main="", col=col.S[cptcol.S]), list(x=c(0, x), y=c(mean.ts+2*sd.ts, y), ...)) 
			} else {
				L <- modifyList(list(type = "n", axes = TRUE, bty = "n", xlab = xlab, ylab = ylabS, lwd=2, yaxs="r", xaxs="r", ylim=ylimS, xlim=xlim, main="", col=col.S[cptcol.S]), list(x=x, y=y, ...)) 				
			}



			do.call(plot, L) 

			premierprint <- FALSE
			
			x2 <- (par("usr")[1]+par("usr")[2]*26)/27
			x1 <- x2*26-par("usr")[2]/0.04

			xlim <- c(x1, x2)
			
			y2 <- (par("usr")[3]+par("usr")[4]*26)/27
			y1 <- y2*26-par("usr")[4]/0.04
			
			ylimS <- c(y1, y2)

			

		} else {
			par(new=TRUE)
			
		}
		
	} else {
		xlim=c(0,70)
	}


	if (show.TSP) {

		x1 <- xlim[1]
		x2 <- nids[dim(nids)[1], 1]/1440
		ty1 <- nids[subset(1:dim(nids)[1], nids[,5]>stages[nustage==TSP.borders[1]])[1], 5]
		ty2 <- nids[subset(1:dim(nids)[1], nids[,5]>stages[nustage==TSP.borders[2]])[1], 5]
		if (show.plot) {
			polygon(x=c(x1, x2, x2, x1), y=c(ty1, ty1, ty2, ty2), col=col.TSP, border=NA)
		}
	}

	if (show.temperatures & show.plot) {
		par(new = TRUE)

		x <- nids[,1]/1440
		y <- nids[,2]

		Lx <- modifyList(list(...), list(x=x, y=y, axes = FALSE, xlab = "", ylab = "", main="")) 
		L <- modifyList(list(type = "l", bty = "n", xlab = "", ylab = "", xlim=xlim, ylim=ylimT, main="", las=1, col=col.temperatures, lty=lty.temperatures, lwd=lwd.temperatures), Lx) 

		do.call(plot, L) 

		axis(side=4, ylim=ylimT, las=1)
		mtext(ylabT, side=4, line=3)
	
		if (show.PT & !is.na(PT[1])) {

			segments(0, PT[1], xlim[2]+xlim[2]*0.05, PT[1], lwd=2, col=col.PT, xpd=TRUE)
			text(xlim[2]+6, PT[1], xpd=TRUE, labels=lab.PT, col=col.PT, cex=0.8)
			if (!is.na(PT[2])) {
				segments(0, PT[1]+PT[2]*2, xlim[2]+xlim[2]*0.05, PT[1]+PT[2]*2, lwd=2, col=col.PT, lty=2, xpd=TRUE)
				segments(0, PT[1]-PT[2]*2, xlim[2]+xlim[2]*0.05, PT[1]-PT[2]*2, lwd=2, col=col.PT, lty=2, xpd=TRUE)
			}
		
		}
	}


	if (show.plot) {
	# je réaffiche la courbe qui a pu être effacée
		par(new = TRUE)

		x <- nids[,1]/1440
		y <- nids[,5]

		Lx <- modifyList(list(...), list(x=x, y=y, axes = FALSE, xlab = "", ylab = "", main="")) 
		L <- modifyList(list(type = "l", bty = "n", xlim=xlim, lwd=2, ylim=ylimS, col=col.S[cptcol.S]), Lx)

		do.call(plot, L)

		if (show.test) {
			segments(0, mean.ts, xlim[2]+0.05*xlim[2], mean.ts, lwd=2, xpd=TRUE)
			segments(0, mean.ts-2*sd.ts,  xlim[2]+0.05*xlim[2], mean.ts-2*sd.ts, lwd=1, lty=2, xpd=TRUE)
			segments(0, mean.ts+2*sd.ts,  xlim[2]+0.05*xlim[2], mean.ts+2*sd.ts, lwd=1, lty=2, xpd=TRUE)
		}
	}

	x1 <- nids[subset(1:dim(nids)[1], nids[,5]>=stages[nustage==TSP.borders[1]])[1], 1]/1440
	x2 <- nids[subset(1:dim(nids)[1], nids[,5]>stages[nustage==TSP.borders[2]])[1]-1, 1]/1440

	indice.debut.tsp <- subset(1:dim(nids)[1], nids[,5]>=stages[nustage==TSP.borders[1]])[1]
	indice.fin.tsp <- subset(1:dim(nids)[1], nids[,5]>stages[nustage==TSP.borders[2]])[1]-1

	length.tsp <- x2-x1
	debut.tsp <- x1
	fin.tsp <- x2

	if (show.TSP & show.plot) {
		text(4, ty1+(ty2-ty1)/3+3, labels="TSP", srt=90, xpd=TRUE)

		y1=0
		y2=0
		arrows(x1, y1, x2, y2, code=3)
		text(x1+(x2-x1)/2, 2, labels="TSP")
	
		segments(x1, 0, x1, stages[nustage==TSP.borders[2]]+1, lwd=1, lty=1)
		segments(x2, 0, x2, stages[nustage==TSP.borders[2]]+1, lwd=1, lty=1)
	}


# Ne sert à rien d'aller là si on n'a aucun se - 17/7/2012
# Je teste individuellement si on a les se de chaque paramètre
if (show.CI & replicate.CI!=1) {

## Nouvelle méthode prenant beaucoup moins de mémoire

# 8/2/2014 dans parssm j'ai les paramètres
Parametre=matrix(rep(NA, length(parssm)*replicate.CI), ncol=length(parssm), dimnames=list(NULL, names(parssm)))
transition.exist <- !(is.na(parssm["transition_P"]) | is.na(parssm["transition_P"]))

for (i in 1:length(parssm)) {
		if (!is.na(res[names(parssm[i])])) {
			Parametre[,i]=rnorm(replicate.CI,parssm[i], res[names(parssm[i])])
		} else {
			Parametre[,i]=rep(parssm[i], replicate.CI)
		}
}



moyenne<-rep(0, dim(nids)[1]-1)
moyenne2<-rep(0, dim(nids)[1]-1)
cpt<-rep(0, dim(nids)[1]-1)

tspl <- NULL
debut <- NULL
fin <- NULL


duree <- 0
duree2 <- 0

cptduree <- 0

taillefin <- NULL

# 21/1/2014
SE.mean.temp.TSP <- NULL
SE.weighted.mean.temp.TSP <- NULL

# 12/12/2014
SE.TSD.mean.temp.TSP <- NULL
SE.WTSD.mean.temp.TSP <- NULL

numerictemplevels <- as.numeric(templevels)

for (j in 1:replicate.CI) {
	if (progress) setTxtProgressBar(pb, (cptseriesfaites-1)*replicate.CI+j)
	
# valeur.r.list <- embryogrowth:::.SSM(as.numeric(templevels), Parametre[j,])
valeur.r.list <- .SSM(numerictemplevels, Parametre[j,])

	valeur.r <- valeur.r.list[[1]]
	valeur.r_L <- valeur.r.list[[2]]

	names(valeur.r)<-templevels
	names(valeur.r_L)<-templevels

  if (transition.exist) {
	  transition_P <- Parametre[j,"transition_P"][[1]]
	  transition_S <- Parametre[j,"transition_S"][[1]]
  } else {
    transition_P <- NA
    transition_S <- NA
  }
	
	
	x1 <- NA
	x2 <- NA

	y <- M0
	
	# 21/1/2014
	serie.taille <- y
	
	k <- dim(nids)[1]-1
	
	if (any(colnames(Parametre)=="rK")) {
		anK <- as.numeric(Parametre[j,"rK"])*mean.ts
	}
	if (any(colnames(Parametre)=="K")) {
		anK <- as.numeric(Parametre[j,"K"])
	}


# Je rajoute la possibilité de finir une série avant la fin - 18/7/2012
# Il faut que j'enregistre la durée d'incubation

	if (stopattest) {

		sizefin <- rnorm(1, mean=mean.ts, sd=sd.ts)
	
		for (i in 1:k) {
			timesunique <- c(nids[i,"Time"], nids[i+1,"Time"])
			if (!transition.exist) {
				transition <- 1
			} else {
				transition <- 1/(1+exp(transition_S*(y-transition_P)))
			}
      # 13/2/2014
			pos <- which.min(abs(numerictemplevels-nids[i,3]))
			a <- as.numeric(valeur.r[pos]*transition+valeur.r_L[pos]*(1-transition))
			
			parms <- c(alpha=a, K=anK)

			out1 <- lsoda(y, timesunique, derivate, parms)			
			y <- as.numeric(out1[2,2])
			# 21/1/2014
			serie.taille <- c(serie.taille, y)
			if (is.na(x1) & y>stages[nustage==TSP.borders[1]]) {
				x1 <- timesunique[2]
				indice.x1 <- i+1
				}
			if (is.na(x2) & y>stages[nustage==TSP.borders[2]]) {
				indice.x2 <- i
				x2 <- timesunique[2]
				tspl <- c(tspl, x2-x1)
				debut <- c(debut, x1)
				fin <- c(fin, x2)
        
				# dans indice.x1 j'ai l'indice du début de la TSP et dans indice.x2 la fin
        Timexxx <- diff(nids[indice.x1:indice.x2, "Time"])
        Massxxx <- diff(serie.taille[indice.x1:indice.x2])
        Tempxxx <- nids[indice.x1:(indice.x2-1), "Temperature"]
        
				SE.mean.temp.TSP <- c(SE.mean.temp.TSP, sum(Timexxx*Tempxxx)/sum(Timexxx))
        SE.weighted.mean.temp.TSP <- c(SE.weighted.mean.temp.TSP, sum(Massxxx*Tempxxx)/
            								sum(Massxxx))
        
        # 12/12/2014
        # J'utilise la nouvelle norme de réaction pour pondérer
        STRN <- .SSM(Tempxxx, SexualisationTRN)[[1]]
        
				SE.TSD.mean.temp.TSP <- c(SE.TSD.mean.temp.TSP, 
				                          sum(Timexxx*Tempxxx*STRN)/sum(Timexxx*STRN))
				
				SE.WTSD.mean.temp.TSP <- c(SE.WTSD.mean.temp.TSP, 
				                           sum(Massxxx*Timexxx*Tempxxx*STRN)/sum(Massxxx*Timexxx*STRN))
			}

			moyenne[i] <- moyenne[i]+y
			moyenne2[i] <- moyenne2[i]+y*y
			cpt[i] <- cpt[i]+1
			
			# je teste si la taille est atteinte
				if (y>=sizefin) {
					duree <- duree+nids[i+1,"Time"]
					duree2 <- duree2+nids[i+1,"Time"]*nids[i+1,"Time"]
					cptduree <- cptduree+1
					break
				}
		}
		
		taillefin <- c(taillefin, y)

		
		if (cptduree==0) {
			cat("\n")
			print(paste("The timeseries", namets, "of temperatures is too short to reach the end of incubation"))
			
			rt <- c(tsp.length.mean=NA, tsp.length.SE=NA)
			rt <- c(rt, tsp.begin.mean=NA, tsp.begin.SE=NA)
			rt <- c(rt, tsp.end.mean=NA, tsp.end.SE=NA)
			rt <- c(rt, incubation.length.mean=NA, incubation.length.SE=NA, incubation.first.third=NA, incubation.second.third=NA )
			rt <- c(rt, SSQ.borders.TSP=NA)
			rt <- c(rt, temperature.TSP.mean=NA)
			rt <- c(rt, temperature.TSP.SE=NA)
			rt <- c(rt, temperature.mean=NA)
			rt <- c(rt, end.size.mean=NA, end.size.SE=NA)
			# 21/1/2014
			rt <- c(rt, temperature.middle.third.mean=NA)
			rt <- c(rt, weighted.TSP.temperature.mean=NA, weighted.TSP.temperature.SE=NA)
			# 25/1/2015
			rt <- c(rt, TSD.temperature.mean=NA, TSD.temperature.SE=NA)
			rt <- c(rt, weighted.TSD.temperature.mean=NA, weighted.TSD.temperature.SE=NA)
			
			rt <- c(rt, LnL=LnL)


			errorcptduree <- TRUE
		
		} else {
			meanduree <- duree/cptduree
			SDduree <- sqrt(duree2/cptduree-(meanduree)^2)
			errorcptduree <- FALSE
		}

	} else {
	# je n'ai pas stopattest donc
	# je vais jusqu'à la fin de la série
	
		for (i in 1:k) {
			timesunique <- c(nids[i,"Time"], nids[i+1,"Time"])
			if (is.na(transition_S) | is.na(transition_P)) {
				transition <- 1
			} else {
				transition <- 1/(1+exp(transition_S*(y-transition_P)))
			}
			pos <- which.min(abs(as.numeric(names(valeur.r))-nids[i,3]))
			a <- as.numeric(valeur.r[pos]*transition+valeur.r_L[pos]*(1-transition))
			
			parms <- c(alpha=a, K=anK)

			out1 <- lsoda(y, timesunique, derivate, parms)			
			y <- as.numeric(out1[2,2])
			# 21/1/2014
			serie.taille <- c(serie.taille, y)

			if (is.na(x1) & y>stages[nustage==TSP.borders[1]]) {
				x1 <- timesunique[2]
				indice.x1 <- i+1
				}
			if (is.na(x2) & y>stages[nustage==TSP.borders[2]]) {
				indice.x2 <- i
				x2 <- timesunique[2]
				tspl <- c(tspl, x2-x1)
				debut <- c(debut, x1)
				fin <- c(fin, x2)
 
				Timexxx <- diff(nids[indice.x1:indice.x2, "Time"])
				Massxxx <- diff(serie.taille[indice.x1:indice.x2])
				Tempxxx <- nids[indice.x1:(indice.x2-1), "Temperature"]
        
				# dans indice.x1 j'ai l'indice du début de la TSP et dans indice.x2 la fin
				SE.mean.temp.TSP <- c(SE.mean.temp.TSP, sum(Timexxx*Tempxxx)/
            								sum(Timexxx))
        SE.weighted.mean.temp.TSP <- c(SE.weighted.mean.temp.TSP, sum(Massxxx*Tempxxx)/
            								sum(Massxxx))
				# 12/12/2014
				# J'utilise la nouvelle norme de reaction pour ponderer
        STRN <- .SSM(Tempxxx, SexualisationTRN)[[1]]

        SE.TSD.mean.temp.TSP <- c(SE.TSD.mean.temp.TSP, 
				                          sum(Timexxx*Tempxxx*STRN)/
				                            sum(Timexxx*STRN))
        
        SE.WTSD.mean.temp.TSP <- c(SE.WTSD.mean.temp.TSP, 
                                  sum(Massxxx*Timexxx*Tempxxx*STRN)/
                                    sum(Massxxx*Timexxx*STRN))
				
			}
			
			moyenne[i] <- moyenne[i]+y
			moyenne2[i] <- moyenne2[i]+y*y
			cpt[i] <- cpt[i]+1
		}
		
		# 27/1/2013 je le sors de la boucle
		errorcptduree <- FALSE

		# dans y j'ai la taille finale
		taillefin <- c(taillefin, y)
	
	}
		
		
}




if (!errorcptduree) {
# si on a une erreur sur la durée, on n'affiche rien
  int.SDmass <- (moyenne2/cpt)-(moyenne/cpt)^2
  
	SDmass <- c(0, sqrt(ifelse(int.SDmass<0,0,int.SDmass)))
  
	rt <- c(tsp.length.mean=length.tsp, tsp.length.SE=sd(tspl)/1440)
	rt <- c(rt, tsp.begin.mean=debut.tsp, tsp.begin.SE=sd(debut)/1440)
	rt <- c(rt, tsp.end.mean=fin.tsp, tsp.end.SE=sd(fin)/1440)
	
	if (stopattest) {
	  # 21/1/2014
	  first.third <- (meanduree/1440)/3
    second.third <- 2*(meanduree/1440)/3
		rt <- c(rt, incubation.length.mean=meanduree/1440, incubation.length.SE=SDduree/1440, 
            incubation.first.third=first.third, incubation.second.third=second.third)
	} else {
	  # 21/1/2014
	  first.third <- (nids[dim(nids)[1],"Time"]/1440)/3
	  second.third <- 2*(nids[dim(nids)[1],"Time"]/1440)/3
		rt <- c(rt, incubation.length.mean=nids[dim(nids)[1],"Time"]/1440, incubation.length.SE=NA, 
            incubation.first.third=first.third, incubation.second.third=second.third)
	}
	# 21/1/2014
	# dans first.third j'ai le temps du début du deuxième tiers en jour
  # dans second.third j'ai le temps de la fin du deuxième tiers en jour
  indice.debut.middle.third <- which.min(abs(nids[,"Time"]-first.third*1440))
	indice.fin.middle.third <- which.min(abs(nids[,"Time"]-second.third*1440))
	
	rt <- c(rt, SSQ.borders.TSP=as.numeric((rt["tsp.begin.mean"]-rt["incubation.first.third"])^2+(rt["tsp.end.mean"]-rt["incubation.second.third"])^2))

  Timexxx <- diff(nids[indice.debut.tsp:indice.fin.tsp, "Time"])
  Massxxx <- diff(nids[indice.debut.tsp:indice.fin.tsp, "Mass"])
  Tempxxx <- nids[indice.debut.tsp:(indice.fin.tsp-1), "Temperature"]
  STRN <- .SSM(Tempxxx, SexualisationTRN)[[1]]
  
	rt <- c(rt, temperature.TSP.mean=sum(Timexxx*Tempxxx)/sum(Timexxx))
	rt <- c(rt, temperature.TSP.SE=sd(SE.mean.temp.TSP))  
	rt <- c(rt, temperature.mean=sum(diff(nids[, 1])*nids[1:(dim(nids)[1]-1), "Temperature"])/nids[dim(nids)[1], "Time"])	
	rt <- c(rt, end.size.mean=mean(taillefin), end.size.SE=sqrt(var(taillefin)/length(taillefin)))
	# 21/1/2014
	rt <- c(rt, temperature.middle.third.mean=sum(diff(nids[indice.debut.middle.third:indice.fin.middle.third, "Time"])*
	                                                                              nids[indice.debut.middle.third:(indice.fin.middle.third-1), "Temperature"])/
	          sum(diff(nids[indice.debut.middle.third:indice.fin.middle.third, "Time"])))
  
    rt <- c(rt, weighted.TSP.temperature.mean=sum(Massxxx*Tempxxx)/sum(Massxxx))
  	rt <- c(rt, weighted.TSP.temperature.SE=sd(SE.weighted.mean.temp.TSP))  
  # 12/12/2014
  # J'utilise la nouvelle norme de réaction pour pondérer
  rt <- c(rt, TSD.temperature.mean=sum(Timexxx*Tempxxx*STRN)/sum(Timexxx*STRN))
  rt <- c(rt, TSD.temperature.SE=sd(SE.TSD.mean.temp.TSP))
  # 25/1/2015
  rt <- c(rt, weighted.TSD.temperature.mean=sum(Massxxx*Timexxx*Tempxxx*STRN)/sum(Massxxx*Timexxx*STRN))
  rt <- c(rt, weighted.TSD.temperature.SE=sd(SE.WTSD.mean.temp.TSP))
  
	rt <- c(rt, LnL=LnL)
	
} else {
# 27/1/2013
# j'ai une erreur sur la durée
# car errorcptduree est TRUE
# SDmass n'a aucun sens alors ?

# print("Pourquoi je viens là ???")
# print(paste("errorcptduree=", errorcptduree))

}

# pas de CI
} else {

	rt <- c(tsp.length.mean=length.tsp, tsp.length.SE=NA)
	rt <- c(rt, tsp.begin.mean=debut.tsp, tsp.begin.SE=NA)
	rt <- c(rt, tsp.end.mean=fin.tsp, tsp.end.SE=NA)
	
	if (stopattest) {
	  # 21/1/2014
	  first.third <- (meanduree/1440)/3
	  second.third <- 2*(meanduree/1440)/3
		rt <- c(rt, incubation.length.mean=meanduree/1440, incubation.length.SE=NA, 
            incubation.first.third=first.third, incubation.second.third=second.third)
	} else {
	  # 21/1/2014
	  first.third <- (nids[dim(nids)[1],"Time"]/1440)/3
	  second.third <- 2*(nids[dim(nids)[1],"Time"]/1440)/3
		rt <- c(rt, incubation.length.mean=nids[dim(nids)[1],"Time"]/1440, incubation.length.SE=NA, 
            incubation.first.third=first.third, incubation.second.third=second.third)
	}
	# 21/1/2014
	# dans first.third j'ai le temps du début du deuxième tiers en jour
	# dans second.third j'ai le temps de la fin du deuxième tiers en jour
	indice.debut.middle.third <- which.min(abs(nids[,"Time"]-first.third*1440))
	indice.fin.middle.third <- which.min(abs(nids[,"Time"]-second.third*1440))
	
	Timexxx <- diff(nids[indice.debut.tsp:indice.fin.tsp, "Time"])
	Massxxx <- diff(nids[indice.debut.tsp:indice.fin.tsp, "Mass"])
	Tempxxx <- nids[indice.debut.tsp:(indice.fin.tsp-1), "Temperature"]
	STRN <- .SSM(Tempxxx, SexualisationTRN)[[1]]
	
	
	rt <- c(rt, SSQ.borders.TSP=as.numeric((rt["tsp.begin.mean"]-rt["incubation.first.third"])^2+(rt["tsp.end.mean"]-rt["incubation.second.third"])^2))

	rt <- c(rt, temperature.TSP.mean=sum(Timexxx*Tempxxx)/sum(Timexxx))
    rt <- c(rt, temperature.TSP.SE=NA)
	rt <- c(rt, temperature.mean=sum(diff(nids[, "Time"])*nids[1:(dim(nids)[1]-1), "Temperature"])/nids[dim(nids)[1], "Time"])
	rt <- c(rt, end.size.mean=taillefin, end.size.SE=NA)
  
  
  
	# 21/1/2014
	rt <- c(rt, temperature.middle.third.mean=sum(diff(nids[indice.debut.middle.third:indice.fin.middle.third, "Time"])*
	                                                nids[indice.debut.middle.third:(indice.fin.middle.third-1), "Temperature"])/
	          sum(diff(nids[indice.debut.middle.third:indice.fin.middle.third, "Time"])))
	
	# 25/1/2015
	rt <- c(rt, weighted.TSP.temperature.mean=sum(Massxxx*Tempxxx)/sum(Massxxx) 
            )
	rt <- c(rt, weighted.TSP.temperature.SE=NA)
	# 12/12/2014
	# J'utilise la nouvelle norme de réaction pour pondérer
	rt <- c(rt, TSD.temperature.mean=sum(Timexxx*Tempxxx*STRN)/sum(Timexxx*STRN))
	rt <- c(rt, TSD.temperature.SE=NA)    
	
	# 26/1/2015
	rt <- c(rt, weighted.TSD.temperature.mean=sum(Massxxx*Timexxx*Tempxxx*STRN)/sum(Massxxx*Timexxx*STRN))
	rt <- c(rt, weighted.TSD.temperature.SE=NA)    
	
	
	rt <- c(rt, LnL=LnL)

}



if (show.stages & show.plot) {
	## on affiche les stades
	for(i in 1:length(stages)) {
		y1=nids[subset(1:dim(nids)[1], nids[,5]>stages[i])[1], 5]
		segments(0, y1, xlim[2]-4, y1, lwd=1, lty=2, col=col.stages)
		text(xlim[2]-3*(nustage[i]%%2), y1, labels=nustage[i], col=col.stages, cex=0.7)
	}

	text(xlim[2]-2, ylimS[2], labels=lab.stages, col=col.stages, cex=0.7)
}




if (show.third & show.plot) {
	
	if (stopattest) {
		## encadre le deuxième tiers de l'incubation
		x1<-(meanduree/3)/(60*24)
		segments(x1, 0, x1, ylimS[2]+1, lwd=1, lty=3, xpd=TRUE)
		x2<-(2*meanduree/3)/(60*24)
		segments(x2, 0, x2, ylimS[2]+1, lwd=1, lty=3, xpd=TRUE)

		arrows(x1, ylimS[2]+1, x2, ylimS[2]+1, code=3, xpd=TRUE)
		text(x1+(x2-x1)/2, ylimS[2]+at.lab.third, labels=lab.third, xpd=TRUE)

	} else {
	# encadre le deuxième tiers de l'incubation - 20/7/2012
	# corrigé car travaillait en indice et pas en temps

		x1<-(nids[dim(nids)[1], 1]/3)/1440
		segments(x1, 0, x1, ylimS[2]+1, lwd=1, lty=3, xpd=TRUE)
		x2<-(2*nids[dim(nids)[1], 1]/3)/1440
		segments(x2, 0, x2, ylimS[2]+1, lwd=1, lty=3, xpd=TRUE)

		arrows(x1, ylimS[2]+1, x2, ylimS[2]+1, code=3, xpd=TRUE)
		text(x1+(x2-x1)/2, ylimS[2]+at.lab.third, labels=lab.third, xpd=TRUE)
	}
	

}

#GOTO 200
if (show.CI & show.plot & !errorcptduree) {

	par(new = TRUE)

	x <- nids[,1]/1440
	y <- nids[,5]-2*SDmass

	Lx <- modifyList(list(...), list(x=x, y=y, axes = FALSE, xlab = "", ylab = "", main="")) 
	L <- modifyList(list(type = "l", bty = "n", xlim=xlim, lwd=2, lty=2, ylim=ylimS, col=col.S[cptcol.S]), Lx)

	do.call(plot, L)

	par(new = TRUE)
	
	y <- nids[,5]+2*SDmass
	Lx <- modifyList(list(...), list(x=x, y=y, axes = FALSE, xlab = "", ylab = "", main=""))
	L <- modifyList(list(type = "l", bty = "n", xlim=xlim, lwd=2, lty=2, ylim=ylimS, col=col.S[cptcol.S]), Lx)

	do.call(plot, L)
#END 200
}

# Je stocke les informations sur le nids

df <- list(data.frame(Time=nids[,1], Mean=nids[,5], CIminus=nids[,5]-2*SDmass, CIplus=nids[,5]+2*SDmass))
names(df) <- namets
listdf <- c(listdf, df)

#END 100 fin du test si on arrive bien au bout avec nouvelle série
}

# fin du test si on fait la série

rt <- list(rt)
names(rt) <- namets
rtT <- c(rtT, rt)

}

# fin de la boucle de toutes les séries
}


# je rétablis les valeurs de par(mar)
# par(mar=parpre)

if (progress) setTxtProgressBar(pb, nbpb)

# cat("\n")

return(invisible(list(summary=rtT, traces=listdf)))

}
}
