% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pln_probit.R
\name{pln_probit}
\alias{pln_probit}
\title{Recursive PLN-Probit Model}
\usage{
pln_probit(
  form_pln,
  form_probit,
  data = NULL,
  par = NULL,
  method = "BFGS",
  init = c("zero", "unif", "norm", "default")[4],
  H = 20,
  verbose = 0
)
}
\arguments{
\item{form_pln}{Formula for the first-stage Poisson lognormal model}

\item{form_probit}{Formula for the second-stage probit model}

\item{data}{Input data, a data frame}

\item{par}{Starting values for estimates}

\item{method}{Optimization algorithm. Without gradient, NM is much faster than BFGS}

\item{init}{Initialization method}

\item{H}{Number of quadrature points}

\item{verbose}{A integer indicating how much output to display during the estimation process.
\itemize{
\item <0 - No ouput
\item 0 - Basic output (model estimates)
\item 1 - Moderate output, basic ouput + parameter and likelihood in each iteration
\item 2 - Extensive output, moderate output + gradient values on each call
}}
}
\value{
A list containing the results of the estimated model, some of which are inherited from the return of maxLik
\itemize{
\item estimates: Model estimates with 95\% confidence intervals. Prefix "pln" means first stage variables.
\item estimate or par: Point estimates
\item variance_type: covariance matrix used to calculate standard errors. Either BHHH or Hessian.
\item var: covariance matrix
\item se: standard errors
\item gradient: Gradient function at maximum
\item hessian: Hessian matrix at maximum
\item gtHg: \eqn{g'H^-1g}, where H^-1 is simply the covariance matrix. A value close to zero (e.g., <1e-3 or 1e-6) indicates good convergence.
\item LL or maximum: Likelihood
\item AIC: AIC
\item BIC: BIC
\item n_obs: Number of observations
\item n_par: Number of parameters
\item LR_stat: Likelihood ratio test statistic for \eqn{\rho=0}
\item LR_p: p-value of likelihood ratio test
\item iterations: number of iterations taken to converge
\item message: Message regarding convergence status.
}

Note that the list inherits all the components in the output of maxLik. See the documentation of maxLik for more details.
}
\description{
Estimate a Poisson Lognormal model and a Probit model with bivariate normally distributed error/heterogeneity terms.\cr\cr
First stage (Poisson Lognormal):
\deqn{E[m_i|w_i,u_i]=exp(\boldsymbol{\alpha}'\mathbf{w_i}+\lambda u_i)}{E[m_i | w_i, u_i] = exp(\alpha' * w_i + \lambda * u_i)}
Second stage (Probit):
\deqn{y_i = 1(\boldsymbol{\beta}'\mathbf{x_i} + {\gamma}m_i + \sigma v_i > 0)}{y_i = 1(\beta' * x_i + \gamma * m_i + \sigma * v_i > 0)}
Endogeneity structure:
\eqn{u_i} and \eqn{v_i} are bivariate normally distributed with a correlation of \eqn{\rho}. \cr\cr
This model is typically well-identified even if w and x are the same set of variables. This model still works if the first-stage dependent variable is not a regressor in the second stage.
}
\examples{
library(MASS)
N = 2000
rho = -0.5
set.seed(1)

x = rbinom(N, 1, 0.5)
z = rnorm(N)

e = mvrnorm(N, mu=c(0,0), Sigma=matrix(c(1,rho,rho,1), nrow=2))
e1 = e[,1]
e2 = e[,2]

m = rpois(N, exp(-1 + x + z + e1))
y = as.numeric(1 + x + z + log(1+m) + e2 > 0)

est = pln_probit(m~x+z, y~x+z+log(1+m))
print(est$estimates, digits=3)
}
\references{
Peng, Jing. (2022) Identification of Causal Mechanisms from Randomized Experiments: A Framework for Endogenous Mediation Analysis. Information Systems Research (Forthcoming), Available at https://doi.org/10.1287/isre.2022.1113
}
\seealso{
Other endogeneity: 
\code{\link{bilinear}()},
\code{\link{biprobit_latent}()},
\code{\link{biprobit_partial}()},
\code{\link{biprobit}()},
\code{\link{linear_probit}()},
\code{\link{pln_linear}()},
\code{\link{probit_linearRE}()},
\code{\link{probit_linear_latent}()},
\code{\link{probit_linear_partial}()},
\code{\link{probit_linear}()}
}
\concept{endogeneity}
