% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/probit_linearRE.R
\name{probit_linearRE}
\alias{probit_linearRE}
\title{Recursive Probit-LinearRE Model}
\usage{
probit_linearRE(
  form_probit,
  form_linear,
  id,
  data = NULL,
  par = NULL,
  method = "BFGS",
  H = 20,
  stopUpdate = F,
  init = c("zero", "unif", "norm", "default")[4],
  verbose = 0
)
}
\arguments{
\item{form_probit}{Formula for the probit model at the individual level}

\item{form_linear}{Formula for the linear model at the individual-time level}

\item{id}{group id, character if data  supplied or numerical vector if data not supplied}

\item{data}{Input data, must be a data.table object}

\item{par}{Starting values for estimates}

\item{method}{Optimization algorithm. Default is BFGS}

\item{H}{Number of quadrature points}

\item{stopUpdate}{Adaptive Gaussian Quadrature disabled if TRUE}

\item{init}{Initialization method}

\item{verbose}{A integer indicating how much output to display during the estimation process.
\itemize{
\item <0 - No ouput
\item 0 - Basic output (model estimates)
\item 1 - Moderate output, basic ouput + parameter and likelihood in each iteration
\item 2 - Extensive output, moderate output + gradient values on each call
}}
}
\value{
A list containing the results of the estimated model, some of which are inherited from the return of maxLik
\itemize{
\item estimates: Model estimates with 95\% confidence intervals
\item estimate or par: Point estimates
\item variance_type: covariance matrix used to calculate standard errors. Either BHHH or Hessian.
\item var: covariance matrix
\item se: standard errors
\item var_bhhh: BHHH covariance matrix, inverse of the outer product of gradient at the maximum
\item se_bhhh: BHHH standard errors
\item gradient: Gradient function at maximum
\item hessian: Hessian matrix at maximum
\item gtHg: \eqn{g'H^-1g}, where H^-1 is simply the covariance matrix. A value close to zero (e.g., <1e-3 or 1e-6) indicates good convergence.
\item LL or maximum: Likelihood
\item AIC: AIC
\item BIC: BIC
\item n_obs: Number of observations
\item n_par: Number of parameters
\item time: Time takes to estimate the model
\item LR_stat: Likelihood ratio test statistic for \eqn{\rho=0}
\item LR_p: p-value of likelihood ratio test
\item iterations: number of iterations taken to converge
\item message: Message regarding convergence status.
}

Note that the list inherits all the components in the output of maxLik. See the documentation of maxLik for more details.
}
\description{
A panel extension of the probit_linear model. The first stage is a probit model at the individual level. The second stage is a panel linear model at the individual-time level with individual-level random effects. The random effect is correlated with the error term in the first stage.\cr\cr
First stage (Probit):
\deqn{m_i=1(\boldsymbol{\alpha}'\mathbf{w_i}+u_i>0)}{m_i = 1(\alpha' * w_i + u_i > 0)}
Second stage (Panel linear model with individual-level random effects):
\deqn{y_{it} = \boldsymbol{\beta}'\mathbf{x_{it}} + {\gamma}m_i + \lambda v_i +\sigma \epsilon_{it}}{y_it = \beta' * x_it + \gamma * m_i + \lambda * v_i + \sigma * \epsilon_it}
Endogeneity structure:
\eqn{u_i} and \eqn{v_i} are bivariate normally distributed with a correlation of \eqn{\rho}. \cr\cr
This model uses Adaptive Gaussian Quadrature to overcome numerical challenges with long panels. w and x can be the same set of variables. Identification can be weak if w are not good predictors of m. This model still works if the first-stage dependent variable is not a regressor in the second stage.
}
\examples{
library(MASS)
library(data.table)
N = 500
period = 5
obs = N*period
rho = -0.5
set.seed(100)

e = mvrnorm(N, mu=c(0,0), Sigma=matrix(c(1,rho,rho,1), nrow=2))
e1 = e[,1]
e2 = e[,2]

t = rep(1:period, N)
id = rep(1:N, each=period)
w = rnorm(N)
m = as.numeric(1+w+e1>0)
m_long = rep(m, each=period)

x = rnorm(obs)
y = 1 + x + m_long + rep(e2, each=period) + rnorm(obs)

dt = data.table(y, x, id, t, m=rep(m, each=period), w=rep(w, each=period))

est = probit_linearRE(m~w, y~x+m, 'id', dt)
print(est$estimates, digits=3)
}
\references{
Chen, H., Peng, J., Li, H., & Shankar, R. (2022). Impact of Refund Policy on Sales of Paid Information Services: The Moderating Role of Product Characteristics. Available at SSRN: https://ssrn.com/abstract=4114972.
}
\seealso{
Other endogeneity: 
\code{\link{bilinear}()},
\code{\link{biprobit_latent}()},
\code{\link{biprobit_partial}()},
\code{\link{biprobit}()},
\code{\link{linear_probit}()},
\code{\link{pln_linear}()},
\code{\link{pln_probit}()},
\code{\link{probit_linear_latent}()},
\code{\link{probit_linear_partial}()},
\code{\link{probit_linear}()}
}
\concept{endogeneity}
