% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_obj_name.r
\name{get_obj_name}
\alias{get_obj_name}
\title{Return the name of an object referenced by a parameter at any parent generation}
\usage{
get_obj_name(obj, n = 0, eval = FALSE, silent = TRUE)
}
\arguments{
\item{obj}{object whose name is of interest, or whose evaluated name is of interest.}

\item{n}{number of parent generations to go back to retrieve the name of the object that leads to \code{obj}
in the function calling chain. See details for more information.}

\item{eval}{whether to evaluate \code{obj} in the \code{n}-th parent generation before
getting the object's name in that environment. See details for more information.}

\item{silent}{when \code{FALSE}, the names of the environments and objects in those environments are printed
as those environments are traversed by this function.}
}
\value{
The name of the object in the \code{n}-th parent generation environment.
If the object in that parent generation is an unnamed environment, \code{"<environment>"} is returned.
}
\description{
Return the name of the object at the specified parent generation leading to a function's parameter.
This is done by iteratively retrieving the name of the object that leads to the function's parameter
at each parent generation.
The object is optionally evaluated in the environment of the specified parent generation
before retrieving its name. In this case, the result is very similar to the deparse() function.
The differences are explained in the Details section.
}
\details{
To better understand the meaning of this function when \code{n} > 0, consider the first example below:  
Function \code{f(x)} is called with parameter \code{z}, which calls \code{g(y)} with parameter \code{x}.
The result of the call to \code{get_obj_name()} from function \code{g()} is \code{"z"}
because that call is telling "give me the name of object \code{y} two levels up from
the current environment, i.e. from the environment of \code{g()}. The name of \code{y} in the environment
of \code{f(x)} is \code{"x"}, and the name of \code{y} in the global environment is \code{"z"}.  

Note that one may think that the result of this function is the same as using deparse(substitute()) where the
object being \code{substitute}d is evaluated at the \code{n}-th parent generation. However, this is not quite so
because \code{substitute(obj, parent.frame(n))} retrieves the object assigned to \code{obj} at the \code{n}-th
parent generation, where \code{obj} \emph{is the name of the variable substituted at that \code{n}-th parent generation}.
On the contrary, \code{get_obj_name(obj, n=2)} \emph{first} looks for the name leading to \code{obj} and then
retrieves its name.

When eval=TRUE, the result of the function is the same as the output of \code{deparse()} except for the following two cases:
\itemize{
\item the result of \code{NULL} is \code{NULL} instead of \code{"NULL"} which is the case with \code{deparse()}
\item when \code{eval=TRUE}, if the object passed to \code{get_obj_name()} evaluates to a name, it returns that name,
without added quotes. For example, if \code{v = "x"} then \code{get_obj_name(v, eval=TRUE)} returns \code{"x"} while
\code{deparse(v)} returns \code{"\"x\""}.
}
}
\examples{
# In its default behaviour (eval=FALSE), get_obj_name() returns
# the name of an object in the n-th parent generation.
# This example shows the difference between get_obj_name() and deparse(substitute())
g <- function(y) { return(list(obj_name=get_obj_name(y, n=2), 
                               substitute=deparse(substitute(y, parent.frame(n=2))) )) }
f <- function(x) { g(x) }
z = 3; 
f(z)           # Returns a list where the first element is "z" and the second element is "y"
               # Note that 'z' is the object leading to object 'y' inside function g()
               # if we follow the function calling chain.

# When eval=TRUE, get_obj_name() behaves the same way as deparse(),
# except for the cases noted above.
g <- function(y) {
  print(get_obj_name(y, n=2, eval=TRUE));
  print(deparse(y))
}
f <- function(x) { g(x) }
z = 3; 
f(z)           # Prints "3" twice, once as the output of get_obj_name(),
               # once as the output of deparse()
}
\seealso{
\link{get_obj_value}
}

