#' Explore and Render Exam Exercises
#'
#' This function allows the user to explore exercises from the `exams.forge.data` package
#' and optionally render them to PDF or HTML. Users can filter exercises by directory (topic)
#' or by a pattern in filenames or file contents.
#'
#' @param action Character. One of `"topic"`, `"file"`, `"pdf"`, or `"html"`.
#'   - `"topic"`: return the list of available exercise topics (subdirectories).
#'   - `"file"`: return the list of exercise filenames relative to `aufgaben/`.
#'   - `"pdf"`: render selected exercises to PDF.
#'   - `"html"`: render selected exercises to HTML.
#' @param pattern Character or `NULL`. A regular expression to filter exercises.
#'   If `topic = TRUE`, the pattern is matched against the directory names (topics).
#'   If `topic = FALSE`, the pattern is searched inside the exercise files.
#' @param topic Logical. If `TRUE` (default), `pattern` is applied to directory names.
#'   If `FALSE`, `pattern` is applied to the content of the exercise files.
#' @param ... Additional arguments passed to `grepl()` when filtering files or contents.
#'
#' @return A character vector of topics or filenames, or `NULL` if no exercises are found.
#'   When rendering (`pdf`/`html`), the files are rendered silently with messages and warnings suppressed.
#'
#' @examples
#' \donttest{
#' view("topic")
#'
#' # List all files in a specific topic
#' view("file", pattern = "ttest", topic = TRUE)
#'
#' if (interactive()) {
#' # Render the files matching a pattern to HTML
#'   view("file", pattern = "hyper")
#'   view("html", pattern = "hyper")
#' }
#' }
#'
#' @importFrom exams exams2html exams2pdf
#' @importFrom exams.forge catif
#' @export
view <- function (action=c("topic", "file", "pdf", "html"), pattern=NULL, topic=TRUE, ...) {
  action <- match.arg(action)
  files <- list.files(
    system.file("aufgaben", package = "exams.forge.data"),
    pattern = "\\.Rmd$", recursive = TRUE, full.names = TRUE
  )
  index <- seq_along(files)
  if (!is.null(pattern)) {
    index <- if (topic) grepl(pattern, dirname(files), ...) else sapply(files, function(f) { any(grepl(pattern, readLines(f, warn = FALSE), ...))})
    index <- which(index)
  }
  if (length(index)==0) {
    warning("No exercises found.")
    return(invisible(NULL))
  }
  findex <- files[index]
  if (action=="topic") {
    topics <- unique(dirname(sub("^.*?/aufgaben/", "", findex)))
    return(topics)
  }
  if (action=="file") {
    return(sub("^.*?/aufgaben/", "", findex))
  }
  if (!interactive()) {
    if (action %in% c("html", "pdf")) {
      message("Skipping HTML/PDF generation in non-interactive session")
      return(invisible(NULL))
    }
  }
  if (action=="html") {
    suppressMessages({
      invisible(suppressMessages(suppressWarnings({
        exams2html(findex)
      })))
    })
  }
  if (action=="pdf") {
    suppressMessages({
      invisible(suppressMessages(suppressWarnings({
        exams2pdf(findex, template=system.file("aufgaben", "plain.tex", package = "exams.forge.data"))
      })))
    })
  }
  return(invisible(files[index]))
  catif(FALSE, "All exercises require `exams.forge`, but CRAN complains about Import: without an @import...")
}



