% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pipeline-epoch.R
\name{epoch}
\alias{epoch}
\title{Epoch (and baseline) pupil data based on custom event message structure}
\usage{
epoch(
  eyeris,
  events,
  limits = NULL,
  label = NULL,
  baseline = FALSE,
  baseline_type = c("sub", "div"),
  baseline_events = NULL,
  baseline_period = NULL,
  hz = NULL,
  verbose = TRUE,
  call_info = NULL,
  calc_baseline = deprecated(),
  apply_baseline = deprecated()
)
}
\arguments{
\item{eyeris}{An object of class \code{eyeris} derived from \code{\link[=load_asc]{load_asc()}}}

\item{events}{Either (1) a single string representing the event message to
perform trial extraction around, using specified \code{limits} to center the epoch
around or no \code{limits} (which then just grabs the data epochs between each
subsequent event string of the same type); (2) a vector containing both
\code{start} and \code{end} event message strings -- here, \code{limits} will be ignored and
the duration of each trial epoch will be the number of samples between each
matched \code{start} and \code{end} event message pair; or (3) a list of 2 dataframes
that manually specify start/end event timestamp-message pairs to pull out of
the raw timeseries data -- here, it is required that each raw timestamp and
event message be provided in the following format:

list(
data.frame(time = c(...), msg = c(...)), # start events
data.frame(time = c(...), msg = c(...)), # end events
1                                        # block number
)

where the first data.frame indicates the \code{start} event timestamp and message
string pairs, and the second data.frame indicates the \code{end} event timestamp
and message string pairs. Additionally, manual epoching only words with
1 block at a time for event-modes \code{2} and \code{3}; thus, please be sure to
explicitly indicate the block number in your input list (for examples,
see above as well as example #9 below for more details)

For event-modes \code{1} and \code{2}, the way in which you pass in the event message
string must conform to a standardized protocol so that \code{eyeris} knows how to
find your events and (optionally) parse any included metadata into the tidy
epoch data outputs. You have two primary choices: either (a) specify a string
followed by a \code{*} wildcard expression (e.g., \verb{"PROBE_START*}), which will
match any messages that have "PROBE_START ..." (... referring to potential
metadata, such as trial number, stim file, etc.); or (b) specify a string
using the \code{eyeris} syntax: (e.g., \code{"PROBE_{type}_{trial}"}), which will match
the messages that follow a structure like this "PROBE_START_1" and
"PROBE_STOP_1", and generate two additional metadata columns: \code{type} and
\code{trial}, which would contain the following values based on these two example
strings: \code{type}: \verb{('START', 'STOP')}, and \code{trial}: \verb{(1, 1)}}

\item{limits}{A vector of 2 values (start, end) in seconds, indicating where
trial extraction should occur centered around any given \code{start} message
string in the \code{events} parameter}

\item{label}{An (optional) string you can provide to customize the name of
the resulting \code{eyeris} class object containing the epoched data frame. If
left as \code{NULL} (default), then list item will be called \code{epoch_xyz}, where
\code{xyz} will be a sanitized version of the original \code{start} event string you
provided for matching. If you choose to specify a \code{label} here, then the
resulting list object name will take the form: \code{epoch_label}. \strong{Warning:
if no \code{label} is specified and there are no event message strings to
sanitize, then you may obtain a strange-looking epoch list element in
your output object (e.g., \verb{$epoch_}, or \verb{$epoch_nana}, etc.). The data
should still be accessible within this nested lists, however, to avoid
ambiguous list objects, we recommend you provide an \code{epoch} label here
to be safe}}

\item{baseline}{\strong{(New)} A single parameter that controls baseline
correction. Set to \code{TRUE} to both calculate and apply baseline correction, or
\code{FALSE} to skip it. This replaces the deprecated \code{calc_baseline} and
\code{apply_baseline} parameters}

\item{baseline_type}{Whether to perform \emph{subtractive} (\code{sub}) or \emph{divisive}
(\code{div}) baseline correction. Defaults to \code{sub}}

\item{baseline_events}{Similar to \code{events}, \code{baseline_events}, you can supply
either (1) a single string representing the event message to center the
baseline calculation around, as indicated by \code{baseline_period}; or (2) a
single vector containing both a \code{start} and an \code{end} event message string --
here, \code{baseline_period} will be ignored and the duration of each baseline
period that the mean will be calculated on will be the number of samples
between each matched \code{start} and \code{end} event message pair, as opposed to a
specified fixed duration (as described in 1). Please note, providing a list
of trial-level start/end message pairs (like in the \code{events} parameter) to
manually indicate unique start/end chunks for baselining is currently
unsupported. Though, we intend to add this feature in a later version of
\code{eyeris}, given it likely won't be a heavily utilized / in demand feature.}

\item{baseline_period}{A vector of 2 values (start, end) in seconds,
indicating the window of data that will be used to perform the baseline
correction, which will be centered around the single string "start" message
string provided in \code{baseline_events}. Again, \code{baseline_period} will be
ignored if both a "start" \strong{and} "end" message string are provided to the
\code{baseline_events} argument}

\item{hz}{Data sampling rate. If not specified, will use the value contained
within the tracker's metadata}

\item{verbose}{A flag to indicate whether to print detailed logging messages
Defaults to \code{TRUE}. Set to \code{False} to suppress messages about the current
processing step and run silently}

\item{call_info}{A list of call information and parameters. If not provided,
it will be generated from the function call}

\item{calc_baseline}{\strong{(Deprecated)} Use \code{baseline} instead}

\item{apply_baseline}{\strong{(Deprecated)} Use \code{baseline} instead}
}
\value{
An \code{eyeris} object with a new nested list of data frames: \verb{$epoch_*}.
The epochs are organized hierarchically by block and preprocessing step.
Each epoch contains the pupil timeseries data for the specified time window
around each event message, along with metadata about the event.

When using \code{bidsify()} to export the data, filenames will include both
epoch and baseline event information for clarity.
}
\description{
Intended to be used as the final preprocessing step. This function creates
data epochs of either fixed or dynamic durations with respect to provided
\code{events} and time \code{limits}, and also includes an intuitive metadata parsing
feature where additional trial data embedded within event messages can easily
be identified and joined into the resulting epoched data frames.
}
\examples{
demo_data <- eyelink_asc_demo_dataset()
eye_preproc <- eyeris::glassbox(demo_data)

# example 1: select 1 second before/after matched event message "PROBE*"
eye_preproc |>
  eyeris::epoch(events = "PROBE*", limits = c(-1, 1))

# example 2: select all samples between each trial
eye_preproc |>
  eyeris::epoch(events = "TRIALID {trial}")

# example 3: grab the 1 second following probe onset
eye_preproc |>
  eyeris::epoch(
    events = "PROBE_START_{trial}",
    limits = c(0, 1)
  )

# example 4: 1 second prior to and 1 second after probe onset
eye_preproc |>
  eyeris::epoch(
    events = "PROBE_START_{trial}",
    limits = c(-1, 1),
    label = "prePostProbe" # custom epoch label name
  )

# example 5: manual start/end event pairs
# note: here, the `msg` column of each data frame is optional
eye_preproc |>
  eyeris::epoch(
    events = list(
      data.frame(time = c(11334491), msg = c("TRIALID 22")), # start events
      data.frame(time = c(11337158), msg = c("RESPONSE_22")), # end events
      1 # block number
    ),
    label = "example5"
  )

# example 6: manual start/end event pairs
# note: set `msg` to NA if you only want to pass in start/end timestamps
eye_preproc |>
  eyeris::epoch(
    events = list(
      data.frame(time = c(11334491), msg = NA), # start events
      data.frame(time = c(11337158), msg = NA), # end events
      1 # block number
    ),
    label = "example6"
  )

## examples with baseline arguments enabled

# example 7: use mean of 1-s preceding "PROBE_START" (i.e. "DELAY_STOP")
# to perform subtractive baselining of the 1-s PROBE epochs.
eye_preproc |>
  eyeris::epoch(
    events = "PROBE_START_{trial}",
    limits = c(0, 1), # grab 0 seconds prior to and 1 second post PROBE event
    label = "prePostProbe", # custom epoch label name
    baseline = TRUE, # Calculate and apply baseline correction
    baseline_type = "sub", # "sub"tractive baseline calculation is default
    baseline_events = "DELAY_STOP_*",
    baseline_period = c(-1, 0)
  )

# example 8: use mean of time period between set start/end event messages
# (i.e. between "DELAY_START" and "DELAY_STOP"). In this case, the
# `baseline_period` argument will be ignored since both a "start" and "end"
# message string are provided to the `baseline_events` argument.
eye_preproc |>
  eyeris::epoch(
    events = "PROBE_START_{trial}",
    limits = c(0, 1), # grab 0 seconds prior to and 1 second post PROBE event
    label = "prePostProbe", # custom epoch label name
    baseline = TRUE, # Calculate and apply baseline correction
    baseline_type = "sub", # "sub"tractive baseline calculation is default
    baseline_events = c(
      "DELAY_START_*",
      "DELAY_STOP_*"
    )
  )

# example 9: additional (potentially helpful) example
start_events <- data.frame(
  time = c(11334491, 11338691),
  msg = c("TRIALID 22", "TRIALID 23")
)
end_events <- data.frame(
  time = c(11337158, 11341292),
  msg = c("RESPONSE_22", "RESPONSE_23")
)
block_number <- 1

eye_preproc |>
  eyeris::epoch(
    events = list(start_events, end_events, block_number),
    label  = "example9"
  )

}
\seealso{
\code{\link[lifecycle:deprecate_soft]{lifecycle::deprecate_warn()}}
}
