% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/directed_erdos_renyi.R, R/sample_edgelist.R,
%   R/undirected_erdos_renyi.R, R/undirected_sbm.R
\name{sample_edgelist.directed_erdos_renyi}
\alias{sample_edgelist.directed_erdos_renyi}
\alias{sample_edgelist}
\alias{sample_edgelist.undirected_factor_model}
\alias{sample_edgelist.directed_factor_model}
\alias{sample_edgelist.undirected_erdos_renyi}
\alias{sample_edgelist.undirected_sbm}
\title{Sample a random edgelist from a random dot product graph}
\usage{
\method{sample_edgelist}{directed_erdos_renyi}(factor_model, ..., allow_self_loops = TRUE)

sample_edgelist(
  factor_model,
  ...,
  poisson_edges = TRUE,
  allow_self_loops = TRUE
)

\method{sample_edgelist}{undirected_factor_model}(
  factor_model,
  ...,
  poisson_edges = TRUE,
  allow_self_loops = TRUE
)

\method{sample_edgelist}{directed_factor_model}(
  factor_model,
  ...,
  poisson_edges = TRUE,
  allow_self_loops = TRUE
)

\method{sample_edgelist}{undirected_erdos_renyi}(
  factor_model,
  ...,
  poisson_edges = FALSE,
  allow_self_loops = TRUE
)

\method{sample_edgelist}{undirected_sbm}(factor_model, ..., allow_self_loops = TRUE)

\method{sample_edgelist}{undirected_sbm}(factor_model, ..., allow_self_loops = TRUE)
}
\arguments{
\item{factor_model}{A \code{\link[=directed_factor_model]{directed_factor_model()}} or
\code{\link[=undirected_factor_model]{undirected_factor_model()}}.}

\item{...}{Ignored. Do not use.}

\item{allow_self_loops}{Logical indicating whether or not
nodes should be allowed to form edges with themselves.
Defaults to \code{TRUE}. When \code{FALSE}, sampling proceeds allowing
self-loops, and these are then removed after the fact.}

\item{poisson_edges}{Logical indicating whether or not
multiple edges are allowed to form between a pair of
nodes. Defaults to \code{TRUE}. When \code{FALSE}, sampling proceeds
as usual, and duplicate edges are removed afterwards.
See Section 2.3 of Rohe et al (2017) for additional
details.}
}
\value{
A single realization of a random Poisson (or Bernoulli)
Dot Product Graph, represented as a \code{\link[tibble:tibble]{tibble::tibble()}} with two
integer columns, \code{from} and \code{to}.

In the undirected case, \code{from} and \code{to} do not encode
information about edge direction, but we will always have
\code{from <= to} for convenience of edge identification.
To avoid handling such considerations yourself, we recommend using
\code{\link[=sample_sparse]{sample_sparse()}}, \code{\link[=sample_igraph]{sample_igraph()}}, and \code{\link[=sample_tidygraph]{sample_tidygraph()}}
over \code{\link[=sample_edgelist]{sample_edgelist()}}.
}
\description{
There are two steps to using the \code{fastRG} package. First,
you must parameterize a random dot product graph by
sampling the latent factors. Use functions such as
\code{\link[=dcsbm]{dcsbm()}}, \code{\link[=sbm]{sbm()}}, etc, to perform this specification.
Then, use \verb{sample_*()} functions to generate a random graph
in your preferred format.
}
\details{
This function implements the \code{fastRG} algorithm as
described in Rohe et al (2017). Please see the paper
(which is short and open access!!) for details.
}
\examples{

library(igraph)
library(tidygraph)

set.seed(27)

##### undirected examples ----------------------------

n <- 100
k <- 5

X <- matrix(rpois(n = n * k, 1), nrow = n)
S <- matrix(runif(n = k * k, 0, .1), nrow = k)

# S will be symmetrized internal here, or left unchanged if
# it is already symmetric

ufm <- undirected_factor_model(
  X, S,
  expected_density = 0.1
)

ufm

### sampling graphs as edgelists ----------------------

edgelist <- sample_edgelist(ufm)
edgelist

### sampling graphs as sparse matrices ----------------

A <- sample_sparse(ufm)

inherits(A, "dsCMatrix")
isSymmetric(A)
dim(A)

B <- sample_sparse(ufm, poisson_edges = FALSE)

inherits(B, "dsCMatrix")
isSymmetric(B)
dim(B)

### sampling graphs as igraph graphs ------------------

sample_igraph(ufm)

### sampling graphs as tidygraph graphs ---------------

sample_tidygraph(ufm)

sample_tidygraph(ufm, poisson_edges = FALSE)

##### directed examples ----------------------------

n2 <- 100

k1 <- 5
k2 <- 3

d <- 50

X <- matrix(rpois(n = n2 * k1, 1), nrow = n2)
S <- matrix(runif(n = k1 * k2, 0, .1), nrow = k1, ncol = k2)
Y <- matrix(rexp(n = k2 * d, 1), nrow = d)

fm <- directed_factor_model(X, S, Y, expected_in_degree = 2)
fm

### sampling graphs as edgelists ----------------------

edgelist2 <- sample_edgelist(fm)
edgelist2

### sampling graphs as sparse matrices ----------------

A2 <- sample_sparse(fm)

inherits(A2, "dgCMatrix")
isSymmetric(A2)
dim(A2)

B2 <- sample_sparse(fm, poisson_edges = FALSE)

inherits(B2, "dgCMatrix")
isSymmetric(B2)
dim(B2)

### sampling graphs as igraph graphs ------------------

# since the number of rows and the number of columns
# in `fm` differ, we will get a bipartite igraph here

# creating the bipartite igraph is slow relative to other
# sampling -- if this is a blocker for
# you please open an issue and we can investigate speedups

dig <- sample_igraph(fm)
is_bipartite(dig)

### sampling graphs as tidygraph graphs ---------------

sample_tidygraph(fm, poisson_edges = FALSE)

}
\references{
Rohe, Karl, Jun Tao, Xintian Han, and Norbert Binkiewicz. 2017.
"A Note on Quickly Sampling a Sparse Matrix with Low Rank Expectation."
Journal of Machine Learning Research; 19(77):1-13, 2018.
\url{https://www.jmlr.org/papers/v19/17-128.html}
}
\seealso{
Other samplers: 
\code{\link{sample_edgelist.matrix}()},
\code{\link{sample_igraph.directed_erdos_renyi}()},
\code{\link{sample_sparse.directed_erdos_renyi}()},
\code{\link{sample_tidygraph.directed_erdos_renyi}()}
}
\concept{samplers}
