landmarkreg <- function(fdobj, ximarks, x0marks=xmeanmarks,
                        WfdPar, monwrd=FALSE)
{
#  Arguments:
#  FDOBJ   ... functional data object for curves to be registered
#  XIMARKS ... N by NL array of times of interior landmarks for
#                 each observed curve
#  XOMARKS ... vector of length NL of times of interior landmarks for
#                 target curve
#  WFDPAR  ... a functional parameter object defining a warping function
#  MONWRD  ... If TRUE, warping functions are estimated by monotone smoothing,
#                 otherwise by regular smoothing.  The latter is faster, but
#                 not guaranteed to produce a strictly monotone warping
#                 function.  If MONWRD is 0 and an error message results
#                 indicating nonmonotonicity, rerun with MONWRD = 1.
#                 Default:  TRUE
#  Returns:
#  FDREG   ... a functional data object for the registered curves
#  WARPFD  ... a functional data object for the warping functions

 #  Last modified 24 September 2008 by Jim Ramsay

  #  check FDOBJ

  if (!(inherits(fdobj,  "fd"))) stop(
		"Argument fdobj  not a functional data object.")

  #  extract information from curve functional data object and its basis

  coef   <- fdobj$coefs
  coefd  <- dim(coef)
  ndim   <- length(coefd)
  ncurve <- coefd[2]
  if (ndim > 2) {
      nvar <- coefd[3]
  } else {
      nvar <- 1
  }

  basisobj <- fdobj$basis
  type     <- basisobj$type
  nbasis   <- basisobj$nbasis
  rangeval <- basisobj$rangeval
  fdParobj <- fdPar(basisobj, 2, 1e-10)

  #  check landmarks

  if (is.vector(ximarks)) ximarks = as.matrix(ximarks)
  ximarksd <- dim(ximarks)
  if (ximarksd[1] != ncurve) stop(
     "Number of rows of third argument wrong.")
  nlandm <- dim(ximarks)[2]
  xmeanmarks <- apply(ximarks,2,mean)

  if (length(x0marks) != nlandm) stop(
     "Number of target landmarks not equal to number of curve landmarks.")
  if (any(ximarks <= rangeval[1]) || any(ximarks >= rangeval[2])) stop(
     "Some landmark values are not within the range.")
 
  #  check WFDPAR

  WfdPar <- fdParcheck(WfdPar)
		
  #  set up WFD0 and WBASIS

  Wfd0   <- WfdPar$fd
  wLfd   <- WfdPar$Lfd
  wbasis <- Wfd0$basis

  #  set up LAMBDA

  lambda <- WfdPar$lambda

  #  check landmark target values

  wrange <- wbasis$rangeval
  if (any(rangeval != wrange)) stop(
		"Ranges for FD and WFDPAR do not match.")

  #  set up analysis

  n   <- min(c(101,10*nbasis))
  x   <- seq(rangeval[1],rangeval[2],length=n)
  wtn <- rep(1,n)

  y       <- eval.fd(x, fdobj)
  yregmat <- y
  hfunmat <- matrix(0,n,ncurve)
  lambda  <- max(lambda,1e-10)

  xval <- c(rangeval[1],x0marks,rangeval[2])
  nwbasis <- wbasis$nbasis
  Wcoef   <- matrix(0,nwbasis,ncurve)
  nval <- length(xval)
  wval <- rep(1,nval)

  #  --------------------------------------------------------------------
  #                  Iterate through curves to register
  #  --------------------------------------------------------------------

  cat("Progress:  Each dot is a curve\n")

  for (icurve in 1:ncurve) {
    cat(".")
    #  set up landmark times for this curve
    yval   <- c(rangeval[1],ximarks[icurve,],rangeval[2])
    #  smooth relation between this curve"s values and target"s values
    if (monwrd) {
       #  use monotone smoother
       conv    <- 1e-4
       iterlim <- 20
       dblev   <- 0
       Wfd     <- smooth.morph(xval, yval, WfdPar, wval)$Wfdobj
       h       <- monfn(x, Wfd)
       b       <- (rangeval[2]-rangeval[1])/(h[n]-h[1])
       a       <- rangeval[1] - b*h[1]
       h       <- a + b*h
       h[1]    <- rangeval[1]
       h[n]    <- rangeval[2]
       wcoefi  <- Wfd$coefs
       Wcoef[,icurve] <- wcoefi
    } else {
       warpfd <- smooth.basis(xval, yval, WfdPar, wval)$fd
       #  set up warping function by evaluating at sampling values
       h <- as.vector(eval.fd(x, warpfd))
       b       <- (rangeval[2]-rangeval[1])/(h[n]-h[1])
       a       <- rangeval[1] - b*h[1]
       h       <- a + b*h
       h[1]    <- rangeval[1]
       h[n]    <- rangeval[2]
       #  check for monotonicity
       deltah <- diff(h)
       if (any(deltah <= 0)) stop(
           paste("Non-increasing warping function estimated for curve",icurve))
    }
    hfunmat[,icurve] <- h

    #  compute h-inverse  in order to register curves

    if (monwrd) {
       wcoef   <- Wfd$coefs
       Wfdinv  <- fd(-wcoef,wbasis)
       WfdParinv <- fdPar(Wfdinv, wLfd, lambda)
       conv    <- 1e-4
       iterlim <- 20
       dblev   <- 0
       Wfdinv  <- smooth.morph(h, x, WfdParinv, wtn,
                              conv, iterlim, dbglev=0)$Wfdobj
       hinv    <- monfn(x, Wfdinv)
       b       <- (rangeval[2]-rangeval[1])/(hinv[n]-hinv[1])
       a       <- rangeval[1] - b*hinv[1]
       hinv    <- a + b*hinv
       hinv[1] <- rangeval[1]
       hinv[n] <- rangeval[2]
   } else {
       hinvfd  <- smooth.basis(h, x, WfdPar)$fd
       hinv    <- as.vector(eval.fd(x, hinvfd))
       b       <- (rangeval[2]-rangeval[1])/(hinv[n]-hinv[1])
       a       <- rangeval[1] - b*hinv[1]
       hinv    <- a + b*hinv
       hinv[1] <- rangeval[1]
       hinv[n] <- rangeval[2]
       deltahinv <- diff(hinv)
       if (any(deltahinv <= 0)) stop(
           paste("Non-increasing warping function estimated for curve",icurve))
    }

    #  compute registered curves

    if (length(dim(coef)) == 2) {
        #  single variable case
        yregfd <- smooth.basis(hinv, y[,icurve], fdParobj, wtn)$fd
        yregmat[,icurve] <- eval.fd(x, yregfd)
    }
    if (length(dim(coef)) == 3) {
        #  multiple variable case
        for (ivar in 1:nvar) {
            # evaluate curve as a function of h at sampling points
            yregfd <- smooth.basis(hinv, y[,icurve,ivar], fdParobj, wtn)$fd
            yregmat[,icurve,ivar] <- eval.fd(x, yregfd)
        }
     }
  }

  cat("\n")

  #  create functional data objects for the registered curves

  fdParobj    <- fdPar(basisobj, 2, 1e-10)
  regfdobj    <- smooth.basis(x, yregmat, fdParobj)$fd
  regnames    <- fdobj$fdnames
  names(regnames)[3] <- paste("Registered",names(regnames)[3])
  regfdobj$fdnames <- regnames

  #  create functional data objects for the warping functions

  warpfdobj   <- smooth.basis(x, hfunmat, fdParobj)$fd
  warpfdnames <- fdobj$fdnames
  names(warpfdnames)[3] <- paste("Warped",names(regnames)[1])
  warpfdobj$fdnames <- warpfdnames

  return( list("regfd" = regfdobj, "warpfd" = warpfdobj) )
}
