% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/historyIndexSparse.R
\name{historyIndexSparse}
\alias{historyIndexSparse}
\title{Functional History Index Model}
\usage{
historyIndexSparse(Y, X, Lag = NULL, optnsY = NULL, optnsX = NULL)
}
\arguments{
\item{Y}{A list which contains functional responses in the form of a list LY and the time points LT at which they are observed (i.e., list(Ly = LY,Lt = LT)).}

\item{X}{A list of lists which contains the observed functional predictors list Lxj and the time points list Ltj at which they are observed. It needs to be of the form \code{list(list(Ly = Lx1,Lt = Lxt1),list(Ly = Lx2,Lt = Lxt2),...)}.}

\item{Lag}{A length \code{length(X)} vector denoting the lags for all predictors.}

\item{optnsY}{A list of options control parameters for the response specified by \code{list(name=value)}. See `Details' in  \code{FPCA}.}

\item{optnsX}{A list of options control parameters for the predictors specified by \code{list(name=value)}. See `Details' in  \code{FPCA}.}
}
\value{
A list of the following:
\item{beta0}{a vector of \code{length(workGridY)} representing the fitted \eqn{\beta_0(t)}.}
\item{beta1}{a list of vectors with the \eqn{j}-th element representing the fitted \eqn{\beta_j(t)}.}
\item{gamma}{a list of vectors with the \eqn{j}-th element representing the fitted \eqn{\gamma_j(s)}.}
\item{workGridY}{a vetor representing the working grid for the response.}
\item{workGridLag}{a list of vectors with the \eqn{j}-th element representing the working grid in \eqn{[0, \Delta_j]}.}
}
\description{
Functional history index model for functional responses and functional predictors.
}
\details{
The functional history index model is defined as 
\eqn{E[Y(t)|X_1(t), \cdots, X_p(t)] = \beta_0(t) + \sum_{j=1}^p\beta_j(t)\int_0^{\Delta_j}\gamma_j(s)X_j(t-s)ds} 
for \eqn{t\in[\max_j\{\Delta_j\}, T]} with a suitable \eqn{T>0}. 
Write \eqn{\alpha_j(t, s)=\beta_j(t)\gamma_j(s)}. It becomes 
\eqn{E[Y(t)|X_1(t), \cdots, X_p(t)] = \beta_0(t) + \sum_{j=1}^p\int_0^{\Delta_j}\alpha_j(t, s)X_j(t-s)ds}. 
For more details we refer to 
\cite{Şentürk, D. and Müller, H.G., (2010). Functional varying coefficient models for longitudinal data. Journal of the American Statistical Association, 105(491), pp.1256-1264.}
}
\examples{
set.seed(1)
### functional covariate X(t) ###
phi1 <- function(t) sin(pi*t / 5) / sqrt(5)
phi2 <- function(t) cos(pi*t / 5) / sqrt(5)
lambdaX <- c(10, 5)
n <- 150
N <- 101
Xi <- matrix(rnorm(2*n), nrow = n, ncol = 2)
denseLt <- list()
denseLy <- list()
t0 <- seq(0, 15, length.out = N)
for (i in 1:n) {
  denseLt[[i]] <- t0
  denseLy[[i]] <- lambdaX[1]*Xi[i, 1]*phi1(t0) + lambdaX[2]*Xi[i, 2]*phi2(t0)
}
denseX0 <- list(Ly = denseLy,Lt = denseLt)

### generate coefficient function gamma(u), beta(u) ###
Lag <- 5
u0 <- t0[t0<=Lag]
t0_out <- t0[t0>=Lag]
gamma_u <- function(u) sqrt(2/5) * cos(pi * u /5)
beta_1 <- function(t) 5*sin(pi*t/10)
beta_0 <- function(t) t^2/2

### functional response Y(t), t in t0_out ### 
denseLt <- list()
denseLy <- list()
for (i in 1:n) {
  denseLt[[i]] <- t0_out
  Xt <- denseX0$Ly[[i]]
  Xtu <- t(sapply((1:N)[t0>=Lag], function(j){
    rev(Xt[(j-length(u0)+1):j])  #history index for X[t-u:t]
  }))
  IntGammaXtu <- apply(Xtu, 1, function(v){
    fdapace::trapzRcpp(u0, gamma_u(u0) * v)
  })
  #append 0 in the first length(u0)-1 element(useless info. in our modeling)
  denseLy[[i]] <- beta_0(t0_out) + IntGammaXtu * beta_1(t0_out) + rnorm(length(t0_out), 0, 0.1) 
}
denseY <- list(Ly = denseLy, Lt = denseLt)

### functional predictor X(t) (adjust for t0_out) ###
denseLt <- list()
denseLy <- list()
for (i in 1:n){
  denseLt[[i]] <- t0_out
  denseLy[[i]] <- denseX0$Ly[[i]][t0>=Lag]
}
denseX <- list(Ly = denseLy,
               Lt = denseLt)
               
### sparify the dense data ###
SparseY <- fdapace::Sparsify(samp = t(sapply(denseY$Ly, function(v) v)), 
                    pts = t0_out,
                    sparsity = sample(10:20, n, replace = TRUE)
)
SparseX <- fdapace::Sparsify(samp = t(sapply(denseX$Ly, function(v) v)),
                    pts = t0_out,
                    sparsity = sample(10:20, n, replace = TRUE))

### model fitting for sparse case ###
Fit_result <- historyIndexSparse(Y = SparseY, X = list(X = SparseX), Lag = Lag)
Fit_result$beta0
Fit_result$beta1
Fit_result$gamma
}
\references{
\cite{Şentürk, D. and Müller, H.G., (2010). Functional varying coefficient models for longitudinal data. Journal of the American Statistical Association, 105(491), pp.1256-1264.}
\cite{Yao, F., Müller, H.G., Wang, J.L. (2005). Functional linear regression analysis for longitudinal data. Annals of Statistics 33, 2873--2903.}
\cite{Hall, P., Horowitz, J.L. (2007). Methodology and convergence rates for functional linear regression. The Annals of Statistics, 35(1), 70--91.}
}
