% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VCAM.R
\name{VCAM}
\alias{VCAM}
\title{Sieve estimation:
B-spline based estimation procedure for time-varying additive models}
\usage{
VCAM(Lt, Ly, X, optnAdd = list(), optnVc = list())
}
\arguments{
\item{Lt}{An \emph{n}-dimensional list of \emph{N_i}-dimensional vectors whose elements consist of longitudinal time points for each \emph{i}-th subject.}

\item{Ly}{An \emph{n}-dimensional list of \emph{N_i}-dimensional vectors whose elements consist of longitudinal response observations of each \emph{i}-subject corresponding to \emph{Lt}.}

\item{X}{An \emph{n} by \emph{d} matrix whose row vectors consist of covariate vector of additive components for each subject.}

\item{optnAdd}{A list of options controls B-spline parameters for additive components, specified by list(name=value). See 'Details'.}

\item{optnVc}{A list of options controls B-spline parameters for varying-coefficient components, specified by list(name=value). See 'Details'.}
}
\value{
A list containing the following fields:
\item{Lt}{The same with input given by \emph{Lt}}
\item{LyHat}{Fitted values corresponding to \emph{Ly}}
\item{phiEst}{An \emph{N} by \emph{d} matrix whose column vectors consist of estimates for each additive component function evaluated at \emph{gridX}.}
\item{beta0Est}{An \emph{M}-dimensional vector for overall mean function estimates evaluated at \emph{gridT}.}
\item{betaEst}{An \emph{M} by \emph{d} matrix whose column vectors consist of estimates for each varying-coefficient components evaluated at \emph{gridT}.}
\item{gridX}{The same with input given by \emph{optnAdd$grid}}
\item{gridT}{The same with input  given by \emph{optnVc$grid}}
}
\description{
Sieve estimation:
B-spline based estimation procedure for time-varying additive models
}
\details{
\code{VCAM} provides a simple algorithm based on B-spline basis to estimate its nonparametric additive and varying-coefficient components.

Available control options for \emph{optnAdd} are 
\describe{
\item{nKnot}{A \emph{d}-dimensional vector which designates the number of knots for each additive component function estimation (default=10).}
\item{order}{A \emph{d}-dimensional vector which designates the order of B-spline basis for each additive component function estimation (default=3).}
\item{grid}{A \emph{N} by \emph{d} matrix whose column vector consist of evaluation grid points for each component function estimation.}
}
and control options for \emph{optnVc} are 
\describe{
\item{nKnot}{A \emph{(d+1)}-dimensional vector which designates the number of knots for overall mean function and each varying-coefficient component function estimation (default=10).}
\item{order}{A \emph{(d+1)}-dimensional vector which designates the order of B-spline basis for overall mean function and each varying-coefficient component function estimation (default=3).}
\item{grid}{A \emph{M} by \emph{(d+1)} matrix whose column vectors consist of evaluation grid points for overall mean function and each varying-coefficient component function estimation.}
}
}
\examples{

library(MASS)

set.seed(100)

n <- 100
d <- 2

Lt <- list()
Ly <- list()

m <- rep(0,2)
S <- matrix(c(1,0.5,0.5,1),nrow=2,ncol=2)
X <- pnorm(mvrnorm(n,m,S))

beta0 <- function(t) 1.5*sin(3*pi*(t+0.5))
beta1 <- function(t) 3*(1-t)^2
beta2 <- function(t) 4*t^3

phi1 <- function(x) sin(2*pi*x)
phi2 <- function(x) 4*x^3-1

for (i in 1:n) {
  Ni <- sample(10:20,1)
  
  Lt[[i]] <- sort(runif(Ni,0,1))
  Ly[[i]] <- beta0(Lt[[i]]) + 
     beta1(Lt[[i]])*phi1(X[i,1]) + beta2(Lt[[i]])*phi2(X[i,2]) + rnorm(Ni,0,0.1)
  
}


vcam <- VCAM(Lt,Ly,X)

op <- par(no.readonly = TRUE)

par(mfrow=c(1,1))
plot(unlist(vcam$LyHat),unlist(Ly),xlab='observed Y',ylab='fitted Y')
abline(coef=c(0,1),col=8)

par(mfrow=c(1,2))
plot(vcam$gridX[,1],vcam$phiEst[,1],type='l',ylim=c(-1,1),xlab='x1',ylab='phi1')
points(vcam$gridX[,1],phi1(vcam$gridX[,1]),col=2,type='l',lty=2,lwd=2)
legend('topright',c('true','est'),lwd=2,lty=c(1,2),col=c(1,2))

plot(vcam$gridX[,2],vcam$phiEst[,2],type='l',ylim=c(-1,3),xlab='x2',ylab='phi2')
points(vcam$gridX[,2],phi2(vcam$gridX[,2]),col=2,type='l',lty=2,lwd=2)
legend('topleft',c('true','est'),lwd=2,lty=c(1,2),col=c(1,2))

par(mfrow=c(1,3))
plot(vcam$gridT,vcam$beta0Est,type='l',xlab='t',ylab='beta0')
points(vcam$gridT,beta0(vcam$gridT),col=2,type='l',lty=2,lwd=2)
legend('topright',c('true','est'),lwd=2,lty=c(1,2),col=c(1,2))

plot(vcam$gridT,vcam$betaEst[,1],type='l',xlab='t',ylab='beta1')
points(vcam$gridT,beta1(vcam$gridT),col=2,type='l',lty=2,lwd=2)
legend('topright',c('true','est'),lwd=2,lty=c(1,2),col=c(1,2))

plot(vcam$gridT,vcam$betaEst[,2],type='l',xlab='t',ylab='beta2')
points(vcam$gridT,beta2(vcam$gridT),col=2,type='l',lty=2,lwd=2)
legend('topright',c('true','est'),lwd=2,lty=c(1,2),col=c(1,2))

par(op)

}
\references{
\cite{Zhang, X. and Wang, J.-L. (2015), "Varying-coefficient additive models for functional data", Biometrika, Vol.102, No.1, p.15-32.}
}
