% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/learn-heavy-tail-bipartite-kcomp.R
\name{learn_heavy_tail_kcomp_bipartite_graph}
\alias{learn_heavy_tail_kcomp_bipartite_graph}
\title{Laplacian matrix of a k-component bipartite graph with heavy-tailed data

Computes the Laplacian matrix of a k-component bipartite graph on the basis of an observed data matrix
whose distribution is assumed to be Student-t.}
\usage{
learn_heavy_tail_kcomp_bipartite_graph(
  X,
  r,
  q,
  k,
  nu = 2.001,
  rho = 1,
  learning_rate = 1e-04,
  maxiter = 1000,
  reltol = 1e-05,
  init = "default",
  verbose = TRUE,
  record_objective = FALSE
)
}
\arguments{
\item{X}{a n x p data matrix, where p is the number of nodes in the graph and n is the number of observations.}

\item{r}{number of nodes in the objects set.}

\item{q}{number of nodes in the classes set.}

\item{k}{number of components of the graph.}

\item{nu}{degrees of freedom of the Student-t distribution.}

\item{rho}{ADMM hyperparameter.}

\item{learning_rate}{gradient descent parameter.}

\item{maxiter}{maximum number of iterations.}

\item{reltol}{relative tolerance as a convergence criteria.}

\item{init}{string denoting how to compute the initial graph or a r x q matrix with initial graph weights.}

\item{verbose}{whether or not to show a progress bar during the iterations.}

\item{record_objective}{whether or not to record the objective function value during iterations.}
}
\value{
A list containing possibly the following elements:
\item{\code{laplacian}}{estimated Laplacian matrix}
\item{\code{adjacency}}{estimated adjacency matrix}
\item{\code{B}}{estimated graph weights matrix}
\item{\code{maxiter}}{number of iterations taken to reach convergence}
\item{\code{convergence}}{boolean flag to indicate whether or not the optimization converged}
\item{\code{dual_residual}}{dual residual value per iteration}
\item{\code{primal_residual}}{primal residual value per iteration}
\item{\code{aug_lag}}{augmented Lagrangian value per iteration}
\item{\code{rho_seq}}{constraint relaxation hyperparameter value per iteration}
\item{\code{elapsed_time}}{time taken per iteration until convergence is reached}
}
\description{
Laplacian matrix of a k-component bipartite graph with heavy-tailed data

Computes the Laplacian matrix of a k-component bipartite graph on the basis of an observed data matrix
whose distribution is assumed to be Student-t.
}
\examples{
library(finbipartite)
library(igraph)
set.seed(42)
r <- 50
q <- 5
p <- r + q

bipartite <- sample_bipartite(r, q, type="Gnp", p = 1, directed=FALSE)
# randomly assign edge weights to connected nodes
E(bipartite)$weight <- 1
Lw <- as.matrix(laplacian_matrix(bipartite))
B <- -Lw[1:r, (r+1):p]
B[,] <- runif(length(B))
B <- B / rowSums(B)
# utils functions
from_B_to_laplacian <- function(B) {
  A <- from_B_to_adjacency(B)
  return(diag(rowSums(A)) - A)
}

from_B_to_adjacency <- function(B) {
  r <- nrow(B)
  q <- ncol(B)
  zeros_rxr <- matrix(0, r, r)
  zeros_qxq <- matrix(0, q, q)
  return(rbind(cbind(zeros_rxr, B), cbind(t(B), zeros_qxq)))
}
Ltrue <- from_B_to_laplacian(B)
X <- MASS::mvrnorm(100*p, rep(0, p), MASS::ginv(Ltrue))
bipartite_graph <- learn_heavy_tail_kcomp_bipartite_graph(X = X,
                                                          r = r,
                                                          q = q,
                                                          k = 1,
                                                          nu = 1e2,
                                                          verbose=FALSE)
}
