% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MiscFuns.R
\name{coefplot}
\alias{coefplot}
\title{Plots confidence intervals}
\usage{
coefplot(object, sd, ci_low, ci_top, x, x.shift = 0, dict, drop, order,
  ci.width = 0.1, ci_level = 0.95, add = FALSE, pt.pch = 20,
  cex = par("cex"), pt.cex = cex, col = 1, pt.col = col,
  ci.col = col, lwd = par("lwd"), ci.lwd = lwd, ci.lty = 1,
  grid = TRUE, grid.par = list(lty = 3, col = "gray"), zero = TRUE,
  zero.par = list(col = "black", lwd = 1), join = FALSE,
  join.par = list(col = pt.col, lwd = lwd), ref.line,
  ref.line.par = list(col = "black", lty = 2), xlim.add, ylim.add,
  only.params = FALSE, only.inter = TRUE, ...)
}
\arguments{
\item{object}{Can be either: i) an estimation object (obtained for example from \code{\link[fixest]{feols}}, ii) a matrix of coefficients table, or iii) a numeric vector of the point estimates -- the latter requiring the extra arguments \code{sd} or \code{ci_low} and \code{ci_top}.}

\item{sd}{The standard errors of the estimates. It may be missing.}

\item{ci_low}{If \code{sd} is not provided, the lower bound of the confidence interval. For each estimate.}

\item{ci_top}{If \code{sd} is not provided, the upper bound of the confidence interval. For each estimate.}

\item{x}{The value of the x-axis. If missing, the names of the argument \code{estimate} are used.}

\item{x.shift}{Shifts the confidence intervals bars to the left or right, depending on the value of \code{x.shift}. Default is 0.}

\item{dict}{(Tex only.) A named character vector. It changes the original variable names to the ones contained in the \code{dict}. E.g. to change the variables named \code{a} and \code{b3} to (resp.) \dQuote{$log(a)$} and to \dQuote{$bonus^3$}, use \code{dict=c(a="$log(a)$",b3="$bonus^3$")}. By default it is equal to \code{getFixest_dict()}, a default dictionary which can be set with \code{\link[fixest]{setFixest_dict}}.}

\item{drop}{Character vector. This element is used if some variables are not to be displayed. This should be a vector of regular expressions (see \code{\link[base]{regex}} help for more info). Each variable satisfying any of the regular expressions will be discarded. This argument is applied post aliasing (see argument \code{dict}). Example: you have the variable \code{x1} to \code{x55} and want to display only \code{x1} to \code{x9}, then you could use \code{drop = "x[[:digit:]]{2}"}. If the first character is an exclamation mark, the effect is reversed (e.g. drop = "!Intercept" means: every variable that does not contain \dQuote{Intercept} is dropped). See details.}

\item{order}{Character vector. This element is used if the user wants the variables to be ordered in a certain way. This should be a vector of regular expressions (see \code{\link[base]{regex}} help for more info). The variables satisfying the first regular expression will be placed first, then the order follows the sequence of regular expressions. This argument is applied post aliasing (see argument \code{dict}). Example: you have the following variables: \code{month1} to \code{month6}, then \code{x1} to \code{x5}, then \code{year1} to \code{year6}. If you want to display first the x's, then the years, then the months you could use: \code{order = c("x", "year")}. If the first character is an exclamation mark, the effect is reversed (e.g. order = "!Intercept" means: every variable that does not contain \dQuote{Intercept} goes first).  See details.}

\item{ci.width}{The width of the extremities of the confidence intervals. Default is \code{0.1}.}

\item{ci_level}{Scalar between 0 and 1: the level of the CI. By default it is equal to 0.95.}

\item{add}{Default is \code{FALSE}, if the intervals are to be added to an existing graph. Note that if it is the case, then the argument \code{x} MUST be numeric.}

\item{pt.pch}{The patch of the coefficient estimates. Default is 20 (circle).}

\item{cex}{Numeric, default is \code{par("cex")}. Expansion factor for the points}

\item{pt.cex}{The size of the coefficient estimates. Default is the other argument \code{cex}.}

\item{col}{The color of the points and the confidence intervals. Default is 1 ("black"). Note that you can set the colors separately for each of them with \code{pt.col} and \code{ci.col}.}

\item{pt.col}{The color of the coefficient estimate. Default is equal to the other argument \code{col}.}

\item{ci.col}{The color of the confidence intervals. Default is equal to the other argument \code{col}.}

\item{lwd}{General liwe with. Default is par("lwd").}

\item{ci.lwd}{The line width of the confidence intervals. Default is equal to the other argument \code{lwd}.}

\item{ci.lty}{The line type of the confidence intervals. Default is 1.}

\item{grid}{Logical, default is \code{TRUE}. Whether a grid should be displayed. You can set the display of the grid with the argument \code{grid.par}.}

\item{grid.par}{List. Parameters of the grid. The default values are: \code{lty = 3} and \code{col = "gray"}. You can add any graphical parameter that will be passed to \code{\link[graphics]{abline}}. You also have two additional arguments: use \code{horiz = FALSE} to disable the horizontal lines, and use \code{vert = FALSE} to disable the vertical lines. Eg: \code{grid.par = list(vert = FALSE, col = "red", lwd = 2)}.}

\item{zero}{Logical, default is \code{TRUE}. Whether the 0-line should be emphasized. You can set the parameters of that line with the argument \code{zero.par}.}

\item{zero.par}{List. Parameters of the zero-line. The default values are \code{col = "black"} and \code{lwd = 1}. You can add any graphical parameter that will be passed to \code{\link[graphics]{abline}}. Example: \code{zero.par = list(col = "darkblue", lwd = 3)}.}

\item{join}{Logical, default depends on the situation. If \code{TRUE}, then the coefficient estimates are joined with a line. By default, it is equal to \code{TRUE} only if: i) interactions are plotted, ii) the x values are numeric and iii) a reference is found.}

\item{join.par}{List. Parameters of the line joining the cofficients. The default values are: \code{col = pt.col} and \code{lwd = lwd}. You can add any graphical parameter that will be passed to \code{\link[graphics]{lines}}. Eg: \code{join.par = list(lty = 2)}.}

\item{ref.line}{Logical, default depends on the situation. It is \code{TRUE} only if: i) interactions are plotted, ii) the x values are numeric and iii) a reference is found. If \code{TRUE}, then a vertical line is drawn at the level of the reference value. You can set the parameters of this line with the argument \code{ref.line.par}.}

\item{ref.line.par}{List. Parameters of the vertical line on the reference. The default values are: \code{col = "black"} and \code{lty = 2}. You can add any graphical parameter that will be passed to \code{\link[graphics]{abline}}. Eg: \code{ref.line.par = list(lty = 1, lwd = 3)}.}

\item{xlim.add}{A numeric vector of length 1 or 2. It represents an extension factor of xlim, in percentage. Eg: \code{xlim.add = c(0, 0.5)} extends \code{xlim} of 50\% on the right. If of lentgh 1, positive values represent the right, and negative values the left (Eg: \code{xlim.add = -0.5} is equivalent to \code{xlim.add = c(0.5, 0)}).}

\item{ylim.add}{A numeric vector of length 1 or 2. It represents an extension factor of ylim, in percentage. Eg: \code{ylim.add = c(0, 0.5)} extends \code{ylim} of 50\% on the top. If of lentgh 1, positive values represent the top, and negative values the bottom (Eg: \code{ylim.add = -0.5} is equivalent to \code{ylim.add = c(0.5, 0)}).}

\item{only.params}{Logical, default is \code{FALSE}. If \code{TRUE} no graphic is displayed, only the values of \code{x} and \code{y} used in the plot are returned.}

\item{only.inter}{Logical, default is \code{TRUE}. If an interaction of the type of \code{var::fe} (see \code{\link[fixest]{feols}} help for details) is found, then only these interactions are plotted. If \code{FALSE}, then interactions are treated as regular coefficients.}

\item{...}{Other arguments to be passed to \code{summary}, if \code{object} is an estimation, and/or to the function \code{plot} or \code{lines} (if \code{add = TRUE}).}
}
\description{
This function plots the results of estimations (coefficients and confidence intervals). It is flexible and handle interactions in a special way.
}
\section{Setting custom default values}{

The function \code{coefplot} dispose of many arguments to parametrize the plots. Most of these arguments can be set once an for all using the function \code{\link[fixest]{setFixest_coefplot}}. See Example 3 below for a demonstration.
}

\examples{

#
# Example 1: Stacking two sets of results on the same graph
#

# Estimation on Iris data with one fixed-effect (Species)
est = feols(Petal.Length ~ Petal.Width + Sepal.Length +
            Sepal.Width | Species, iris)


# First graph with clustered standard-errors
# (the default when fixed-effects are present)
coefplot(est, x.shift = -.2)

# 'x.shift' was used to shift the coefficients on the left.


# Second set of results: this time with
#  standard-errors that are not clustered.
coefplot(est, se = "standard", x.shift = .2,
         add = TRUE, col = 2, ci.lty = 2, pch=15)

 # Note that we used 'se', an argument that will
 #  be passed to summary.fixest

legend("topright", col = 1:2, pch = 20, lwd = 1, lty = 1:2,
       legend = c("Clustered", "Standard"), title = "Standard-Errors")


#
# Example 2: Interactions
#


# Now we estimate and plot the "yearly" treatment effects

data(base_did)
base_inter = base_did

# We interact the variable 'period' with the variable 'treat'
est_did = feols(y ~ x1 + i(treat, period, 5) | id+period, base_inter)

# You could have written the following formula instead:
# y ~ x1 + treat::period(5) | id+period

# In the estimation, the variable treat is interacted
#  with each value of period but 5, set as a reference

# When estimations contain interactions, as before,
#  the default behavior of coefplot changes,
#  it now only plots interactions:
coefplot(est_did)

# We can see that the graph is different from before:
#  - only interactions are shown,
#  - the reference is present,
#  - the estimates are joined.
# => this is fully flexible

coefplot(est_did, ref.line = FALSE, join = FALSE)

# Now to display all coefficients, use 'only.inter'
coefplot(est_did, only.inter = FALSE)

#
# What if the interacted variable is not numeric?

# Let's create a "month" variable
all_months = c("aug", "sept", "oct", "nov", "dec", "jan",
               "feb", "mar", "apr", "may", "jun", "jul")
base_inter$period_month = all_months[base_inter$period]

# The new estimation
est = feols(y ~ x1 + i(treat, period_month, "oct") | id+period, base_inter)
# Since 'period_month' of type character, coefplot sorts it
coefplot(est)

# To respect a plotting order, use a factor
base_inter$month_factor = factor(base_inter$period_month, levels = all_months)
est = feols(y ~ x1 + i(treat, month_factor, "oct") | id+period, base_inter)
coefplot(est)


#
# Example 3: Setting defaults
#

# coefplot has many arguments, which makes it highly flexible.
# If you don't like the default style of coefplot. No worries,
# you can set *your* default by using the function
# setFixest_coefplot()

dict = c("Petal.Length"="Length (Petal)", "Petal.Width"="Width (Petal)",
         "Sepal.Length"="Length (Sepal)", "Sepal.Width"="Width (Sepal)")

setFixest_coefplot(ci.col = 2, pt.col = "darkblue", ci.lwd = 3,
                   pt.cex = 2, pt.pch = 15, ci.width = 0, dict = dict)

est = feols(Petal.Length ~ Petal.Width + Sepal.Length +
                Sepal.Width | Species, iris)

# Tadaaa!
coefplot(est)

# To reset to the default settings:
setFixest_coefplot()
coefplot(est)


}
\seealso{
See \code{\link[fixest]{setFixest_coefplot}} to set the default values of \code{coefplot}, and the estimation functions: e.g. \code{\link[fixest]{feols}}, \code{\link[fixest]{fepois}}, \code{\link[fixest]{feglm}}, \code{\link[fixest]{fenegbin}}.
}
\author{
Laurent Berge
}
