% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ibdEstimate.R
\name{ibdEstimate}
\alias{ibdEstimate}
\title{Pairwise relatedness estimation}
\usage{
ibdEstimate(
  x,
  ids = typedMembers(x),
  param = c("kappa", "delta"),
  markers = NULL,
  start = NULL,
  tol = sqrt(.Machine$double.eps),
  beta = 0.5,
  sigma = 0.5,
  contourPlot = FALSE,
  levels = NULL,
  verbose = TRUE
)
}
\arguments{
\item{x}{A \code{ped} object or a list of such.}

\item{ids}{Either a vector with ID labels, or a data frame/matrix with two
columns, where each row contains the ID labels of two individuals. The
entries are coerced to characters, and must match uniquely against the ID
labels of \code{x}. By default, all pairs of members of \code{x} are included.}

\item{param}{Either "kappa" (default) or "delta"; indicating which set of
coefficients should be estimated.}

\item{markers}{A vector with names or indices of markers attached to x,
indicating which markers to include. If NULL (default), all markers are
used.}

\item{start}{A probability vector (i.e., with nonnegative entries and sum 1)
of length 3 (if \code{param = "kappa"}) or 9 (if \code{param = "delta"}), indicating
the initial value of for the optimisation. By default, \code{start} is set to
\verb{(1/3, 1/3, 1/3)} if \code{param = "kappa"} and \verb{(1/9, ..., 1/9)} if \code{param = "delta"}.}

\item{tol, beta, sigma}{Control parameters for the optimisation routine; can
usually be left untouched.}

\item{contourPlot}{A logical. If TRUE, contours of the log-likelihood
function are plotted overlaying the IBD triangle.}

\item{levels}{(Only relevant if \code{contourPlot = TRUE}.) A numeric vector of
levels at which to draw contour lines. If NULL (default), the levels are
chosen automatically.}

\item{verbose}{A logical.}
}
\value{
An object of class \code{ibdEst}, which is basically a dataframe with
either 6 columns (if \code{param = "kappa"}) or 12 columns (if \code{param = "delta"}). The first three columns are \code{id1} (label of first individual),
\code{id2} (label of second individual) and \code{N} (the number of markers with no
missing alleles). The remaining columns contain the coefficient estimates.
}
\description{
Estimate the IBD coefficients \eqn{\kappa = (\kappa_0, \kappa_1,
\kappa_2)}{(k0, k1, k2)} or the condensed identity coefficients \eqn{\Delta =
(\Delta_1, ..., \Delta_9)}{(d1, ..., d9)} between a pair (or several pairs)
of pedigree members, using maximum likelihood methods.
}
\details{
It should be noted that this procedure estimates the \emph{realised} identity
coefficients of each pair, i.e., the actual fractions of the autosomes in
each IBD state. These may deviate substantially from the theoretical pedigree
coefficients.

Maximum likelihood estimation of relatedness coefficients originates with
Thompson (1975). Optimisation of \eqn{\kappa} is done in the \eqn{(\kappa_0,
\kappa_2)}{(k0, k2)}-plane and restricted to the triangle defined by
\deqn{\kappa_0 \ge 0, \kappa_2 \ge 0, \kappa_0 + \kappa_2 \le 1}{k0 >= 0, k2
>= 0, k0 + k2 <= 1}. Optimisation of \eqn{\Delta} is done in unit simplex of
R^8, using the first 8 coefficients.

The implementation optimises the log-likelihood using a projected gradient
descent algorithm, combined with a version of Armijo line search.
}
\examples{

### Example 1: Siblings
x = nuclearPed(2)

# Simulate 100 markers
x = markerSim(x, N = 100, alleles = 1:4, seed = 123, verbose = FALSE)

# Estimate kappa (expectation: (0.25, 0.5, 0.25)
ibdEstimate(x, ids = 3:4)

# Plot contours of the log-likelihood function
ibdEstimate(x, ids = 3:4, contourPlot = TRUE)

### Example 2: Full sib mating
y = fullSibMating(1)

# Simulate 200 SNP markers
y = markerSim(y, N = 1000, alleles = 1:10, seed = 123, verbose = FALSE)

# Estimate
ibdEstimate(y, ids = 5:6, param = "delta")


}
\references{
\itemize{
\item E. A. Thompson (1975). \emph{The estimation of pairwise relationships.} Annals
of Human Genetics 39.
\item E. A. Thompson (2000). \emph{Statistical Inference from Genetic Data on
Pedigrees.} NSF-CBMS Regional Conference Series in Probability and
Statistics. Volume 6.
}
}
\seealso{
\code{\link[=ibdBootstrap]{ibdBootstrap()}}
}
\author{
Magnus Dehli Vigeland
}
