% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/funDataMethods.R
\name{flipFuns}
\alias{flipFuns}
\title{Flip functional data objects}
\usage{
flipFuns(refObject, newObject, ...)
}
\arguments{
\item{refObject}{An object of class \code{funData}, \code{irregFunData} or
\code{multiFunData} that serves as reference. It must have the same number
of observations as \code{newObject} or have only one observation. In this
case, all observations in \code{newObject} are flipped with respect to this
single observation.}

\item{newObject}{An object of class \code{funData}, \code{irregFunData} or
\code{multiFunData} that is to be flipped with respect to \code{refObject}.}

\item{...}{Further parameters passed to \code{\link{norm}}.}
}
\value{
An object of the same class as \code{newData} with flipped 
  observations.
}
\description{
This function flips an object \code{newObject} of class \code{funData}, 
\code{irregFunData} or \code{multiFunData} with respect to a reference object
\code{refObject} of the same class (or of class \code{funData}, if 
\code{newObject} is irregular). This is particularly useful when dealing with
functional principal components, as they are only defined up to a sign 
change. For details, see below.
}
\details{
Functional principal component analysis is an important tool in functional 
data analysis. Just as eigenvectors, eigenfunctions (or functional principal 
components) are only defined up to a sign change. This may lead to 
difficulties in simulation studies or when bootstrapping pointwise confidence
bands, as in these cases one wants the estimates to have the same 
"orientation" as the true function (in simulation settings) or the 
non-bootstrapped estimate (when calculating bootstrap confidence bands). This
function allows to flip (i.e. multiply by \eqn{-1}{-1}) all observations in 
\code{newObject} that have a different orientation than their counterparts in
\code{refData}.

Technically, the function compares the distance between \code{newObject} and 
\code{refObject} \deqn{|||f_\mathrm{new} - f_\mathrm{ref}|||}{||| f_{new} - 
f_{ref}|||} and the distance between  \code{newObject} and 
\code{-1 * refObject} \deqn{|||f_\mathrm{new} + f_\mathrm{ref}|||.}{||| f_{new} + 
f_{ref}|||.} If \code{newObject} is closer to \code{-1 * refObject}, it is 
flipped, i.e. multiplied by -1.
}
\section{Warning}{

The function is currently implemented only for functional data with one- and 
two-dimensional domains.
}
\examples{

### Univariate
argvals <- seq(0,2*pi,0.01)
refData <- funData(argvals, rbind(sin(argvals))) # one observation as reference
newData <- funData(argvals, outer(sample(c(-1,1), 11, replace = TRUE) * seq(0.75, 1.25, by = 0.05),
                               sin(argvals)))

oldpar <- par(no.readonly = TRUE)
par(mfrow = c(1,2))

plot(newData, col = "grey", main = "Original data")
plot(refData, col = "red", lwd = 2, add = TRUE)

plot(flipFuns(refData, newData), col = "grey", main = "Flipped data")
plot(refData, col = "red", lwd = 2, add = TRUE)

### Univariate (irregular)
ind <- replicate(11, sort(sample(1:length(argvals), sample(5:10,1)))) # sample observation points
argvalsIrreg <- lapply(ind, function(i){argvals[i]})
argvalsIrregAll <- unique(sort(unlist(argvalsIrreg)))
 # one observation as reference (fully observed)
refDataFull <- funData(argvals, rbind(sin(argvals)))
 # one observation as reference (irregularly observed)
refDataIrreg <- irregFunData(argvals = list(argvalsIrregAll), X = list(sin(argvalsIrregAll)))
newData <- irregFunData(argvals = argvalsIrreg, X = mapply(function(x, a, s){s * a * sin(x)},
     x = argvalsIrreg, a = seq(0.75, 1.25, by = 0.05), s = sample(c(-1,1), 11, replace = TRUE)))

plot(newData, col = "grey", main = "Original data (regular reference)")
plot(refDataFull, col = "red", lwd = 2, add = TRUE)

plot(flipFuns(refDataFull, newData), col = "grey", main = "Flipped data")
plot(refDataFull, col = "red", lwd = 2, add = TRUE)

plot(newData, col = "grey", main = "Original data (irregular reference)")
plot(refDataIrreg, col = "red", lwd = 2, add = TRUE)

plot(flipFuns(refDataIrreg, newData), col = "grey", main = "Flipped data")
plot(refDataIrreg, col = "red", lwd = 2, add = TRUE)

### Multivariate
refData <- multiFunData(funData(argvals, rbind(sin(argvals))), # one observation as reference
                        funData(argvals, rbind(cos(argvals)))) 
sig <- sample(c(-1,1), 11, replace = TRUE) 
newData <- multiFunData(funData(argvals, outer(sig * seq(0.75, 1.25, by = 0.05), sin(argvals))),
                        funData(argvals, outer(sig * seq(0.75, 1.25, by = 0.05), cos(argvals))))
                        
par(mfrow = c(2,2))

plot(newData[[1]], col = topo.colors(11), main = "Original data")
plot(refData[[1]], col = "red", lwd = 2, add = TRUE)

plot(newData[[2]], col = topo.colors(11), main = "Original data")
plot(refData[[2]], col = "red", lwd = 2, add = TRUE)

plot(flipFuns(refData, newData)[[1]], col = topo.colors(11), main = "Flipped data")
plot(refData[[1]], col = "red", lwd = 2, add = TRUE)

plot(flipFuns(refData, newData)[[2]], col = topo.colors(11), main = "Flipped data")
plot(refData[[2]], col = "red", lwd = 2, add = TRUE)

par(oldpar)
}
\seealso{
\code{\linkS4class{funData}}, \code{\linkS4class{irregFunData}},
  \code{\linkS4class{multiFunData}}, \code{\link{Arith.funData}}
}

