% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gasfluxes.R
\name{selectfluxes}
\alias{selectfluxes}
\title{Select a flux estimate}
\usage{
selectfluxes(dat, select, f.detect = NULL, t.meas = NULL,
  tol = 5e-05, ...)
}
\arguments{
\item{dat}{a data.table as returned by \code{\link{gasfluxes}}. The function modifies it by reference.}

\item{select}{character; specify a ruleset for selection of the final flux value, see details.}

\item{f.detect}{detection limit for HMR method. This can be determined by a simple simulation (see examples) or for four data points the approximation in Parkin et al. (2012) can be used.}

\item{t.meas}{a vector or single value giving the measurement time factor that relates to kappa.max. It is suggested to use the time difference between the first and last sample taken from the closed chamber. The unit should be consistent with the units of \code{f.detect} and kappa (e.g., h if kappa is in 1/h).}

\item{tol}{the relative tolerance \code{abs((linear.f0 - HMR.f0)/HMR.f0)} below which the linear flux estimate and the HMR flux estimate are considered equal in the "kappa.max" algorithm. This is to protect against HMR fits that equal the linear fit and have extremely high standard errors. Defaults to \code{tol = 5e-5}.}

\item{\dots}{further parameters}
}
\value{
A data.table with the with following columns added to the function input: selected flux estimate, the corresponding standard error and p-value and the method with which the selected flux was estimated. 
 For the  "kappa.max" method the "kappa.max" values are included. These columns are also added to the input data.table by reference.
}
\description{
Selects the appropriate flux estimate from linear, robust linear and non-linear calculated fluxes.
}
\details{
Available selection algorithms currently are
\describe{
\item{"RF2011"}{The algorithm used, e.g., in Leiber-Sauheitl 2014 (doi:10.5194/bg-11-749-2014).
This overwrites the methods parameter. The factor guarding against degenerate HMR fits
can be set via the ellipsis as \code{gfactor}. Default is \code{gfactor = 4}. This method is not recommended any more and only provided for reproducibility of old results.}
\item{"RF2011new"}{The same rules as "RF2011", but using the improved fitting function for HMR,
which results in larger SE and p-values. Thus, it is less likely to select the HMR result.
This method is not recommended any more and only provided for reproducibility of old results.}
\item{"kappa.max"}{The selection algorithm restricts the use of HMR by imposing a maximal value for kappa "kappa.max", 
depending on the quotient of the linear flux estimate and the minimal detectable flux (f.detect), as well as 
the chamber closure time (t.meas). kappa.max = f.lin/f.detect/t.meas. This is currently the recommended algorithm.}
}

Other selection algorithms could be implemented, but selection can always be done as a postprocessing step. E.g., if many data points are available for each flux measurement it is probably most sensible to use AICc.
}
\examples{
\dontrun{
res <- gasfluxes(fluxMeas[1:499], 
                 .id = "serie", .V = "V", .A = "A",
                 .times = "time", .C = "C",
                 methods = c("linear", "robust linear", "HMR"), verbose = FALSE, plot = FALSE)
selectfluxes(res, "RF2011new")
res[method == "HMR", .N] #2

### estimate f.detect by simulation ###
#ambient concentration:
C0 <- 320/1000 * 28 * 273.15 / 22.4 / (273.15 + 15) #mg N / m^3
#uncertainty of GC measurement:
sdGC <- 5/1000 * 28 * 273.15 / 22.4 / (273.15 + 15) #mg N / m^3 
#create simulated concentrations corresponding to 1 hour flux measurements with zero fluxes:
set.seed(42)
sim <- data.frame(t = seq(0, 1, length.out = 4), C = rnorm(4e3, mean = C0, sd = sdGC), 
                  id = rep(1:1e3, each = 4), A = 1, V = 0.52)
#fit HMR model:                  
simflux <- gasfluxes(sim, .id = "id", .times = "t", methods = c("HMR", "linear"), plot = FALSE) 
simflux[, f0 := HMR.f0]
simflux[is.na(f0), f0 := linear.f0]
#dection limit as 97.5 \% quantile (95 \% confidence):
f.detect <- simflux[, quantile(f0, 0.975)] #0.03 mg N / m^2 / h

# example using the kappa.max (ref. Hueppi et al., 2018) with a single t.meas value
t.meas <- max(fluxMeas$time[1:499]) #1
selectfluxes(res, "kappa.max", f.detect = f.detect, t.meas = t.meas)
res[method == "HMR", .N] # 11        

# example using the kappa.max with a vector for t.meas
t.meas <- fluxMeas[1:499][, max(time), by = serie][["V1"]]
selectfluxes(res, "kappa.max", f.detect = f.detect, t.meas = t.meas)
res[method == "HMR", .N] # 10                  
}

}
\references{
Parkin, T.B., Venterea, R.T., Hargreaves, S.K., 2012. Calculating the Detection Limits of Chamber-based Soil Greenhouse Gas Flux Measurements. Journal of Environmental Quality 41, 705-715.

Hueppi, R., Felber, R., Krauss, M., Six, J., Leifeld, J., Fuss, R., 2018. Restricting the nonlinearity parameter in soil greenhouse gas flux calculation for more reliable flux estimates. PLOS ONE 13(7): e0200876. https://doi.org/10.1371/journal.pone.0200876
}
