% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_plots.R
\name{bar_terminal}
\alias{bar_terminal}
\title{Comparative plot for the Pathmox terminal nodes}
\usage{
bar_terminal(
  x,
  .LV,
  .bycoef = FALSE,
  .cex.names = 1,
  .cex.axis = 1.2,
  .cex.main = 1,
  ...
)
}
\arguments{
\item{x}{An object of the class \code{"plstree"}.}

\item{.LV}{A string indicating the name of the dependent  latent variable. The label
must be the same as used to define the structural model in the (lavaan syntax).}

\item{.bycoef}{Logical value indicating if the comparison is done by nodes or by path
coefficients. By default, \code{FALSE} means that the comparison is done 
by nodes.}

\item{.cex.names}{Expansion factor for axis names (bar labels).}

\item{.cex.axis}{Expansion factor for numeric axis labels.}

\item{.cex.main}{Allows fixing the size of the main. It is equal to 1 to default.}

\item{\dots}{Further arguments are ignored.}
}
\description{
\code{bar_terminal} returns the path coefficient bar plots of the Pathmox terminal 
nodes.
}
\details{
This function aims to visualize, using bar plots, the path coefficients of
the dependnet latent construct associated with the terminal nodes. 
The user indicates the dependnet latent construct they want to visualize. 
This is done using the same label as used in the structural model definition
(lavaan syntax). The comparison is done by analyzing the path coefficient 
values for each node, or the values estimated in each node for each path coefficient. 
In the former, the plot also returns the R^2. In the latter, the bar corresponding to the 
node with the highest path coefficient value shows in a different color. 
By default the comparison is done by analyzing the path coefficient values for each node.
}
\examples{
 \dontrun{
# Example of PATHMOX approach in customer satisfaction analysis 
# (Spanish financial company).
# Model with 5 LVs (4 common factor: Image (IMAG), Value (VAL), 
# Satisfaction (SAT), and Loyalty (LOY); and 1 composite construct: 
# Quality (QUAL)

# load library and dataset csibank
library(genpathmx)
data("csibank")

# Define the model using the lavaan syntax. Use a set of regression formulas to define 
# first the structural model and then the measurement model

CSImodel <- "
# Structural model
VAL  ~ QUAL
SAT  ~ IMAG  + QUAL + VAL
LOY  ~ IMAG + SAT

# Measurement model
# Composite
QUAL <~ qual1 + qual2 + qual3 + qual4 + qual5 + qual6 + qual7 
     
# Common factor
IMAG =~ imag1 + imag2 + imag3 + imag4 + imag5 + imag6 
VAL  =~ val1  + val2  + val3  + val4
SAT  =~ sat1  + sat2  + sat3           
LOY  =~ loy1  + loy2  + loy3           

"

# Run pathmox on one single variable 
age = csibank[,2]

# Transform age into an ordered factor
age = factor(age, levels = c("<=25", "26-35", "36-45", "46-55",
                                      "56-65", ">=66"),ordered = T)
                                     
csi.pathmox.age = pls.pathmox(
 .model = CSImodel ,
 .data  = csibank,
 .catvar= age,
 .alpha = 0.05,
 .deep = 1
)  

# Visualize the bar plot by comparing the nodes
bar_terminal(csi.pathmox.age, .LV = "SAT")

# Visualize the bar plot by comparing path coefficients
bar_terminal(csi.pathmox.age, .LV = "SAT", .bycoef = TRUE)

}

}
\references{
Lamberti, G., Aluja, T. B., and Sanchez, G. (2016). The Pathmox approach for PLS path 
modeling segmentation. \emph{Applied Stochastic Models in Business and Industry}, \bold{32}(4), 453-468.
\doi{10.1002/asmb.2168}

Lamberti, G. (2015). \emph{Modeling with Heterogeneity}, PhD Dissertation.

Sanchez, G. (2009). \emph{PATHMOX Approach: Segmentation Trees in
Partial Least Squares Path Modeling}, PhD Dissertation.
}
\seealso{
\code{\link{summary.plstree}}, \code{\link{print.plstree}}, \code{\link{pls.pathmox}},  
\code{\link{plot.plstree}}, and \code{\link{bar_impvar}}
}
\author{
Giuseppe Lamberti
}
