% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gisco_get.R
\name{gisco_get}
\alias{gisco_get}
\alias{gisco_get_coastallines}
\alias{gisco_get_communes}
\alias{gisco_get_countries}
\alias{gisco_get_lau}
\alias{gisco_get_nuts}
\alias{gisco_get_urban_audit}
\title{Get geospatial data from GISCO API}
\source{
\url{https://gisco-services.ec.europa.eu/distribution/v2/}
}
\usage{
gisco_get_coastallines(
  year = "2016",
  epsg = "4326",
  cache = TRUE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  resolution = "20"
)

gisco_get_communes(
  year = "2016",
  epsg = "4326",
  cache = TRUE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  spatialtype = "RG",
  country = NULL
)

gisco_get_countries(
  year = "2016",
  epsg = "4326",
  cache = TRUE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  resolution = "20",
  spatialtype = "RG",
  country = NULL,
  region = NULL
)

gisco_get_lau(
  year = "2016",
  epsg = "4326",
  cache = TRUE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  country = NULL,
  gisco_id = NULL
)

gisco_get_nuts(
  year = "2016",
  epsg = "4326",
  cache = TRUE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  resolution = "20",
  spatialtype = "RG",
  country = NULL,
  nuts_id = NULL,
  nuts_level = "all"
)

gisco_get_urban_audit(
  year = "2020",
  epsg = "4326",
  cache = TRUE,
  update_cache = FALSE,
  cache_dir = NULL,
  verbose = FALSE,
  spatialtype = "RG",
  country = NULL,
  level = NULL
)
}
\arguments{
\item{year}{Release year. See \strong{Release years available} on \code{\link{gisco_get}}.}

\item{epsg}{projection of the map: 4-digit \href{https://epsg.io/}{EPSG code}.
One of:
\itemize{
\item "4258": ETRS89
\item "4326": WGS84
\item "3035": ETRS89 / ETRS-LAEA
\item "3857": Pseudo-Mercator
}}

\item{cache}{A logical whether to do caching. Default is \code{TRUE}.}

\item{update_cache}{A logical whether to update cache. Default is \code{FALSE}.
When set to \code{TRUE} it would force a fresh download of the source
\code{.geojson} file.}

\item{cache_dir}{A path to a cache directory. See Details on \code{\link{gisco_get}}.}

\item{verbose}{Display information. Useful for debugging,
default is \code{FALSE}.}

\item{resolution}{Resolution of the geospatial data. One of
\itemize{
\item "60": 1:60million
\item "20": 1:20million
\item "10": 1:10million
\item "03": 1:3million
\item "01": 1:1million
}}

\item{spatialtype}{Type of geometry to be returned:
\itemize{
\item \strong{"RG"}: Regions - \code{MULTIPOLYGON/POLYGON} object.
\item \strong{"LB"}: Labels - \code{POINT} object.
\item \strong{"BN"}: Boundaries - \code{LINESTRING} object.
\item \strong{"COASTL"}: coastlines - \code{LINESTRING} object.
\item \strong{"INLAND"}: inland boundaries - \code{LINESTRING} object.
}}

\item{country}{Optional. A character vector of country codes.
See Details on \code{\link{gisco_get}}.}

\item{region}{Optional. A character vector of UN M49 region codes or
European Union membership. Possible values are "Africa", "Americas",
"Asia", "Europe", "Oceania" or "EU" for countries belonging to the European
Union (as per 2021). See \strong{About world regions} and \code{\link{gisco_countrycode}}}

\item{gisco_id}{Optional. A character vector of GISCO_ID LAU values.}

\item{nuts_id}{Optional. A character vector of NUTS IDs.}

\item{nuts_level}{NUTS level. One of "0" (Country-level), "1", "2" or "3".
See \url{https://ec.europa.eu/eurostat/web/nuts/background}.}

\item{level}{Level of Urban Audit. Possible values are "CITIES", "FUA",
"GREATER_CITIES" or \code{NULL}, that would download the full dataset.}
}
\value{
A \code{sf} object specified by \code{spatialtype}.
}
\description{
Loads a simple feature (\code{sf}) object from GISCO API entry point or your
local library.
}
\details{
\code{country} only available on specific datasets. Some \code{spatialtype} options
(as "BN", "COASTL", "INLAND") may not include country-level identifiers.

\code{country} could be either a vector of country names, a vector of ISO3
country codes or a vector of Eurostat country codes. Mixed types
(as \code{c("Turkey","US","FRA")}) would not work.

Sometimes cached files may be corrupt. On that case, try re-downloading
the data setting \code{update_cache = TRUE}.

Set \code{cache_dir = "path/to/dir"} or \verb{options(gisco_cache_dir = "path/to/dir)}.
If you experience any problem on download, try to download
the corresponding \code{.geojson} file by any other method and save it on
your \code{cache_dir}.

For a complete list of files available check \code{\link{gisco_db}}.
}
\note{
Please check the download and usage provisions on \code{\link[=gisco_attributions]{gisco_attributions()}}.
}
\section{About world regions}{
Regions are defined as per the geographic regions defined by the
UN (see \url{https://unstats.un.org/unsd/methodology/m49/}.
Under this scheme Cyprus is assigned to Asia. You may use
\code{region = "EU"} to get the EU members (reference date: 2021).
}

\section{Release years available}{
\itemize{
\item \strong{\code{gisco_get_coastallines}}: one of "2006", "2010", "2013" or "2016".
\item \strong{\code{gisco_get_communes}}: one of "2001", "2004", "2006", "2008", "2010",
"2013" or "2016".
\item \strong{\code{gisco_get_countries}}: one of "2001", "2006", "2010", "2013", "2016"
or "2020".
\item \strong{\code{gisco_get_lau}}: one of "2016", "2017", "2018" or "2019"
}

\itemize{
\item \strong{\code{gisco_get_nuts}}: one of "2003", "2006", "2010", "2013", "2016"
or "2021".
}

\itemize{
\item \strong{\code{gisco_get_urban_audit}}: one of "2001", "2004", "2014", "2018"
or "2020".
}
}

\examples{

library(sf)

##################################
# Example - gisco_get_coastallines
##################################

coastlines <- gisco_get_coastallines()
plot(st_geometry(coastlines), col = "palegreen", border = "lightblue3")
title(
  main = "Coastal Lines",
  sub = gisco_attributions(),
  line = 1
)

###############################
# Example - gisco_get_countries
###############################

sf_world <- gisco_get_countries()
plot(st_geometry(sf_world), col = "seagreen2")
title(sub = gisco_attributions(), line = 1)


sf_africa <- gisco_get_countries(region = "Africa")
plot(st_geometry(sf_africa),
  col = c("springgreen4", "darkgoldenrod1", "red2")
)
title(sub = gisco_attributions(), line = 1)

sf_benelux <-
  gisco_get_countries(country = c("Belgium", "Netherlands", "Luxembourg"))
plot(st_geometry(sf_benelux),
  col = c("grey10", "orange", "deepskyblue2")
)
title(sub = gisco_attributions(), line = 1)

##########################
# Example - gisco_get_nuts
##########################

nuts1 <- gisco_get_nuts(
  resolution = "20",
  year = "2016",
  epsg = "4326",
  nuts_level = "1",
  country = "ITA"
)
nuts2 <- gisco_get_nuts(
  resolution = "20",
  year = "2016",
  epsg = "4326",
  nuts_level = "2",
  country = "ITA"
)
nuts3 <- gisco_get_nuts(
  resolution = "20",
  year = "2016",
  epsg = "4326",
  nuts_level = "3",
  country = "ITA"
)

plot(st_geometry(nuts3),
  border = "grey60",
  lty = 3
)

plot(st_geometry(nuts2),
  lwd = 2,
  border = "red2",
  add = TRUE
)

plot(st_geometry(nuts1),
  lwd = 3,
  border = "springgreen4",
  add = TRUE
)

box()
title(
  main = "NUTS Levels on Italy",
  sub = gisco_attributions(),
  cex.sub = 0.7,
  line = 1
)
legend(
  "topright",
  legend = c("NUTS 1", "NUTS 2", "NUTS 3"),
  col = c("springgreen4", "red2", "grey60"),
  lty = c(1, 1, 3),
  lwd = c(3, 2, 1),
  bty = "n",
  y.intersp = 2
)
}
\seealso{
\code{\link{gisco_db}}, \code{\link[=gisco_attributions]{gisco_attributions()}}, \code{\link{gisco_coastallines}}

\code{\link[=gisco_countrycode]{gisco_countrycode()}}, \code{\link{gisco_countries}}

\code{\link{gisco_nuts}}
}
\author{
dieghernan, \url{https://github.com/dieghernan/}
}
\concept{political}
