% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{llmc}
\alias{llmc}
\title{Negative Log-Likelihood for the McDonald (Mc)/Beta Power Distribution}
\usage{
llmc(par, data)
}
\arguments{
\item{par}{A numeric vector of length 3 containing the distribution parameters
in the order: \code{gamma} (\eqn{\gamma > 0}), \code{delta} (\eqn{\delta \ge 0}),
\code{lambda} (\eqn{\lambda > 0}).}

\item{data}{A numeric vector of observations. All values must be strictly
between 0 and 1 (exclusive).}
}
\value{
Returns a single \code{double} value representing the negative
log-likelihood (\eqn{-\ell(\theta|\mathbf{x})}). Returns \code{Inf}
if any parameter values in \code{par} are invalid according to their
constraints, or if any value in \code{data} is not in the interval (0, 1).
}
\description{
Computes the negative log-likelihood function for the McDonald (Mc)
distribution (also known as Beta Power) with parameters \code{gamma}
(\eqn{\gamma}), \code{delta} (\eqn{\delta}), and \code{lambda} (\eqn{\lambda}),
given a vector of observations. This distribution is the special case of the
Generalized Kumaraswamy (GKw) distribution where \eqn{\alpha = 1} and
\eqn{\beta = 1}. This function is suitable for maximum likelihood estimation.
}
\details{
The McDonald (Mc) distribution is the GKw distribution (\code{\link{dmc}})
with \eqn{\alpha=1} and \eqn{\beta=1}. Its probability density function (PDF) is:
\deqn{
f(x | \theta) = \frac{\lambda}{B(\gamma,\delta+1)} x^{\gamma \lambda - 1} (1 - x^\lambda)^\delta
}
for \eqn{0 < x < 1}, \eqn{\theta = (\gamma, \delta, \lambda)}, and \eqn{B(a,b)}
is the Beta function (\code{\link[base]{beta}}).
The log-likelihood function \eqn{\ell(\theta | \mathbf{x})} for a sample
\eqn{\mathbf{x} = (x_1, \dots, x_n)} is \eqn{\sum_{i=1}^n \ln f(x_i | \theta)}:
\deqn{
\ell(\theta | \mathbf{x}) = n[\ln(\lambda) - \ln B(\gamma, \delta+1)]
+ \sum_{i=1}^{n} [(\gamma\lambda - 1)\ln(x_i) + \delta\ln(1 - x_i^\lambda)]
}
This function computes and returns the \emph{negative} log-likelihood, \eqn{-\ell(\theta|\mathbf{x})},
suitable for minimization using optimization routines like \code{\link[stats]{optim}}.
Numerical stability is maintained, including using the log-gamma function
(\code{\link[base]{lgamma}}) for the Beta function term.
}
\examples{
\donttest{
# Assuming existence of rmc, grmc, hsmc functions for Mc distribution

# Generate sample data from a known Mc distribution
set.seed(123)
true_par_mc <- c(gamma = 2, delta = 3, lambda = 0.5)
# Use rmc for data generation
sample_data_mc <- rmc(100, gamma = true_par_mc[1], delta = true_par_mc[2],
                      lambda = true_par_mc[3])
hist(sample_data_mc, breaks = 20, main = "Mc(2, 3, 0.5) Sample")

# --- Maximum Likelihood Estimation using optim ---
# Initial parameter guess
start_par_mc <- c(1.5, 2.5, 0.8)

# Perform optimization (minimizing negative log-likelihood)
mle_result_mc <- stats::optim(par = start_par_mc,
                              fn = llmc, # Use the Mc neg-log-likelihood
                              method = "BFGS", # Or "L-BFGS-B" with lower=1e-6
                              hessian = TRUE,
                              data = sample_data_mc)

# Check convergence and results
if (mle_result_mc$convergence == 0) {
  print("Optimization converged successfully.")
  mle_par_mc <- mle_result_mc$par
  print("Estimated Mc parameters:")
  print(mle_par_mc)
  print("True Mc parameters:")
  print(true_par_mc)
} else {
  warning("Optimization did not converge!")
  print(mle_result_mc$message)
}

# --- Compare numerical and analytical derivatives (if available) ---
# Requires 'numDeriv' package and analytical functions 'grmc', 'hsmc'
if (mle_result_mc$convergence == 0 &&
    requireNamespace("numDeriv", quietly = TRUE) &&
    exists("grmc") && exists("hsmc")) {

  cat("\nComparing Derivatives at Mc MLE estimates:\n")

  # Numerical derivatives of llmc
  num_grad_mc <- numDeriv::grad(func = llmc, x = mle_par_mc, data = sample_data_mc)
  num_hess_mc <- numDeriv::hessian(func = llmc, x = mle_par_mc, data = sample_data_mc)

  # Analytical derivatives (assuming they return derivatives of negative LL)
  ana_grad_mc <- grmc(par = mle_par_mc, data = sample_data_mc)
  ana_hess_mc <- hsmc(par = mle_par_mc, data = sample_data_mc)

  # Check differences
  cat("Max absolute difference between gradients:\n")
  print(max(abs(num_grad_mc - ana_grad_mc)))
  cat("Max absolute difference between Hessians:\n")
  print(max(abs(num_hess_mc - ana_hess_mc)))

} else {
   cat("\nSkipping derivative comparison for Mc.\n")
   cat("Requires convergence, 'numDeriv' package and functions 'grmc', 'hsmc'.\n")
}

}

}
\references{
McDonald, J. B. (1984). Some generalized functions for the size distribution
of income. \emph{Econometrica}, 52(3), 647-663.

Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized
distributions. \emph{Journal of Statistical Computation and Simulation},

Kumaraswamy, P. (1980). A generalized probability density function for
double-bounded random processes. \emph{Journal of Hydrology}, \emph{46}(1-2), 79-88.
}
\seealso{
\code{\link{llgkw}} (parent distribution negative log-likelihood),
\code{\link{dmc}}, \code{\link{pmc}}, \code{\link{qmc}}, \code{\link{rmc}},
\code{grmc} (gradient, if available),
\code{hsmc} (Hessian, if available),
\code{\link[stats]{optim}}, \code{\link[base]{lbeta}}
}
\author{
Lopes, J. E.
}
\keyword{distribution}
\keyword{likelihood}
\keyword{mcdonald}
\keyword{optimize}
