% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg.R
\name{predict.gkwreg}
\alias{predict.gkwreg}
\title{Predictions from a Fitted Generalized Kumaraswamy Regression Model}
\usage{
\method{predict}{gkwreg}(
  object,
  newdata = NULL,
  type = "response",
  na.action = stats::na.pass,
  at = 0.5,
  elementwise = NULL,
  family = NULL,
  ...
)
}
\arguments{
\item{object}{An object of class \code{"gkwreg"}, typically the result of a
call to \code{\link{gkwreg}}.}

\item{newdata}{An optional data frame containing the variables needed for prediction.
If omitted, predictions are made for the data used to fit the model.}

\item{type}{A character string specifying the type of prediction. Options are:
\describe{
\item{\code{"response"} or \code{"mean"}}{Predicted mean response (default).}
\item{\code{"link"}}{Linear predictors for each parameter before applying
inverse link functions.}
\item{\code{"parameter"}}{Parameter values on their original scale (after
applying inverse link functions).}
\item{\code{"alpha"}, \code{"beta"}, \code{"gamma"}, \code{"delta"},
\code{"lambda"}}{Values for a specific distribution parameter.}
\item{\code{"variance"}}{Predicted variance of the response.}
\item{\code{"density"} or \code{"pdf"}}{Density function values at points
specified by \code{at}.}
\item{\code{"probability"} or \code{"cdf"}}{Cumulative distribution function
values at points specified by \code{at}.}
\item{\code{"quantile"}}{Quantiles corresponding to probabilities specified
by \code{at}.}
}}

\item{na.action}{Function determining how to handle missing values in \code{newdata}.
Default is \code{stats::na.pass}, which returns \code{NA} for cases with missing
predictors.}

\item{at}{Numeric vector of values at which to evaluate densities, probabilities,
or for which to compute quantiles, depending on \code{type}. Required for
\code{type = "density"}, \code{type = "probability"}, or \code{type = "quantile"}.
Defaults to 0.5.}

\item{elementwise}{Logical. If \code{TRUE} and \code{at} has the same length as
the number of observations, applies each value in \code{at} to the corresponding
observation. If \code{FALSE} (default), applies all values in \code{at} to each
observation, returning a matrix.}

\item{family}{Character string specifying the distribution family to use for
calculations. If \code{NULL} (default), uses the family from the fitted model.
Options match those in \code{\link{gkwreg}}: \code{"gkw"}, \code{"bkw"},
\code{"kkw"}, \code{"ekw"}, \code{"mc"}, \code{"kw"}, \code{"beta"}.}

\item{...}{Additional arguments (currently not used).}
}
\value{
The return value depends on the \code{type} argument:
\itemize{
\item For \code{type = "response"}, \code{type = "variance"}, or individual
parameters (\code{type = "alpha"}, etc.): A numeric vector of length equal
to the number of rows in \code{newdata} (or the original data).

\item For \code{type = "link"} or \code{type = "parameter"}: A data frame with
columns for each parameter and rows corresponding to observations.

\item For \code{type = "density"}, \code{type = "probability"}, or
\code{type = "quantile"}:
\itemize{
\item If \code{elementwise = TRUE}: A numeric vector of length equal to
the number of rows in \code{newdata} (or the original data).
\item If \code{elementwise = FALSE}: A matrix where rows correspond to
observations and columns correspond to the values in \code{at}.
}
}
}
\description{
Computes predictions and related quantities from a fitted Generalized
Kumaraswamy (GKw) regression model object. This method can extract fitted values,
predicted means, linear predictors, parameter values, variances, densities,
probabilities, and quantiles based on the estimated model. Predictions can be made
for new data or for the original data used to fit the model.
}
\details{
The \code{predict.gkwreg} function provides a flexible framework for obtaining
predictions and inference from fitted Generalized Kumaraswamy regression models.
It handles all subfamilies of GKw distributions and respects the parametrization
and link functions specified in the original model.

\subsection{Prediction Types}{
The function supports several types of predictions:
\itemize{
\item \strong{Response/Mean}: Computes the expected value of the response variable
based on the model parameters. For most GKw family distributions, this requires
numerical integration or special formulas.

\item \strong{Link}: Returns the linear predictors for each parameter without
applying inverse link functions. These are the values \eqn{\eta_j = X\beta_j}
for each parameter \eqn{j}.

\item \strong{Parameter}: Computes the distribution parameter values on their original
scale by applying the appropriate inverse link functions to the linear predictors.
For example, if alpha uses a log link, then \eqn{\alpha = \exp(X\beta_\alpha)}.

\item \strong{Individual Parameters}: Extract specific parameter values (alpha, beta,
gamma, delta, lambda) on their original scale.

\item \strong{Variance}: Estimates the variance of the response based on the model
parameters. For some distributions, analytical formulas are used; for others,
numerical approximations are employed.

\item \strong{Density/PDF}: Evaluates the probability density function at specified
points given the model parameters.

\item \strong{Probability/CDF}: Computes the cumulative distribution function at
specified points given the model parameters.

\item \strong{Quantile}: Calculates quantiles corresponding to specified probabilities
given the model parameters.
}
}

\subsection{Link Functions}{
The function respects the link functions specified in the original model for each
parameter. The supported link functions are:
\itemize{
\item \code{"log"}: \eqn{g(\mu) = \log(\mu)}, \eqn{g^{-1}(\eta) = \exp(\eta)}
\item \code{"logit"}: \eqn{g(\mu) = \log(\mu/(1-\mu))}, \eqn{g^{-1}(\eta) = 1/(1+\exp(-\eta))}
\item \code{"probit"}: \eqn{g(\mu) = \Phi^{-1}(\mu)}, \eqn{g^{-1}(\eta) = \Phi(\eta)}
\item \code{"cauchy"}: \eqn{g(\mu) = \tan(\pi*(\mu-0.5))}, \eqn{g^{-1}(\eta) = 0.5 + (1/\pi) \arctan(\eta)}
\item \code{"cloglog"}: \eqn{g(\mu) = \log(-\log(1-\mu))}, \eqn{g^{-1}(\eta) = 1 - \exp(-\exp(\eta))}
\item \code{"identity"}: \eqn{g(\mu) = \mu}, \eqn{g^{-1}(\eta) = \eta}
\item \code{"sqrt"}: \eqn{g(\mu) = \sqrt{\mu}}, \eqn{g^{-1}(\eta) = \eta^2}
\item \code{"inverse"}: \eqn{g(\mu) = 1/\mu}, \eqn{g^{-1}(\eta) = 1/\eta}
\item \code{"inverse-square"}: \eqn{g(\mu) = 1/\sqrt{\mu}}, \eqn{g^{-1}(\eta) = 1/\eta^2}
}
}

\subsection{Family-Specific Constraints}{
The function enforces appropriate constraints for each distribution family:
\itemize{
\item \code{"gkw"}: All 5 parameters (\eqn{\alpha, \beta, \gamma, \delta, \lambda}) are used.
\item \code{"bkw"}: \eqn{\lambda = 1} is fixed.
\item \code{"kkw"}: \eqn{\gamma = 1} is fixed.
\item \code{"ekw"}: \eqn{\gamma = 1, \delta = 0} are fixed.
\item \code{"mc"}: \eqn{\alpha = 1, \beta = 1} are fixed.
\item \code{"kw"}: \eqn{\gamma = 1, \delta = 0, \lambda = 1} are fixed.
\item \code{"beta"}: \eqn{\alpha = 1, \beta = 1, \lambda = 1} are fixed.
}
}

\subsection{Parameter Bounds}{
All parameters are constrained to their valid ranges:
\itemize{
\item \eqn{\alpha, \beta, \gamma, \lambda > 0}
\item \eqn{0 < \delta < 1}
}
}

\subsection{Using with New Data}{
When providing \code{newdata}, ensure it contains all variables used in the model's
formula. The function extracts the terms for each parameter's model matrix and applies
the appropriate link functions to calculate predictions. If any variables are missing,
the function will attempt to substitute reasonable defaults or raise an error if
critical variables are absent.
}

\subsection{Using for Model Evaluation}{
The function is useful for model checking, generating predicted values for plotting,
and evaluating the fit of different distribution families. By specifying the \code{family}
parameter, you can compare predictions under different distributional assumptions.
}
}
\examples{
\donttest{
# Generate a sample dataset (n = 1000)
set.seed(123)
n <- 1000

# Create predictors
x1 <- runif(n, -2, 2)
x2 <- rnorm(n)
x3 <- factor(rbinom(n, 1, 0.4))

# Simulate Kumaraswamy distributed data
# True parameters with specific relationships to predictors
true_alpha <- exp(0.7 + 0.3 * x1)
true_beta <- exp(1.2 - 0.2 * x2 + 0.4 * (x3 == "1"))

# Generate random responses
y <- rkw(n, alpha = true_alpha, beta = true_beta)

# Ensure responses are strictly in (0, 1)
y <- pmax(pmin(y, 1 - 1e-7), 1e-7)

# Create data frame
df <- data.frame(y = y, x1 = x1, x2 = x2, x3 = x3)

# Split into training and test sets
set.seed(456)
train_idx <- sample(n, 800)
train_data <- df[train_idx, ]
test_data <- df[-train_idx, ]

# ====================================================================
# Example 1: Basic usage - Fit a Kumaraswamy model and make predictions
# ====================================================================

# Fit the model
kw_model <- gkwreg(y ~ x1 | x2 + x3, data = train_data, family = "kw")

# Predict mean response for test data
pred_mean <- predict(kw_model, newdata = test_data, type = "response")

# Calculate prediction error
mse <- mean((test_data$y - pred_mean)^2)
cat("Mean Squared Error:", mse, "\n")

# ====================================================================
# Example 2: Different prediction types
# ====================================================================

# Create a grid of values for visualization
x1_grid <- seq(-2, 2, length.out = 100)
grid_data <- data.frame(x1 = x1_grid, x2 = 0, x3 = 0)

# Predict different quantities
pred_mean <- predict(kw_model, newdata = grid_data, type = "response")
pred_var <- predict(kw_model, newdata = grid_data, type = "variance")
pred_params <- predict(kw_model, newdata = grid_data, type = "parameter")
pred_alpha <- predict(kw_model, newdata = grid_data, type = "alpha")
pred_beta <- predict(kw_model, newdata = grid_data, type = "beta")

# Plot predicted mean and parameters against x1
plot(x1_grid, pred_mean,
  type = "l", col = "blue",
  xlab = "x1", ylab = "Predicted Mean", main = "Mean Response vs x1"
)
plot(x1_grid, pred_var,
  type = "l", col = "red",
  xlab = "x1", ylab = "Predicted Variance", main = "Response Variance vs x1"
)
plot(x1_grid, pred_alpha,
  type = "l", col = "purple",
  xlab = "x1", ylab = "Alpha", main = "Alpha Parameter vs x1"
)
plot(x1_grid, pred_beta,
  type = "l", col = "green",
  xlab = "x1", ylab = "Beta", main = "Beta Parameter vs x1"
)

# ====================================================================
# Example 3: Computing densities, CDFs, and quantiles
# ====================================================================

# Select a single observation
obs_data <- test_data[1, ]

# Create a sequence of y values for plotting
y_seq <- seq(0.01, 0.99, length.out = 100)

# Compute density at each y value
dens_values <- predict(kw_model,
  newdata = obs_data,
  type = "density", at = y_seq, elementwise = FALSE
)

# Compute CDF at each y value
cdf_values <- predict(kw_model,
  newdata = obs_data,
  type = "probability", at = y_seq, elementwise = FALSE
)

# Compute quantiles for a sequence of probabilities
prob_seq <- seq(0.1, 0.9, by = 0.1)
quant_values <- predict(kw_model,
  newdata = obs_data,
  type = "quantile", at = prob_seq, elementwise = FALSE
)

# Plot density and CDF
plot(y_seq, dens_values,
  type = "l", col = "blue",
  xlab = "y", ylab = "Density", main = "Predicted PDF"
)
plot(y_seq, cdf_values,
  type = "l", col = "red",
  xlab = "y", ylab = "Cumulative Probability", main = "Predicted CDF"
)

# ====================================================================
# Example 4: Prediction under different distributional assumptions
# ====================================================================

# Fit models with different families
beta_model <- gkwreg(y ~ x1 | x2 + x3, data = train_data, family = "beta")
gkw_model <- gkwreg(y ~ x1 | x2 + x3 | 1 | 1 | x3, data = train_data, family = "gkw")

# Predict means using different families
pred_kw <- predict(kw_model, newdata = test_data, type = "response")
pred_beta <- predict(beta_model, newdata = test_data, type = "response")
pred_gkw <- predict(gkw_model, newdata = test_data, type = "response")

# Calculate MSE for each family
mse_kw <- mean((test_data$y - pred_kw)^2)
mse_beta <- mean((test_data$y - pred_beta)^2)
mse_gkw <- mean((test_data$y - pred_gkw)^2)

cat("MSE by family:\n")
cat("Kumaraswamy:", mse_kw, "\n")
cat("Beta:", mse_beta, "\n")
cat("GKw:", mse_gkw, "\n")

# Compare predictions from different families visually
plot(test_data$y, pred_kw,
  col = "blue", pch = 16,
  xlab = "Observed", ylab = "Predicted", main = "Predicted vs Observed"
)
points(test_data$y, pred_beta, col = "red", pch = 17)
points(test_data$y, pred_gkw, col = "green", pch = 18)
abline(0, 1, lty = 2)
legend("topleft",
  legend = c("Kumaraswamy", "Beta", "GKw"),
  col = c("blue", "red", "green"), pch = c(16, 17, 18)
)

# ====================================================================
# Example 5: Working with linear predictors and link functions
# ====================================================================

# Extract linear predictors and parameter values
lp <- predict(kw_model, newdata = test_data, type = "link")
params <- predict(kw_model, newdata = test_data, type = "parameter")

# Verify that inverse link transformation works correctly
# For Kumaraswamy model, alpha and beta use log links by default
alpha_from_lp <- exp(lp$alpha)
beta_from_lp <- exp(lp$beta)

# Compare with direct parameter predictions
cat("Manual inverse link vs direct parameter prediction:\n")
cat("Alpha difference:", max(abs(alpha_from_lp - params$alpha)), "\n")
cat("Beta difference:", max(abs(beta_from_lp - params$beta)), "\n")

# ====================================================================
# Example 6: Elementwise calculations
# ====================================================================

# Generate probabilities specific to each observation
probs <- runif(nrow(test_data), 0.1, 0.9)

# Calculate quantiles for each observation at its own probability level
quant_elementwise <- predict(kw_model,
  newdata = test_data,
  type = "quantile", at = probs, elementwise = TRUE
)

# Calculate probabilities at each observation's actual value
prob_at_y <- predict(kw_model,
  newdata = test_data,
  type = "probability", at = test_data$y, elementwise = TRUE
)

# Create Q-Q plot
plot(sort(prob_at_y), seq(0, 1, length.out = length(prob_at_y)),
  xlab = "Empirical Probability", ylab = "Theoretical Probability",
  main = "P-P Plot", type = "l"
)
abline(0, 1, lty = 2, col = "red")

# ====================================================================
# Example 7: Predicting for the original data
# ====================================================================

# Fit a model with original data
full_model <- gkwreg(y ~ x1 + x2 + x3 | x1 + x2 + x3, data = df, family = "kw")

# Get fitted values using predict and compare with model's fitted.values
fitted_from_predict <- predict(full_model, type = "response")
fitted_from_model <- full_model$fitted.values

# Compare results
cat(
  "Max difference between predict() and fitted.values:",
  max(abs(fitted_from_predict - fitted_from_model)), "\n"
)

# ====================================================================
# Example 8: Handling missing data
# ====================================================================

# Create test data with some missing values
test_missing <- test_data
test_missing$x1[1:5] <- NA
test_missing$x2[6:10] <- NA

# Predict with different na.action options
pred_na_pass <- tryCatch(
  predict(kw_model, newdata = test_missing, na.action = na.pass),
  error = function(e) rep(NA, nrow(test_missing))
)
pred_na_omit <- tryCatch(
  predict(kw_model, newdata = test_missing, na.action = na.omit),
  error = function(e) rep(NA, nrow(test_missing))
)

# Show which positions have NAs
cat("Rows with missing predictors:", which(is.na(pred_na_pass)), "\n")
cat("Length after na.omit:", length(pred_na_omit), "\n")
}

\donttest{
## Example 1: Simple Kumaraswamy regression model ----
set.seed(123)
n <- 1000
x1 <- runif(n, -2, 2)
x2 <- rnorm(n)

# True regression coefficients
alpha_coef <- c(0.8, 0.3, -0.2) # Intercept, x1, x2
beta_coef <- c(1.2, -0.4, 0.1) # Intercept, x1, x2

# Generate linear predictors and transform to parameters using inverse link (exp)
eta_alpha <- alpha_coef[1] + alpha_coef[2] * x1 + alpha_coef[3] * x2
eta_beta <- beta_coef[1] + beta_coef[2] * x1 + beta_coef[3] * x2
alpha_true <- exp(eta_alpha)
beta_true <- exp(eta_beta)

# Generate responses from Kumaraswamy distribution (assuming rkw is available)
y <- rkw(n, alpha = alpha_true, beta = beta_true)
# Create data frame
df1 <- data.frame(y = y, x1 = x1, x2 = x2)

# Fit Kumaraswamy regression model using extended formula syntax
# Model alpha ~ x1 + x2 and beta ~ x1 + x2
kw_reg <- gkwreg(y ~ x1 + x2 | x1 + x2, data = df1, family = "kw", silent = TRUE)

# Display summary
summary(kw_reg)

## Example 2: Generalized Kumaraswamy regression ----
set.seed(456)
x1 <- runif(n, -1, 1)
x2 <- rnorm(n)
x3 <- factor(rbinom(n, 1, 0.5), labels = c("A", "B")) # Factor variable

# True regression coefficients
alpha_coef <- c(0.5, 0.2) # Intercept, x1
beta_coef <- c(0.8, -0.3, 0.1) # Intercept, x1, x2
gamma_coef <- c(0.6, 0.4) # Intercept, x3B
delta_coef <- c(0.0, 0.2) # Intercept, x3B (logit scale)
lambda_coef <- c(-0.2, 0.1) # Intercept, x2

# Design matrices
X_alpha <- model.matrix(~x1, data = data.frame(x1 = x1))
X_beta <- model.matrix(~ x1 + x2, data = data.frame(x1 = x1, x2 = x2))
X_gamma <- model.matrix(~x3, data = data.frame(x3 = x3))
X_delta <- model.matrix(~x3, data = data.frame(x3 = x3))
X_lambda <- model.matrix(~x2, data = data.frame(x2 = x2))

# Generate linear predictors and transform to parameters
alpha <- exp(X_alpha \%*\% alpha_coef)
beta <- exp(X_beta \%*\% beta_coef)
gamma <- exp(X_gamma \%*\% gamma_coef)
delta <- plogis(X_delta \%*\% delta_coef) # logit link for delta
lambda <- exp(X_lambda \%*\% lambda_coef)

# Generate response from GKw distribution (assuming rgkw is available)
y <- rgkw(n, alpha = alpha, beta = beta, gamma = gamma, delta = delta, lambda = lambda)

# Create data frame
df2 <- data.frame(y = y, x1 = x1, x2 = x2, x3 = x3)

# Fit GKw regression with parameter-specific formulas
# alpha ~ x1, beta ~ x1 + x2, gamma ~ x3, delta ~ x3, lambda ~ x2
gkw_reg <- gkwreg(y ~ x1 | x1 + x2 | x3 | x3 | x2, data = df2, family = "gkw")

# Compare true vs. estimated coefficients
print("Estimated Coefficients (GKw):")
print(coef(gkw_reg))
print("True Coefficients (approx):")
print(list(
  alpha = alpha_coef, beta = beta_coef, gamma = gamma_coef,
  delta = delta_coef, lambda = lambda_coef
))

## Example 3: Beta regression for comparison ----
set.seed(789)
x1 <- runif(n, -1, 1)

# True coefficients for Beta parameters (gamma = shape1, delta = shape2)
gamma_coef <- c(1.0, 0.5) # Intercept, x1 (log scale for shape1)
delta_coef <- c(1.5, -0.7) # Intercept, x1 (log scale for shape2)

# Generate linear predictors and transform (default link is log for Beta params here)
X_beta_eg <- model.matrix(~x1, data.frame(x1 = x1))
gamma_true <- exp(X_beta_eg \%*\% gamma_coef)
delta_true <- exp(X_beta_eg \%*\% delta_coef)

# Generate response from Beta distribution
y <- rbeta_(n, gamma_true, delta_true)

# Create data frame
df_beta <- data.frame(y = y, x1 = x1)

# Fit Beta regression model using gkwreg
# Formula maps to gamma and delta: y ~  x1 | x1
beta_reg <- gkwreg(y ~ x1 | x1,
  data = df_beta, family = "beta",
  link = list(gamma = "log", delta = "log")
) # Specify links if non-default

## Example 4: Model comparison using AIC/BIC ----
# Fit an alternative model, e.g., Kumaraswamy, to the same beta-generated data
kw_reg2 <- try(gkwreg(y ~ x1 | x1, data = df_beta, family = "kw"))

print("AIC Comparison (Beta vs Kw):")
c(AIC(beta_reg), AIC(kw_reg2))
print("BIC Comparison (Beta vs Kw):")
c(BIC(beta_reg), BIC(kw_reg2))

## Example 5: Predicting with a fitted model

# Use the Beta regression model from Example 3
newdata <- data.frame(x1 = seq(-1, 1, length.out = 20))

# Predict expected response (mean of the Beta distribution)
pred_response <- predict(beta_reg, newdata = newdata, type = "response")

# Predict parameters (gamma and delta) on the scale of the link function
pred_link <- predict(beta_reg, newdata = newdata, type = "link")

# Predict parameters on the original scale (shape1, shape2)
pred_params <- predict(beta_reg, newdata = newdata, type = "parameter")

# Plot original data and predicted mean response curve
plot(df_beta$x1, df_beta$y,
  pch = 20, col = "grey", xlab = "x1", ylab = "y",
  main = "Beta Regression Fit (using gkwreg)"
)
lines(newdata$x1, pred_response, col = "red", lwd = 2)
legend("topright", legend = "Predicted Mean", col = "red", lty = 1, lwd = 2)
}

}
\references{
Cordeiro, G. M., & de Castro, M. (2011). A new family of generalized distributions.
\emph{Journal of Statistical Computation and Simulation}, \strong{81}(7), 883-898.

Kumaraswamy, P. (1980). A generalized probability density function for double-bounded
random processes. \emph{Journal of Hydrology}, \strong{46}(1-2), 79-88.

Ferrari, S. L. P., & Cribari-Neto, F. (2004). Beta regression for modelling rates and
proportions. \emph{Journal of Applied Statistics}, \strong{31}(7), 799-815.

Jones, M. C. (2009). Kumaraswamy's distribution: A beta-type distribution with some
tractability advantages. \emph{Statistical Methodology}, \strong{6}(1), 70-81.
}
\seealso{
\code{\link{gkwreg}} for fitting Generalized Kumaraswamy regression models,
\code{\link{fitted.gkwreg}} for extracting fitted values,
\code{\link{residuals.gkwreg}} for calculating residuals,
\code{\link{summary.gkwreg}} for model summaries,
\code{\link{coef.gkwreg}} for extracting coefficients.
}
\author{
Lopes, J. E. and contributors
}
\keyword{models}
\keyword{predict}
\keyword{regression}
