% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/synsort.R
\name{synsort}
\alias{synsort}
\title{Sorting functions for synoptic tables}
\usage{
synsort(syn1, syn2 = syn1, cluster, method = "allspec", min1 = 0,
  min2 = 0, relate2 = "entire")
}
\arguments{
\item{syn1}{Input synoptic table 1 (as dataframe) with priority entries for sorting.
Usually dataframe from \code{\link{syntable}} function output,
but function should work with every synoptic table input, as long as formats are
appropriate. The values of this table will be displayed in the final output table.}

\item{syn2}{Optional second input table with additional sorting criteria. Note that
values of second input table will be considered in sorting, but not be displayed
in final synoptic table with \code{method = "allspec"}.}

\item{cluster}{Integer vector with classification cluster identity. Ensure matching order
of cluster identity and samples in dataframe for correct allocation of cluster numbers to samples.}

\item{method}{Sorting algorithm (\code{method = c("allspec", "p_diff", "n_diff", "pn_diff", "accspec", "all_diff")}).
See Details.}

\item{min1}{Treshold minimum value for considering species of \code{syn1} in ordering algorithm.
Species below that minimum will neither be considered in algorithm nor displayed in final
synoptic table, but will be listed in the \code{$others} output.}

\item{min2}{Treshold minimum value for considering species of \code{syn2} in ordering algorithm.
Species below that minimum will neither be considered in algorithm nor displayed in final
synoptic table, but will be listed in the \code{$others} vector.}

\item{relate2}{Specifies relation of given second table minimum values to either related to
entire dataset (default) or to each cluster only (\code{relate2 = c("entire", "cluster"))}.}
}
\value{
Returns a list composed of:
  \item{\code{$output}}{sorting method description}
  \item{\code{$species}}{species sorting criteria}
  \item{\code{$samplesize}}{sample sizes in clusters}
  \item{\code{$syntable}}{sorted synoptic table}
  \item{\code{$others}}{species that failed to be included in the final table due to
treshold values given by min1 and min2}
   \item{\code{$differential}}{In case of combined sorting with considering differential species character,
   a table with differential character of species.}
}
\description{
Synoptic tables are a tool for interpretation of cluster species composition.
This function provides sorting options for synoptic tables, sorting criteria can be either
values in synoptic tables, such as frequencies, as well as combined criteria with considering
differential character, too.
Sorting algorithm aims to sort species in given cluster column order to blocked structure.
Thereby, species with high frequencies and/or differential character are displayed blocked
for each cluster or several neighbouring clusters.
}
\section{Details}{

Six types of synoptic tables can be created with this function.

 \describe{\item{\code{method = "allspec"}}{creates a sorted synoptic table basing on numeric input tables,
 e.g. common percentage frequency tables. Sorting criteria can be either given by one input table (\code{syn1}), as well
 as by two input tables (\code{syn1, syn2}). Thereby, only values of \code{syn1} will be shown in the final sorted table,
 but values of \code{syn2} will be considered, too. The second minimum treshold (\code{min2}) for values in \code{syn2}
 will be either applied on single clusters (e.g. recommended for phi values with specifying \code{relate2 = "cluster"})
 for having minimum phi/cluster or e.g. on total frequencies in entire dataset for excluding rare species from
 synoptic table, applying the minimum treshold on the entire dataset (\code{relate2 = "entire"}).}}

 With including differential species character as sorting criterion (\code{method = c("p_diff", "n_diff", "pn_diff", "accspec", "all_diff")}),
 input table \code{syn1} must be numeric, the second one with information on differential character (output from \code{\link{syntable}} function with
\code{type="diffspec"}). Again, algorithm detects highest cluster values of species in \code{syn1} as base for sorting,
but will sort them considering differentiating character criterion (from second
input table \code{syn2}). Species with high values in \code{syn1} AND differential character will then be listed
on the top of a species block. Within differentiating species, prevalence of diagnostic character
is considered by favoring positive and/or cluster-specific differential character. Available types are:
\describe{
 \item{\code{method = "p_diff"}}{creates a synoptic table of diagnostic species with numerical values of input table
 \code{syn1}}
 \item{\code{method = "accspec"}}{creates a synoptic table of non-diagnostic species with numerical values of input table
 \code{syn1}}
 \item{\code{method = "all_diff"}}{showing all diagnostic and non-diagnostic species}}
}

\examples{
## Synoptic table of Scheden vegetation data:
library(cluster)
pam1 <- pam(schedenveg, 4)

## Unordered synoptic tables
# Unordered synoptiv percentage frequency table
unordered <- syntable(schedenveg, pam1$clustering, abund = "perc",
                      type = "percfreq")
# Differential species analysis
differential <- syntable(schedenveg, pam1$clustering, abund = "perc",
                         type = "diffspec")
# Fidelity phi
phitable <- syntable(schedenveg, pam1$clustering, abund = "perc",
                     type = "phi")

## Common complete synoptic table: sort by percentage frequency,
## show all species
sorted <- synsort(syn1 = unordered$syntable, cluster = pam1$clustering,
                  method = "allspec", min1 = 0)
sorted             # view results

## Synoptic table, with only positive differentiating species with
## minimum 25\% frequency in table
positive <- synsort(syn1 = unordered$syntable, syn2 = differential$syntable,
                    cluster = pam1$clustering, method = "p_diff", min1 = 25)
positive           # view results

## Synoptic table, with percentage frequency (only species >25\%) and
## differential character.
complete <- synsort(syn1 = unordered$syntable, syn2 = differential$syntable,
                    cluster = pam1$clustering, method = "all_diff", min1 = 25)
complete

## Synoptic table, species with minimum phi-value of 0.3, show
## percentage frequency
phi_complete <- synsort(syn1 = unordered$syntable, syn2 = phitable$syntable,
                        cluster = pam1$clustering, method = "allspec", min1 = 25, min2 = 0.3)
phi_complete

## Synoptic table with only phi values
phi_table <- synsort(syn1 = phitable$syntable, cluster = pam1$clustering,
                     method = "allspec", min1 = 0.3)
phitable

## Synoptic table showing diagnostic phi phi-values (>=0.3) and
## differential character
phi_diff <- synsort(syn1 = phitable$syntable, syn2 = differential$syntable,
                    cluster = pam1$clustering, method = "all_diff", min1 = 0.3)
phi_diff
}
\references{
Bruelheide, H. (2000): A new measure of fidelity and its application to defining species groups. - \emph{Journal
of Vegetation Science} 11: 167-178.

Chytry, M., Tichy, L., Holt, J., Botta-Dukat, Z. (2002): Determination of diagnostic species with statistical fidelity measures.
\emph{Journal of Vegetation Science} 13: 79-90.

Sokal, R.R. & Rohlf, F.J. (1995): Biometry. 3rd edition Freemann, New York.

Tsiripidis, I., Bergmeier, E., Fotiadis, G. & Dimopoulos, P. (2009): A new algorithm for the determination
of differential taxa. - \emph{Journal of Vegetation Science} 20: 233-240.
}
\seealso{
\code{\link{syntable}}
}
\author{
Jenny Schellenberg (\email{jschell@gwdg.de})
}
