% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/syntable.R
\name{syntable}
\alias{syntable}
\title{Synoptic tables and calculation of cluster-wise frequencies, fidelity and
differential species character}
\usage{
syntable(spec, cluster, abund = "perc", type = "percfreq")
}
\arguments{
\item{spec}{Species matrix or dataframe with species in columns and samples in rows.
Values must be numeric, with point "." as decimal character, or integer.  Missing values, NA or NaN
are not allowed. Species and sample names must be defined as column- and rownames, respectively.}

\item{cluster}{Integer vector with classification cluster identity. Ensure matching order of
cluster identity and samples in dataframe for correct allocation of cluster numbers to samples.}

\item{abund}{Data input type. Define whether input species matrix or dataframe is presence/absence
data (\code{abund = "freq"}) or percentage cover (\code{abund = "perc"}, default).}

\item{type}{Type of synoptic table output \code{type = c("percfreq", "totalfreq", "mean",
"median", "diffspec", "phi")}. See Details.}
}
\value{
The function returns a list of result components.
  \item{\code{$syntable }}{unordered synoptic table for given species and clusters}
  \item{\code{$samplesize }}{total samples in clusters}

Additionally for differential species character calculation:
  \item{\code{$onlydiff }}{Synoptic table only with differential species}
  \item{\code{$others }}{List of non-differential species}
  \item{\code{$differentials }}{Lists differential species for each cluster}
}
\description{
Synoptic tables are a tool for the visualization and interpretation of previously
defined plant species groups (clusters), e.g. from cluster analysis or classification methods.
They help to determine characteristic patterning of species occurrences in plant communities
by calculating cluster-wise percentage or absolute frequencies, mean/median cover values, fidelity
(phi) or differential species character.
\code{syntable} function calculates an unordered synoptic table for plant community analysis, using
an input species-sample dataframe and a numeric vector of cluster identity input.
The unordered output table can be sorted automatically with \code{\link[goeveg]{synsort}} function
in this package.
}
\section{Details}{

For synoptic table calculation, six types are available.
  \itemize{\item{\code{type = "percfreq" }}{Default, creates a percentage frequency table}
  \item{\code{type = "totalfreq" }}{Creates an absolute frequency table}
  \item{\code{type = "mean" }}{Calculates mean of species values given in \code{spec} per cluster}
  \item{\code{type = "median" }}{Calculates median of species values given in \code{spec} per
   cluster}
  \item{\code{type = "diffspec" }}{Calculates differential character of species according to
   Tsiripidis et al. 2009, with resulting character p = positive, n = negative, pn = positive-
   negative or no differential character (-). Consider that differential character is always
   restricted to some and not necessarily all of the other units, thus considering percentage
   frequency is essential for correct interpretation of the diagnostic species character.}
  \item{\code{type = "phi" }}{Calculates fidelity measure phi (algorithm basing on Sokal & Rohlf
   1995, Bruelheide 2000). Values are ranging between -1 and 1 with high values near 1 indicating
   high fidelity.}}

For sorting the output synoptic table, use \code{\link{synsort}} function, providing several
options.
}

\examples{
## Synoptic table of Scheden vegetation data
library(cluster)
pam1 <- pam(schedenveg, 4)  # PAM clustering with 4 clusters output

## 1) unordered synoptic percentage frequency table
unordered <- syntable(schedenveg, pam1$clustering, abund = "perc",
                      type = "percfreq")
unordered                   # view results

## 2) differential species analysis
differential <- syntable(schedenveg, pam1$clustering, abund = "perc",
                         type = "diffspec")
# show complete table with differential character of species
differential$syntable
# list differential species for second cluster
differential$differentials$group2

## 3) Synoptic table with phi fidelity
phitable <- syntable(schedenveg, pam1$clustering, abund = "perc",
                     type = "phi")
phitable
}
\references{
Bruelheide, H. (2000): A new measure of fidelity and its application to defining species groups.
 \emph{Journal of Vegetation Science} \strong{11}: 167-178. \doi{https://doi.org/10.2307/3236796}

Chytry, M., Tichy, L., Holt, J., Botta-Dukat, Z. (2002): Determination of diagnostic species with
 statistical fidelity measures. \emph{Journal of Vegetation Science} \strong{13}: 79-90. \doi{https://doi.org/10.1111/j.1654-1103.2002.tb02025.x}

Sokal, R.R. & Rohlf, F.J. (1995): Biometry. 3rd edition Freemann, New York.

Tsiripidis, I., Bergmeier, E., Fotiadis, G. & Dimopoulos, P. (2009): A new algorithm for the
determination of differential taxa. \emph{Journal of Vegetation Science} \strong{20}: 233-240. \doi{https://doi.org/10.1111/j.1654-1103.2009.05273.x}
}
\seealso{
\code{\link{synsort}}
}
\author{
Jenny Schellenberg (\email{jschell@gwdg.de})
}
