% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/flm_est.R
\name{flm_est}
\alias{flm_est}
\title{Estimation of functional linear models}
\usage{
flm_est(X, Y, est_method = "fpcr_l1s", p = NULL, q = NULL,
  thre_p = 0.99, thre_q = 0.99, lambda = NULL, X_fpc = NULL,
  Y_fpc = NULL, compute_residuals = TRUE, centered = FALSE,
  int_rule = "trapezoid", cv_verbose = FALSE, ...)
}
\arguments{
\item{X, Y}{samples of functional/scalar predictors and functional/scalar
response. Either \code{\link[fda.usc]{fdata}} objects (for functional
variables) or vectors of length \code{n} (for scalar variables).}

\item{est_method}{either \code{"fpcr"} (Functional Principal Components
Regression; FPCR), \code{"fpcr_l2"} (FPCR with ridge penalty),
\code{"fpcr_l1"} (FPCR with lasso penalty) or \code{"fpcr_l1s"}
(FPCR with lasso-selected FPC). If \code{X} is scalar, \code{flm_est}
only considers \code{"fpcr"} as estimation method. See details below.
Defaults to \code{"fpcr_l1s"}.}

\item{p, q}{index vectors indicating the specific FPC to be
considered for the truncated bases expansions of \code{X} and \code{Y},
respectively. If a single number for \code{p} is provided, then
\code{p <- 1:max(p)} internally (analogously for \code{q}) and the first
\code{max(p)} FPC are considered. If \code{NULL} (default), then a
data-driven selection of \code{p} and \code{q} is done. See details below.}

\item{thre_p, thre_q}{thresholds for the \emph{proportion} of variance
that is explained, \emph{at least}, by the first \eqn{p} and \eqn{q} FPC
of \code{X} and \code{Y}, respectively. These thresholds are employed
for an (initial) automatic selection of \eqn{p} and \eqn{q}.
Default to \code{0.99}. \code{thre_p} (\code{thre_q}) is ignored if
\code{p} (\code{q}) is provided.}

\item{lambda}{regularization parameter \eqn{\lambda} for the estimation
methods \code{"fpcr_l2"}, \code{"fpcr_l1"}, and \code{"fpcr_l1s"}. If
\code{NULL} (default), it is chosen with \code{\link[goffda]{cv_glmnet}}.}

\item{X_fpc, Y_fpc}{FPC decompositions of \code{X} and \code{Y}, as
returned by \code{\link[goffda]{fpc}}. Computed if not provided.}

\item{compute_residuals}{whether to compute the fitted values \code{Y_hat}
and its \code{Y_hat_scores}, and the \code{residuals} of the fit
and its \code{residuals_scores}. Defaults to \code{TRUE}.}

\item{centered}{flag to indicate if \code{X} and \code{Y} have been
centered or not, in order to avoid their recentering. Defaults to
\code{FALSE}.}

\item{int_rule}{quadrature rule for approximating the definite
unidimensional integral: trapezoidal rule (\code{int_rule = "trapezoid"})
and extended Simpson rule (\code{int_rule = "Simpson"}) are available.
Defaults to \code{"trapezoid"}.}

\item{cv_verbose}{flag to display information about the estimation procedure
(passed to \code{\link{cv_glmnet}}). Defaults to \code{FALSE}.}

\item{...}{further parameters to be passed to \code{\link{cv_glmnet}}
(and then to \code{\link[glmnet]{cv.glmnet}}) such as \code{cv_1se},
\code{cv_nlambda} or \code{cv_parallel}, among others.}
}
\value{
A list with the following entries:
\item{Beta_hat}{estimated \eqn{\beta}, a matrix with values
\eqn{\hat\beta (s, t)} evaluated at the grid points for \code{X}
and \code{Y}. Its size is \code{c(length(X$argvals), length(Y$argvals))}.}
\item{Beta_hat_scores}{the matrix of coefficients of \code{Beta_hat}
(resulting from projecting it into the tensor basis of \code{X_fpc} and
\code{Y_fpc}), with dimension \code{c(p_hat, q_thre)}.}
\item{H_hat}{hat matrix of the associated fitted multivariate
linear model, a matrix of size \code{c(n, n)}. \code{NULL} if
\code{est_method = "fpcr_l1"}, since lasso estimation does not provide it
explicitly.}
\item{p_thre}{index vector indicating the FPC of \code{X}
considered for estimating the model. Chosen by \code{thre_p} or equal
to \code{p} if given.}
\item{p_hat}{index vector of the FPC considered by the methods
\code{"fpcr_l1"} and \code{"fpcr_l1s"} methods after further selection
of the FPC considered in \code{p_thre}. For methods \code{"fpcr"} and
\code{"fpcr_l2"}, \code{p_hat} equals \code{p_thre}.}
\item{q_thre}{index vector indicating the FPC of \code{Y}
considered for estimating the model. Chosen by \code{thre_q} or equal
to \code{q} if given. Note that zeroing by lasso procedure only affects
in the rows.}
\item{est_method}{the estimation method employed.}
\item{Y_hat}{fitted values, either an \code{\link[fda.usc]{fdata}}
object or a vector, depending on \code{Y}.}
\item{Y_hat_scores}{the matrix of coefficients of \code{Y_hat}, with
dimension \code{c(n, q_thre)}.}
\item{residuals}{residuals of the fitted model, either an
\code{\link[fda.usc]{fdata}} object or a vector, depending on \code{Y}.}
\item{residuals_scores}{the matrix of coefficients of
\code{residuals}, with dimension \code{c(n, q_thre)}.}
\item{X_fpc, Y_fpc}{FPC of \code{X} and \code{Y}, as
returned by \code{\link{fpc}} with \code{n_fpc = n}.}
\item{lambda}{regularization parameter \eqn{\lambda} used for the
estimation methods \code{"fpcr_l2"}, \code{"fpcr_l1"}, and
\code{"fpcr_l1s"}.}
\item{cv}{cross-validation object returned by
\code{\link{cv_glmnet}}.}
}
\description{
Estimation of the linear operator relating a
functional predictor \eqn{X} with a functional response \eqn{Y} in the
linear model
\deqn{Y(t) = \int_a^b \beta(s, t) X(s)\,\mathrm{d}s + \varepsilon(t),}{
Y(t) = \int_a^b \beta(s, t) X(s) ds + \epsilon(t),}
where \eqn{X} is a random variable in the Hilbert space of
square-integrable functions in \eqn{[a, b]}, \eqn{L^2([a, b])},
\eqn{Y} and \eqn{\varepsilon}{\epsilon} are random variables
in \eqn{L^2([c, d])}, and \eqn{s \in [a, b]} and \eqn{t \in [c, d]}.

The linear, Hilbert--Schmidt, integral operator is parametrized by
the bivariate kernel \eqn{\beta \in L^2([a, b]) \otimes
L^2([c, d])}. Its estimation is done through the truncated expansion
of \eqn{\beta} in the tensor product of the data-driven
bases of the Functional Principal Components (FPC) of
\eqn{X} and \eqn{Y}, and through the fitting of the resulting multivariate
linear model. The FPC basis for \eqn{X} is truncated in \eqn{p}
components, while the FPC basis for \eqn{Y} is truncated in \eqn{q}
components. Automatic selection of \eqn{p} and \eqn{q} is detailed below.

The particular cases in which either \eqn{X} or \eqn{Y} are
\emph{constant} functions give either a scalar predictor or response.
The simple linear model arises if both \eqn{X} and \eqn{Y} are scalar,
for which \eqn{\beta} is a constant.
}
\details{
\code{flm_est} deals seamlessly with either functional or scalar inputs
for the predictor and response. In the case of scalar inputs, the
corresponding dimension-related arguments (\code{p}, \code{q},
\code{thre_p} or \code{thre_q}) will be ignored as in these cases either
\eqn{p = 1} or \eqn{q = 1}.

The function translates the functional linear model into a multivariate
model with multivariate response and then estimates the
\eqn{p \times q}{p x q} matrix of coefficients of \eqn{\beta} in the
tensor basis of the FPC of \code{X} and \code{Y}. The following estimation
methods are implemented:
\itemize{
  \item \code{"fpcr"}: Functional Principal Components Regression (FPCR);
  see details in Ramsay and Silverman (2005).
  \item \code{"fpcr_l2"}: FPCR, with ridge penalty on the associated
  multivariate linear model.
  \item \code{"fpcr_l1"}: FPCR, with lasso penalty on the associated
  multivariate linear model.
  \item \code{"fpcr_l1s"}: FPCR, with FPC selected by lasso regression
  on the associated multivariate linear model.
}
The last three methods are explained in García-Portugués et al. (2021).

The \eqn{p} FPC of \code{X} and \eqn{q} FPC of \code{Y} are determined
as follows:
\itemize{
  \item If \code{p = NULL}, then \code{p} is set as
  \code{p_thre <- 1:j_thre}, where \code{j_thre} is the \eqn{j}-th FPC of
  \code{X} for which the cumulated proportion of explained variance is
  greater than \code{thre_p}. If \code{p != NULL}, then \code{p_thre <- p}.
  \item If \code{q = NULL}, then the same procedure is followed with
  \code{thre_q}, resulting \code{q_thre}.
}
Once \code{p_thre} and \code{q_thre} have been obtained, the methods
\code{"fpcr_l1"} and \code{"fpcr_l1s"} perform a second selection
of the FPC that are effectively considered in the estimation of \eqn{\beta}.
This subset of FPC (of \code{p_thre}) is encoded in \code{p_hat}. No further
selection of FPC is done for the methods \code{"fpcr"} and \code{"fpcr_l2"}.

The flag \code{compute_residuals} controls if \code{Y_hat},
\code{Y_hat_scores}, \code{residuals}, and \code{residuals_scores} are
computed. If \code{FALSE}, they are set to \code{NULL}. \code{Y_hat} equals
\eqn{\hat Y_i(t) = \int_a^b \hat \beta(s, t) X_i(s) \,\mathrm{d}s}{
\hat Y_i(t) = \int_a^b \hat \beta(s, t) X_i(s) ds} and \code{residuals}
stands for \eqn{\hat \varepsilon_i(t) = Y_i(t) - \hat Y_i(t)}{
\hat \epsilon_i(t) = Y_i(t) - \hat Y_i(t)}, both for
\eqn{i = 1, \ldots, n}. \code{Y_hat_scores} and\cr \code{residuals_scores}
are the \eqn{n\times q}{n x q} matrices of coefficients (or scores) of these
functions in the FPC of \code{Y}.

Missing values on \code{X} and \code{Y} are automatically removed.
}
\examples{
## Quick example of functional response and functional predictor

# Generate data
set.seed(12345)
n <- 50
X_fdata <- r_ou(n = n, t = seq(0, 1, l = 201), sigma = 2)
epsilon <- r_ou(n = n, t = seq(0, 1, l = 201), sigma = 0.5)
Y_fdata <- 2 * X_fdata + epsilon

# Lasso-selection FPCR (p and q are estimated)
flm_est(X = X_fdata, Y = Y_fdata, est_method = "fpcr_l1s")
\donttest{
## Functional response and functional predictor

# Generate data
set.seed(12345)
n <- 50
X_fdata <- r_ou(n = n, t = seq(0, 1, l = 201), sigma = 2)
epsilon <- r_ou(n = n, t = seq(0, 1, l = 201), sigma = 0.5)
Y_fdata <- 2 * X_fdata + epsilon

# FPCR (p and q are estimated)
fpcr_1 <- flm_est(X = X_fdata, Y = Y_fdata, est_method = "fpcr")
fpcr_1$Beta_hat_scores
fpcr_1$p_thre
fpcr_1$q_thre

# FPCR (p and q are provided)
fpcr_2 <- flm_est(X = X_fdata, Y = Y_fdata, est_method = "fpcr",
                  p = c(1, 5, 2, 7), q = 2:1)
fpcr_2$Beta_hat_scores
fpcr_2$p_thre
fpcr_2$q_thre

# Ridge FPCR (p and q are estimated)
l2_1 <- flm_est(X = X_fdata, Y = Y_fdata, est_method = "fpcr_l2")
l2_1$Beta_hat_scores
l2_1$p_hat

# Ridge FPCR (p and q are provided)
l2_2 <- flm_est(X = X_fdata, Y = Y_fdata, est_method = "fpcr_l2",
                p = c(1, 5, 2, 7), q = 2:1)
l2_2$Beta_hat_scores
l2_2$p_hat

# Lasso FPCR (p and q are estimated)
l1_1 <- flm_est(X = X_fdata, Y = Y_fdata, est_method = "fpcr_l1")
l1_1$Beta_hat_scores
l1_1$p_thre
l1_1$p_hat

# Lasso estimator (p and q are provided)
l1_2 <- flm_est(X = X_fdata, Y = Y_fdata, est_method = "fpcr_l1",
                p = c(1, 5, 2, 7), q = 2:1)
l1_2$Beta_hat_scores
l1_2$p_thre
l1_2$p_hat

# Lasso-selection FPCR (p and q are estimated)
l1s_1 <- flm_est(X = X_fdata, Y = Y_fdata, est_method = "fpcr_l1s")
l1s_1$Beta_hat_scores
l1s_1$p_thre
l1s_1$p_hat

# Lasso-selection FPCR (p and q are provided)
l1s_2 <- flm_est(X = X_fdata, Y = Y_fdata, est_method = "fpcr_l1s",
                 p = c(1, 5, 2, 7), q = 1:4)
l1s_2$Beta_hat_scores
l1s_2$p_thre
l1s_2$p_hat

## Scalar response

# Generate data
set.seed(12345)
n <- 50
beta <- r_ou(n = 1, t = seq(0, 1, l = 201), sigma = 0.5, x0 = 3)
X_fdata <- fdata_cen(r_ou(n = n, t = seq(0, 1, l = 201), sigma = 2))
epsilon <- rnorm(n, sd = 0.25)
Y <- drop(inprod_fdata(X_fdata1 = X_fdata, X_fdata2 = beta)) + epsilon

# FPCR
fpcr_4 <- flm_est(X = X_fdata, Y = Y, est_method = "fpcr")
fpcr_4$p_hat

# Ridge FPCR
l2_4 <- flm_est(X = X_fdata, Y = Y, est_method = "fpcr_l2")
l2_4$p_hat

# Lasso FPCR
l1_4 <- flm_est(X = X_fdata, Y = Y, est_method = "fpcr_l1")
l1_4$p_hat

# Lasso-selection FPCR
l1s_4 <- flm_est(X = X_fdata, Y = Y, est_method = "fpcr_l1s")
l1s_4$p_hat

## Scalar predictor

# Generate data
set.seed(12345)
n <- 50
X <- rnorm(n)
epsilon <- r_ou(n = n, t = seq(0, 1, l = 201), sigma = 0.5)
beta <- r_ou(n = 1, t = seq(0, 1, l = 201), sigma = 0.5, x0 = 3)
beta$data <- matrix(beta$data, nrow = n, ncol = ncol(beta$data),
                    byrow = TRUE)
Y_fdata <- beta * X + epsilon

# FPCR
fpcr_4 <- flm_est(X = X, Y = Y_fdata, est_method = "fpcr")
plot(beta, col = 2)
lines(beta$argvals, drop(fpcr_4$Beta_hat))
}
}
\references{
García-Portugués, E., Álvarez-Liébana, J., Álvarez-Pérez, G. and
Gonzalez-Manteiga, W. (2021). A goodness-of-fit test for the functional
linear model with functional response. \emph{Scandinavian Journal of
Statistics}, 48(2):502--528. \doi{10.1111/sjos.12486}

Ramsay, J. and Silverman, B. W. (2005). \emph{Functional Data Analysis}.
Springer-Verlag, New York.
}
\author{
Eduardo García-Portugués and Javier Álvarez-Liébana.
}
