#' Plot quantile-quantile (QQ) graphs from residuals of linear models.
#'
#' This function takes a linear model (simple or mixed effects) and plots a QQ graph after running `rstudent` from \code{\link[stats]{rstudent}} to generate a table of Studentised model residuals on an ordinary (\code{\link{simple_model}}),  mixed model (\code{\link{mixed_model}} or \code{\link{mixed_model_slopes}}. The graph plots studentised residuals from the model (sample) on Y axis & Theoretical quantiles on X axis.
#' 
#' For generalised additive models fit with `mgcv`, scaled Pearson residuals are plotted.
#'
#' The function uses \code{\link{ggplot2}[geom_qq]} and \code{\link{ggplot2}[geom_qq_line]} geometries. Symbols receive "ok_orange" colour by default.
#'
#' @param Model name of a saved model generated by \code{simple_model} or \code{mixed_model}.
#' @param symsize size of symbols, default set to 3.
#' @param s_alpha fractional opacity of symbols, default set to 0.8 (i.e., 80% opacity).
#' @param fontsize parameter of \code{base_size} of fonts in \code{theme_classic}, default set to size 20.
#' @param symthick thickness of symbol border, default set to `fontsize`/22.
#' @param linethick thickness of line, default set to `fontsize`/22.
#' @param SingleColour colour of symbols (default = `#E69F00`, which is `ok_orange`)
#'
#' @return This function returns a \code{ggplot2} object of class "gg" and "ggplot".
#' @export plot_qqmodel
#' @import ggplot2
#' @importFrom stats rstudent
#' @importFrom mgcv residuals.gam
#'
#' @examples
#' #Basic usage
#' m1 <- simple_model(data = data_2w_Festing, 
#' Y_value = "GST", 
#' Fixed_Factor = c("Treatment", "Strain"))
#' plot_qqmodel(m1)
#'

plot_qqmodel <- function(Model, symsize = 3, s_alpha = 0.8, fontsize = 20, symthick, linethick, SingleColour = "#E69F00") {
  if(missing(symthick)) {symthick = fontsize/22}
  if(missing(linethick)) {linethick = fontsize/22}
  mod <- rstu <- NULL
  g <- inherits(Model, "gam", which = TRUE)
  l <- inherits(Model, c("lm"), which = TRUE)
  gl <- inherits(Model, c("glm"), which = TRUE)
  m <- inherits(Model, c("lmerModLmerTest", "merMod", "lme"))
  if (g == 1) {
    mod <- data.frame(rstu = mgcv::residuals.gam(Model, 
                                                 type = "pearson"))
    suppressWarnings(P <- ggplot2::ggplot(mod,
                                          aes(sample = rstu))+
                       geom_qq_line(na.rm = TRUE,
                                    size = linethick)+
                       geom_qq(na.rm = TRUE,
                               shape = 21, 
                               fill = SingleColour,
                               size = symsize,
                               stroke = symthick,
                               alpha = s_alpha)+
                       labs(x = "Theoretical quantile",
                            y = "Pearson residual")+
                       theme_classic(base_size = fontsize)+
                       theme(strip.background = element_blank()))
  } else {
    if (l == 1) {
      mod <- data.frame(rstu = stats::rstudent(Model))
    }
    if (gl == 1) {
      mod <- data.frame(rstu = stats::rstudent(Model))
    }
    if (m) {
      mod <- data.frame(rstu = stats::rstudent(Model))
    }
    suppressWarnings(P <- ggplot2::ggplot(mod,
                                          aes(sample = rstu))+
                       geom_qq_line(na.rm = TRUE,
                                    size = linethick)+
                       geom_qq(na.rm = TRUE,
                               shape = 21, 
                               fill = SingleColour,
                               size = symsize,
                               stroke = symthick,
                               alpha = s_alpha)+
                       labs(x = "Theoretical quantile",
                            y = "Studentised residual")+
                       theme_classic(base_size = fontsize)+
                       theme(strip.background = element_blank()))
  }
  P
}
