#' Load CRAN packages, and their dependencies, as current on given date 
#' 
#' Groundhog maintains a separate local package library where it stores 
#' version-controlled packages, with multiple versions of the same package saved side-by-side. 
#' The `date` argument in the `groundhog.library()` function determines
#' the version of the package that is loaded (the most recently published version on CRAN on that date).
#' If that version of the package is not available in the local `groundhog` library, 
#' it is automatically installed. `groundhog.library()` thus substitutes both `library()` 
#' and `install.packages()`. No changes to how R manages packages are made (e.g., no change to 
#' .libPaths(), to .Rprofile, or to R Studio global settings). 
#' Therefore, to discontinue relying on `groundhog` for package management, all you do is go back to 
#' executing the `install.packages()` and `library()` functions, instead of the `groundhog.library()` function.
#'  
#'@param pkg character string or vector with name of target package(s). Single package names need not be in quotes.
#'@param date character string  (yyyy-mm-dd), or date value, with the date which determines the 
#'version of the package, and all dependencies, to be loaded (and installed if needed). The most recent
#'date accepted is 2 days prior to when the code is executed.
#'@param quiet.install logical, defaults to `TRUE`. When set to `FALSE`, displays
#'  output generated by `install.packages()` when installing from source
#'@param include.suggests logical, defaults to `FALSE`. When set to `TRUE`, installs/
#'  loads dependencies classified in the DESCRIPTION file as `suggested`.
#'@param ignore.deps an optional character vector containing dependencies which 
#'   may be already loaded in the R session and even if the loaded version does not match
#'   the version implied by the entered date, groundhog.library() will proceed and ignore 
#'   this conflict. If one version of a package is loaded, and a different is needed for
#'   groundhog, the default behavior is to stop the request and ask the user to restart
#'   the R session to unload all packages. This will bypass that requirement.
#'@param force.source logical (defaults to `FALSE`). When set to `TRUE`, will not attempt 
#'   installing binary from CRAN or MRAN and instead download source file and install it.
#'@param force.install logical (defaults to `FALSE`). When set to `TRUE`, will delete  
#'   existing package files in groundhog folder, and install anew.
#'@param tolerate.R.version optional character string containing an R version which 
#' `groundhog.library()` will not throw an error for using, even if the date entered 
#' corresponds to a more recent major R release.  
#'@return a character vector containing all active packages for the session,
#'   with their version number, under the format `pkg_vrs`.
#'@examples
#' \dontrun{
#' groundhog.library("magrittr", "2020-07-12")
#'
#' pkgs <- c('pwr','metafor')
#' groundhog.library(pkgs, "2020-02-12")
#' 
#' #Allow using R 3.6.3 despite entering a date that corresponds to R 4.0
#' groundhog.library('rio', '2021-04-12', tolerate.R.version='3.6.3')
#' 
#' }
#'
#' @importFrom utils capture.output
#'
#' @details For more information about groundhog check out [groundhogr.com](http://groundhogr.com)
#' @export
#'
  groundhog.library <- function(pkg, date,  quiet.install = TRUE,  include.suggests = FALSE,  ignore.deps=c(), force.source = FALSE,
                              force.install = FALSE, tolerate.R.version="" )
    {
    
  #1) Validation     
    #1.1) Is date valid?
        validate.date(date) #Function defined in utils.R
    
    #1.2) Set of ignorable conflicts
        ignore.deps <- c(ignore.deps_default() , ignore.deps) #Add any ignore.deps explicitly stated to the default set in utils
        
    #1.3) put package name in quotes if it is not an object and was not put in quotes
        pkg.catch <- try(typeof(pkg),silent=TRUE)
        if (class(pkg.catch)=="try-error") {
          pkg <- as.character(substitute(pkg))
          message2()
          message1("     There is no object ",pkg, " in your environment, so will try loading package '",pkg,
                   "'.\n     To avoid seeing this message when using groundhog.library(), enter package name in quotes.\n\n")  
        } 
          
  #2 Loop running groundhog
        for (pkgk in pkg) {
              groundhog.library.single(pkgk, date, quiet.install ,  include.suggests ,  
                                        ignore.deps, force.source , force.install, tolerate.R.version)
                }
  

  } #End of groundhog.library
        