% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/substitution.R
\name{sub_values}
\alias{sub_values}
\title{Substitute targeted values in the table body}
\usage{
sub_values(
  data,
  columns = everything(),
  rows = everything(),
  values = NULL,
  pattern = NULL,
  fn = NULL,
  replacement = NULL,
  escape = TRUE
)
}
\arguments{
\item{data}{\emph{The gt table data object}

\verb{obj:<gt_tbl>} // \strong{required}

This is the \strong{gt} table object that is commonly created through use of the
\code{\link[=gt]{gt()}} function.}

\item{columns}{\emph{Columns to target}

\verb{<column-targeting expression>} // \emph{default:} \code{everything()}

The columns to which substitution operations are constrained. Can either
be a series of column names provided in \code{c()}, a vector of column indices,
or a select helper function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}},
\code{\link[=contains]{contains()}}, \code{\link[=matches]{matches()}}, \code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}).}

\item{rows}{\emph{Rows to target}

\verb{<row-targeting expression>} // \emph{default:} \code{everything()}

In conjunction with \code{columns}, we can specify which of their rows should
form a constraint for targeting operations. The default \code{\link[=everything]{everything()}}
results in all rows in \code{columns} being formatted. Alternatively, we can
supply a vector of row IDs within \code{c()}, a vector of row indices, or a
select helper function (e.g. \code{\link[=starts_with]{starts_with()}}, \code{\link[=ends_with]{ends_with()}}, \code{\link[=contains]{contains()}},
\code{\link[=matches]{matches()}}, \code{\link[=num_range]{num_range()}}, and \code{\link[=everything]{everything()}}. We can also use
expressions to filter down to the rows we need
(e.g., \verb{[colname_1] > 100 & [colname_2] < 50}).}

\item{values}{\emph{Values to match on}

\verb{scalar<character|numeric|integer>} // \emph{default:} \code{NULL} (\code{optional})

The specific value or values that should be replaced with a \code{replacement}
value. If \code{pattern} is also supplied then \code{values} will be ignored.}

\item{pattern}{\emph{Regex pattern to match with}

\verb{scalar<character>} // \emph{default:} \code{NULL} (\code{optional})

A regex pattern that can target solely those values in \code{character}-based
columns. If \code{values} is also supplied, \code{pattern} will take precedence.}

\item{fn}{\emph{Function to return logical values}

\verb{<function>} // \emph{default:} \code{NULL} (\code{optional})

A supplied function that operates on \code{x} (the data in a column) and should
return a logical vector that matches the length of \code{x} (i.e., number of
rows in the input table). If either of \code{values} or \code{pattern} is also
supplied, \code{fn} will take precedence.}

\item{replacement}{\emph{Replacement value for matches}

\verb{scalar<character|numeric|integer>} // \emph{default:} \code{NULL} (\code{optional})

The replacement value for any cell values matched by either \code{values} or
\code{pattern}. Must be a character or numeric vector of length 1.}

\item{escape}{\emph{Text escaping}

\verb{scalar<logical>} // \emph{default:} \code{TRUE}

An option to escape replacement text according to the final output format
of the table. For example, if a LaTeX table is to be generated then LaTeX
escaping would be performed on the replacements during rendering. By
default this is set to \code{TRUE} but setting to \code{FALSE} would be useful in the
case where replacement text is crafted for a specific output format in
mind.}
}
\value{
An object of class \code{gt_tbl}.
}
\description{
Should you need to replace specific cell values with custom text,
\code{sub_values()} can be good choice. We can target cells for replacement
through value, regex, and custom matching rules.
}
\section{Targeting cells with \code{columns} and \code{rows}}{


Targeting of values is done through \code{columns} and additionally by \code{rows} (if
nothing is provided for \code{rows} then entire columns are selected). The
\code{columns} argument allows us to target a subset of cells contained in the
resolved columns. We say resolved because aside from declaring column names
in \code{c()} (with bare column names or names in quotes) we can use
\strong{tidyselect}-style expressions. This can be as basic as supplying a select
helper like \code{starts_with()}, or, providing a more complex incantation like

\code{where(~ is.numeric(.x) && max(.x, na.rm = TRUE) > 1E6)}

which targets numeric columns that have a maximum value greater than
1,000,000 (excluding any \code{NA}s from consideration).

By default all columns and rows are selected (with the \code{everything()}
defaults). Cell values that are incompatible with a given substitution
function will be skipped over. So it's safe to select all columns with a
particular substitution function (only those values that can be substituted
will be), but, you may not want that. One strategy is to work on the bulk of
cell values with one substitution function and then constrain the columns for
later passes with other types of substitution (the last operation done to a
cell is what you get in the final output).

Once the columns are targeted, we may also target the \code{rows} within those
columns. This can be done in a variety of ways. If a stub is present, then we
potentially have row identifiers. Those can be used much like column names in
the \code{columns}-targeting scenario. We can use simpler \strong{tidyselect}-style
expressions (the select helpers should work well here) and we can use quoted
row identifiers in \code{c()}. It's also possible to use row indices (e.g., \code{c(3, 5, 6)}) though these index values must correspond to the row numbers of the
input data (the indices won't necessarily match those of rearranged rows if
row groups are present). One more type of expression is possible, an
expression that takes column values (can involve any of the available columns
in the table) and returns a logical vector. This is nice if you want to base
the substitution on values in the column or another column, or, you'd like to
use a more complex predicate expression.
}

\section{Examples}{


Let's create an input table with three columns. This contains an assortment
of values that could potentially undergo some substitution via
\code{sub_values()}.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl <-
  dplyr::tibble(
    num_1 = c(-0.01, 74, NA, 0, 500, 0.001, 84.3),
    int_1 = c(1L, -100000L, 800L, 5L, NA, 1L, -32L),
    lett = LETTERS[1:7]
  )

tbl
#> # A tibble: 7 x 3
#>     num_1   int_1 lett 
#>     <dbl>   <int> <chr>
#> 1  -0.01        1 A    
#> 2  74     -100000 B    
#> 3  NA         800 C    
#> 4   0           5 D    
#> 5 500          NA E    
#> 6   0.001       1 F    
#> 7  84.3       -32 G
}\if{html}{\out{</div>}}

Values in the table body cells can be replaced by specifying which values
should be replaced (in \code{values}) and what the replacement value should be.
It's okay to search for numerical or character values across all columns and
the replacement value can also be of the \code{numeric} or \code{character} types.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl |>
  gt() |>
  sub_values(values = c(74, 500), replacement = 150) |>
  sub_values(values = "B", replacement = "Bee") |>
  sub_values(values = 800, replacement = "Eight hundred")
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_sub_values_1.png" alt="This image of a table was generated from the first code example in the `sub_values()` help file." style="width:100\%;">
}}

We can also use the \code{pattern} argument to target cell values for replacement
in \code{character}-based columns.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl |>
  gt() |>
  sub_values(pattern = "A|C|E", replacement = "Ace")
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_sub_values_2.png" alt="This image of a table was generated from the second code example in the `sub_values()` help file." style="width:100\%;">
}}

For the most flexibility, it's best to use the \code{fn} argument. With that you
need to ensure that the function you provide will return a logical vector
when invoked on a column of cell values, taken as \code{x} (and, the length of
that vector must match the length of \code{x}).

\if{html}{\out{<div class="sourceCode r">}}\preformatted{tbl |>
  gt() |>
  sub_values(
    fn = function(x) x >= 0 & x < 50,
    replacement = "Between 0 and 50"
  )
}\if{html}{\out{</div>}}

\if{html}{\out{
<img src="https://raw.githubusercontent.com/rstudio/gt/master/images/man_sub_values_3.png" alt="This image of a table was generated from the third code example in the `sub_values()` help file." style="width:100\%;">
}}
}

\section{Function ID}{

3-35
}

\section{Function Introduced}{

\code{v0.8.0} (November 16, 2022)
}

\seealso{
Other data formatting functions: 
\code{\link{data_color}()},
\code{\link{fmt}()},
\code{\link{fmt_auto}()},
\code{\link{fmt_bins}()},
\code{\link{fmt_bytes}()},
\code{\link{fmt_chem}()},
\code{\link{fmt_country}()},
\code{\link{fmt_currency}()},
\code{\link{fmt_date}()},
\code{\link{fmt_datetime}()},
\code{\link{fmt_duration}()},
\code{\link{fmt_email}()},
\code{\link{fmt_engineering}()},
\code{\link{fmt_flag}()},
\code{\link{fmt_fraction}()},
\code{\link{fmt_icon}()},
\code{\link{fmt_image}()},
\code{\link{fmt_index}()},
\code{\link{fmt_integer}()},
\code{\link{fmt_markdown}()},
\code{\link{fmt_number}()},
\code{\link{fmt_partsper}()},
\code{\link{fmt_passthrough}()},
\code{\link{fmt_percent}()},
\code{\link{fmt_roman}()},
\code{\link{fmt_scientific}()},
\code{\link{fmt_spelled_num}()},
\code{\link{fmt_tf}()},
\code{\link{fmt_time}()},
\code{\link{fmt_units}()},
\code{\link{fmt_url}()},
\code{\link{sub_large_vals}()},
\code{\link{sub_missing}()},
\code{\link{sub_small_vals}()},
\code{\link{sub_zero}()}
}
\concept{data formatting functions}
