/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the COPYING file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#ifndef H5FDdevelop_H
#define H5FDdevelop_H
#include "H5FDpublic.h"
#define H5FD_CLASS_VERSION 0x01
#define H5FD_MEM_FHEAP_HDR H5FD_MEM_OHDR
#define H5FD_MEM_FHEAP_IBLOCK H5FD_MEM_OHDR
#define H5FD_MEM_FHEAP_DBLOCK H5FD_MEM_LHEAP
#define H5FD_MEM_FHEAP_HUGE_OBJ H5FD_MEM_DRAW
#define H5FD_MEM_FSPACE_HDR H5FD_MEM_OHDR
#define H5FD_MEM_FSPACE_SINFO H5FD_MEM_LHEAP
#define H5FD_MEM_SOHM_TABLE H5FD_MEM_OHDR
#define H5FD_MEM_SOHM_INDEX H5FD_MEM_BTREE
#define H5FD_MEM_EARRAY_HDR H5FD_MEM_OHDR
#define H5FD_MEM_EARRAY_IBLOCK H5FD_MEM_OHDR
#define H5FD_MEM_EARRAY_SBLOCK H5FD_MEM_BTREE
#define H5FD_MEM_EARRAY_DBLOCK H5FD_MEM_LHEAP
#define H5FD_MEM_EARRAY_DBLK_PAGE H5FD_MEM_LHEAP
#define H5FD_MEM_FARRAY_HDR H5FD_MEM_OHDR
#define H5FD_MEM_FARRAY_DBLOCK H5FD_MEM_LHEAP
#define H5FD_MEM_FARRAY_DBLK_PAGE H5FD_MEM_LHEAP
#define H5FD_FLMAP_SINGLE                                                                                    \
    {                                                                                                        \
        H5FD_MEM_SUPER,                                                                      \
            H5FD_MEM_SUPER,                                                                        \
            H5FD_MEM_SUPER,                                                                        \
            H5FD_MEM_SUPER,                                                                         \
            H5FD_MEM_SUPER,                                                                        \
            H5FD_MEM_SUPER,                                                                        \
            H5FD_MEM_SUPER                                                                         \
    }
#define H5FD_FLMAP_DICHOTOMY                                                                                 \
    {                                                                                                        \
        H5FD_MEM_SUPER,                                                                      \
            H5FD_MEM_SUPER,                                                                        \
            H5FD_MEM_SUPER,                                                                        \
            H5FD_MEM_DRAW,                                                                         \
            H5FD_MEM_DRAW,                                                                        \
            H5FD_MEM_SUPER,                                                                        \
            H5FD_MEM_SUPER                                                                         \
    }
#define H5FD_FLMAP_DEFAULT                                                                                   \
    {                                                                                                        \
        H5FD_MEM_DEFAULT,                                                                    \
            H5FD_MEM_DEFAULT,                                                                      \
            H5FD_MEM_DEFAULT,                                                                      \
            H5FD_MEM_DEFAULT,                                                                       \
            H5FD_MEM_DEFAULT,                                                                      \
            H5FD_MEM_DEFAULT,                                                                      \
            H5FD_MEM_DEFAULT                                                                       \
    }
typedef struct H5FD_t H5FD_t;
typedef struct H5FD_class_t {
    unsigned version;
    H5FD_class_value_t value;
    const char *name;
    haddr_t maxaddr;
    H5F_close_degree_t fc_degree;
    herr_t (*terminate)(void);
    hsize_t (*sb_size)(H5FD_t *file);
    herr_t (*sb_encode)(H5FD_t *file, char *name, unsigned char *p);
    herr_t (*sb_decode)(H5FD_t *f, const char *name, const unsigned char *p);
    size_t fapl_size;
    void *(*fapl_get)(H5FD_t *file);
    void *(*fapl_copy)(const void *fapl);
    herr_t (*fapl_free)(void *fapl);
    size_t dxpl_size;
    void *(*dxpl_copy)(const void *dxpl);
    herr_t (*dxpl_free)(void *dxpl);
    H5FD_t *(*open)(const char *name, unsigned flags, hid_t fapl, haddr_t maxaddr);
    herr_t (*close)(H5FD_t *file);
    int (*cmp)(const H5FD_t *f1, const H5FD_t *f2);
    herr_t (*query)(const H5FD_t *f1, unsigned long *flags);
    herr_t (*get_type_map)(const H5FD_t *file, H5FD_mem_t *type_map);
    haddr_t (*alloc)(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, hsize_t size);
    herr_t (*free)(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, hsize_t size);
    haddr_t (*get_eoa)(const H5FD_t *file, H5FD_mem_t type);
    herr_t (*set_eoa)(H5FD_t *file, H5FD_mem_t type, haddr_t addr);
    haddr_t (*get_eof)(const H5FD_t *file, H5FD_mem_t type);
    herr_t (*get_handle)(H5FD_t *file, hid_t fapl, void **file_handle);
    herr_t (*read)(H5FD_t *file, H5FD_mem_t type, hid_t dxpl, haddr_t addr, size_t size, void *buffer);
    herr_t (*write)(H5FD_t *file, H5FD_mem_t type, hid_t dxpl, haddr_t addr, size_t size, const void *buffer);
    herr_t (*read_vector)(H5FD_t *file, hid_t dxpl, uint32_t count, H5FD_mem_t types[], haddr_t addrs[],
                          size_t sizes[], void *bufs[]);
    herr_t (*write_vector)(H5FD_t *file, hid_t dxpl, uint32_t count, H5FD_mem_t types[], haddr_t addrs[],
                           size_t sizes[], const void *bufs[]);
    herr_t (*read_selection)(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, size_t count, hid_t mem_spaces[],
                             hid_t file_spaces[], haddr_t offsets[], size_t element_sizes[],
                             void *bufs[]);
    herr_t (*write_selection)(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, size_t count, hid_t mem_spaces[],
                              hid_t file_spaces[], haddr_t offsets[], size_t element_sizes[],
                              const void *bufs[]);
    herr_t (*flush)(H5FD_t *file, hid_t dxpl_id, hbool_t closing);
    herr_t (*truncate)(H5FD_t *file, hid_t dxpl_id, hbool_t closing);
    herr_t (*lock)(H5FD_t *file, hbool_t rw);
    herr_t (*unlock)(H5FD_t *file);
    herr_t (*del)(const char *name, hid_t fapl);
    herr_t (*ctl)(H5FD_t *file, uint64_t op_code, uint64_t flags, const void *input, void **output);
    H5FD_mem_t fl_map[H5FD_MEM_NTYPES];
} H5FD_class_t;
typedef struct H5FD_free_t {
    haddr_t             addr;
    hsize_t             size;
    struct H5FD_free_t *next;
} H5FD_free_t;
struct H5FD_t {
    hid_t               driver_id;
    const H5FD_class_t *cls;
    unsigned long       fileno;
    unsigned            access_flags;
    unsigned long       feature_flags;
    haddr_t             maxaddr;
    haddr_t             base_addr;
    hsize_t threshold;
    hsize_t alignment;
    hbool_t paged_aggr;
};
typedef hid_t (*H5FD_init_t)(void);
#ifdef __cplusplus
extern "C" {
#endif
H5_DLL hid_t   H5FDperform_init(H5FD_init_t op);
H5_DLL hid_t   H5FDregister(const H5FD_class_t *cls);
H5_DLL htri_t  H5FDis_driver_registered_by_name(const char *driver_name);
H5_DLL htri_t  H5FDis_driver_registered_by_value(H5FD_class_value_t driver_value);
H5_DLL herr_t  H5FDunregister(hid_t driver_id);
H5_DLL H5FD_t *H5FDopen(const char *name, unsigned flags, hid_t fapl_id, haddr_t maxaddr);
H5_DLL herr_t  H5FDclose(H5FD_t *file);
H5_DLL int     H5FDcmp(const H5FD_t *f1, const H5FD_t *f2);
H5_DLL herr_t  H5FDquery(const H5FD_t *f, unsigned long *flags);
H5_DLL haddr_t H5FDalloc(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, hsize_t size);
H5_DLL herr_t  H5FDfree(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, hsize_t size);
H5_DLL haddr_t H5FDget_eoa(H5FD_t *file, H5FD_mem_t type);
H5_DLL herr_t  H5FDset_eoa(H5FD_t *file, H5FD_mem_t type, haddr_t eoa);
H5_DLL haddr_t H5FDget_eof(H5FD_t *file, H5FD_mem_t type);
H5_DLL herr_t  H5FDget_vfd_handle(H5FD_t *file, hid_t fapl, void **file_handle);
H5_DLL herr_t  H5FDread(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, size_t size,
                        void *buf);
H5_DLL herr_t  H5FDwrite(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, haddr_t addr, size_t size,
                         const void *buf);
H5_DLL herr_t  H5FDread_vector(H5FD_t *file, hid_t dxpl_id, uint32_t count, H5FD_mem_t types[],
                               haddr_t addrs[], size_t sizes[], void *bufs[]);
H5_DLL herr_t  H5FDwrite_vector(H5FD_t *file, hid_t dxpl_id, uint32_t count, H5FD_mem_t types[],
                                haddr_t addrs[], size_t sizes[], const void *bufs[]);
H5_DLL herr_t  H5FDread_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count,
                                  hid_t mem_spaces[], hid_t file_spaces[], haddr_t offsets[],
                                  size_t element_sizes[], void *bufs[]);
H5_DLL herr_t  H5FDwrite_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count,
                                   hid_t mem_spaces[], hid_t file_spaces[], haddr_t offsets[],
                                   size_t element_sizes[], const void *bufs[]);
H5_DLL herr_t  H5FDread_vector_from_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count,
                                              hid_t mem_spaces[], hid_t file_spaces[], haddr_t offsets[],
                                              size_t element_sizes[], void *bufs[]);
H5_DLL herr_t  H5FDwrite_vector_from_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count,
                                               hid_t mem_spaces[], hid_t file_spaces[], haddr_t offsets[],
                                               size_t element_sizes[], const void *bufs[]);
H5_DLL herr_t  H5FDread_from_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count,
                                       hid_t mem_space_ids[], hid_t file_space_ids[], haddr_t offsets[],
                                       size_t element_sizes[], void *bufs[]);
H5_DLL herr_t  H5FDwrite_from_selection(H5FD_t *file, H5FD_mem_t type, hid_t dxpl_id, uint32_t count,
                                        hid_t mem_space_ids[], hid_t file_space_ids[], haddr_t offsets[],
                                        size_t element_sizes[], const void *bufs[]);
H5_DLL herr_t  H5FDflush(H5FD_t *file, hid_t dxpl_id, hbool_t closing);
H5_DLL herr_t  H5FDtruncate(H5FD_t *file, hid_t dxpl_id, hbool_t closing);
H5_DLL herr_t  H5FDlock(H5FD_t *file, hbool_t rw);
H5_DLL herr_t  H5FDunlock(H5FD_t *file);
H5_DLL herr_t  H5FDdelete(const char *name, hid_t fapl_id);
H5_DLL herr_t  H5FDctl(H5FD_t *file, uint64_t op_code, uint64_t flags, const void *input, void **output);
#ifdef __cplusplus
}
#endif
#endif
