/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5EAmodule.h" 
#define H5EA_TESTING

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5EApkg.h"     
#include "H5FLprivate.h" 
#include "H5VMprivate.h" 

#define H5EA__TEST_BOGUS_VAL 42

typedef struct H5EA__test_ctx_t {
    uint32_t        bogus; 
    H5EA__ctx_cb_t *cb;    
} H5EA__test_ctx_t;

static void  *H5EA__test_crt_context(void *udata);
static herr_t H5EA__test_dst_context(void *ctx);
static herr_t H5EA__test_fill(void *nat_blk, size_t nelmts);
static herr_t H5EA__test_encode(void *raw, const void *elmt, size_t nelmts, void *ctx);
static herr_t H5EA__test_decode(const void *raw, void *elmt, size_t nelmts, void *ctx);
static herr_t H5EA__test_debug(FILE *stream, int indent, int fwidth, hsize_t idx, const void *elmt);
static void  *H5EA__test_crt_dbg_context(H5F_t H5_ATTR_UNUSED *f, haddr_t H5_ATTR_UNUSED obj_addr);
static herr_t H5EA__test_dst_dbg_context(void *_ctx);

const H5EA_class_t H5EA_CLS_TEST[1] = {{
    H5EA_CLS_TEST_ID,           
    "Testing",                  
    sizeof(uint64_t),           
    H5EA__test_crt_context,     
    H5EA__test_dst_context,     
    H5EA__test_fill,            
    H5EA__test_encode,          
    H5EA__test_decode,          
    H5EA__test_debug,           
    H5EA__test_crt_dbg_context, 
    H5EA__test_dst_dbg_context  
}};

H5FL_DEFINE_STATIC(H5EA__test_ctx_t);

H5FL_DEFINE_STATIC(H5EA__ctx_cb_t);

static void *
H5EA__test_crt_context(void *_udata)
{
    H5EA__test_ctx_t *ctx;                                  
    H5EA__ctx_cb_t   *udata     = (H5EA__ctx_cb_t *)_udata; 
    void             *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    if (NULL == (ctx = H5FL_MALLOC(H5EA__test_ctx_t)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "can't allocate extensible array client callback context");

    
    ctx->bogus = H5EA__TEST_BOGUS_VAL;
    ctx->cb    = udata;

    
    ret_value = ctx;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__test_dst_context(void *_ctx)
{
    H5EA__test_ctx_t *ctx = (H5EA__test_ctx_t *)_ctx; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(H5EA__TEST_BOGUS_VAL == ctx->bogus);

    
    ctx = H5FL_FREE(H5EA__test_ctx_t, ctx);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__test_fill(void *nat_blk, size_t nelmts)
{
    uint64_t fill_val = H5EA_TEST_FILL; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(nat_blk);
    assert(nelmts);

    H5VM_array_fill(nat_blk, &fill_val, sizeof(uint64_t), nelmts);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__test_encode(void *raw, const void *_elmt, size_t nelmts, void *_ctx)
{
    H5EA__test_ctx_t *ctx       = (H5EA__test_ctx_t *)_ctx; 
    const uint64_t   *elmt      = (const uint64_t *)_elmt;  
    herr_t            ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(raw);
    assert(elmt);
    assert(nelmts);
    assert(H5EA__TEST_BOGUS_VAL == ctx->bogus);

    
    if (ctx->cb) {
        if ((*ctx->cb->encode)(elmt, nelmts, ctx->cb->udata) < 0)
            HGOTO_ERROR(H5E_EARRAY, H5E_BADVALUE, FAIL, "extensible array testing callback action failed");
    }

    
    while (nelmts) {
        
        UINT64ENCODE(raw, *elmt);

        
        elmt++;

        
        nelmts--;
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__test_decode(const void *_raw, void *_elmt, size_t nelmts, void H5_ATTR_NDEBUG_UNUSED *_ctx)
{
#ifndef NDEBUG
    H5EA__test_ctx_t *ctx = (H5EA__test_ctx_t *)_ctx; 
#endif                                                
    uint64_t      *elmt = (uint64_t *)_elmt;          
    const uint8_t *raw  = (const uint8_t *)_raw;      

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(raw);
    assert(elmt);
    assert(nelmts);
    assert(H5EA__TEST_BOGUS_VAL == ctx->bogus);

    
    while (nelmts) {
        
        UINT64DECODE(raw, *elmt);

        
        elmt++;

        
        nelmts--;
    }

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5EA__test_debug(FILE *stream, int indent, int fwidth, hsize_t idx, const void *elmt)
{
    char temp_str[128]; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(stream);
    assert(elmt);

    
    snprintf(temp_str, sizeof(temp_str), "Element #%llu:", (unsigned long long)idx);
    Rfprintf(stream, "%*s%-*s %llu\n", indent, "", fwidth, temp_str,
            (unsigned long long)*(const uint64_t *)elmt);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5EA__test_crt_dbg_context(H5F_t H5_ATTR_UNUSED *f, haddr_t H5_ATTR_UNUSED obj_addr)
{
    H5EA__ctx_cb_t *ctx; 
    void           *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    
    if (NULL == (ctx = H5FL_MALLOC(H5EA__ctx_cb_t)))
        HGOTO_ERROR(H5E_EARRAY, H5E_CANTALLOC, NULL,
                    "can't allocate extensible array client callback context");

    
    ret_value = ctx;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5EA__test_dst_dbg_context(void *_ctx)
{
    H5EA__ctx_cb_t *ctx = (H5EA__ctx_cb_t *)_ctx; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(_ctx);

    
    ctx = H5FL_FREE(H5EA__ctx_cb_t, ctx);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5EA__get_cparam_test(const H5EA_t *ea, H5EA_create_t *cparam)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ea);
    assert(cparam);

    
    cparam->raw_elmt_size             = ea->hdr->cparam.raw_elmt_size;
    cparam->max_nelmts_bits           = ea->hdr->cparam.max_nelmts_bits;
    cparam->idx_blk_elmts             = ea->hdr->cparam.idx_blk_elmts;
    cparam->sup_blk_min_data_ptrs     = ea->hdr->cparam.sup_blk_min_data_ptrs;
    cparam->data_blk_min_elmts        = ea->hdr->cparam.data_blk_min_elmts;
    cparam->max_dblk_page_nelmts_bits = ea->hdr->cparam.max_dblk_page_nelmts_bits;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

int
H5EA__cmp_cparam_test(const H5EA_create_t *cparam1, const H5EA_create_t *cparam2)
{
    int ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(cparam1);
    assert(cparam2);

    
    if (cparam1->raw_elmt_size < cparam2->raw_elmt_size)
        HGOTO_DONE(-1);
    else if (cparam1->raw_elmt_size > cparam2->raw_elmt_size)
        HGOTO_DONE(1);

    if (cparam1->max_nelmts_bits < cparam2->max_nelmts_bits)
        HGOTO_DONE(-1);
    else if (cparam1->max_nelmts_bits > cparam2->max_nelmts_bits)
        HGOTO_DONE(1);

    if (cparam1->idx_blk_elmts < cparam2->idx_blk_elmts)
        HGOTO_DONE(-1);
    else if (cparam1->idx_blk_elmts > cparam2->idx_blk_elmts)
        HGOTO_DONE(1);

    if (cparam1->sup_blk_min_data_ptrs < cparam2->sup_blk_min_data_ptrs)
        HGOTO_DONE(-1);
    else if (cparam1->sup_blk_min_data_ptrs > cparam2->sup_blk_min_data_ptrs)
        HGOTO_DONE(1);

    if (cparam1->data_blk_min_elmts < cparam2->data_blk_min_elmts)
        HGOTO_DONE(-1);
    else if (cparam1->data_blk_min_elmts > cparam2->data_blk_min_elmts)
        HGOTO_DONE(1);

    if (cparam1->max_dblk_page_nelmts_bits < cparam2->max_dblk_page_nelmts_bits)
        HGOTO_DONE(-1);
    else if (cparam1->max_dblk_page_nelmts_bits > cparam2->max_dblk_page_nelmts_bits)
        HGOTO_DONE(1);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
