/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5HFpkg.h"     
#include "H5MFprivate.h" 
#include "H5VMprivate.h" 

H5FL_DEFINE(H5HF_direct_t);

herr_t
H5HF__man_dblock_create(H5HF_hdr_t *hdr, H5HF_indirect_t *par_iblock, unsigned par_entry, haddr_t *addr_p,
                        H5HF_free_section_t **ret_sec_node)
{
    H5HF_free_section_t *sec_node;            
    H5HF_direct_t       *dblock = NULL;       
    haddr_t              dblock_addr;         
    size_t               free_space;          
    herr_t               ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);

    
    if (NULL == (dblock = H5FL_MALLOC(H5HF_direct_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                    "memory allocation failed for fractal heap direct block");

    
    memset(&dblock->cache_info, 0, sizeof(H5AC_info_t));

    
    dblock->hdr = hdr;
    if (H5HF__hdr_incr(hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL, "can't increment reference count on shared heap header");

    
    if (par_iblock) {
        unsigned par_row = par_entry / hdr->man_dtable.cparam.width; 

        
        dblock->block_off = par_iblock->block_off;
        dblock->block_off += hdr->man_dtable.row_block_off[par_row];
        dblock->block_off +=
            hdr->man_dtable.row_block_size[par_row] * (par_entry % hdr->man_dtable.cparam.width);
        H5_CHECKED_ASSIGN(dblock->size, size_t, hdr->man_dtable.row_block_size[par_row], hsize_t);
    } 
    else {
        
        dblock->block_off = 0;
        dblock->size      = hdr->man_dtable.cparam.start_block_size;
    } 
    dblock->file_size = 0;
    free_space        = dblock->size - H5HF_MAN_ABS_DIRECT_OVERHEAD(hdr);

    
    
    if ((dblock->blk = H5FL_BLK_MALLOC(direct_block, dblock->size)) == NULL)
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "memory allocation failed");
    memset(dblock->blk, 0, dblock->size);

    dblock->write_buf  = NULL;
    dblock->write_size = 0;

    
    if (H5F_USE_TMP_SPACE(hdr->f)) {
        if (HADDR_UNDEF == (dblock_addr = H5MF_alloc_tmp(hdr->f, (hsize_t)dblock->size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                        "file allocation failed for fractal heap direct block");
    } 
    else {
        if (HADDR_UNDEF == (dblock_addr = H5MF_alloc(hdr->f, H5FD_MEM_FHEAP_DBLOCK, (hsize_t)dblock->size)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                        "file allocation failed for fractal heap direct block");
    } 

    
    dblock->parent = par_iblock;
    if (dblock->parent) {
        if (H5HF__man_iblock_attach(dblock->parent, par_entry, dblock_addr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTATTACH, FAIL, "can't attach direct block to parent indirect block");
        dblock->fd_parent = par_iblock;
    } 
    else
        dblock->fd_parent = hdr;
    dblock->par_entry = par_entry;

    
    if (NULL == (sec_node = H5HF__sect_single_new((dblock->block_off + H5HF_MAN_ABS_DIRECT_OVERHEAD(hdr)),
                                                  free_space, dblock->parent, dblock->par_entry)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't create section for new direct block's free space");

    
    if (ret_sec_node)
        
        *ret_sec_node = sec_node;
    else {
        
        if (H5HF__space_add(hdr, sec_node, 0) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't add direct block free space to global list");
    } 

    
    if (H5AC_insert_entry(hdr->f, H5AC_FHEAP_DBLOCK, dblock_addr, dblock, H5AC__NO_FLAGS_SET) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTINIT, FAIL, "can't add fractal heap direct block to cache");

    
    if (H5HF__hdr_inc_alloc(hdr, dblock->size) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't increase allocated heap size");

    
    if (addr_p)
        *addr_p = dblock_addr;

done:
    if (ret_value < 0)
        if (dblock)
            if (H5HF__man_dblock_dest(dblock) < 0)
                HDONE_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to destroy fractal heap direct block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_dblock_destroy(H5HF_hdr_t *hdr, H5HF_direct_t *dblock, haddr_t dblock_addr, bool *parent_removed)
{
    hsize_t  dblock_size;                      
    unsigned cache_flags = H5AC__NO_FLAGS_SET; 
    herr_t   ret_value   = SUCCEED;            

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(dblock);

    
    if (hdr->filter_len > 0) {
        
        if (dblock->parent == NULL)
            
            dblock_size = (hsize_t)hdr->pline_root_direct_size;
        else {
            H5HF_indirect_t *par_iblock; 
            unsigned         par_entry;  

            
            par_iblock = dblock->parent;
            par_entry  = dblock->par_entry;

            
            dblock_size = (hsize_t)par_iblock->filt_ents[par_entry].size;
        } 
    }     
    else
        dblock_size = (hsize_t)dblock->size;

    
    if (parent_removed)
        *parent_removed = false;

    
    if (hdr->man_dtable.curr_root_rows == 0) {
        
        assert(hdr->man_dtable.table_addr == dblock_addr);
        assert(hdr->man_dtable.cparam.start_block_size == dblock->size);

        
        assert(!H5HF__man_iter_ready(&hdr->next_block));

        
        if (H5HF__hdr_empty(hdr) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTSHRINK, FAIL, "can't make heap empty");
    } 
    else {
        
        hdr->man_alloc_size -= dblock->size;

        
        if ((dblock->block_off + dblock->size) == hdr->man_iter_off)
            
            if (H5HF__hdr_reverse_iter(hdr, dblock_addr) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTRELEASE, FAIL, "can't reverse 'next block' iterator");

        
        if (dblock->parent) {
            
            if (H5AC_destroy_flush_dependency(dblock->fd_parent, dblock) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTUNDEPEND, FAIL, "unable to destroy flush dependency");
            dblock->fd_parent = NULL;

            
            if (parent_removed && 1 == dblock->parent->nchildren)
                *parent_removed = true;

            if (H5HF__man_iblock_detach(dblock->parent, dblock->par_entry) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTATTACH, FAIL, "can't detach from parent indirect block");
            dblock->parent    = NULL;
            dblock->par_entry = 0;
        } 
    }     

    
    dblock->file_size = dblock_size;
    cache_flags |= H5AC__DIRTIED_FLAG | H5AC__DELETED_FLAG;

    
    if (!H5F_IS_TMP_ADDR(hdr->f, dblock_addr))
        cache_flags |= H5AC__FREE_FILE_SPACE_FLAG;

done:
    
    if (H5AC_unprotect(hdr->f, H5AC_FHEAP_DBLOCK, dblock_addr, dblock, cache_flags) < 0)
        HDONE_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap direct block");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_dblock_new(H5HF_hdr_t *hdr, size_t request, H5HF_free_section_t **ret_sec_node)
{
    haddr_t dblock_addr;         
    size_t  min_dblock_size;     
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(request > 0);

    
    if (request < hdr->man_dtable.cparam.start_block_size)
        min_dblock_size = hdr->man_dtable.cparam.start_block_size;
    else {
        min_dblock_size = ((size_t)1) << (1 + H5VM_log2_gen((uint64_t)request));
        assert(min_dblock_size <= hdr->man_dtable.cparam.max_direct_size);
    } 

    
    if (min_dblock_size < H5HF_MAN_ABS_DIRECT_OVERHEAD(hdr) + request)
        min_dblock_size *= 2;

    
    if (!H5_addr_defined(hdr->man_dtable.table_addr) &&
        min_dblock_size == hdr->man_dtable.cparam.start_block_size) {
        
        if (H5HF__man_dblock_create(hdr, NULL, 0, &dblock_addr, ret_sec_node) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "can't allocate fractal heap direct block");

        
        hdr->man_dtable.curr_root_rows = 0;
        hdr->man_dtable.table_addr     = dblock_addr;
        if (hdr->filter_len > 0) {
            hdr->pline_root_direct_size        = hdr->man_dtable.cparam.start_block_size;
            hdr->pline_root_direct_filter_mask = 0;
        } 

        
        if (H5HF__hdr_adjust_heap(hdr, (hsize_t)hdr->man_dtable.cparam.start_block_size,
                                  (hssize_t)hdr->man_dtable.row_tot_dblock_free[0]) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTEXTEND, FAIL, "can't increase space to cover root direct block");
    } 
    
    else {
        H5HF_indirect_t *iblock;     
        unsigned         next_row;   
        unsigned         next_entry; 
        size_t           next_size;  

        
        if (H5HF__hdr_update_iter(hdr, min_dblock_size) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTUPDATE, FAIL, "unable to update block iterator");

        
        if (H5HF__man_iter_curr(&hdr->next_block, &next_row, NULL, &next_entry, &iblock) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "unable to retrieve current block iterator location");
        assert(next_row < iblock->nrows);
        H5_CHECKED_ASSIGN(next_size, size_t, hdr->man_dtable.row_block_size[next_row], hsize_t);

        
        if (min_dblock_size > next_size) {
            Rfprintf(Rstderr,
                    "%s: Skipping direct block sizes not supported, min_dblock_size = %llu, next_size = %llu\n",
                    __func__, (unsigned long long)min_dblock_size, (unsigned long long)next_size);
            HGOTO_ERROR(H5E_HEAP, H5E_UNSUPPORTED, FAIL, "skipping direct block sizes not supported yet");
        } 

        
        if (H5HF__hdr_inc_iter(hdr, (hsize_t)next_size, 1) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTINC, FAIL, "can't increment 'next block' iterator");

        
        if (H5HF__man_dblock_create(hdr, iblock, next_entry, &dblock_addr, ret_sec_node) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "can't allocate fractal heap direct block");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

H5HF_direct_t *
H5HF__man_dblock_protect(H5HF_hdr_t *hdr, haddr_t dblock_addr, size_t dblock_size,
                         H5HF_indirect_t *par_iblock, unsigned par_entry, unsigned flags)
{
    H5HF_direct_t         *dblock;           
    H5HF_dblock_cache_ud_t udata;            
    H5HF_direct_t         *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(H5_addr_defined(dblock_addr));
    assert(dblock_size > 0);

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    udata.par_info.hdr    = hdr;
    udata.par_info.iblock = par_iblock;
    udata.par_info.entry  = par_entry;

    
    udata.f = hdr->f;

    
    udata.dblock_size = dblock_size;

    
    if (hdr->filter_len > 0) {
        if (par_iblock == NULL) {
            udata.odi_size    = hdr->pline_root_direct_size;
            udata.filter_mask = hdr->pline_root_direct_filter_mask;
        } 
        else {
            
            assert(H5_addr_eq(par_iblock->ents[par_entry].addr, dblock_addr));

            
            udata.odi_size    = par_iblock->filt_ents[par_entry].size;
            udata.filter_mask = par_iblock->filt_ents[par_entry].filter_mask;
        } 
    }     
    else {
        udata.odi_size    = dblock_size;
        udata.filter_mask = 0;
    } 

    
    udata.decompressed = false;
    udata.dblk         = NULL;

    
    if (NULL ==
        (dblock = (H5HF_direct_t *)H5AC_protect(hdr->f, H5AC_FHEAP_DBLOCK, dblock_addr, &udata, flags)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, NULL, "unable to protect fractal heap direct block");

    
    ret_value = dblock;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_dblock_locate(H5HF_hdr_t *hdr, hsize_t obj_off, H5HF_indirect_t **ret_iblock, unsigned *ret_entry,
                        bool *ret_did_protect, unsigned flags)
{
    haddr_t          iblock_addr;         
    H5HF_indirect_t *iblock;              
    bool             did_protect;         
    unsigned         row, col;            
    unsigned         entry;               
    herr_t           ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(hdr);
    assert(hdr->man_dtable.curr_root_rows); 
    assert(ret_iblock);
    assert(ret_did_protect);

    
    assert((flags & (unsigned)(~H5AC__READ_ONLY_FLAG)) == 0);

    
    if (H5HF__dtable_lookup(&hdr->man_dtable, obj_off, &row, &col) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTCOMPUTE, FAIL, "can't compute row & column of object");

    
    iblock_addr = hdr->man_dtable.table_addr;

    
    if (NULL == (iblock = H5HF__man_iblock_protect(hdr, iblock_addr, hdr->man_dtable.curr_root_rows, NULL, 0,
                                                   false, flags, &did_protect)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap indirect block");

    
    while (row >= hdr->man_dtable.max_direct_rows) {
        H5HF_indirect_t *new_iblock;      
        bool             new_did_protect; 
        unsigned         nrows;           
        unsigned         cache_flags = H5AC__NO_FLAGS_SET; 

        
        nrows = (H5VM_log2_gen(hdr->man_dtable.row_block_size[row]) - hdr->man_dtable.first_row_bits) + 1;
        assert(nrows < iblock->nrows); 

        
        entry = (row * hdr->man_dtable.cparam.width) + col;

        
        iblock_addr = iblock->ents[entry].addr;

        
        if (!H5_addr_defined(iblock_addr)) {
            if (H5HF__man_iblock_create(hdr, iblock, entry, nrows, nrows, &iblock_addr) < 0)
                HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, FAIL, "can't allocate fractal heap indirect block");

            
            cache_flags |= H5AC__DIRTIED_FLAG;
        } 

        
        if (NULL == (new_iblock = H5HF__man_iblock_protect(hdr, iblock_addr, nrows, iblock, entry, false,
                                                           flags, &new_did_protect)))
            HGOTO_ERROR(H5E_HEAP, H5E_CANTPROTECT, FAIL, "unable to protect fractal heap indirect block");

        
        if (H5HF__man_iblock_unprotect(iblock, cache_flags, did_protect) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTUNPROTECT, FAIL, "unable to release fractal heap indirect block");

        
        iblock      = new_iblock;
        did_protect = new_did_protect;

        
        if (H5HF__dtable_lookup(&hdr->man_dtable, (obj_off - iblock->block_off), &row, &col) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTCOMPUTE, FAIL, "can't compute row & column of object");
        assert(row < iblock->nrows); 
    }                                

    
    if (ret_entry)
        *ret_entry = (row * hdr->man_dtable.cparam.width) + col;
    *ret_iblock      = iblock;
    *ret_did_protect = did_protect;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_dblock_delete(H5F_t *f, haddr_t dblock_addr, hsize_t dblock_size)
{
    unsigned dblock_status = 0;       
    herr_t   ret_value     = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(dblock_addr));
    assert(dblock_size > 0);

    
    if (H5AC_get_entry_status(f, dblock_addr, &dblock_status) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTGET, FAIL, "unable to check metadata cache status for direct block");

    
    if (dblock_status & H5AC_ES__IN_CACHE) {
        
        assert(!(dblock_status & H5AC_ES__IS_PINNED));
        assert(!(dblock_status & H5AC_ES__IS_PROTECTED));

        
        if (H5AC_expunge_entry(f, H5AC_FHEAP_DBLOCK, dblock_addr, H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTREMOVE, FAIL, "unable to remove direct block from cache");
    } 

    
    
    if (!H5F_IS_TMP_ADDR(f, dblock_addr))
        
        
        if (H5MF_xfree(f, H5FD_MEM_FHEAP_DBLOCK, dblock_addr, dblock_size) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to free fractal heap direct block file space");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__man_dblock_dest(H5HF_direct_t *dblock)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dblock);

    
    assert(dblock->hdr != NULL);
    if (H5HF__hdr_decr(dblock->hdr) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL, "can't decrement reference count on shared heap header");
    if (dblock->parent)
        if (H5HF__iblock_decr(dblock->parent) < 0)
            HGOTO_ERROR(H5E_HEAP, H5E_CANTDEC, FAIL,
                        "can't decrement reference count on shared indirect block");

    
    dblock->blk = H5FL_BLK_FREE(direct_block, dblock->blk);

    
    dblock = H5FL_FREE(H5HF_direct_t, dblock);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
