#' Run one or more Markov Model
#' 
#' Runs one or more unevaluated Markov Models. When more 
#' than one model is provided, all models should have the 
#' same states and state value names.
#' 
#' 
#' A usual situation where more than one model needs to be 
#' run is when comparing different care startegies.
#' 
#' In order to compute comparisons Markov Models must be 
#' similar (same states and state value names). Thus models 
#' should only differ through parameters, transition matrix 
#' cell values and values attached to states (but not state 
#' value names).
#' 
#' The initial number of individuals in each state and the 
#' number of cycle will be the same for all models.
#' 
#' Internally this function does 2 operations: first 
#' evaluating parameters, transition matrix, state values 
#' and computing individual counts through 
#' \code{\link{eval_model}}; and then using individual 
#' counts and evaluated state values to compute values at 
#' each cycle through \code{compute_values}.
#' 
#' @param ... One or more \code{uneval_model} object.
#' @param parameters Optional. An object generated by 
#'   \code{\link{define_parameters}}.
#' @param init numeric vector, same length as number of 
#'   model states. Number of individuals in each model state
#'   at the beginning.
#' @param cycles positive integer. Number of Markov Cycles 
#'   to compute.
#' @param cost Names or expression to compute cost on the
#'   cost-effectiveness plane.
#' @param effect Names or expression to compute effect on
#'   the cost-effectiveness plane.
#' @param base_model Name of base model used as reference.
#'   By default the model with the lowest effectiveness.
#' @param method Counting method.
#' @param list_models List of models, only used by 
#'   \code{run_models_} to avoid using \code{...}.
#'   
#' @return A list of evaluated models with computed values.
#' @export
#' 
#' @example inst/examples/example_run_models.R
#'   
run_models <- function(...,
                       parameters = define_parameters(),
                       init = c(1000L, rep(0L, get_state_number(get_states(list(...)[[1]])) - 1)),
                       cycles = 1,
                       method = c("beginning", "end",
                                  "half-cycle", "life-table"),
                       cost, effect, base_model = NULL) {
  list_models <- list(...)
  
  method <- match.arg(method)
  
  run_models_(
    list_models = list_models,
    parameters = parameters,
    init = init,
    cycles = cycles,
    method = method,
    cost = lazyeval::lazy(cost),
    effect = lazyeval::lazy(effect),
    base_model = base_model
  )
}

#' @export
#' @rdname run_models
run_models_ <- function(list_models,
                        parameters,
                        init,
                        cycles,
                        method,
                        cost, effect, base_model) {
  
  stopifnot(
    all(unlist(lapply(list_models,
                      function(x) "uneval_model" %in% class(x)))),
    ! missing(cost),
    ! missing(effect)
  )
  
  list_ce <- list(
    cost,
    effect
  )
  names(list_ce) <- c(".cost", ".effect")
  ce <- c(
    lazyeval::lazy_dots(),
    list_ce
  )
  
  model_names <- names(list_models)
  
  if (is.null(model_names)) {
    message("No named model -> generating names.")
    model_names <- as.character(utils::as.roman(seq_along(list_models)))
    names(list_models) <- model_names
  }
  
  if (any(model_names == "")) {
    warning("Not all models are named -> generating names.")
    model_names <- as.character(utils::as.roman(seq_along(list_models)))
    names(list_models) <- model_names
  }
  
  stopifnot(
    all("uneval_model" %in% unlist(lapply(list_models, class))),
    list_all_same(lapply(list_models,
                         function(x) sort(get_state_names(x)))),
    list_all_same(lapply(list_models,
                         function(x) sort(get_state_value_names(x))))
  )
  
  stopifnot(
    length(init) == get_state_number(list_models[[1]]),
    all(init >= 0)
  )
  
  if (is.null(names(init)))
    names(init) <- get_state_names(list_models[[1]])
  
  stopifnot(
    all(sort(names(init)) == sort(get_state_names(list_models[[1]])))
  )
  
  eval_model_list <- lapply(list_models, eval_model, 
                            parameters = parameters,
                            init = init, 
                            cycles = cycles,
                            method = method)
  
  list_res <- lapply(eval_model_list, get_total_state_values)
  
  for (n in model_names){
    list_res[[n]]$.model_names <- n
  }
  
  res <- Reduce(dplyr::bind_rows, list_res)
  
  res <- dplyr::mutate_(res, .dots = ce)
  
  if (is.null(base_model)) {
    base_model <- get_base_model(res)
  }
  
  structure(
    res,
    eval_model_list = eval_model_list,
    uneval_model_list = list_models,
    class = c("eval_model_list", class(res)),
    parameters = parameters,
    init = init,
    cycles = cycles,
    method = method,
    ce = ce,
    base_model = base_model
  )
}

#' Get Markov Model Parameters
#' 
#' 
#' For internal use.
#' 
#' @param x An \code{eval_model_list}
#'   object.
#'   
#' @return An \code{uneval_parameters} or
#'   \code{eval_parameters} object.
get_parameters <- function(x){
  UseMethod("get_parameters")
}

get_parameters.default <- function(x){
  attr(x, "parameters")
}

get_total_state_values <- function(x) {
  # faster than as.data.frame or dplyr::as_data_frame
  res <- as.list(colSums((x$values)[- 1]))
  class(res) <- "data.frame"
  attr(res, "row.names") <- c(NA, -1)
  res
}

get_base_model <- function(x, ...) {
  UseMethod("get_base_model")
}

get_base_model.default <- function(x, ...) {
  x$.model_names[which(x$.effect == min(x$.effect))[1]]
}

get_base_model.eval_model_list <- function(x, ...) {
  attr(x, "base_model")
}
