% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{hpaBinary}
\alias{hpaBinary}
\title{Perform semi-nonparametric binary choice model estimation}
\usage{
hpaBinary(
  formula,
  data,
  K = 1L,
  z_mean_fixed = NA_real_,
  z_sd_fixed = NA_real_,
  z_constant_fixed = 0,
  is_z_coef_first_fixed = TRUE,
  is_x0_probit = TRUE,
  is_sequence = FALSE,
  x0 = numeric(0),
  cov_type = "sandwich",
  boot_iter = 100L,
  is_parallel = FALSE
)
}
\arguments{
\item{formula}{an object of class "formula" (or one that can be coerced to that class):
a symbolic description of the model to be fitted.}

\item{data}{data frame containing the variables in the model.}

\item{K}{non-negative integer representing polynomial degree.}

\item{z_mean_fixed}{numeric value for binary choice equation random error density mean parameter. 
Set it to NA (default) if this parameter should be estimated rather then fixed.}

\item{z_sd_fixed}{numeric value for binary choice equation random error density \code{sd} parameter. 
Set it to NA (default) if this parameter should be estimated rather then fixed.}

\item{z_constant_fixed}{numeric value for binary choice equation constant parameter. 
Set it to NA (default) if this parameter should be estimated rather then fixed.}

\item{is_z_coef_first_fixed}{bool value indicating weather binary equation first independend variable
coefficient should be fixed (TRUE) or estimated (FALSE).}

\item{is_x0_probit}{logical; if \code{TRUE} (default) then initial points for optimization routine will be
obtained by probit model estimated via \link[stats]{glm} function.}

\item{is_sequence}{if TRUE then function calculates models with polynomial degrees from 0 to K
each time using initial values obtained from the previous step. In this case function will
return the list of models where i-th list element correspond to model calculated under K=(i-1).}

\item{x0}{numeric vector of optimization routine initial values.
Note that \code{x0=c(pol_coefficients[-1], mean, sd, coefficients)}.}

\item{cov_type}{string value determinign the type of covariance matrix to be
returned and used for summary. If \code{cov_type = "hessian"} then negative 
inverse of hessian matrix will be applied. If \code{cov_type = "gop"} then
inverse of jacobian outer products will be applied. 
If \code{cov_type = "sandwich"} (default) then sandwich covariance matrix 
estimator will be applied. If \code{cov_type = "bootstrap"} then bootstrap
with \code{boot_iter} iterations will be applied.}

\item{boot_iter}{the number of bootstrap iterations
for \code{cov_type = "bootstrap"} covariance matrix estimator type.}

\item{is_parallel}{if TRUE then multiple cores will be 
used for some calculations.}
}
\value{
This function returns an object of class "hpaBinary".\cr \cr
An object of class "hpaBinary" is a list containing the following components:
\itemize{
\item \code{optim} - \code{\link[stats]{optim}} function output.
\item \code{x1} - numeric vector of distribution parameters estimates.
\item \code{mean} - mean (mu) parameter of density function estimate.
\item \code{sd} - sd (sigma) parameter of density function estimate.
\item \code{pol_coefficients} - polynomial coefficients estimates.
\item \code{pol_degrees} - the same as \code{K} input parameter.
\item \code{coefficients} - regression (single index) coefficients estimates.
\item \code{cov_matrix} - estimated parameters covariance matrix estimate.
\item \code{marginal_effects} - marginal effects matrix where columns are variables and rows are observations.
\item \code{results} - numeric matrix representing estimation results.
\item \code{log-likelihood} - value of Log-Likelihood function.
\item \code{AIC} - AIC value.
\item \code{errors_exp} - random error expectation estimate.
\item \code{errors_var} - random error variance estimate.
\item \code{dataframe} - dataframe containing variables mentioned in \code{formula} without \code{NA} values.
\item \code{model_Lists} - lists containing information about fixed parameters and parameters indexes in \code{x1}.
\item \code{n_obs} - number of observations.
\item \code{z_latent} - latent variable (signle index) estimates.
\item \code{z_prob} - probabilities of positive outcome (i.e. 1) estimates.}
}
\description{
This function performs semi-nonparametric single index binary choice model estimation
via hermite polynomial densities approximation.
}
\details{
Semi-nonparametric (SNP) approach has been implemented via densities
hermite polynomial approximation

Densities hermite polynomial approximation approach has been proposed by A. Gallant and D. W. Nychka in 1987.
The main idea is to approximate unknown distribution density with hermite polynomial of degree \code{pol_degree}. 
In this framework hermite polynomial represents adjusted (to insure integration to 1) product of squared polynomial and normal distribution densities.
Parameters \code{mean} and \code{sd} determine means and standard deviations of normal distribution density functions
which are parts of this polynomial.
For more information please refer to the literature listed below.

Parameters \code{mean}, \code{sd}, \code{given_ind}, \code{omit_ind}
should have the same length as \code{pol_degrees} parameter.

The first polynomial coefficient (zero powers) set to identity constant
for identification reasons.

Note that if \code{is_z_coef_first_fixed} value is TRUE then the coefficient for the first
independent variable in \code{formula} will be fixed to 1.

Parameter \code{sd} will be scale adjusted in order to
provide better initial point for optimization routine. Please, extract
\code{sd} adjusted value from this function's output list.

All variables mentioned in \code{formula} should be numeric vectors.

The function calculates standard errors via sandwich estimator and significance 
levels are reported taking into account quasi maximum likelihood estimator (QMLE) 
asymptotic normality. If ones wants to switch from QMLE to semi-nonparametric estimator (SNPE) during
hypothesis testing then covariance matrix should be reestimated using bootstrap.

This function maximizes log-likelihood function via \code{\link[stats]{optim}}
setting it's \code{method} argument to "BFGS". Please set \code{is_grad} to \code{TRUE}
in order to sufficiently improve optimization convergence by the cost of greater
computational burden.
}
\examples{
\donttest{
## Estimate survival probability on Titanic

library("titanic")

#Prepare data set converting  
#all variables to numeric vectors
h <- data.frame("male" = as.numeric(titanic_train$Sex == "male"))
	h$class_1 <- as.numeric(titanic_train$Pclass == 1)
	h$class_2 <- as.numeric(titanic_train$Pclass == 2)
	h$class_3 <- as.numeric(titanic_train$Pclass == 3)
	h$sibl <- titanic_train$SibSp
	h$survived <- titanic_train$Survived
	h$age <- titanic_train$Age
	h$parch <- titanic_train$Parch
	h$fare <- titanic_train$Fare
	
#Estimate model parameters
model_hpa_1 <- hpaBinary(survived ~class_1 + class_2 +
	male + age + sibl + parch + fare,
	K = 3, data = h)
#get summary
summary(model_hpa_1)

#Get predicted probabilities
pred_hpa_1 <- predict(model_hpa_1)

#Calculate number of correct predictions
hpa_1_correct_0 <- sum((pred_hpa_1 < 0.5) & (model_hpa_1$dataframe$survived == 0))
hpa_1_correct_1 <- sum((pred_hpa_1 >= 0.5) & (model_hpa_1$dataframe$survived == 1))
hpa_1_correct <- hpa_1_correct_1 + hpa_1_correct_0

#Plot random errors density approximation
plot(model_hpa_1)
}

\donttest{
##Estimate parameters on data simulated from student distribution

library("mvtnorm")
set.seed(123)

#Simulate independent variables from normal distribution
n <- 5000
X <- rmvnorm(n=n, mean = c(0,0), 
sigma = matrix(c(1,0.5,0.5,1), ncol=2))

#Simulate random errors from student distribution
epsilon <- rt(n, 5) * (3 / sqrt(5))

#Calculate latent and observable variables values
z_star <- 1 + X[, 1] + X[, 2] + epsilon
z <- as.numeric((z_star > 0))

#Store the results into dataframe
h <- as.data.frame(cbind(z,X))
names(h) <- c("z", "x1", "x2")

#Estimate model parameters
model <- hpaBinary(formula = z ~ x1 + x2, data=h, K = 4)
summary(model)

#Get predicted probabibilities of 1 values
predict(model)

#Plot density function approximation
plot(model)
}

\dontshow{
##Estimate parameters on data simulated from student distribution

library("mvtnorm")
set.seed(123)

#Simulate independent variables from normal distribution
n <- 200
X <- rmvnorm(n=n, mean = c(0,0), 
sigma = matrix(c(1,0.5,0.5,1), ncol=2))

#Simulate random errors from student distribution
epsilon <- rt(n, 5) * (3 / sqrt(5))

#Calculate latent and observable variables values
z_star <- 1 + X[, 1] + X[, 2] + epsilon
z <- as.numeric((z_star > 0))

#Store the results into dataframe
h <- as.data.frame(cbind(z,X))
names(h) <- c("z", "x1", "x2")

#Estimate model parameters
model <- hpaBinary(formula = z ~ x1 + x2, data=h, K = 3)
summary(model)

#Get predicted probabibilities of 1 values
predict(model)

#Plot density function approximation
plot(model)
}
}
\references{
A. Gallant and D. W. Nychka (1987) <doi:10.2307/1913241>
}
\seealso{
\link[hpa]{summary.hpaBinary}, \link[hpa]{predict.hpaBinary}, \link[hpa]{plot.hpaBinary},
\link[hpa]{AIC.hpaBinary}, \link[hpa]{logLik.hpaBinary}
}
