#' FUNCTION to plot idiograms of karyotypes with and without centromere
#' @description This function reads a data.frame passed to \code{dfChrSize} with columns: \code{chrName} (mono/holo) and
#' \code{shortArmSize} and \code{longArmSize} for monocentrics or a column \code{chrSize} for holocentrics and produces a plot of idiograms. If more
#' than one species, a column named \code{OTU} is needed.
#'
#' @description Optionally, it reads another data.frame passed to \code{dfMarkPos} with the position of
#' marks (sites). Examples: \code{\link{markposDFs}}. Another data.frame for mark characteristics
#' can be used \code{\link{dfMarkColor}} or a character vector passed to \code{mycolors}
#'
#' @param dfChrSize mandatory data.frame, with columns: \code{OTU} (optional), \code{chrName} (mandatory),
#'   \code{shortArmSize}, \code{longArmSize} for monocen. or \code{chrSize} for holocen.
#' @param dfMarkPos data.frame of marks (sites): columns: \code{OTU} (opt), \code{chrName},
#'   \code{markName} (name of site), \code{chrRegion} (for monocen. and opt for whole arm (w) in holocen.), \code{markDistCen} (for monocen.),
#'   \code{markPos} (for holocen.), \code{markSize}; column \code{chrRegion}:
#'   use \code{p} for short arm, \code{q} for long arm, \code{cen} for centromeric mark and \code{w} for whole chr. mark; column
#'   \code{markDistCen}: use distance from
#'   centromere to mark, not necessary for cen. marks (cen), w, p, q (when whole arm). See also param. \code{markDistType}
#' @param dfCenMarks data.frame, specific for centromeric marks. columns: \code{chrName}
#'   and \code{markName}. See also \code{dfMarkPos} for another option to pass cen. marks
#' @param dfMarkColor data.frame, optional, specifying colors and style for marks (sites);
#'   columns: \code{markName}, \code{markColor}, \code{style}. \code{style} accepts: \code{square}, \code{squareLeft}, \code{dots}, \code{cM},
#'    \code{"cMLeft"}, \code{"cenStyle"}, \code{"upArrow"}, \code{"downArrow"}.
#'   (if column \code{style} missing all (except 5S) are plotted as in param. \code{defaultStyleMark}).
#' @param mycolors character vector, optional, i.e. \code{c("blue",} \code{} \code{"red",} \code{} \code{"green")} for specifying color of marks in order of appearance. if diverges with number of marks will be recycled if \code{dfMarkColor} present, mycolors will be ignored. To know the order of your marks use something like: \code{unique(c(dfMarkPos$markName,} \code{dfCenMarks$markName) ) }
#' @param addMissingOTUAfter character, when you want to add space (ghost OTUs) after one or several OTUs, pass the names of OTUs preceding the desired space in a character vector i.e. \code{c("species one",} \code{} \code{"species five")}
#' @param missOTUspacings numeric, when you use \code{addMissingOTUAfter} this numeric vector should have the same length and corresponds to the number of free spaces (ghost OTUs) to add after each OTU respectively
#' @param moveKarHor character, OTUs' names of karyotypes that should be moved horizontally. See \code{mkhValue}
#' @param mkhValue numeric, value to move kar. hor. See \code{moveKarHor}
#' @param anchor boolean, when TRUE, plots a parent progeny structure in karyotypes in \code{moveKarHor}
#' @param anchorVsizeF numeric, factor to modify vertical size of anchor \code{0.5} (default). Size itself is equal to \code{karHeiSpace}
#' @param moveAnchorV numeric, displace anchor vertical portion to right or left. See \code{anchor}
#' @param moveAnchorH numeric, displace anchor horizontal portion to right or left. See \code{anchor}
#' @param pchAnchor numeric, symbol for anchor, see \code{?points} and \code{anchor}
#' @param orderChr character, replaces \code{orderBySize - deprecated} when \code{"size"}, sorts chromosomes by total
#'   length from the largest to the smallest. \code{"original"}: preserves d.f. order. \code{"name"}: sorts alphabetically; \code{"group"}: sorts by group name
#' @param centromereSize numeric, optional, this establishes the apparent size of cen. in the plot in \eqn{\mu}m. Automatic when \code{NA}. Default: \code{NA}
#' @param origin, For non-monocentric chr. (for holocentrics only) Use \code{"b"} (default) if distance to mark in (\code{"markPos"} column in \code{"dfMarkPos"}) data.frame measured from bottom of chromosome, use \code{"t"} for distance to mark from top of chr.
#' @param efZero, numeric, numbers below this one will be considered as zero, for comparison purposes. Defaults to \code{1e-5}
#' @param cMBeginCenter, boolean, start position of \code{cM} and \code{cMLeft} marks. If \code{TRUE}, starts in the center (width) of chr. . Defaults to \code{FALSE}
#' @param arrowhead numeric, proportion of head of arrow (mark styles: \code{upArrow,} \code{} \code{downArrow}). Defaults to \code{0.3}
#' @param shrinkArrow numeric, proportion, shrinks body of arrow. Defaults to \code{0.3333}
#' @param arrowheadWidthShrink numeric, proportion, shrinks head of arrow. Defaults to \code{0.1}
#' @param arrowsToSide boolean, when \code{FALSE} use a centered arrow, instead of an arrow next to chr. margins (\code{TRUE}, default). See \code{arrowsBothChrt}
#' @param markDistType character, if \code{"cen"} = the distance you provided in data.frame (\code{dfMarkPos}) column \code{markDistCen}
#' or \code{markPos}  is to the center of the mark, if \code{"beg"} = the distance you provided is to the
#'   beginning of the mark (Default)
#' @param chrWidth numeric, relative chromosome width. Defaults to \code{0.5}
#' @param specialChrWidth numeric, relative chromosome width. Defaults to \code{0.5} for OTUs in \code{specialOTUNames}
#' @param chrSpacing numeric, horizontal spacing among chromosomes, see also  \code{chrWidth}. Defaults to \code{0.5}
#' @param specialChrSpacing numeric, horizontal spacing among chromosomes for OTUs in \code{specialOTUNames}, see also  \code{chrWidth}. Defaults to \code{0.5}
#' @param chrColor character, main color for chromosomes. Defaults to \code{"gray"}
#' @param chrBorderColor character, color for border of chromosomes, defaults to \code{chrColor}
#' @param cenColor character, color for centromeres, if GISH use \code{NULL} or \code{NA}. Defaults to \code{chrColor}
#' @param fixCenBorder boolean, when \code{TRUE} uses \code{chrColor} as centromere (and cen. mark) border color. See also \code{cenColor},
#' \code{chrColor}, \code{colorBorderMark}, \code{borderOfWhiteMarks}. No default value.
#' @param roundedCen boolean, when \code{FALSE}, cen has triangular aspect. When \code{TRUE}, it has rounded aspect (Default).
#' @param squareness numeric, shape of vertices of chromosomes and square marks,
#'   higher values more squared. Defaults to \code{4}
#' @param karHeight numeric, vertical size of karyotypes. See also  \code{karHeiSpace}. Defaults to \code{2}
#' @param karHeiSpace numeric, vertical size of karyotypes including spacing. Proportional to \code{karHeight}, if overlap, increase. Defautl value \code{2.5}
#' @param karSepar boolean, reduce distance among karyotypes \code{FALSE} = equally
#'   sized karyotypes or \code{TRUE} = equally spaced karyotypes. Incompatible with \code{addMissingOTUAfter}
#' @param amoSepar numeric, depends on \code{karSepar=TRUE}, if zero your
#'   karyotypes will have no distance among them, if overlap,
#'   increase this and \code{karHeiSpace}
#' @param chrId character, print name of chromosome, \code{"original"} uses the original
#'   name in OTU column of dfChrSize, \code{"simple"} (just 1 to ...) or \code{""} (none).
#' @param chrIdPatternRem character, regex pattern to remove from chr. names
#' @param distTextChr numeric, distance from name of chromosome to chromosome,
#'   also affects vertical separation of indices. Defaults to \code{1}
#' @param groupUp boolean, when \code{TRUE} when groups present, they appear over the chr. name. Defaults to \code{FALSE}
#' @param groupName boolean, when \code{TRUE} (default), shows group names. When \code{FALSE} only line
#' @param groupSepar numeric, factor for affecting chr. spacing \code{chrSpacing} among groups. Defaults to \code{0.5}
#' @param chromatids boolean, when \code{TRUE} shows separated chromatids. Defaults to \code{TRUE}
#' @param arrowsBothChrt boolean, when \code{TRUE} (default) (for \code{chromatids=TRUE}) shows \code{upArrow,} \code{} \code{ downArrow} styles of marks in both chromatids when \code{arrowsToSide=TRUE}.
#' @param holocenNotAsChromatids boolean, when \code{TRUE} and \code{chromatids=TRUE} does not plot holocen kar. with chromatids. Defaults to \code{FALSE}. A value of \code{TRUE} modifies excHoloFrArrToSide to \code{TRUE} always.
#' @param excHoloFrArrToSide boolean, when \code{arrowsToSide=TRUE}, excludes holocen. from this behaviour, plotting a centered arrow only.
#' @param xModifier numeric, for \code{chromatids=TRUE}, separation among chromatids
#' @param xModMonoHoloRate numeric, factor to shrink xModifier for holocen. 5 means 5 times smaller (quotient).
#' @param indexIdTextSize numeric, font size of chr. and kar. indices and
#'   chromosome name. Defaults to \code{1}
#' @param OTUTextSize numeric, font size of OTU name (species). Defaults to \code{1}. When \code{OTUasNote} is \code{TRUE}, use  \code{notesTextSize} instead
#' @param legend character, \code{""} for no legend; \code{"inline"} prints labels near
#'   chromosomes; \code{"aside"} prints legend to the right of karyotypes (default). See \code{markLabelSpacer}
#' @param remSimiMarkLeg boolean, when \code{legend="aside"}, if you use \code{pattern}, you can have several marks with same name. When \code{TRUE} this remove this pseudoduplicates from legend. Be sure that this pseudoduplicates have the same color, otherwise you should use \code{FALSE} (default).
#' @param bannedMarkName character, character string or vector with mark names to be removed from plot. Not the marks but the labels.
#' @param legendWidth numeric, factor to increase width of squares and of legend. Defaults to \code{1.7}
#' @param legendHeight numeric, factor to increase height of squares and dots of legend. Automatic.
#' @param defaultStyleMark character, default style of mark, only used when \code{style} column of \code{dfMarkColor} data.frame is missing or in absence of this data.frame. Use \code{"square"} (default), \code{"squareLeft"}, \code{"dots"}, \code{"cM"} \code{"cM"}, \code{"cMLeft"},\code{"cenStyle"}, \code{"upArrow"}, \code{"downArrow"}.
#' @param colorBorderMark character, without default, pass a name of a color to use as border of marks. See \code{borderOfWhiteMarks}
#' @param borderOfWhiteMarks boolean, if \code{TRUE} (Default) uses black border for white marks. See \code{dfMarkColor}. Does not apply to marks with style \code{cenStyle}
#' @param lwd.mimicCen thickness of lines of \code{cenStyle} marks; affects only lateral borders. Defaults to \code{lwd.chr}
#' @param defCenStyleCol character, color of outer part of \code{cenStyle} marks. Defaults to \code{white}
#' @param protruding numeric, when style of mark is \code{"cM"}, fraction of chrWidth to stretch marker. Defaults to \code{0.2}. Introduced in 1.13
#' @param markLabelSize numeric, only if legend != (not) "", size of the text of
#'   labels of marks (legend). Defaults to \code{1}
#' @param markLabelSpacer numeric, only if \code{legend="aside"}, space from the
#'   rightmost chr. to legend. Defaults to \code{1}
#' @param legendYcoord numeric, modify Y position of legend when \code{legend="aside"}
#' @param pattern REGEX pattern to remove from names of marks
#' @param chrIndex character, add arm ratio with \code{"AR"} and centromeric index with \code{"CI"}, or \code{"both"} (Default), or \code{""} for none
#' @param chrSize boolean, when \code{TRUE} adds total chr size under each chr. Defaults to \code{FALSE}
#' @param chrNameUp boolean, when \code{TRUE} adds secondary chromosome name from col. \code{chrNameUp} over chrs. Defaults to \code{FALSE}
#' @param classMbName character, name of "chromosome" when in Mbp. Defaults to \code{"Pm"}. See \code{MbUnit}
#' @param classcMName character, name of "chromosome" when OTU in \code{specialOTUNames}. Defaults to \code{"L.G."}
#' @param classChrName character, name of "chromosome" when in micrometers (apparently). Defaults to \code{"Chr."}. See \code{specialOTUnames}, \code{classMbName}, \code{classcMName}
#' @param classChrNameUp character, name of "chromosome" for col. \code{"chrNameUp"}. Defaults to \code{"Type"}
#' @param classGroupName character, name of groups. Defaults to \code{""}
#' @param nsmall numeric, rounding decimals for \code{chrSize} parameter. Defaults to \code{1}
#' @param chrSizeMbp boolean, when \code{TRUE} adds total Mbp chr. size to each chr. provided, there is a \code{Mbp} column in \code{dfChrSize} data.frame. Defaults to \code{FALSE}. If data in columns \code{shortArmSize}, or col. \code{chrSize} is in millions ("Mbp"). Use \code{chrSize=TRUE} not this one (not column \code{Mbp}, you don't need this).
#' @param markPer character, name of mark to calculate % of mark in chr. and add it to plot. See \code{perAsFraction}
#' @param showMarkPos boolean, adds position of marks under karyotype (fraction 0-1) when \code{TRUE}. Defaults to \code{FALSE}
#' @param bToRemove, character, bands to remove from calc. of pos.
#' @param perAsFraction boolean, when \code{TRUE} % is shown as fraction. Defaults to \code{FALSE}. See \code{markPer}
#' @param nameChrIndexPos numeric, modify position of name of chr. indices
#' @param karIndex logical, add karyotype indices A (intrachromosomal -
#'   centromere pos.) and A2 (interchromosomal asymmetry, variation among
#'   chromosome sizes)
#' @param karIndexPos numeric, move karyotype index
#' @param notesLeft deprecated, use a data.frame for \code{leftNotes}
#' @param notesPos numeric, move right notes to the right or left (x axis)
#' @param notesPosY numeric, move right notes to the down or up (y axis)
#' @param leftNotesPos numeric, move left notes to the right or left (x axis)
#' @param leftNotesPosY numeric, move left notes to the down or up (y axis)
#' @param morpho character, if \code{"both"} (default) prints the Guerra and Levan classif of cen. position, use also \code{"Guerra"} or  \code{"Levan"} or \code{""} for none. See also \code{?armRatioCI}.
#' @param addOTUName boolean, if \code{TRUE} adds OTU (species) name to karyotype
#' @param OTUfont numeric, \code{1} for normal,  \code{2} for bold,  \code{3} for italics,  \code{4} for bold-italics
#' @param OTUfamily character, font family for OTU name.
#' @param OTUasNote boolean, if \code{TRUE} adds OTU (species) name to the right, see \code{notes}
#' @param revOTUs boolean, The order of species is the one in the main
#'   data.frame, use \code{TRUE} to reverse
#' @param ruler boolean, display ruler to the left of karyotype, when \code{FALSE} no ruler
#' @param useMinorTicks boolean, display minor ticks between labeled ticks in ruler. See \code{miniTickFactor}. Defaults to \code{FALSE}. (ticks without label)
#' @param miniTickFactor numeric, number of minor ticks for each labeled tick. See \code{useMinorTicks}. Defaults to \code{10}
#' @param rulerPos numeric, absolute position of ruler, corresponds to \code{pos}
#'   argument of \code{axis} R plot
#' @param ruler.tck numeric, tick size of ruler, corresponds to \code{tck} argument of
#'   \code{axis} R plot. Defaults to \code{-0.02}

#' @param rulerNumberPos numeric, modify position of numbers of ruler. Defaults to \code{0.5}
#' @param rulerNumberSize numeric, size of number's font in ruler. Defaults to \code{1}
#' @param ceilingFactor numeric, affects number of decimals for ceiling. Affects max. value of ruler. Defaults to \code{0}. When \code{threshold} is greater than \code{35} this may have to be negative. Introduced in 1.13
#' @param rulerInterval numeric, intervals in ruler. No default, automatic. Introduced in 1.13
#' @param rulerIntervalcM numeric, intervals in ruler of OTU in \code{specialOTUNames}. No default. Introduced in 1.13
#' @param rulerIntervalMb numeric, intervals in ruler of OTU with data in Mb (>\code{MbThreshold}) and absent from \code{specialOTUNames}. No default. Introduced in 1.13
#' @param yTitle character, units for common title. Defaults to \eqn{\mu m}
#' @param specialOTUNames character vector, normally title of ruler is micrometer or Mb (big numbers). Use this param. to be able to put a different unit in ruler title. See \code{"specialyTitle"}
#' @param specialyTitle, character, title of ruler if OTU is in \code{specialOTUNames}. Will not apply if \code{MbThreshold} met. In that case use \code{MbUnit}
#' @param xlimLeftMod numeric, modifies \code{xlim} left argument of plot
#' @param xlimRightMod numeric, \code{xlim} right side modification by adding space to the right
#'   of idiograms. Defaults to \code{2}
#' @param ylimBotMod numeric, modify \code{ylim} bottom argument of plot
#' @param ylimTopMod numeric, modify \code{ylim} top argument of plot
#' @param lwd.cM thickness of cM marks. Defaults to \code{lwd.chr}
#' @param lwd.chr thickness of border of chr., marks; ruler. Also thick of cM marks if \code{lwd.cM} absent Defaults to \code{0.5}
#' @param MbThreshold, numeric, if greater than this number (defaults to \code{10000}), \code{MbUnit} will apply and \code{specialyTitle} will not.
#' @param threshold, this is the max. value allowed for the main two significative digits, otherwise scale will shrink. For example, after 35 \eqn{\mu m} (Default), apparent size will be 3.5 and scale interval will change. See also \code{ceilingFactor}, you may have to use \code{-1}. Introduced in 1.13
#' @param MbUnit, character, text of units of title when \code{MbThreshold} met and OTU not in \code{specialOTUNames}. See \code{specialyTitle}
#' Defaults to \code{"Mb"}, but anything can be used. Introduced in 1.13. See \code{specialyTitle}
#' @param xPosRulerTitle, numeric, modify position of ruler title. See \code{yTitle,} \code{} \code{ specialyTitle,} \code{} \code{ MbUnit}. Defaults to \code{2.6}. A value of \code{2.6} means \code{2.6} times the value of \code{chrSpacing} to the left, from the first chr.
#' @param yPosRulerTitle, numeric, affects vertical position of ruler title. Defaults to \code{0}
#' @param rulerTitleSize, numeric font size of units of ruler. See also \code{xPosRulerTitle}
#' @param n, numeric vertices number for round corners
#' @param notes, data.frame, (to the right), with columns \code{OTU} and \code{note} for adding notes to each OTU, they appear to the right of chromosomes
#' @param leftNotes, data.frame, (to the left), with columns \code{OTU} and \code{note} for adding notes to each OTU, they appear to the left of chromosomes
#' @param notesTextSize numeric, font size of notes, see \code{notes}
#' @param propWidth, boolean, defaults to \code{FALSE}. Diminishes chr. width with increasing number of OTUs
#' @param asp, numeric, y x aspect of plot. Defaults to \code{1}
#' @param defaultFontFamily character. use this as the font family. No default value.
#' @param circularPlot boolean, if \code{TRUE} chromosomes are plotted in concentric circles. Defaults to \code{FALSE}
#' @param shrinkFactor numeric, for \code{circularPlot=TRUE} percentage of usage of circle. Defaults to \code{0.9}
#' @param separFactor numeric, for \code{circularPlot=TRUE} modify separation of concentric karyotypes. Defaults to \code{1.5}
#' @param labelSpacing numeric, for \code{circularPlot=TRUE}. Spacing of mark labels. Defaults to \code{0.7}
#' @param labelOutwards boolean, inline labels projected outwards
#' @param chrLabelSpacing numeric, for \code{circularPlot=TRUE}. Spacing of chr. labels. Defaults to \code{0.5}
#' @param OTUlabelSpacing numeric, for \code{circularPlot=TRUE}. Spacing for OTU names. Defaults to \code{0.3}
#' @param radius numeric, for \code{circularPlot=TRUE}. Affects radius of karyotypes. Defaults to \code{0.5}
#' @param OTUsrt numeric, for \code{circularPlot=TRUE}. Angle to use for OTU names. Defaults to \code{0}
#' @param OTUplacing character, for \code{circularPlot=TRUE}. location of OTU name. Defaults to \code{"first"} plots name near
#' first chr. \code{"number"} places number near 1st chr. and index and name to the right or center.
#' \code{"simple"} place name to the right or center without numbering. See also \code{OTUcentered}
#' @param useOneDot boolean, for \code{circularPlot=TRUE}. use one dot instead of two in style of marks \code{dots}. Defaults to \code{TRUE}
#' @param circleCenter numeric, for \code{circularPlot=TRUE}. Coordinate X of center of circles. Affects \code{legend="aside"} position. Defaults to \code{1}
#' @param circleCenterY numeric, for \code{circularPlot=TRUE}. Coordinate Y of center of circles. Affects \code{legend="aside"} position. Defaults to \code{1}
#' @param OTULabelSpacerx numeric, for \code{circularPlot=TRUE} and \code{OTUplacing="number" or "simple"}. Modifies x names position
#' @param OTULabelSpacery numeric, for \code{circularPlot=TRUE} and \code{OTUplacing="number" or "simple"}. Modifies y names position
#' @param OTUcentered boolean, for \code{circularPlot=TRUE} and \code{OTUplacing="number" or "simple"}. OTU name in center of circle
#' when \code{TRUE}, otherwise, to the right.
#' @param OTUjustif numeric, for \code{circularPlot=TRUE} and \code{OTUplacing="number" or "simple"}. Justification of OTU name. \code{0} = left
#' (Default); use \code{0.5} for centered. See \code{?text} -> \code{adj}
#' @param OTUlegendHeight numeric, for \code{circularPlot=TRUE} and \code{OTUplacing="number" or "simple"}. Modifies y names separation
#' @param callPlot boolean, create new plot in your device. Defaults to \code{TRUE}
#' @param rotation numeric, anti-clockwise rotation, defaults to \code{0.5} which rotates chr. from top to -90 degrees. (-0.5*\eqn{\pi} )
#' @param roundness deprecated, use \code{squareness}
#' @param ... accepts other arguments for the plot, see, \code{?plot}
#'
#' @keywords data.frame chromosome
#'
#' @importFrom graphics par plot segments mtext
#' @importFrom dplyr bind_rows
#' @importFrom grDevices col2rgb
#'
#' @export
#'
#' @examples
#' data(dfOfChrSize)
#' plotIdiograms(dfOfChrSize, ylimBotMod = .75, rulerPos=-.5)
#' plotIdiograms(dfOfChrSize, circularPlot = TRUE, chrLabelSpacing = 1)
#' plotIdiograms(dfChrSizeHolo, rulerPos=-.5)
#' @seealso \code{\link{asymmetry}}
#' @seealso \code{\link{armRatioCI}}
#' @seealso \code{\link{chrbasicdatamono}}
#' @seealso \code{\link{chrbasicdataHolo}}
#' @seealso \code{\link{markposDFs}}
#' @seealso \code{\link{markdataholo}}
#' @seealso \code{\link{dfMarkColor}}
#' @return plot
plotIdiograms <- function(dfChrSize, # karyotype

  defaultFontFamily,
  revOTUs=FALSE,
  karHeight=2,karHeiSpace=2.5,
  karSepar=TRUE,
  amoSepar=10,
  addMissingOTUAfter=NA,missOTUspacings=0,
  moveKarHor,
  anchor,
  anchorVsizeF=.5,
  pchAnchor=23,
  moveAnchorV=0,
  moveAnchorH=0,
  mkhValue=.5,
  n=50,
  notes,
  leftNotes,
  notesTextSize=.4,
  notesLeft,
  notesPos=.5,
  notesPosY=0,
  leftNotesPos=.5,
  leftNotesPosY=0,

  propWidth=FALSE,

  # Units
  MbThreshold=10000,
  threshold=35,
  # divisor,
  MbUnit="Mb",
  yTitle="\u00B5m",
  specialyTitle="cM",
  specialOTUNames="",

  #OTU names
  addOTUName=TRUE,
  OTUTextSize=1,
  OTUfont,
  OTUfamily,
  # OTUasNote=TRUE,
  OTUasNote=FALSE,

  # chromosomes
  orderChr="size",
  chrId="original",
  classMbName =  "Pm.",
  classcMName =  "L.G.",
  classChrName=  "Chr.",
  classChrNameUp = "Type",
  classGroupName = "",
  chrNameUp=FALSE,
  chrIdPatternRem,
  # chrId="simple",
  indexIdTextSize=1,
  distTextChr=1,
  groupUp=FALSE,
  groupName=TRUE,
  groupSepar=0.5,
  # chromatids=FALSE,
  chromatids=TRUE,
  arrowsBothChrt=TRUE,

  holocenNotAsChromatids=FALSE,
  excHoloFrArrToSide=FALSE,

  xModifier,
  xModMonoHoloRate,
  chrWidth=0.5,
  chrSpacing=0.5,
  specialChrWidth=0.3, specialChrSpacing=0.7,

  chrColor="gray",
  chrBorderColor,
  centromereSize=NA,
  cenColor,
  fixCenBorder,
  roundedCen=TRUE,
  squareness=4,
  lwd.chr=0.5,
  lwd.cM,

  #marks
  dfMarkPos, dfCenMarks,
  defaultStyleMark="square",
  markDistType="beg",
  protruding=0.2,
  # markDistType="cen",
  origin="b",
  efZero=1e-5,
  # origin="t",
  cMBeginCenter=FALSE,
  arrowhead = .3,
  shrinkArrow = .3333,
  arrowheadWidthShrink = .1,
  arrowsToSide=TRUE,

  dfMarkColor,
  mycolors,
  borderOfWhiteMarks=TRUE,
  colorBorderMark,
  lwd.mimicCen,
  defCenStyleCol,
  pattern="",

  # mark labels
  legend="aside",
  remSimiMarkLeg=FALSE,
  bannedMarkName,
  # legend="inline",
  legendWidth=1.7, legendHeight=NA,
  markLabelSize=1,
  markLabelSpacer=1,
  legendYcoord=0,

  #indices
  chrSize=FALSE,
  nsmall=1,
  chrSizeMbp=FALSE,
  markPer="",
  showMarkPos=FALSE,
  bToRemove="",
  perAsFraction=FALSE,
  chrIndex="both",
  morpho="both",
  nameChrIndexPos=2,
  karIndex=TRUE,
  # karIndex=FALSE,
  karIndexPos=.5,

  # rulers
  ruler=TRUE,
  useMinorTicks=FALSE,
  miniTickFactor=10,
  rulerPos=0,
  ruler.tck=-0.02,
  rulerNumberPos=0.5,
  rulerNumberSize=1,
  rulerInterval,
  rulerIntervalcM,
  rulerIntervalMb,
  ceilingFactor=0,
  xPosRulerTitle=2.6,
  yPosRulerTitle=0,
  rulerTitleSize=1,

  # margins
  xlimLeftMod=1, xlimRightMod=2,
  ylimBotMod=.2, ylimTopMod=.2,
  callPlot=TRUE,
  asp=1,

  # c plot
  circularPlot=FALSE,
  # circularPlot=TRUE,

  shrinkFactor=.9,
  separFactor=1.5,
  labelSpacing=.7,
  labelOutwards=FALSE,
  chrLabelSpacing=.5,
  # radius=1,
  radius=.5,

  rotation=0.5,
  useOneDot=TRUE,
  circleCenter=1,
  circleCenterY=1,

  # c. p OTU
  OTUlabelSpacing=.3,
  OTUsrt=0,
  OTUplacing="first",
  # useOneDot=FALSE,
  OTULabelSpacerx=0,
  OTULabelSpacery=0,
  OTUcentered=TRUE,
  OTUjustif=0,
  OTUlegendHeight=NA,
  roundness,
  ...) {

  xfactor <- yfactor <- 1

  chrWFactor<-specialChrWidth/chrWidth

  if(circularPlot){
    chrWidth<-chrWidth*4
    legendHeight<-legendHeight*4
    # legendWidth<-legendWidth*2
    specialChrWidth<-specialChrWidth*4
  }

  if(!missing(defaultFontFamily)){
    defaultFontFamily2<-defaultFontFamily
  } else {
    defaultFontFamily2<-"sans"
  }

  if(missing(xModifier)) {
    xModifier <- chrWidth/12
  }
  if(missing(xModMonoHoloRate)){
    xModifierMono<-xModifier
    xModifierHolo<-xModifier
  } else {
    xModifierMono<-xModifier
    xModifierHolo<-xModifier/as.numeric(xModMonoHoloRate)
  }

  if(missing(defCenStyleCol) ) {
    defCenStyleCol <- "white"
  }

  if(!missing(roundness)){
    crayon::red("roundness is deprecated and was substituted with new param. 'squareness'")
  }


  if(!missing(notesLeft)){
    crayon::red("notesLeft is deprecated and was substituted with new param. 'leftNotes', use a data.frame")
  }



  if(!missing(lwd.cM)){
    lwd.cM2 <- lwd.cM
  } else {
    lwd.cM2 <- lwd.chr
  }
  if(!missing(lwd.mimicCen)){
    lwd.mimicCen2 <- lwd.mimicCen
  } else {
    lwd.mimicCen2 <- lwd.chr
  }

  OTUfont2<-ifelse( !missing(OTUfont),   OTUfont,   1)
  OTUfamily2<-ifelse( !missing(OTUfamily), OTUfamily, defaultFontFamily2)

  if(!missing(dfChrSize)){
    dfChrSizeInternal<-makeNumCols(dfChrSize)
  } else {
    message(crayon::red("Missing mandatory dfChrSize data.frame"))
    return(NA)
  }

  if(!missing(dfMarkPos) ) {

    #
    #   rename column markArm if necessary
    #

    if("markArm" %in% colnames(dfMarkPos)  ){
      message(crayon::red(paste(c("Column markArm in d.f. of marks renamed to chrRegion")))
      ) # mess
      colnames(dfMarkPos)[which(names(dfMarkPos)=="markArm")]<-"chrRegion"
    }

    dfMarkPos[dfMarkPos==""]<-NA
    copyDfMarkPosInternal1 <- dfMarkPosInternal <- makeNumCols(dfMarkPos)

    if(is.null(copyDfMarkPosInternal1$markPos)){
      copyDfMarkPosInternal1$markPos<-NA
    }
    if(is.null(copyDfMarkPosInternal1$markSize)){
      copyDfMarkPosInternal1$markSize<-NA
    }
    if(is.null(copyDfMarkPosInternal1$markDistCen)){
      copyDfMarkPosInternal1$markDistCen<-NA
    }

    dfCenMarksInternal <- copyDfMarkPosInternal1[which(copyDfMarkPosInternal1$chrRegion=="cen"),]

    if(nrow(dfCenMarksInternal)==0 ){
      remove(dfCenMarksInternal)
    }
    dfpGISHInternal <- copyDfMarkPosInternal1[which(copyDfMarkPosInternal1$chrRegion %in% "p" &
                                                  is.na(copyDfMarkPosInternal1$markSize) &
                                                  is.na(copyDfMarkPosInternal1$markDistCen)
                                                ),]
    if(nrow(dfpGISHInternal)==0 ){
      remove(dfpGISHInternal)
    }

    dfqGISHInternal <- copyDfMarkPosInternal1[which(copyDfMarkPosInternal1$chrRegion %in% "q" &
                                                  is.na(copyDfMarkPosInternal1$markSize) &
                                                  is.na(copyDfMarkPosInternal1$markDistCen)
    ),]
    if(nrow(dfqGISHInternal)==0 ){
      remove(dfqGISHInternal)
    }

    dfwholeGISHInternal <- copyDfMarkPosInternal1[which(copyDfMarkPosInternal1$chrRegion %in% "w" &
                                                  is.na(copyDfMarkPosInternal1$markSize) &
                                                  (is.na(copyDfMarkPosInternal1$markDistCen) |
                                                   is.na(copyDfMarkPosInternal1$markPos) )
    ),]

    if(nrow(dfwholeGISHInternal)==0 ){
      remove(dfwholeGISHInternal)
    }

} # df mark pos

  ##############################################################################
  #
  #   adds name of otu when missing
  #
  ##############################################################################

  if (exists("dfMarkPosInternal")) {

    listOfdfMarkPosInternal<-dfToListColumn(dfMarkPosInternal)
    dfMarkPosInternal <- dplyr::bind_rows(listOfdfMarkPosInternal, .id = "OTU")

  } # df of marks

  if(!missing(dfCenMarks)  ) {
    dfCenMarksInternal2<-makeNumCols(dfCenMarks)

    parlistOfdfMarkPosDataCen2<-dfToListColumn(dfCenMarksInternal2)
    dfCenMarksInternal2 <- dplyr::bind_rows(parlistOfdfMarkPosDataCen2, .id = "OTU")
    remove(parlistOfdfMarkPosDataCen2)

  }

  if (exists("dfCenMarksInternal")) {

    parlistOfdfMarkPosDataCen <- dfToListColumn(dfCenMarksInternal)
    dfCenMarksInternal <- dplyr::bind_rows(parlistOfdfMarkPosDataCen, .id = "OTU")

  } # df of marks

  cendfs <- mget(ls(pattern = "^dfCenMarksInternal" ) )

  if(length(cendfs) ) {
    dfCenMarksInternal <- suppressWarnings(dplyr::bind_rows(cendfs) )
  }

  #
  #   mark style
  #

  if(!missing(dfMarkColor)){
    dfMarkColorInternal<-makeNumCols(dfMarkColor)
  }

  message(crayon::black(paste("Making checks\n")) )
  message(crayon::black(paste("In case of error see messages and the help ?functionName\n")) )

  #
  #   dfChrSizeInternal
  #

  #################################################################### LISTS

  #
  # transform dfs to list of df dfChrSizeInternal
  #

  if (!"OTU" %in% colnames(dfChrSizeInternal) ) {
    addOTUName<-FALSE
    OTUasNote <-FALSE
  }

  listOfdfChromSize <- dfToListColumn(dfChrSizeInternal)

  #
  #   reconstitute dfChrSizeInternal OTU
  #

  #
  #   Classify data.frames from list as monocen or holocen Add attribute cenType
  #

listOfdfChromSize <- addAttributesDfChrSize(listOfdfChromSize,threshold,specialOTUNames,centromereSize,MbThreshold)

  dfChrSizeInternal <- dplyr::bind_rows(listOfdfChromSize, .id = "OTU")
  dfChrSizeInternal <- makeNumCols(dfChrSizeInternal)

  # important must be after bind_rows
  listOfdfChromSize <- lapply(listOfdfChromSize, function(x) makeNumCols(x))

  #
  #     calculate armRatioCI only for chr. with centromere attr cenType =  holocen.
  #

  #
  #    generate Chromosome indexes for Monocen
  #

  if(chrIndex=="both" | chrIndex=="AR"| chrIndex=="CI" | morpho=="both" | morpho=="Guerra" | morpho == "Levan" | chrSize == TRUE | markPer!="" | showMarkPos) {
    for (i in 1:length(listOfdfChromSize)) {
      if(attr(listOfdfChromSize[[i]], "cenType")=="monocen"){ # only for monocen

        listOfdfChromSize[[i]] <- armRatioCI(listOfdfChromSize[[i]])

        if(attr(listOfdfChromSize[[i]], "indexStatus")=="failure"){

          if("OTU" %in% colnames(listOfdfChromSize[[i]])){
            message(crayon::red(paste("in",unique(listOfdfChromSize[[i]]$OTU) ) ) )
          } # otu
          message(crayon::red("\nFix measures or use chrIndex=\"\", and morpho=\"\" ")
          ) #m
        } # if failure
      } # monocen
    } # for
  } # if chrIndex

  ##############################################################################
  #
  #   names of holo and mono
  #

{
  monocenNames<-makeVectorNames(listOfdfChromSize,"cenType","monocen")

  holocenNames<-makeVectorNames(listOfdfChromSize,"cenType","holocen")
}

  ##########################################3
  #
  #     gish  p
  #
  #########################################

# message(crayon::green("GISH data loading"))

  if (exists("dfpGISHInternal")) {

    listOfdfpGISHInternal<-dfToListColumn(dfpGISHInternal)

    # monocen

    listOfdfpGISHInternalMonocen<-listOfdfpGISHInternal[which(names(listOfdfpGISHInternal) %in% monocenNames)]

    # names(listOfdfpGISHInternalMonocen)
    if(length(listOfdfpGISHInternalMonocen)==0){
      remove(listOfdfpGISHInternalMonocen)
    } else {
      listOfdfpGISHInternalMonocen<-Filter(function(x) {nrow(x) >= 1}, listOfdfpGISHInternalMonocen)
      dfpGISHInternalMonocen <- dplyr::bind_rows(listOfdfpGISHInternalMonocen, .id = "OTU")
      # dfpGISHInternalMonocen$chrRegion<-"p"
    } # else

    # P marks of Holocen MUST NOt exist

    checkArmHolocenError(listOfdfpGISHInternal,holocenNames)

  } #   if (exists("dfpGISHInternal")){

  ##########################################3
  #
  #     gish  q
  #
  #########################################

  if (exists("dfqGISHInternal")){

    listOfdfqGISHInternal<-dfToListColumn(dfqGISHInternal)

    # monocen

    listOfdfqGISHInternalMonocen<-listOfdfqGISHInternal[which(names(listOfdfqGISHInternal) %in% monocenNames)]

    if(length(listOfdfqGISHInternalMonocen)==0){
      remove(listOfdfqGISHInternalMonocen)
    } else {
      listOfdfqGISHInternalMonocen<-Filter(function(x) {nrow(x) >= 1}, listOfdfqGISHInternalMonocen)
      dfqGISHInternalMonocen <- dplyr::bind_rows(listOfdfqGISHInternalMonocen, .id = "OTU")
    } # else

    # q marks of Holocen MUST NOt exist

    checkArmHolocenError(listOfdfqGISHInternal,holocenNames)

  } #   if (exists("dfpGISHInternal")){

  ##########################################3
  #
  #     gish whole
  #
  #########################################

  if(exists("dfwholeGISHInternal")){

    listOfdfwholeGISHInternal<-dfToListColumn(dfwholeGISHInternal)

    ###########################################################################################################################3
    #
    # MONOCEN GISH TO P Q CEN
    #

    listOfdfwholeGISHMonocen<-listOfdfwholeGISHInternal[which(names(listOfdfwholeGISHInternal) %in% monocenNames)]

    if(length(listOfdfwholeGISHMonocen)==0){
      remove(listOfdfwholeGISHMonocen)
    } else {
      listOfdfwholeGISHMonocen<-Filter(function(x) {nrow(x) >= 1}, listOfdfwholeGISHMonocen)

      #
      #   p part
      #

      listOfdfpGISHInternalMonocen2 <- listOfdfwholeGISHMonocen

      dfpGISHInternalMonocen2 <- dplyr::bind_rows(listOfdfpGISHInternalMonocen2, .id = "OTU")
      dfpGISHInternalMonocen2$chrRegion<-"p"
      dfpGISHInternalMonocen2$chrRegionOrig<-"w"

      #
      #   q part
      #

      listOfdfqGISHInternalMonocen2 <- listOfdfwholeGISHMonocen

      dfqGISHInternalMonocen2 <- dplyr::bind_rows(listOfdfqGISHInternalMonocen2, .id = "OTU")
      dfqGISHInternalMonocen2$chrRegion<-"q"
      dfqGISHInternalMonocen2$chrRegionOrig<-"w"

      #
      # cen part
      #

      listOfdfCenMarksInternal2 <- listOfdfwholeGISHMonocen

      dfCenMarksInternal2 <- dplyr::bind_rows(listOfdfCenMarksInternal2, .id = "OTU")
      dfCenMarksInternal2$chrRegion<-"cen"
      # dfCenMarksInternal2$chrRegionOrig<-"w" leaving this hides w names completely in inline

      cendfs <- mget(ls(pattern = "^dfCenMarksInternal" ) )
      if(length(cendfs) ) {
        dfCenMarksInternal <- suppressWarnings(dplyr::bind_rows(cendfs) )
      }

    } # else

    #
    # HOLOCEN
    #
    ###########################################################################################################

    listOfdfwholeGISHHolocen<-listOfdfwholeGISHInternal[which(names(listOfdfwholeGISHInternal) %in% holocenNames)]

    if(length(listOfdfwholeGISHHolocen)==0){
      remove(listOfdfwholeGISHHolocen)
    } else {

    dfwholeGISHHolocen <- dplyr::bind_rows(listOfdfwholeGISHHolocen, .id = "OTU")

    #
    # remake chrom sizes df
    #

    dfwholeGISHHolocen$markSize <- dfChrSizeInternal[match(interaction(dfwholeGISHHolocen[c("OTU","chrName")] ),
                                                                 interaction(dfChrSizeInternal[c("OTU","chrName") ] )
                                                                 ),]$chrSize
    dfwholeGISHHolocen$markPos <- 0

    if(markDistType=="cen") { # center
      dfwholeGISHHolocen$markPos <- dfChrSizeInternal[match(interaction(dfwholeGISHHolocen[c("OTU","chrName")] ),
                                                           interaction(dfChrSizeInternal[c("OTU","chrName") ] )
      ),]$chrSize/2
    }

    #
    #   merge dfMarkPosInternal and dfwholeGISHHolocen
    #

    if(exists("dfMarkPosInternal") & exists("dfwholeGISHHolocen") ) {
      dfMarkPosInternal <- dplyr::bind_rows(dfMarkPosInternal,dfwholeGISHHolocen)
    }
    if(!exists("dfMarkPosInternal") & exists("dfwholeGISHHolocen") ) {
      dfMarkPosInternal <- dfwholeGISHHolocen
    }

    } #     if(length(listOfdfwholeGISHHolocen)==0){

  }  #  end   if(exists("dfwholeGISHInternal")){

  #################################################################################################################3
  #
  #   merge p
  #

  gishMonocenDfsP <- mget(ls(pattern = "^dfpGISHInternalMonocen" ) )

  if(length(gishMonocenDfsP) ) {
    MdfpGISHInternalMonocen <- suppressWarnings(dplyr::bind_rows(gishMonocenDfsP) )
  }

  if(exists("MdfpGISHInternalMonocen") ) {
    MdfpGISHInternalMonocen <- markDistCenGISHfix(MdfpGISHInternalMonocen,dfChrSizeInternal,"shortArmSize",markDistType)
  } # p gish

  ############################################################################################
  # q

  gishMonocenDfsQ <- mget(ls(pattern = "^dfqGISHInternalMonocen" ) )

  if(length(gishMonocenDfsQ) ) {
    MdfqGISHInternalMonocen <- suppressWarnings(dplyr::bind_rows(gishMonocenDfsQ) )
  }

  if(exists("MdfqGISHInternalMonocen") ) {
    MdfqGISHInternalMonocen<-markDistCenGISHfix(MdfqGISHInternalMonocen,dfChrSizeInternal,"longArmSize",markDistType)
  } # q gish

  ##################################################################################################
  #
  #       merging p and q
  #
  ##################################################################################################
#
  gishMonocenDfsPQ <- mget(ls(pattern = "^Mdf" ) )

  if(length(gishMonocenDfsPQ) ) {
    dfMarkPosInternal2 <- suppressWarnings(dplyr::bind_rows(gishMonocenDfsPQ) )
  }

  #
  #    merge dfMarkPosInternal2 dfMarkPosInternal
  #

  mDfMarkPosI <- mget(ls(pattern = "^dfMarkPosInternal" ) )
#  mDfMarkPosI860 <<- mDfMarkPosI

  if(length(mDfMarkPosI) ) {
    dfMarkPosInternal <- suppressWarnings(dplyr::bind_rows(mDfMarkPosI) )
  }

  #
  #     DF OF marks to list
  #

  if (exists("dfMarkPosInternal")) {

    dfMarkPosInternal <- unique(dfMarkPosInternal)

    listOfdfMarkPosInternal<-dfToListColumn(dfMarkPosInternal)

    #
    #              monocen marks list
    #

    parlistOfdfMarkPosMonocen<-listOfdfMarkPosInternal[which(names(listOfdfMarkPosInternal) %in% monocenNames)]
#    parlistOfdfMarkPosMonocen908<<-parlistOfdfMarkPosMonocen

    if(length(parlistOfdfMarkPosMonocen)==0){
      remove(parlistOfdfMarkPosMonocen)
    } else {
      for (i in 1:length(parlistOfdfMarkPosMonocen)){
      parlistOfdfMarkPosMonocen[[i]]  <- parlistOfdfMarkPosMonocen[[i]][which(parlistOfdfMarkPosMonocen[[i]]$chrRegion!="cen"),]
      } # for
      parlistOfdfMarkPosMonocen<-Filter(function(x) {nrow(x) >= 1}, parlistOfdfMarkPosMonocen)
    }# else

    #
    #                holocen marks list
    #

    parlistOfdfMarkPosHolocen <- listOfdfMarkPosInternal[which(names(listOfdfMarkPosInternal) %in% holocenNames)]

    if(length(parlistOfdfMarkPosHolocen)==0){
      remove(parlistOfdfMarkPosHolocen)
    }

} # end missing dfMarkPosInternal

  #
  #   df of cen marks to list
  #

  if (exists("dfCenMarksInternal")){

    parlistOfdfMarkPosDataCen<-dfToListColumn(dfCenMarksInternal)

    parlistOfdfMarkPosDataCen<-parlistOfdfMarkPosDataCen[which(names(parlistOfdfMarkPosDataCen) %in% monocenNames)]
    if(length(parlistOfdfMarkPosDataCen)==0){
      remove(parlistOfdfMarkPosDataCen)
    } else {
      #
      #   remove columns without info.
      #
      for (i in 1:length(parlistOfdfMarkPosDataCen)){
        parlistOfdfMarkPosDataCen[[i]][parlistOfdfMarkPosDataCen[[i]]==""]<-NA
        parlistOfdfMarkPosDataCen[[i]]<-  parlistOfdfMarkPosDataCen[[i]][, !apply(is.na(parlistOfdfMarkPosDataCen[[i]]), 2, all)]
      } # for
    } # else
  } # end missing dfCenMarksInternal

  #
  #   for each d.f. of dfmarkpos check columns
  #

  ############################################################################################################################
  #
  #   Monocen check marks
  #

  if(exists("parlistOfdfMarkPosMonocen")){
    message(crayon::black("\nChecking mandatory columns from dfMarkPos: chrName, markName, chrRegion,markDistCen\n (column OTU  is necessary if more than one species)\nmarkSize can be absent when cM style"
    ) )# cat

    for (i in 1:length(parlistOfdfMarkPosMonocen ) ) {

      parlistOfdfMarkPosMonocen[[i]][parlistOfdfMarkPosMonocen[[i]]==""] <- NA
      parlistOfdfMarkPosMonocen[[i]] <- parlistOfdfMarkPosMonocen[[i]][, !apply(is.na(parlistOfdfMarkPosMonocen[[i]]), 2, all)]

      #
      #   rename column markpos if necessary
      #

      if(!"markDistCen" %in% colnames(parlistOfdfMarkPosMonocen[[i]]) & "markPos" %in% colnames(parlistOfdfMarkPosMonocen[[i]])  ){
        message(crayon::red(paste(c("Columns markPos in d.f. of marks of OTU",names(parlistOfdfMarkPosMonocen)[[i]] ,"renamed to markDistCen")))
        ) # mess
        colnames(parlistOfdfMarkPosMonocen[[i]])[which(names(parlistOfdfMarkPosMonocen[[i]])=="markPos")]<-"markDistCen"
      }

      #
      #   REMOVE GISH DATA incomplete duplicated data
      #

      parlistOfdfMarkPosMonocen[[i]] <- parlistOfdfMarkPosMonocen[[i]][setdiff(1:length(parlistOfdfMarkPosMonocen[[i]]$chrRegion),
                                                                         which(parlistOfdfMarkPosMonocen[[i]]$chrRegion %in% "p" &
                                                                                 is.na(parlistOfdfMarkPosMonocen[[i]]$markSize) &
                                                                                 is.na(parlistOfdfMarkPosMonocen[[i]]$markDistCen)
      ) ) ,]

      parlistOfdfMarkPosMonocen[[i]] <- parlistOfdfMarkPosMonocen[[i]][setdiff(1:length(parlistOfdfMarkPosMonocen[[i]]$chrRegion),
                                                                         which(parlistOfdfMarkPosMonocen[[i]]$chrRegion %in% "q" &
                                                                         is.na(parlistOfdfMarkPosMonocen[[i]]$markSize) &
                                                                         is.na(parlistOfdfMarkPosMonocen[[i]]$markDistCen)
      ) ) ,]

      parlistOfdfMarkPosMonocen[[i]] <- parlistOfdfMarkPosMonocen[[i]][setdiff(1:length(parlistOfdfMarkPosMonocen[[i]]$chrRegion),
                                                                         which(parlistOfdfMarkPosMonocen[[i]]$chrRegion %in% "w"
      ) ) ,]

      #
      #   column error check
      #

      # if(length (setdiff(c("chrName", "markName", "chrRegion","markDistCen","markSize"),
      if(length (setdiff(c("chrName", "markName", "chrRegion","markDistCen"),
                         colnames(parlistOfdfMarkPosMonocen[[i]]) ) )>0 ) {
        message(crayon::red(paste(c("ERROR Missing columns in d.f. of marks of OTU",names(parlistOfdfMarkPosMonocen)[[i]] ,":",
                                setdiff(c("chrName", "markName", "chrRegion","markDistCen"), # rem markSize
                                        colnames(parlistOfdfMarkPosMonocen[[i]]) ) ) , sep="\n", collapse = " " )
        )
        ) # cat
        message(crayon::red(paste("\nERRORS PRESENT, see above, dfMarksPos of OTU", names(parlistOfdfMarkPosMonocen)[[i]] ,"REMOVED\n")
        ) )#m
        parlistOfdfMarkPosMonocen[[i]]<-NA
      } # fi setdiff
      #
      #   column without error
      #
      else { # if no error

        corr_index<-which(names(listOfdfChromSize) %in% names(parlistOfdfMarkPosMonocen)[[i]] )
        divisor2<-as.numeric(attr(listOfdfChromSize[[corr_index]],"divisor"))
          parlistOfdfMarkPosMonocen[[i]]$markDistCen<-parlistOfdfMarkPosMonocen[[i]]$markDistCen/divisor2

          if("markSize" %in% colnames(parlistOfdfMarkPosMonocen[[i]])){
              parlistOfdfMarkPosMonocen[[i]]$markSize<-parlistOfdfMarkPosMonocen[[i]]$markSize/divisor2
          }

        message(crayon::black(paste("\nOK marks of OTU",names(parlistOfdfMarkPosMonocen)[[i]],"checked \n")
                ) ) #m
        if(markDistType=="cen") { # this is from center
          #
          #   fix bug when markDistType is cen (center) but cM style of marks have NA in markSize column
          #
          # halfMarkSize<-ifelse(is.na(parlistOfdfMarkPosMonocen[[i]]$markSize/2),0,(parlistOfdfMarkPosMonocen[[i]]$markSize/2) )
          if("markSize" %in% colnames(parlistOfdfMarkPosMonocen[[i]])){
          parlistOfdfMarkPosMonocen[[i]]$markDistCen <- psum(parlistOfdfMarkPosMonocen[[i]]$markDistCen,
                                                          ( - parlistOfdfMarkPosMonocen[[i]]$markSize/2),
                                                          na.rm=TRUE)
          }
        } # if
      } # else No Error
    } # for each data.frame of Marks of Monocen

  parlistOfdfMarkPosMonocen<-parlistOfdfMarkPosMonocen[!is.na(parlistOfdfMarkPosMonocen)]
  # do as before with holo 27/09

} # fi parlistOfdfMarkPosMonocen

  ##################################################################################################################
  #
  #   holocen check mark
  #

  if(exists("parlistOfdfMarkPosHolocen")){
    message(crayon::black("\nChecking mandatory columns from dfMarkPos (without cen.): chrName, markName, markPos\n (column OTU  is necessary if more than one species)\nmarkSize column is not necessary for style of mark cM"
    ) )# mess

    for (i in 1:length(parlistOfdfMarkPosHolocen ) ) {

      parlistOfdfMarkPosHolocen[[i]][parlistOfdfMarkPosHolocen[[i]]==""]<-NA
      parlistOfdfMarkPosHolocen[[i]]<-  parlistOfdfMarkPosHolocen[[i]][, !apply(is.na(parlistOfdfMarkPosHolocen[[i]]), 2, all)]

      #
      #   REMOVE GISH DATA incomplete duplicated data
      #

      parlistOfdfMarkPosHolocen[[i]] <- parlistOfdfMarkPosHolocen[[i]][setdiff(1:length(parlistOfdfMarkPosHolocen[[i]]$chrName),
                                                                         which(parlistOfdfMarkPosHolocen[[i]]$chrRegion %in% "w" &
                                                                                 is.na(parlistOfdfMarkPosHolocen[[i]]$markSize )
      ) ) ,]

      #
      #   rename column markdistcen if necessary
      #

      if(!"markPos" %in% colnames(parlistOfdfMarkPosHolocen[[i]]) & "markDistCen" %in% colnames(parlistOfdfMarkPosHolocen[[i]])  ){
        message(crayon::red(paste(c("Columns markDistCen in d.f. of marks of OTU",names(parlistOfdfMarkPosHolocen)[[i]] ,"renamed to markPos")))
        ) # mess
        colnames(parlistOfdfMarkPosHolocen[[i]])[which(names(parlistOfdfMarkPosHolocen[[i]])=="markDistCen")]<-"markPos"
      }

      #
      #   column error
      #

      # if(length (setdiff(c("chrName", "markName", "markPos","markSize"),
      if(length (setdiff(c("chrName", "markName", "markPos"),
                           colnames(parlistOfdfMarkPosHolocen[[i]]) ) )>0 ){
          message(crayon::red(paste(c("ERROR Missing columns:",
                                      # setdiff(c("chrName", "markName", "markPos","markSize"),
                                              setdiff(c("chrName", "markName", "markPos"),
                                              colnames(parlistOfdfMarkPosHolocen[[i]]) ) ) , sep="\n", collapse = " " )
          )
          ) # cat
          message(crayon::red(paste("\nERRORS PRESENT, see above, dfMarksPos of OTU", names(parlistOfdfMarkPosHolocen)[[i]] ,"REMOVED\n")
          ) ) #m
          # parlistOfdfMarkPosHolocen<-parlistOfdfMarkPosHolocen[-i]
          parlistOfdfMarkPosHolocen[[i]]<-NA
        } # fi
      #
      #   column without error
      #

      else { # if no error
        message(crayon::black(paste("\nOK marks of OTU",names(parlistOfdfMarkPosHolocen)[[i]],"checked \n")
        ) ) #m
        if(any(is.na(parlistOfdfMarkPosHolocen[[i]]$markPos))){
          message(crayon::blue(paste("\nholocen. mark(s) without pos. might get unexpected results\n")
          ))
        }
        if(origin=="t"){
          parlistOfdfMarkPosHolocen[[i]]$markPos2<-parlistOfdfMarkPosHolocen[[i]]$markPos
          parlistOfdfMarkPosHolocen[[i]]$chrSize<-
            dfChrSizeInternal[match(interaction( parlistOfdfMarkPosHolocen[[i]][c("OTU", "chrName")]),
                                    interaction( dfChrSizeInternal[c("OTU", "chrName")] )
                                    ),]$chrSize

          if(markDistType=="beg"){
            if("markSize" %in% colnames(parlistOfdfMarkPosHolocen[[i]])){
              # markSize2<-ifelse(is.na( parlistOfdfMarkPosHolocen[[i]]$markSize  ) ,0, ( parlistOfdfMarkPosHolocen[[i]]$markSize ) )
              parlistOfdfMarkPosHolocen[[i]]$markPos<- psum(parlistOfdfMarkPosHolocen[[i]]$chrSize,
                                                         - parlistOfdfMarkPosHolocen[[i]]$markPos2,
                                                         - parlistOfdfMarkPosHolocen[[i]]$markSize,
                                                         na.rm=TRUE)
            } # markSize column exist

          } else if(markDistType=="cen"){
              if("markSize" %in% colnames(parlistOfdfMarkPosHolocen[[i]])){
                # halfMarkSize<-ifelse(is.na(parlistOfdfMarkPosHolocen[[i]]$markSize/2) ,0, ( (parlistOfdfMarkPosHolocen[[i]]$markSize/2) ) )
                parlistOfdfMarkPosHolocen[[i]]$markPos<-psum( parlistOfdfMarkPosHolocen[[i]]$chrSize,
                                                           - parlistOfdfMarkPosHolocen[[i]]$markPos2,
                                                           (- parlistOfdfMarkPosHolocen[[i]]$markSize/2),
                                                           na.rm=TRUE)
              } # col markSize exists
          } # cen

        } else if (origin=="b") { # if t else b

          if(markDistType=="cen") { # center
            if("markSize" %in% colnames(parlistOfdfMarkPosHolocen[[i]])){

            parlistOfdfMarkPosHolocen[[i]]$markPos <- psum(parlistOfdfMarkPosHolocen[[i]]$markPos,
                                                        (- parlistOfdfMarkPosHolocen[[i]]$markSize/2),
                                                        na.rm=TRUE)
            } # if col markSize exist
          } # cen
        } # origin b
      } # else No Error
    } # for each data.frame of Marks of Monocen

  parlistOfdfMarkPosHolocen<-parlistOfdfMarkPosHolocen[!is.na(parlistOfdfMarkPosHolocen)]

  for (i in 1:length(parlistOfdfMarkPosHolocen)) {

    corr_index<-which(names(listOfdfChromSize) %in% names(parlistOfdfMarkPosHolocen)[[i]] )
    divisor2<-as.numeric(attr(listOfdfChromSize[[corr_index]],"divisor"))

    # if(attr(parlistOfdfMarkPosHolocen[i],"name") %in% MbNames ){
      parlistOfdfMarkPosHolocen[[i]]$markPos <-parlistOfdfMarkPosHolocen[[i]]$markPos/divisor2
      if("markSize" %in% colnames(parlistOfdfMarkPosHolocen[[i]])){
        parlistOfdfMarkPosHolocen[[i]]$markSize<-parlistOfdfMarkPosHolocen[[i]]$markSize/divisor2
      }
    # } # if
  } # for
} # fi holocen exists

  ################################################################################################################################
  #
  #   cen Mark check
  #

  if(exists("parlistOfdfMarkPosDataCen")){
    message(crayon::black("\nChecking mandatory columns from dfCenMarks: chrName, markName\n (column OTU  is necessary if more than one species)\n")
    ) # mess

    for (i in 1:length(parlistOfdfMarkPosDataCen)){
      #
      #   columns with error
      #

    if(length(setdiff(c("chrName", "markName"),
                      colnames(parlistOfdfMarkPosDataCen[[i]]) ) )>0 ){
      message(crayon::red(paste(c("ERROR Missing columns:",
                              setdiff(c("chrName", "markName"),
                                      colnames(parlistOfdfMarkPosDataCen[[i]]) ),"in OTU", names(parlistOfdfMarkPosDataCen)[[i]]   ), sep="\n", collapse = " " )
      )
      ) # cat
      message(crayon::red(paste("\nERRORS PRESENT, see above, dfCenMarks of OTU", names(parlistOfdfMarkPosDataCen)[[i]] ,"REMOVED\n")
      ) ) #m
      parlistOfdfMarkPosDataCen[[i]]<-NA

    } # fi

      #
      #   columns without error
      #

    else { # if no error
      message(crayon::black(paste("\nOK cen. marks of OTU",names(parlistOfdfMarkPosDataCen)[[i]],"checked \n")
      ))# mess
    } # else
    } # for

  parlistOfdfMarkPosDataCen<-parlistOfdfMarkPosDataCen[!is.na(parlistOfdfMarkPosDataCen)]

  } # fi   if(exists("parlistOfdfMarkPosDataCen"))

  ##############################################################################################################
  #
  #   OTU cross check of d.fs
  #

  if(exists("parlistOfdfMarkPosMonocen")){

    parlistOfdfMarkPosMonocen<-filterExtraOTU(listOfdfChromSize,parlistOfdfMarkPosMonocen)

  } # exists

  if(exists("parlistOfdfMarkPosHolocen")){
    # message(crayon::black("\n####\ndfMarkPos exists, if error will be removed\n") )

    parlistOfdfMarkPosHolocen<-filterExtraOTU(listOfdfChromSize,parlistOfdfMarkPosHolocen)

  } # exists

  #
  #     check chromosomes names  from d.f. marks to chr. size. d.f.
  #

  if(exists("parlistOfdfMarkPosMonocen") ) {
#    # listOfdfChromSize1120<<-listOfdfChromSize
#    # parlistOfdfMarkPosMonocen1121<<-parlistOfdfMarkPosMonocen
    listOfChecksChr<-checkNameChrDfMarks(listOfdfChromSize,parlistOfdfMarkPosMonocen)
#    # listOfChecksChr123<<-listOfChecksChr
    listOfdfChromSize<-listOfChecksChr[[1]]
    parlistOfdfMarkPosMonocen<-listOfChecksChr[[2]]

    if(length(parlistOfdfMarkPosMonocen)==0){
      remove(parlistOfdfMarkPosMonocen)
    } else {

      allMarkNames<-unique(listOfChecksChr[[3]])

      if(length(listOfChecksChr[[4]])>0){
        allMarkMaxSize<-max(listOfChecksChr[[4]], na.rm=TRUE)
      }
    }
  } # parlistOfdfMarkPosMonocen


  if(exists("parlistOfdfMarkPosHolocen") ) {

    listOfChecksChr<-checkNameChrDfMarks(listOfdfChromSize,parlistOfdfMarkPosHolocen)
    listOfdfChromSize<-listOfChecksChr[[1]]

    parlistOfdfMarkPosHolocen<-listOfChecksChr[[2]]

    if(length(parlistOfdfMarkPosHolocen)==0){
      remove(parlistOfdfMarkPosHolocen)
    } else {

      if(exists("allMarkNames")) {
#        # allMarkNames1<<-allMarkNames
        allMarkNames<-unique(c(allMarkNames,listOfChecksChr[[3]] ) )
      } else {
        allMarkNames<-unique(listOfChecksChr[[3]] )
#        # allMarkNames2<<-allMarkNames
      }

      if(length(listOfChecksChr[[4]])>0){
        if (exists("allMarkMaxSize")){
          allMarkMaxSize<-max(c(allMarkMaxSize,max(listOfChecksChr[[4]], na.rm=TRUE) ), na.rm=TRUE)
        } else {
          allMarkMaxSize<-max(listOfChecksChr[[4]], na.rm=TRUE)
        }
      }
    }
  } # holocen

    if(exists("parlistOfdfMarkPosDataCen") ) {
      listOfChecksChr<-checkNameChrDfMarks(listOfdfChromSize,parlistOfdfMarkPosDataCen)
      listOfdfChromSize<-listOfChecksChr[[1]]
      parlistOfdfMarkPosDataCen<-listOfChecksChr[[2]]

      if(length(parlistOfdfMarkPosDataCen)==0){
        remove(parlistOfdfMarkPosDataCen)
      } else {

      if(exists("allMarkNames")){
        allMarkNames<-unique(c(allMarkNames,listOfChecksChr[[3]] ) )
      } else {
        allMarkNames<-unique(listOfChecksChr[[3]])
      }
        cenMarkNames<-unique(listOfChecksChr[[3]])

      } # parlistOfdfMarkPosDataCen 0
    } # parlistOfdfMarkPosDataCen

  #
  #   remake dfMarkPosInternal (for per. mark) after filtering from lists
  #

  mlists <- ls(pattern = "^parlistOfdfMarkPos" )
  if(length(mlists)){
  plist <- lapply(mget(mlists ), function(x) unname(x) )
  dfMarkPosInternal <- suppressWarnings(dplyr::bind_rows(plist) )
  }

  #############################################################################################################
  #
  #   check compatibility of columns dfMarkColor
  #

  if(!missing(cenColor)) {
    cenColor2<-cenColor

    if( is.null(cenColor)  ){
      cenColor2<-NULL
    }

  } else {
    cenColor2<-chrColor
  }


  {
  if(exists("dfMarkColorInternal") ) {

#   dfMarkColorInternal1311<<-dfMarkColorInternal

    message(crayon::black("\n####\nChecking mandatory columns from dfMarkColor: markName, markColor\n"
    ) )#cat

    #
    #   create style column when missing
    #

    if(!"style" %in% colnames(dfMarkColorInternal) ) {
      message(crayon::red(paste0("\ndfMarkColor style column missing, created with string: ",defaultStyleMark,"\n")
                          ) ) # m
      dfMarkColorInternal$style <- defaultStyleMark    # "square" or user default
    }

    if (length( setdiff(c("markName", "markColor","style"),
                        colnames(dfMarkColorInternal) ) )>0 ){
      message(crayon::red(paste(c("ERROR Missing column:",
                              setdiff(c("markName", "markColor","style"),
                                      colnames(dfMarkColorInternal) ) ) , sep="\n", collapse = " ")
      )
      )# cat
      remove(dfMarkColorInternal)
      message(crayon::red("\nError in dfMarkColor, REMOVED\n") )
    } #fi
    else {  # column names ok
#      dfMarkColorInternal1212<<-dfMarkColorInternal

      # missing color for constric cenStyle

      tryCatch(dfMarkColorInternal[which(dfMarkColorInternal$markColor==""),]$markColor <- NA, error= function(e){NA})

      tryCatch(bannedMarkName2 <- dfMarkColorInternal[which(is.na(dfMarkColorInternal$markColor) ) , ]$markName, error= function(e){NA} )

      tryCatch(dfMarkColorInternal[which(is.na(dfMarkColorInternal$markColor) &
                     dfMarkColorInternal$style %in% "cenStyle"),]$markColor<-chrColor,
               error= function (e) NA )
#     dfMarkColorInternal1245<<-dfMarkColorInternal

      if(exists("allMarkNames")) {
#        # allMarkNames1248<<-allMarkNames
        dfMarkColorInternal <- dfMarkColorInternal[which(dfMarkColorInternal$markName %in% allMarkNames) ,]
        if (nrow(dfMarkColorInternal)==0){
          message(crayon::red("\nError in dfMarkColor markNames respect to Marks pos. data.frames, dfMarkColor REMOVED\n")
          )#cat
          remove(dfMarkColorInternal)
        } else if ( length(setdiff(allMarkNames,unique(dfMarkColorInternal$markName) ) )>0) { # nrow not 0
          message(crayon::black("\nColors provided in to dfMarkColor are not enough, internal colors will be used.\n") )
          dfMarkColorInternal <- makedfMarkColor(dfMarkColorInternal,allMarkNames, c(chrColor,cenColor2) )
        } else { # nrow not 0
          message(crayon::black("\nCheck OK\n") )
        }
#        dfMarkColorInternal1232 <<-dfMarkColorInternal

      } # fi # end allmarknames exist
      else { # all Mark Names does not exist
        message(crayon::red("\nError in dfMarkColor Names respect to Marks data.frames, dfMarkColor REMOVED\n")
        )
        remove(dfMarkColorInternal)
      } # else
    } # else column names ok
  } #fi exists dfMarkColor ... continues
  else if (missing(mycolors) ) { # if dfMarkColor not exist and missing mycolors
      if(exists("allMarkNames")){
        dfMarkColorInternal<-makedfMarkColor(idiogramFISH::dfMarkColor,allMarkNames,c(chrColor,cenColor2),defaultStyleMark )
        } # allmarknames
    } else if (!missing(mycolors) ) { # if dfMarkColor not exist , mycolors exist
      if(exists("allMarkNames")){
        dfMarkColorInternal<-makedfMarkColorMycolors(allMarkNames, mycolors, c(chrColor,cenColor2),defaultStyleMark )
      }
  } # elif
  }

  if(exists("dfMarkColorInternal")){
    dfMarkColorInternal$markBorderColor<-dfMarkColorInternal$markColor

    if(!missing(colorBorderMark)){

      colorBorderMarkFiltered<-colorBorderMark[sapply(colorBorderMark, function(X) {
        tryCatch(is.matrix(col2rgb(X)),
                 error = function(e) {message(crayon::red(paste("Color",X,"invalid, removed from colorBorderMark") ) ); return(FALSE) })
      } )]
      colorBorderMarkFiltered<-colorBorderMarkFiltered[!is.na(colorBorderMarkFiltered)]

      if(length(colorBorderMarkFiltered)>0 ) {
        dfMarkColorInternal$markBorderColor<-colorBorderMarkFiltered
      }

    } # colorBorderMark

    if(borderOfWhiteMarks){
      tryCatch(dfMarkColorInternal[which(dfMarkColorInternal$markColor %in% "white" &
                                           !dfMarkColorInternal$style %in% "cenStyle"),]$markBorderColor<-"black",
               error= function (e) NA)
    }
  }

  if (exists("dfMarkColorInternal") & legend=="inline"){
    dfMarkColorInternalCopy<-dfMarkColorInternal
    dfMarkColorInternalArrowsLabels<-dfMarkColorInternalCopy[which(dfMarkColorInternalCopy$style %in% c("downArrow","upArrow") ),]
    if(nrow(dfMarkColorInternalArrowsLabels)>0){
      tryCatch(dfMarkColorInternalArrowsLabels[which(dfMarkColorInternalArrowsLabels$style %in% "downArrow"),]$style<-"cMLeft",error = function(e) {""})
      tryCatch(dfMarkColorInternalArrowsLabels[which(dfMarkColorInternalArrowsLabels$style %in% "upArrow"),]$style<-"cM",error = function(e) {""})
      tryCatch(dfMarkColorInternalArrowsLabels$markName <- paste0(dfMarkColorInternalArrowsLabels$markName," "),error = function(e) {""} )
      dfMarkColorInternal <- dplyr::bind_rows(dfMarkColorInternal,dfMarkColorInternalArrowsLabels)
#      # dfMarkColorInternal1 <<-dfMarkColorInternal
#      # dfMarkColorInternalArrowsLabels1<<-dfMarkColorInternalArrowsLabels
    }
  }

  if(!missing(bannedMarkName) & exists("bannedMarkName2")  ){
#    bannedMarkName1301<<-bannedMarkName
    tryCatch(bannedMarkName<-c(bannedMarkName,bannedMarkName2), error= function(e){NA} )
  } else if(exists("bannedMarkName2") ){
    bannedMarkName<-bannedMarkName2
  } else {
    bannedMarkName<-NULL
  }

#  bannedMarkName1308<<-bannedMarkName
# dfMarkColorInternal1402<<-dfMarkColorInternal

  #
  #   CREATION OF CHILD DATAFRAMES MARKS
  #

  ###################
  # REQUIRES STYLE
  ###################

  # substituted by:

  if(exists("parlistOfdfMarkPosMonocen") & exists("dfMarkColorInternal")  ) {
    speListOfdfMarkPosMonocenArrowsLabels<-list()
#    parlistOfdfMarkPosMonocen1322<<-parlistOfdfMarkPosMonocen

    for (i in 1:length(parlistOfdfMarkPosMonocen)){
      if(class(parlistOfdfMarkPosMonocen[[i]])=="data.frame" ) {

        parlistOfdfMarkPosMonocen[[i]]$style <- dfMarkColorInternal$style[match(parlistOfdfMarkPosMonocen[[i]]$markName, dfMarkColorInternal$markName)]

        if (legend=="inline") {
          speListOfdfMarkPosMonocenArrowsLabels[[i]]<-parlistOfdfMarkPosMonocen[[i]]
          speListOfdfMarkPosMonocenArrowsLabels[[i]]<-speListOfdfMarkPosMonocenArrowsLabels[[i]][which(speListOfdfMarkPosMonocenArrowsLabels[[i]]$style %in% c("downArrow","upArrow") ),]
          if(nrow(speListOfdfMarkPosMonocenArrowsLabels[[i]])>0) {
            tryCatch(speListOfdfMarkPosMonocenArrowsLabels[[i]][which(speListOfdfMarkPosMonocenArrowsLabels[[i]]$style %in% "downArrow"),]$style<-"cMLeft",error = function(e) {""})

            tryCatch(speListOfdfMarkPosMonocenArrowsLabels[[i]][which(speListOfdfMarkPosMonocenArrowsLabels[[i]]$style %in% "upArrow"),]$style<-"cM",error = function(e) {""})

            tryCatch({
              downleftp <- which(speListOfdfMarkPosMonocenArrowsLabels[[i]]$style %in% "cMLeft" &
                                 speListOfdfMarkPosMonocenArrowsLabels[[i]]$chrRegion %in% "p" )
              speListOfdfMarkPosMonocenArrowsLabels[[i]][downleftp,]$markDistCen<-
                speListOfdfMarkPosMonocenArrowsLabels[[i]][downleftp,]$markDistCen+
                speListOfdfMarkPosMonocenArrowsLabels[[i]][downleftp,]$markSize
                },    error = function(e) {""} )
            tryCatch({
              uprightq <- which(speListOfdfMarkPosMonocenArrowsLabels[[i]]$style %in% "cM" &
                                   speListOfdfMarkPosMonocenArrowsLabels[[i]]$chrRegion %in% "q" )
              speListOfdfMarkPosMonocenArrowsLabels[[i]][uprightq,]$markDistCen<-
                speListOfdfMarkPosMonocenArrowsLabels[[i]][uprightq,]$markDistCen+
                speListOfdfMarkPosMonocenArrowsLabels[[i]][uprightq,]$markSize
            },    error = function(e) {""} )

            tryCatch(speListOfdfMarkPosMonocenArrowsLabels[[i]]$markName <- paste0(speListOfdfMarkPosMonocenArrowsLabels[[i]]$markName," ")
                     ,error = function(e) {""} )

            parlistOfdfMarkPosMonocen[[i]] <- dplyr::bind_rows(parlistOfdfMarkPosMonocen[[i]],speListOfdfMarkPosMonocenArrowsLabels[[i]])
          } # arrows present
        } # inline

        tryCatch(parlistOfdfMarkPosMonocen[[i]]$protruding <- dfMarkColorInternal$protruding[match(parlistOfdfMarkPosMonocen[[i]]$markName,
                                                                                                dfMarkColorInternal$markName)]
                 ,error = function(e) {""}
                 )



      if(nrow(parlistOfdfMarkPosMonocen[[i]][which(parlistOfdfMarkPosMonocen[[i]]$style %in% "cenStyle"),])>0) {
        if(!"markSize" %in% colnames(parlistOfdfMarkPosMonocen[[i]])){
          parlistOfdfMarkPosMonocen[[i]]$markSize<-NA
        }
        for (m in 1:nrow(parlistOfdfMarkPosMonocen[[i]][which(parlistOfdfMarkPosMonocen[[i]]$style %in% "cenStyle"),] ) ) {
          if( is.na(parlistOfdfMarkPosMonocen[[i]][which(parlistOfdfMarkPosMonocen[[i]]$style %in% "cenStyle"),]$markSize[m]) ) {

            parlistOfdfMarkPosMonocen[[i]][which(parlistOfdfMarkPosMonocen[[i]]$style %in% "cenStyle"),]$markSize[m] <-
                dfChrSizeInternal[match(interaction( parlistOfdfMarkPosMonocen[[i]][which(
                parlistOfdfMarkPosMonocen[[i]]$style %in% "cenStyle"),][m,c("OTU", "chrName")]),
                                      interaction( dfChrSizeInternal[c("OTU", "chrName")] )
              ),]$centromereSize

            } # if
          } # for
        } # if nrow
      } # if data.frame
    } # for each monocen
  } # fi exists monocen

  if(exists("parlistOfdfMarkPosHolocen") & exists("dfMarkColorInternal")  ) {

    speListOfdfMarkPosHolocenArrowsLabels<-list()
#    parlistOfdfMarkPosHolocen1390<<-parlistOfdfMarkPosHolocen
    for (i in 1:length(parlistOfdfMarkPosHolocen)) {
      if(class(parlistOfdfMarkPosHolocen[[i]])=="data.frame" ) {

        parlistOfdfMarkPosHolocen[[i]]$style<-dfMarkColorInternal$style[match(parlistOfdfMarkPosHolocen[[i]]$markName, dfMarkColorInternal$markName)]

        if (legend=="inline"){
          speListOfdfMarkPosHolocenArrowsLabels[[i]]<-parlistOfdfMarkPosHolocen[[i]]
          speListOfdfMarkPosHolocenArrowsLabels[[i]]<-speListOfdfMarkPosHolocenArrowsLabels[[i]][which(speListOfdfMarkPosHolocenArrowsLabels[[i]]$style %in% c("downArrow","upArrow") ),]
          if(nrow(speListOfdfMarkPosHolocenArrowsLabels[[i]])>0){
            tryCatch(speListOfdfMarkPosHolocenArrowsLabels[[i]][which(speListOfdfMarkPosHolocenArrowsLabels[[i]]$style %in% "downArrow"),]$style<-"cMLeft",error = function(e) {""})
            tryCatch(speListOfdfMarkPosHolocenArrowsLabels[[i]][which(speListOfdfMarkPosHolocenArrowsLabels[[i]]$style %in% "upArrow"),]$style<-"cM",error = function(e) {""})

            tryCatch({
              downleft <- which(speListOfdfMarkPosHolocenArrowsLabels[[i]]$style %in% "cMLeft" )

              speListOfdfMarkPosHolocenArrowsLabels[[i]][downleft,]$markPos <-
                speListOfdfMarkPosHolocenArrowsLabels[[i]][downleft,]$markPos+
                speListOfdfMarkPosHolocenArrowsLabels[[i]][downleft,]$markSize
            },    error = function(e) {""} )

            tryCatch(speListOfdfMarkPosHolocenArrowsLabels[[i]]$markName <- paste0(speListOfdfMarkPosHolocenArrowsLabels[[i]]$markName," "),error = function(e) {""} )
            parlistOfdfMarkPosHolocen[[i]] <- dplyr::bind_rows(parlistOfdfMarkPosHolocen[[i]],speListOfdfMarkPosHolocenArrowsLabels[[i]])
          }
        } # inline

        tryCatch(parlistOfdfMarkPosHolocen[[i]]$protruding <- dfMarkColorInternal$protruding[match(parlistOfdfMarkPosHolocen[[i]]$markName,
                                                                                                dfMarkColorInternal$markName)]
                 ,error = function(e) {""}
        )


      if(nrow(parlistOfdfMarkPosHolocen[[i]][which(parlistOfdfMarkPosHolocen[[i]]$style %in% "cenStyle"),])>0) {
        if(!"markSize" %in% colnames(parlistOfdfMarkPosHolocen[[i]]) ) {
          parlistOfdfMarkPosMonocen[[i]]$markSize<-NA
        }
        for (m in 1:nrow(parlistOfdfMarkPosHolocen[[i]][which(parlistOfdfMarkPosHolocen[[i]]$style %in% "cenStyle"),] ) ) {
          if( is.na(parlistOfdfMarkPosHolocen[[i]][which(parlistOfdfMarkPosHolocen[[i]]$style %in% "cenStyle"),]$markSize[m]) ) {
            parlistOfdfMarkPosHolocen[[i]][which(parlistOfdfMarkPosHolocen[[i]]$style %in% "cenStyle"),]$markSize[m]<-

              dfChrSizeInternal[match(interaction( parlistOfdfMarkPosHolocen[[i]][which(
                parlistOfdfMarkPosHolocen[[i]]$style %in% "cenStyle"),][m,c("OTU", "chrName")]),
                interaction( dfChrSizeInternal[c("OTU", "chrName")] )
              ),]$centromereSize

            } # if
          } # for
        } # if nrow
      } # if data.frame
    } # for
  } # fi

  #################################################################################
  #
  #       MISSING OTUs
  #

  if(!is.na(addMissingOTUAfter[1]) ){
    if (length(missOTUspacings) != length(addMissingOTUAfter) ){
      missOTUspacings<-rep(missOTUspacings, abs(length(addMissingOTUAfter)/length(missOTUspacings)) ) [1:length(addMissingOTUAfter)]
    }
    for (i in 1:length(addMissingOTUAfter)){
    listOfdfChromSize <- append(listOfdfChromSize,
                                rep(NA,missOTUspacings[[i]]),
                                (which(names(listOfdfChromSize)==addMissingOTUAfter[[i]]) )
                                ) # append
    }
  } # fi

  #
  #     reverse
  #

  listOfdfChromSize<-rev(listOfdfChromSize)

  if(revOTUs){
    listOfdfChromSize<-rev(listOfdfChromSize)
  }

  #
  #	change of size based on number of sps
  #

  if(propWidth){
    chrWidth  <- chrWidth/length(listOfdfChromSize)
    chrSpacing<- chrSpacing/length(listOfdfChromSize)
  }  # dfMarkPosSq to listOfdfMarkPosInternalSq


  #######################################################
  #
  #   If Marks missing, rename duplicates of chrNames
  #
  #######################################################

  for (s in 1:length(listOfdfChromSize) ){
    OTUname<-names(listOfdfChromSize[s])
    if (exists("parlistOfdfMarkPosHolocen") ){
      OTUparlistOfdfMarkPosHolocen<-parlistOfdfMarkPosHolocen[which(names(parlistOfdfMarkPosHolocen) %in% OTUname)]
      if(length(OTUparlistOfdfMarkPosHolocen)==0){
        remove(OTUparlistOfdfMarkPosHolocen)
      }
    }
    if (exists("parlistOfdfMarkPosMonocen") ){
      OTUparlistOfdfMarkPosMonocen<-parlistOfdfMarkPosMonocen[which(names(parlistOfdfMarkPosMonocen) %in% OTUname)]
      if(length(OTUparlistOfdfMarkPosMonocen)==0){
        remove(OTUparlistOfdfMarkPosMonocen)
      }
    }
    if (exists("parlistOfdfMarkPosDataCen") ){
      OTUparlistOfdfMarkPosDataCen<-parlistOfdfMarkPosDataCen[which(names(parlistOfdfMarkPosDataCen) %in% OTUname)]
      if(length(OTUparlistOfdfMarkPosDataCen)==0){
        remove(OTUparlistOfdfMarkPosDataCen)
      }
    }
    mybooleanChrName <- !exists("OTUparlistOfdfMarkPosHolocen") & !exists("OTUparlistOfdfMarkPosMonocen") & !exists("OTUparlistOfdfMarkPosDataCen")
    dfChromSize<-fixChrNameDupDF(listOfdfChromSize[s], mybooleanChrName)
    listOfdfChromSize[[s]]<-dfChromSize[[1]]
  }

  #####################
  #   total size of chr
  #####################

  #
  #   add column total to data.frames
  #

  listOfdfChromSize <- addChrSizeColumn(listOfdfChromSize)

  {
    totalLength<-lapply(listOfdfChromSize, function(x) tryCatch(x$chrSize, error=function(e) NA)  )
    ifelse(
      inherits(totalLength, "matrix"),
      totalLength <- base::split(totalLength, col(totalLength) )
      ,NA)
    normalizeToOne<-karHeight/max(unlist(totalLength) , na.rm=TRUE)
  }

  ##############################################################################
  # order by size
  ##############################################################################

    if(orderChr=="size") {
        orderlist<-lapply(totalLength, function(x) order(x, decreasing = TRUE) )
    } else if(orderChr=="name"){
        orderlist<-lapply(listOfdfChromSize, function(x) tryCatch(order(x$chrName), error=function(e) NA ) )
    } else if(orderChr=="original" | orderChr=="group") {
        orderlist<-lapply(listOfdfChromSize, function(x) tryCatch(1:max(order(x$chrName) ), error=function(e) NA ) )
    }

    ##################################################
    #
    #   add column of new chro index to data.frames
    #
    ##################################################

  listOfdfChromSize <- addNeworderColumn(listOfdfChromSize,orderlist)

    ###################################################
    #     groups
    ###################################################

    if("group" %in% colnames(dfChrSizeInternal)) {
      message(crayon::blue("group column present - remove column if not using") )
      grouporderlist<-lapply(listOfdfChromSize, function(x) tryCatch(order(x$group), error=function(e) NA ) )

      if(orderChr=="group") {

        for (s in 1:length(listOfdfChromSize)){
          if(class(listOfdfChromSize[[s]])=="data.frame") {
          listOfdfChromSize[[s]]<-listOfdfChromSize[[s]][grouporderlist[[s]], ]
          listOfdfChromSize[[s]]$neworder<-1:nrow(listOfdfChromSize[[s]])
          } #fi
        } # end for

      } # order

    } # if group colname

    ##
    # order by group and chromosome
    ##

    ##################################################
    #
    #   important - add new indexer to df of marks - adds neworder column to listOfdfChromSizeCenType
    #
    ##################################################

    if(exists("parlistOfdfMarkPosHolocen") ){

      parlistOfdfMarkPosHolocen<-newOrderColumn(listOfdfChromSize,parlistOfdfMarkPosHolocen)

    } # end if presence of parlistOfdfMarkPosHolocen order

    if(exists("parlistOfdfMarkPosMonocen") ) {

      parlistOfdfMarkPosMonocen<-newOrderColumn(listOfdfChromSize,parlistOfdfMarkPosMonocen)

    } # end if presence of parlistOfdfMarkPosMonocen

    ######################################################
    #
    #   important - add new indexer to d.f DataCen
    #
    ######################################################

    if (exists("parlistOfdfMarkPosDataCen")){

      parlistOfdfMarkPosDataCen<-newOrderColumn(listOfdfChromSize,parlistOfdfMarkPosDataCen)

    }  # end presence of dfCenMarksInternal

  #######################
  #
  #      main plot
  #
  #######################

  {
    # Monocen
    num_species<-length(listOfdfChromSize)
  }

 # processing for main plot

    # Monocen
{
    rownumber<-2
    chromosome_ns<-sapply(listOfdfChromSize, function(x) nrow(x) )

    # listOfdfChromSize ->     chromosome_ns ->     arms_number ->     armRepVector

    arms_number<-sapply(chromosome_ns, function(x) x*2)

    armRepVector<-lapply(arms_number, function(x) return(tryCatch( c(rep(rownumber,ceiling(x/rownumber) ) ), error=function(e) NA ) ))

    rownumber<-1
    chromRepVector<-lapply(chromosome_ns, function(x) return(tryCatch( c(rep(rownumber,ceiling(x/rownumber) ) ), error=function(e) NA ) ))

    #
    #   creating x y main coords fro chr.
    #

    ym<-xm<-y<-x<-list()
    segX0<-segX1<-segY0<-segY1<-list()
}
########################################################################################################################################
  if(!missing(moveKarHor)) {
    moveKarHor2<-moveKarHor
    } else {# if
    moveKarHor2<-""
    }

  for (s in 1:num_species) {
      if(class(listOfdfChromSize[[s]])=="data.frame"){
        #######################################################################################################

      if(attr(listOfdfChromSize[[s]], "cenType")=="monocen") {       # monocen

        if(attr(listOfdfChromSize[[s]],"ytitle")=="cM"){
          chrWidth2  <-specialChrWidth
          chrSpacing2<-specialChrSpacing
        } else {
          chrWidth2<-chrWidth
          chrSpacing2<-chrSpacing
        }

        croxright<-  croxleft<-  croytop<-croybot <- list()

      for (i in 1:length(armRepVector[[s]] )){

        centromereSize2<-as.numeric(attr(listOfdfChromSize[[s]],"centromere"))


        croybot[i]<-tryCatch(list(c(karHeight,
                           rep((karHeight-(listOfdfChromSize[[s]][,"longArmSize"]*normalizeToOne)[i]),2),
                           karHeight
                          )#c
                        ), error=function(e) NA ) # list

        croytop[i]<-tryCatch( list(c((karHeight+(centromereSize2*normalizeToOne)+(listOfdfChromSize[[s]][,"shortArmSize"]*normalizeToOne)[i] ),
                           rep((karHeight+(centromereSize2*normalizeToOne)),2),
                           (karHeight+(centromereSize2*normalizeToOne)+(listOfdfChromSize[[s]][,"shortArmSize"]*normalizeToOne)[i] )
        ) # c
        ), error=function(e) NA) # list
      } # for

      crox<-matrix(rep(NA,length(armRepVector[[s]])*4),ncol=4, nrow=length(armRepVector[[s]]) )

#      listOfdfChromSize1715<<-listOfdfChromSize

      for (i in 1:length(armRepVector[[s]])){
        crox[i,] <- c(
          rep( ( (chrSpacing2 * i + ( i * chrWidth2) ) + chrWidth2),2),
          rep( ( (chrSpacing2 * i + ( i * chrWidth2) )  ),2)
        ) # c rox
      } # for

      if(names(listOfdfChromSize)[s] %in% moveKarHor2 ) {
        crox<-crox+mkhValue

        if(!missing(anchor)) {
          if(anchor){
            x1<-min(crox)-mkhValue
            x2<-min(crox)-chrWidth2
            y<- ( (min(unlist(croytop) )+max(unlist(croybot)) ) /2 ) +  karHeiSpace*(s-1)
            yh<-y+karHeiSpace*anchorVsizeF
            yl<-y-karHeiSpace*anchorVsizeF
            segX0[[s]]<-c(x1+moveAnchorV,x1+moveAnchorV)
            segY0[[s]]<-c(y,yh)
            segX1[[s]]<-c(x2+moveAnchorH,x1+moveAnchorV)
            segY1[[s]]<-c(y,yl)

          }
        }
      }

# crox1721<<-crox

      #
      #   groups
      #

      if(circularPlot==FALSE & "group" %in% colnames(listOfdfChromSize[[s]])  ) {
        lens <- rle(listOfdfChromSize[[s]]$group)$lengths
        names(lens)<-rle(listOfdfChromSize[[s]]$group)$values
#       lens1715<<-lens
        clens<-cumsum(lens)
        clens2<-clens[!is.na(names(clens))]
        lens2<-lens[!is.na(names(clens))]

        # groupSepar=.5
#       crox1718<<-crox
        if(length(clens2)>0 ) {
          for (i in 1:(length(clens2) ) ) {
            groupSize <- lens2[[i]]
            if( clens2[[i]] - lens2[[i]] != tryCatch(clens2[[i-1]], error=function(e){0} ) ) {

            tryCatch(crox[(clens2[[i]]-groupSize+1):nrow(crox),] <- crox[(clens2[[i]]-groupSize+1):nrow(crox),] + chrSpacing2*groupSepar, error=function(e){"nothing before group"})
            } # if
          }
          for (i in 1:(length(clens2) ) ){
            tryCatch(crox[(clens2[[i]]+1):nrow(crox),] <- crox[(clens2[[i]]+1):nrow(crox),] + chrSpacing2*groupSepar, error=function(e){"nothing after group" })
          }
        } # if
#       crox1739<<-crox
      }

      xm[[s]] <- rbind(crox,crox)

      x[[s]]  <- base::split(xm[[s]], row(xm[[s]]))

      ifelse(any(is.na(x[[s]]) ), x[[s]]<-NA,"") # ifelseinloop

      ym[[s]] <- t(sapply (c(croybot,croytop ), function (x) {length (x) ; return (x)} ) ) + karHeiSpace*(s-1)

      } # fi is monocen
        ########################################################################################### HOLOCEN

      if (attr(listOfdfChromSize[[s]], "cenType")=="holocen" ) {
        if(attr(listOfdfChromSize[[s]],"ytitle")=="cM"){
          chrWidth2  <-specialChrWidth
          chrSpacing2<-specialChrSpacing
        } else {
          chrWidth2<-chrWidth
          chrSpacing2<-chrSpacing
        }

      croxright <-  croxleft <- croytop <- list()

      for (i in 1:length(chromRepVector[[s]])){

        croytop[i]<-tryCatch( list(c(   (karHeight/2+(listOfdfChromSize[[s]][,"chrSize"]*normalizeToOne)[i]   ), # 0+
                                        rep(karHeight/2,2),                                                              # 0,2
                                        (karHeight/2+(listOfdfChromSize[[s]][,"chrSize"]*normalizeToOne)[i]  )   # 0+
        ) # c
        ),error=function(e) NA)  # list
      } # for

      crox <- matrix(rep(NA,length(chromRepVector[[s]])*4),ncol=4, nrow=length(chromRepVector[[s]]) )

      for (i in 1:length(chromRepVector[[s]])){
        crox[i,] <- c(
                      rep( ( (chrSpacing2*i+( i*chrWidth2))+chrWidth2),2),
                      rep( ( (chrSpacing2*i+( i*chrWidth2))     ),2)
                    ) # c rox
      } # for

        if(names(listOfdfChromSize)[s] %in% moveKarHor2 ){
          crox<-crox+mkhValue

          if(!missing(anchor)) {
            if(anchor){
              x1<-min(crox)-mkhValue
              x2<-min(crox)-chrWidth2
              y<- ( (min(unlist(croytop) )+max(unlist(croytop)) ) /2 ) +  karHeiSpace*(s-1)
              yh<-y+karHeiSpace*anchorVsizeF
              yl<-y-karHeiSpace*anchorVsizeF
              # segments(c(x1,x1),c(y,yh),c(x2,x1),c(y,yl))
              segX0[[s]]<-c(x1+moveAnchorV,x1+moveAnchorV)
              segY0[[s]]<-c(y,yh)
              segX1[[s]]<-c(x2+moveAnchorH,x1+moveAnchorV)
              segY1[[s]]<-c(y,yl)

            }
          }
        }

      #
      # group
      #

      if(circularPlot==FALSE & "group" %in% colnames(listOfdfChromSize[[s]])  ) {
        lens <- rle(listOfdfChromSize[[s]]$group)$lengths
        names(lens)<-rle(listOfdfChromSize[[s]]$group)$values
#        lens1715<<-lens
        clens<-cumsum(lens)
        clens2<-clens[!is.na(names(clens))]
        lens2<-lens[!is.na(names(clens))]

        # groupSepar=.5
#        crox1718<<-crox
        if(length(clens2)>0 ) {
          for (i in 1:(length(clens2) ) ) {
            groupSize<-lens2[[i]]

            if( clens2[[i]] - lens2[[i]] != tryCatch(clens2[[i-1]], error=function(e){0} ) ) {
            tryCatch(crox[(clens2[[i]]-groupSize+1):nrow(crox),] <- crox[(clens2[[i]]-groupSize+1):nrow(crox),] + chrSpacing2*groupSepar, error=function(e){"nothing before group"})
            } #if
          }

          for (i in 1:(length(clens2) ) ){


            tryCatch(crox[(clens2[[i]]+1):nrow(crox),] <- crox[(clens2[[i]]+1):nrow(crox),] + chrSpacing2*groupSepar, error=function(e){"nothing after group" })
          }

        } # if
#        crox1739<<-crox
      }

      xm[[s]] <- crox
      x[[s]]  <- base::split(xm[[s]], row(xm[[s]]))
      ifelse(any(is.na(x[[s]]) ),x[[s]]<-NA,"") # ifelseinloop

      # listOfdfChromSize[[s]]<-x[[s]]

      ym[[s]] <- t(sapply ( croytop, function (x) {length (x) ; return (x)}) ) + (karHeiSpace*(s-1)  )

    } # holocen if

  } # data.frame
  # message(crayon::green(paste0("main plot calc section end" ) ) )

} # for species
    ###################################################################################################

   if (length(ym)==1){
      karSepar=FALSE
    }

    #
    #	reducing distance among OTUs
    #

  names(ym)<-names(listOfdfChromSize) # important here


    ymCopyC<-ymCopy2<-ymCopy<-ym # important must stay here before modifying ym

    if(is.na(addMissingOTUAfter[1])){
      if(karSepar){
        for (s in 1:(length(ym)-1)) {
          diffnext<-abs( min(ym[[s+1]] ) - max(ym[[s]]) )

          ym[[s+1]]   =ym[[s+1]]   -diffnext
          tryCatch(segY0[[s+1]]<-segY0[[s+1]]-diffnext, error=function(e){})
                   tryCatch(segY1[[s+1]]<-segY1[[s+1]]-diffnext, error=function(e){})

          ifelse(amoSepar>diffnext,amoSepar2<-diffnext,amoSepar2<-amoSepar)

          ym[[s+1]]=ym[[s+1]]+amoSepar2
          tryCatch(segY0[[s+1]]<-segY0[[s+1]]+amoSepar2, error=function(e){})
                   tryCatch(segY1[[s+1]]<-segY1[[s+1]]+amoSepar2, error=function(e){} )

        }
      }
    }

for (s in 1:length(ym)) {
  if(all(is.null(ym[[s]])) ){
    # y[[s]] <-NA
    xm[[s]]<-ym[[s]]<-x[[s]] <-NA
    listOfdfChromSize[[s]]<-NA
  } # if
  attr(listOfdfChromSize[[s]],"groupPresence") <- 0
} # for

{
  areNA<-which(is.na(ym))

  listOfdfChromSizenoNA <- removeNAFromList(listOfdfChromSize,areNA)

  xmnoNA <- removeNAFromList(xm,areNA)

    for (i in 1:length(xmnoNA)){
      attr(xmnoNA[[i]],"cenType") <- attr(listOfdfChromSizenoNA[[i]],"cenType")
    }

  ymnoNA <- removeNAFromList(ym,areNA)
  #######################################################
  #
  #  modif x, creation y
  #
  #######################################################

  x<-removeNAFromList(x,areNA)

  y<-lapply(1:length(ymnoNA), function(s) base::split(ymnoNA[[s]], row(ymnoNA[[s]] )) )

  names(xm)<-names(listOfdfChromSize)

  names(x)<-names(listOfdfChromSizenoNA)
  names(y)<-names(listOfdfChromSizenoNA)

#  yInternalbefore<<-y
#  listOfdfChromSizeI1<<-listOfdfChromSize

for (s in 1:length(y) ){
    # lenYS<-length(y[[s]] )
   # if (names(listOfdfChromSizenoNA[s]) %in% c(monocenNames,holocenNames)){

    lenCS<-length(listOfdfChromSizenoNA[[s]][,"chrName"] )

    for (c in 1:lenCS )  {
      attr(y[[s]][[c]], "chrName1") <- listOfdfChromSizenoNA[[s]][,"chrName"][c]
    }
    if (length(y[[s]])>lenCS ){
      for (c in 1:lenCS )  {
      attr(y[[s]][[c+lenCS]], "chrName1") <- listOfdfChromSizenoNA[[s]][,"chrName"][c]
      }
    }
   # } # mono holo check
}

} # chunk


    #
    #     mark percentages %
    #

    if (exists("dfMarkPosInternal") & markPer!="" ) {
      perList <- perMark(dfMarkPosInternal,listOfdfChromSizenoNA)
#      perList1657<<-perList
    } # exist

    if (exists("dfMarkPosInternal") & showMarkPos ) {
      posTib <- posCalc(dfMarkPosInternal,listOfdfChromSizenoNA,bToRemove)
    } # exist


    {
      # monocenVector2<-integer()
      for(i in 1:length(listOfdfChromSizenoNA)){
        if(class(listOfdfChromSizenoNA[[i]])=="data.frame" ){
          if(attr(listOfdfChromSizenoNA[[i]],"cenType")=="monocen"){
            # monocenVector2<-c(monocenVector2,i)
            attr(listOfdfChromSizenoNA[[i]],"positionnoNA") <- i
          }
        }
      }

      # holocenVector2<-integer()
      for(i in 1:length(listOfdfChromSizenoNA)){
        if(class(listOfdfChromSizenoNA[[i]])=="data.frame" ){
          if(attr(listOfdfChromSizenoNA[[i]],"cenType")=="holocen"){
            # holocenVector2<-c(holocenVector2,i)
            attr(listOfdfChromSizenoNA[[i]],"positionnoNA") <- i
          }
        }
      }

    }

    for (s in 1:length(listOfdfChromSizenoNA) ) {
      attr(y[[s]], "positionnoNA")<- attr(listOfdfChromSizenoNA[[s]],"positionnoNA")
    }

#    yInternal<<-y
#    xInternal<<-x
#    ymInternal<<-ym
#    xmInternal<<-xm


    {
      monocenVector2<-integer()
      for(i in 1:length(listOfdfChromSize)){
        if(class(listOfdfChromSize[[i]])=="data.frame" ){
          if(attr(listOfdfChromSize[[i]],"cenType")=="monocen"){
            monocenVector2<-c(monocenVector2,i)
            attr(listOfdfChromSize[[i]],"position") <- i
          }
        }
      }

      monocenNames2<-names(listOfdfChromSize)[monocenVector2]
      listOfdfChromSizeMonocen<-listOfdfChromSize[monocenVector2]
      if(length(listOfdfChromSizeMonocen)==0){
        remove(listOfdfChromSizeMonocen)
      }

      holocenVector2<-integer()
      for(i in 1:length(listOfdfChromSize)){
        if(class(listOfdfChromSize[[i]])=="data.frame" ){
          if(attr(listOfdfChromSize[[i]],"cenType")=="holocen"){
            holocenVector2<-c(holocenVector2,i)
            attr(listOfdfChromSize[[i]],"position") <- i
          }
        }
      }
      holocenNames2<-names(listOfdfChromSize)[holocenVector2]
      listOfdfChromSizeHolocen<-listOfdfChromSize[holocenVector2]

      if(length(listOfdfChromSizeHolocen)==0){
        remove(listOfdfChromSizeHolocen)
      }
    }

    if(!missing(chrBorderColor)) {
      chrBorderColor2<-chrBorderColor
    } else {
      chrBorderColor2<-chrColor
    }

#{ # plot types
    xsizeplot<-(max(unlist(x), na.rm=TRUE)+xlimRightMod )- ( (min(unlist(x), na.rm=TRUE)-(xlimLeftMod)) )
    ysizeplot<- max(unlist(y), na.rm=TRUE)

    if(squareness < 1) {
      squareness<- 1
    }
    # plot types

      yInter<-intercalate(y,monocenNames)
      names(yInter)<-names(y)

      ylistNewChrSimple<-yVertoHor(yInter,monocenNames)
      names(ylistNewChrSimple)<-names(y)

      ylistTransChrSimple<-transYList(ylistNewChrSimple,shrinkFactor,monocenNames)
      names(ylistTransChrSimple)<-names(ylistNewChrSimple)

      if(squareness > 20) {  ########################################  squareness > 20 #################### > 20

        if(circularPlot==FALSE) {

        #####################################################################################################################
          if(callPlot){
            graphics::plot("",xlim=c( (min(unlist(x), na.rm=TRUE)-xlimLeftMod),(max(unlist(x), na.rm=TRUE)+xlimRightMod ) ),
                       ylim = c( ylimBotMod*-1 , ( (max(unlist(y), na.rm = TRUE) )+ylimTopMod) ) ,
                       ylab = "", xaxt='n',
                       xlab="", yaxt='n',main = NULL, frame.plot = FALSE, asp=asp, ...
                       )
             }
        ######################################################################################################################
          if(!missing(anchor) & !missing(moveKarHor) ){
            lapply(1:length(segX0), function(s) mapply(function(w,x,y,z) graphics::segments(w,
                                                                                   x,
                                                                                   y,
                                                                                   z
                                                                                   ),
                                                   w=segX0[[s]],
                                                   x=segY0[[s]],
                                                   y=segX1[[s]],
                                                   z=segY1[[s]]
            ) #m
            ) # l

            lapply(1:length(segX0), function(s) mapply(function(x,y) graphics::points(x,
                                                                                      y,
                                                                                      pch=pchAnchor,
                                                                                      bg="black"
                                                                                      ),
            y=segY0[[s]][1],
            x=segX0[[s]][1]
            ) #m
            ) # l

          }

          if(chromatids==FALSE){

          lapply(1:length(y), function(s) mapply(function(x,y) graphics::polygon(x=x,
                                                                                 y=y,
                                                                                 col=chrColor,
                                                                                 lwd=lwd.chr,
                                                                                 border=chrBorderColor2),
                                                 x=x[[s]],
                                                 y=y[[s]]
                                          ) #m
                 ) # l

          } # ct FALSE
          if (chromatids & holocenNotAsChromatids) {
            for (s in 1:length(y) ) {
              if(attr(listOfdfChromSizenoNA[[s]],"cenType")=="holocen"){

                  mapply(function(x,y) graphics::polygon(x=x,
                                                                                       y=y,
                                                                                       col=chrColor,
                                                                                       lwd=lwd.chr,
                                                                                       border=chrBorderColor2),
                                                       x=x[[s]],
                                                       y=y[[s]]
                  ) #m

              } # holocen
            } # for
          } # if
          if (chromatids) { # CHROMAT TRUE

            XSA<-YSA<-XLA<-YLA<-list()

            XHO1<-XHO2<-YHO1<-YHO2<-list()

            for (s in 1:length(y) ) {

              if(class(listOfdfChromSizenoNA[[s]])=="data.frame") {
                ########################################################

                if(attr(listOfdfChromSizenoNA[[s]], "cenType")=="monocen" ) {                 ############# monocen ###########



                  chrtXchrtYSA <- mapXYchromatidSA( (length(y[[s]])/2)+1 ,
                                                    length(y[[s]]),
                                                    y[[s]],
                                                    x[[s]],
                                                    xModifierMono )

                  chrtXchrtYLA <- mapXYchromatidLA( 1,
                                                   (length(y[[s]])/2) ,
                                                    y[[s]],
                                                    x[[s]],
                                                    xModifierMono )



                  XSA[[s]]<-chrtXchrtYSA$shortArmChrtx
                  YSA[[s]]<-chrtXchrtYSA$shortArmChrty

                  XLA[[s]]<-chrtXchrtYLA$longArmChrtx
                  YLA[[s]]<-chrtXchrtYLA$longArmChrty

                  attr(YSA[[s]], "positionnoNA")<- attr(listOfdfChromSizenoNA[[s]],"positionnoNA")
                  # attr(YLA[[s]], "positionnoNA")<- attr(listOfdfChromSizenoNA[[s]],"positionnoNA")

                  # important integration of long arm and short arms info

                  YSA[[s]][sapply(YSA[[s]], is.null)]<-YLA[[s]][!sapply(YLA[[s]], is.null)]

                  XSA[[s]][sapply(XSA[[s]], is.null)]<-XLA[[s]][!sapply(XLA[[s]], is.null)]

                  # short arm indices are in the second half of list, so YLA is a smaller list

                  names(XSA)[s]<-names(YSA)[s]<-names(y[s])

                  for (a in 1: length(YSA[[s]]) ){
                    # attr(roundedY[[s]][[a]],"chrName1")<- attr(y[[s]][[a]],"chrName1")
                    names(YSA[[s]])[a]<- names(y[[s]][a])
                    names(XSA[[s]])[a]<- names(y[[s]][a])
                  }

              } else if (attr(listOfdfChromSizenoNA[[s]], "cenType")=="holocen" & holocenNotAsChromatids==FALSE ) { # if monocen else  #####  holo

                  # mapXYchromatidHolo <- function(start,end,y,x,xModifier ){
                  chrtXchrtYHolo<-mapXYchromatidHolo(1 ,
                                                 (length(y[[s]]) ) ,
                                    y[[s]],
                                    x[[s]],
                                    xModifierHolo
                    )

                    XHO1[[s]]<-chrtXchrtYHolo$xCT1
                    XHO2[[s]]<-chrtXchrtYHolo$xCT2
                    YHO1[[s]]<-chrtXchrtYHolo$yCT1
                    YHO2[[s]]<-chrtXchrtYHolo$yCT2

                    attr(YHO1[[s]], "positionnoNA")<- attr(listOfdfChromSizenoNA[[s]],"positionnoNA")
                    attr(YHO2[[s]], "positionnoNA")<- attr(listOfdfChromSizenoNA[[s]],"positionnoNA")

                    names(YHO1)[s]<-names(XHO1)[s]<-names(YHO2)[s]<-names(XHO2)[s]<-names(y[s])

                    for (a in 1: length(YHO1[[s]])){
                      names(YHO1[[s]])[a]<- names(y[[s]][a])
                      names(XHO1[[s]])[a]<- names(y[[s]][a])
                      names(YHO2[[s]])[a]<- names(y[[s]][a])
                      names(XHO2[[s]])[a]<- names(y[[s]][a])
                    }
                } # holo holocenNotAsChromatids F
              } # df
            } # for s

            #make YSA same length as Y

            if(length(YSA)>0) { # monocen plot chromatid sq

            # this is not only SA, were integrated

            YSA<-YSA[lengths(YSA) != 0]
            XSA<-XSA[lengths(XSA) != 0]

              lapply(1:length(YSA), function(s) mapply(function(x,y) graphics::polygon(x=x, y=y,
                                                                                     col=chrColor,
                                                                                     lwd=lwd.chr,
                                                                                     border=chrBorderColor2),
                                                     x=XSA[[s]],
                                                     y=YSA[[s]]
                   )#m
              ) #l

            } # if YSA

            if(length(YHO1)>0){ # PLOT HOLOCEN SQ CHROMATID

              YHO1 <- YHO1[lengths(YHO1) != 0]
              XHO1 <- XHO1[lengths(XHO1) != 0]
              YHO2 <- YHO2[lengths(YHO2) != 0]
              XHO2 <- XHO2[lengths(XHO2) != 0]

              # LEFT CHROMATID
              lapply(1:length(YHO1), function(s) mapply(function(x,y) graphics::polygon(x=x, y=y,
                                                                                     col=chrColor,
                                                                                     lwd=lwd.chr,
                                                                                     border=chrBorderColor2),
                                                     x=XHO1[[s]],
                                                     y=YHO1[[s]]
              )#m
              ) #l
              # RIGHT CHROMATID
              lapply(1:length(YHO1), function(s) mapply(function(x,y) graphics::polygon(x=x, y=y,
                                                                                     col=chrColor,
                                                                                     lwd=lwd.chr,
                                                                                     border=chrBorderColor2),
                                                     x=XHO2[[s]],
                                                     y=YHO2[[s]]
              )#m
              ) #l
            } # if YHO1


          } # chromatids FALSE TRUE

        } else { # circular false # circular TRUE sq > 20

          #
          #   x horizontal to vertical
          #

          xlistNewChr<-xHortoVer(x)

          circleMaps <- applyMapCircle(radius,circleCenter,circleCenterY,separFactor,ylistTransChrSimple,xlistNewChr,n,0,
                                       chrWidth,rotation=rotation)

          if(callPlot){

            graphics::plot("",xlim=c( (min(unlist(circleMaps), na.rm=TRUE)-xlimLeftMod),
                                      (max(unlist(circleMaps), na.rm=TRUE)+xlimRightMod )
            ),
            ylim = c( min (unlist(circleMaps), na.rm = TRUE) +ylimBotMod*-1 ,
                      ( (max(unlist(circleMaps), na.rm = TRUE) )+ylimTopMod)
            ) ,
            ylab = "",
            xaxt='n',
            xlab="",
            yaxt='n',
            main = NULL, frame.plot = FALSE,
            asp=asp,
            ...
            ) # plot
          } # callPlot

          #
          #   plot chr
          #

          drawPlot(circleMaps,chrColor,lwd.chr)

          #
          #   add OTU names
          #

          if(addOTUName) {

            firstXchrEach <- lapply(xlistNewChr, `[[`, 1)
            firstYchrEach <- lapply(ylistTransChrSimple, `[[`, 1)

            circleMapsOTUname <- mapOTUnames(firstYchrEach , firstXchrEach, ylistNewChrSimple, n, radius, circleCenter, circleCenterY,separFactor,
                                           OTUlabelSpacing, chrWidth,rotation=rotation)
            addOTUnames(circleMapsOTUname,OTUTextSize,OTUsrt,OTUplacing,OTUfont2,OTUfamily2,
                      circleCenter,OTULabelSpacerx,circleCenterY,OTULabelSpacery,
                      OTUlegendHeight*normalizeToOne,radius,chrWidth,normalizeToOne,OTUcentered,OTUjustif,separFactor,labelSpacing)
          }

          #
          # Plot chr names
          #

          if(chrId!="") {
            listXChrCenter <- mapChrCenter(xlistNewChr)
            listYChrCenter <- mapChrCenter(ylistTransChrSimple)
            names(listYChrCenter)<-names(ylistTransChrSimple)
            chrNames <- applyMapCircle(radius,circleCenter,circleCenterY,separFactor,
                                       listYChrCenter,listXChrCenter,n,-chrLabelSpacing,chrWidth,
                                       specialOTUNames=specialOTUNames,chrWFactor=chrWFactor,rotation=rotation)
            plotChrNames(chrNames, indexIdTextSize, chrId,monocenNames,chrColor)
          }

        } # cP true if else

      } else {         # if squareness > 20 ###########         else             ########          squareness <= 20

        pts<- seq(-pi/2, pi*1.5, length.out = n*4)
        ptsl<-split(pts, sort(rep(1:4, each=length(pts)/4, len=length(pts))) )

        yMod<-y

        roundedY<-roundedX<-list()

        XSARO<-YSARO<-XLARO<-YLARO<-list()

        XHO1Ro<-YHO1Ro<-XHO2Ro<-YHO2Ro<-list()

        for (s in 1:length(yMod) ) {

          if(class(listOfdfChromSizenoNA[[s]])=="data.frame") {
            ########################################################

          if(attr(listOfdfChromSizenoNA[[s]], "cenType")=="monocen" ) {                 ############# monocen ###########
            if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
              chrWidth2  <-specialChrWidth
            } else {
              chrWidth2 <- chrWidth
            }
            r2<- chrWidth2/(squareness*2)

            if(chromatids==FALSE | circularPlot==TRUE) {

                  shortxyCoords<-mapXY(( (length(yMod[[s]])/2)+1 ) , length(yMod[[s]]),
                                           y[[s]],yMod[[s]],x[[s]],
                                           yfactor,r2,
                                       ptsl[[1]],ptsl[[2]],ptsl[[4]],ptsl[[3]]
                                      )

                  longxyCoords<-mapXY( 1 , (length(yMod[[s]])/2 ) ,
                                          y[[s]], yMod[[s]] ,x[[s]],
                                          yfactor,r2,
                                        ptsl[[1]],ptsl[[2]],ptsl[[4]],ptsl[[3]]
                                      )

                  # integration of X coords of short and long
                  shortxyCoords$roundedX[sapply(shortxyCoords$roundedX, is.null)] <-
                    longxyCoords$roundedX[!sapply(longxyCoords$roundedX, is.null)]

                  # integration of Y coords of short and long
                  shortxyCoords$roundedY[sapply(shortxyCoords$roundedY, is.null)] <-
                    longxyCoords$roundedY[!sapply(longxyCoords$roundedY, is.null)]

#                  shortxyCoordsINEW<<-shortxyCoords

                                 # this is not short only
                  roundedX[[s]]<-shortxyCoords$roundedX
                  roundedY[[s]]<-shortxyCoords$roundedY
                  attr(roundedY[[s]], "positionnoNA")<- attr(listOfdfChromSizenoNA[[s]],"positionnoNA")
                  attr(roundedY[[s]], "cenType")<- attr(listOfdfChromSizenoNA[[s]],"cenType")

                  for (a in 1: length(roundedY[[s]])){
                    # attr(roundedY[[s]][[a]],"chrName1")<- attr(y[[s]][[a]],"chrName1")
                    names(roundedY[[s]])[a]<- names(y[[s]][a])
                    names(roundedX[[s]])[a]<- names(y[[s]][a])
                  }
                  names(roundedX)[s]<-names(roundedY)[s]<-names(y[s])

            } else if(chromatids==TRUE & circularPlot==FALSE) { # chromatids FALSE TRUE

              pts<- seq(-pi/2, pi*1.5, length.out = n*4)

              chrtXchrtYSARo <- mapXYchromatidSARo(
                (length(y[[s]])/2)+1  ,
                (length(y[[s]]) )
                ,y[[s]],
                x[[s]],
                r2,
                xModifierMono ,
                pts)
#              chrtXchrtYSARo2157<<-chrtXchrtYSARo

              chrtXchrtYLARo <- mapXYchromatidLARo(
                1 ,
                (length(y[[s]])/2 )
                ,y[[s]],
                x[[s]],
                r2,
                xModifierMono,
                pts)

              XSARO[[s]] <- chrtXchrtYSARo$RoundedSAChrtx
              YSARO[[s]] <-chrtXchrtYSARo$RoundedSAChrty
              XLARO[[s]] <- chrtXchrtYLARo$RoundedLAChrtx
              YLARO[[s]] <-chrtXchrtYLARo$RoundedLAChrty

#              listOfdfChromSizenoNA2179<<-listOfdfChromSizenoNA
              attr(YSARO[[s]], "positionnoNA") <- attr(listOfdfChromSizenoNA[[s]],"positionnoNA")

              # important integration of long arm and short arms info
              YSARO[[s]][sapply(YSARO[[s]], is.null)] <- YLARO[[s]][!sapply(YLARO[[s]], is.null)]

              XSARO[[s]][sapply(XSARO[[s]], is.null)] <- XLARO[[s]][!sapply(XLARO[[s]], is.null)]
              # short arm indices are in the second half of list, so YLA is a smaller list

              names(XSARO)[s]<-names(YSARO)[s]<-names(y[s])

              for (a in 1: length(YSARO[[s]]) ){
                # attr(roundedY[[s]][[a]],"chrName1")<- attr(y[[s]][[a]],"chrName1")
                names(YSARO[[s]])[a]<- names(y[[s]][a])
                names(XSARO[[s]])[a]<- names(y[[s]][a])
              }

            } # chromatids TRUE

    } else if (attr(listOfdfChromSizenoNA[[s]], "cenType")=="holocen") { # if monocen else  holocen #########################

            if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
              chrWidth2  <-specialChrWidth
            } else {
              chrWidth2<-chrWidth
            }

            r2<- chrWidth2/(squareness*2)

            if(chromatids==FALSE | holocenNotAsChromatids | circularPlot==TRUE) {

            xyCoords<-mapXY(1 , (length(yMod[[s]]) ) ,
                                y[[s]], yMod[[s]] ,
                                x[[s]],
                                yfactor,r2,
                                # pts_1,pts_2,pts_3,pts_4
                            ptsl[[1]],ptsl[[2]],ptsl[[4]],ptsl[[3]]
                            )

            roundedX[[s]]<-xyCoords$roundedX
            roundedY[[s]]<-xyCoords$roundedY
            attr(roundedY[[s]], "positionnoNA")<- attr(listOfdfChromSizenoNA[[s]],"positionnoNA")
            attr(roundedY[[s]], "cenType")<- attr(listOfdfChromSizenoNA[[s]],"cenType")

            for (a in 1: length(roundedY[[s]])){
              # attr(roundedY[[s]][[a]],"chrName1")<- attr(y[[s]][[a]],"chrName1")
              names(roundedY[[s]])[a]<- names(y[[s]][a])
              names(roundedX[[s]])[a]<- names(y[[s]][a])
            }
            names(roundedX)[s]<-names(roundedY)[s]<-names(y[s])

            } else if( chromatids==TRUE & holocenNotAsChromatids==FALSE | circularPlot==FALSE) { # chromatids FALSE TRUE

              pts<- seq(-pi/2, pi*1.5, length.out = n*4)

              chrtXchrtYHoloRo<-mapXYchromatidHoloRo(1 ,
                                                 (length(y[[s]]) ) ,
                                                 y[[s]],
                                                 x[[s]],
                                                 r2,
                                                 xModifierHolo,
                                                 pts
              )

              XHO1Ro[[s]]<-chrtXchrtYHoloRo$holoRightx
              YHO1Ro[[s]]<-chrtXchrtYHoloRo$holoRighty

              XHO2Ro[[s]]<-chrtXchrtYHoloRo$holoLeftx
              YHO2Ro[[s]]<-chrtXchrtYHoloRo$holoLefty

              attr(YHO1Ro[[s]], "positionnoNA")<- attr(listOfdfChromSizenoNA[[s]],"positionnoNA")
              attr(YHO2Ro[[s]], "positionnoNA")<- attr(listOfdfChromSizenoNA[[s]],"positionnoNA")

              names(YHO1Ro)[s]<-names(XHO1Ro)[s]<-names(YHO2Ro)[s]<-names(XHO2Ro)[s]<-names(y[s])

              for (a in 1: length(YHO1Ro[[s]])){
                names(YHO1Ro[[s]])[a]<- names(y[[s]][a])
                names(XHO1Ro[[s]])[a]<- names(y[[s]][a])
                names(YHO2Ro[[s]])[a]<- names(y[[s]][a])
                names(XHO2Ro[[s]])[a]<- names(y[[s]][a])
              }

            } # chrt FALSE

          } # holocen ################################################################################# END HOLOCEN
    } # d.f.

   } # for species

  if(exists("roundedY") ){
    roundedY<-roundedY[!is.na(roundedY)]
    roundedX<-roundedX[!is.na(roundedX)]
  }

  if(circularPlot==FALSE){

    ##################################################################################################################### <20
    if(callPlot){
    graphics::plot("",xlim=c( (min(unlist(x), na.rm=TRUE)-xlimLeftMod),(max(unlist(x), na.rm=TRUE)+xlimRightMod ) ),
                   ylim = c( ylimBotMod*-1 ,( (max(unlist(y), na.rm = TRUE) )+ylimTopMod) ) , ylab = "", xaxt='n',
                   xlab="", yaxt='n',main = NULL, frame.plot = FALSE, asp=asp, ...)
    }
    ######################################################################################################################
    if(!missing(anchor) & !missing(moveKarHor) ){
      lapply(1:length(segX0), function(s) mapply(function(w,x,y,z) graphics::segments(w,
                                                                                      x,
                                                                                      y,
                                                                                      z
      ),
      w=segX0[[s]],
      x=segY0[[s]],
      y=segX1[[s]],
      z=segY1[[s]]
      ) #m
      ) # l

      lapply(1:length(segX0), function(s) mapply(function(x,y) graphics::points(x,
                                                                                y,
                                                                                pch=pchAnchor,
                                                                                bg="black"
                                                                                ),
      y=segY0[[s]][1],
      x=segX0[[s]][1]
      ) #m
      ) # l

    }
      if (chromatids==FALSE ) {
            lapply(1:length(y), function(s) mapply(function(x,y) graphics::polygon(x=x, y=y,
                                                                                     col=chrColor,
                                                                                     lwd=lwd.chr,
                                                                                     border=chrBorderColor2),
                                                     x=roundedX[[s]],
                                                     y=roundedY[[s]]
                                              )#m
            ) #l
      }
      if (chromatids & holocenNotAsChromatids) {

      for (s in 1:length(roundedY) ) {
        if(length(attr(roundedY[[s]],"cenType"))>0){
        if(attr(roundedY[[s]],"cenType")=="holocen"){

          mapply(function(x,y) graphics::polygon(x=x,
                                                 y=y,
                                                 col=chrColor,
                                                 lwd=lwd.chr,
                                                 border=chrBorderColor2),
                 x=roundedX[[s]],
                 y=roundedY[[s]]
          ) #m

        } # holocen
        }
      } # for
    } # if
     if (chromatids) { #        CHRT TRUE

        # this is not only SA, arms were integrated
        if(length(YSARO)>0) {
        YSARO<-YSARO[lengths(YSARO) != 0]
        XSARO<-XSARO[lengths(XSARO) != 0]

        lapply(1:length(YSARO), function(s) mapply(function(x,y) graphics::polygon(x=x, y=y,
                                                                                 col=chrColor,
                                                                                 lwd=lwd.chr,
                                                                                 border=chrBorderColor2),
                                                 x=XSARO[[s]],
                                                 y=YSARO[[s]]
          )#m
         ) #l
        } # YHARO len

        if(length(YHO1Ro)>0){ # PLOT HOLOCEN SQ CHROMATID

          YHO1Ro <- YHO1Ro[lengths(YHO1Ro) != 0]
          XHO1Ro <- XHO1Ro[lengths(XHO1Ro) != 0]
          YHO2Ro <- YHO2Ro[lengths(YHO2Ro) != 0]
          XHO2Ro <- XHO2Ro[lengths(XHO2Ro) != 0]

          # LEFT CHROMATID
          lapply(1:length(YHO1Ro), function(s) mapply(function(x,y) graphics::polygon(x=x, y=y,
                                                                                    col=chrColor,
                                                                                    lwd=lwd.chr,
                                                                                    border=chrBorderColor2),
                                                    x=XHO1Ro[[s]],
                                                    y=YHO1Ro[[s]]
          )#m
          ) #l
          # RIGHT CHROMATID
          lapply(1:length(YHO1Ro), function(s) mapply(function(x,y) graphics::polygon(x=x, y=y,
                                                                                    col=chrColor,
                                                                                    lwd=lwd.chr,
                                                                                    border=chrBorderColor2),
                                                    x=XHO2Ro[[s]],
                                                    y=YHO2Ro[[s]]
          )#m
          ) #l
        } # if YHO1Ro

      } # CHRT T
   } else {  # cP FALSE # cP TRUE         ####################################################         circular - squareness < 20

    yInterLong<-intercalate(roundedY,monocenNames)
    names(yInterLong)<-names(y)

    ylistNewChrLong<-yVertoHor(yInterLong,monocenNames )
    names(ylistNewChrLong)<-names(y)

    #
    #   x horizontal to vertical
    #

    xlistNewChr <- xHortoVer(roundedX)

    ylistTransChr <- transYList(ylistNewChrLong,shrinkFactor,monocenNames)

    names(ylistTransChr)<-names(ylistNewChrLong)
#    ylistTransChrI<<-ylistTransChr
    #
      circleMaps<-applyMapCircle(radius,circleCenter,circleCenterY,separFactor,ylistTransChr,xlistNewChr,n,0,chrWidth,rotation=rotation)

      if(callPlot){
      graphics::plot("",xlim=c( (min(unlist(circleMaps), na.rm=TRUE)-xlimLeftMod),
                                    (max(unlist(circleMaps), na.rm=TRUE)+xlimRightMod )
          ),
          ylim = c( min (unlist(circleMaps), na.rm = TRUE) +ylimBotMod*-1 ,
                    ( (max(unlist(circleMaps), na.rm = TRUE) )+ylimTopMod)
                    ) ,
          ylab = "",
          xaxt='n',
          xlab="",
          yaxt='n',
          main = NULL,
          frame.plot = FALSE,
          asp=asp,
          ...
      ) # plot
      }

      #
      #   plot chr
      #

      drawPlot(circleMaps,chrColor,lwd.chr)

      #
      #   add OTU names
      #

      if(addOTUName){

      firstXchrEach <- lapply(xlistNewChr, `[[`, 1)

      firstYchrEach <- lapply(ylistTransChr, `[[`, 1)
#      #      chrNamesI<<-chrNames

      #
      #  OTU name place
      #

      circleMapsOTUname <- mapOTUnames(firstYchrEach , firstXchrEach, ylistNewChrLong, n, radius, circleCenter,
                                       circleCenterY,separFactor, OTUlabelSpacing, chrWidth,rotation=rotation)

      addOTUnames(circleMapsOTUname,OTUTextSize,OTUsrt,OTUplacing,OTUfont2,OTUfamily2,
                  circleCenter,OTULabelSpacerx,circleCenterY,OTULabelSpacery,
                  OTUlegendHeight*normalizeToOne,radius,chrWidth,normalizeToOne,OTUcentered,OTUjustif,separFactor,labelSpacing)
      }

      #
      # Plot chr. names
      #

      if(chrId!=""){
        listYChrCenter <- mapChrCenter(ylistTransChr)
        names(listYChrCenter)<-names(ylistTransChr)

#        #      listYChrCenterI<<-listYChrCenter

        listXChrCenter <- mapChrCenter(xlistNewChr)

       chrNames <- applyMapCircle(radius,circleCenter,circleCenterY,separFactor,
                                  listYChrCenter,listXChrCenter,n,
                                  -chrLabelSpacing, # inner
                                  chrWidth,
                                  specialOTUNames=specialOTUNames,chrWFactor=chrWFactor,rotation=rotation)

       plotChrNames(chrNames, indexIdTextSize, chrId,monocenNames ,chrColor)

       } # if
} # circularplot ####################################################################################################

} # squareness if squareness <20 end plot

  ####################################################################################
  #
  #                                                ruler calculate
  #
  ####################################################################################
    # listOfdfChromSize<-rev(listOfdfChromSize)


# message(crayon::red(paste("back2089",rulerIntervalback)))
if(circularPlot==FALSE){
  if (ruler) {
    if (length( which(names(listOfdfChromSize) %in% monocenNames) )>0 ) {
    # for (x in monocenVector2){

    maxShortRound<-lapply(1:length(listOfdfChromSizeMonocen),
                            function(x) {
                            return(tryCatch(customCeiling(max(listOfdfChromSizeMonocen[[x]]$shortArmSize),
                                         ceilingFactor), error=function(e) NA ) )
                            }
                          )
    names(maxShortRound) <- names(listOfdfChromSizeMonocen)

    maxLongRound <-lapply(1:length(listOfdfChromSizeMonocen),
                          function(x)
                            tryCatch(customCeiling(max(listOfdfChromSizeMonocen[[x]]$longArmSize ),
                                  ceilingFactor), error=function(e) NA ) )


    fromZerotoMaxLong<-fromZerotoMaxShort<-list()

    for (i in 1:length(maxShortRound)) {

      corr_index<-which(names(listOfdfChromSize) %in% names(listOfdfChromSizeMonocen)[[i]] )
      divisor2<-as.numeric(attr(listOfdfChromSize[[corr_index]],"divisor"))

      if ( attr(listOfdfChromSizeMonocen[[i]], "ytitle" )=="cM" ) {
        if (!missing(rulerIntervalcM)) {
          if (rulerIntervalcM > divisor2/11 ) {
            rulerInterval2<-rulerIntervalcM/divisor2
          } else {
            message(crayon::red("rulerIntervalCM too small, using default"))
            rulerInterval2<-1
          }
        } else {
          rulerInterval2<-1
        }
      } else if( attr(listOfdfChromSizeMonocen[[i]], "ytitle" )=="Mb" ) {
        if (!missing(rulerIntervalMb)) {
          if (rulerIntervalMb > divisor2/11 ) {
            rulerInterval2<-rulerIntervalMb/(divisor2)
          } else {
            message(crayon::red("rulerIntervalMb too small, using default"))
            rulerInterval2<-1
          }
        } else {
          rulerInterval2<-1
        }
      } else if(attr(listOfdfChromSizeMonocen[[i]], "ytitle" )=="notMb" ) {
        # rulerInterval2<-rulerInterval2/divisor2
        rulerInterval2 <- ifelse(missing(rulerInterval),
                                 1,
                                 rulerInterval/divisor2
        )
      }

      if(rulerInterval2 > maxShortRound[[i]]){
        message(crayon::red(paste0("rulerInterval too big. Use smaller rulerInterval; rulerIntervalMb or rulerIntervalcM
                                   if you have Mb data or specialOTUNames, respectively" ) )
                )
      }

      # listOfdfChromSizeMonocen  - >  maxShortRound  ->  fromZerotoMaxShort

      fromZerotoMaxShort[[i]]<- tryCatch(seq(
        from = 0,
        to = (maxShortRound[[i]] +
        (rulerInterval2 * ifelse(maxShortRound[[i]] %% rulerInterval2>0,1,0) ) # ifelseinloop
        - maxShortRound[[i]] %% rulerInterval2),
        by = rulerInterval2), error=function(e) NA
        ) # try

      fromZerotoMaxLong[[i]]<- tryCatch(seq(
        from = 0,
          to = (maxLongRound[[i]] + (rulerInterval2 * ifelse(maxLongRound[[i]] %% rulerInterval2>0,1,0) ) # ifelseinloop
          - maxLongRound[[i]] %% rulerInterval2),
          by = rulerInterval2),
        error=function(e) NA
      ) # try

      if(!is.na(centromereSize)){
        centromereSize2<-centromereSize
      } else {
        centromereSize2<-divisor2
      }

      attr(fromZerotoMaxShort[[i]],"centromere") <- centromereSize2/divisor2

      remove(rulerInterval2)

    } # for maxshortround

    names(fromZerotoMaxShort)<-names(maxShortRound)

    ycoordLongRound <-lapply(1:length(fromZerotoMaxLong), function(x) {
      pos <- as.numeric(attr(listOfdfChromSizeMonocen[[x]],"position") )
      unlist(
        lapply(1:length(fromZerotoMaxLong[[x]]), function(y)
          (karHeight - (fromZerotoMaxLong[[x]][y] * normalizeToOne) ) + (karHeiSpace*(pos-1))
        ) #l
      ) #u
    } # ycoordLongRound
    ) # l

    names(ycoordLongRound)<-(monocenNames2)

    ycoordShortRound  <-lapply(1:length(fromZerotoMaxShort), function(x){
      pos<-as.numeric(attr(listOfdfChromSizeMonocen[[x]],"position") )
      unlist(
        lapply(1:length(fromZerotoMaxShort[[x]]), function(y)
          # (karHeight+(centromereSize*normalizeToOne)+(fromZerotoMaxShort[[x]][y]*normalizeToOne))+(1*karHeiSpace*(pos-1))
          (karHeight+(as.numeric(attr(fromZerotoMaxShort[[x]],"centromere"))*normalizeToOne)+(fromZerotoMaxShort[[x]][y]*normalizeToOne))+(1*karHeiSpace*(pos-1))

        ) # l
      ) # u
    }
    ) #l

    names(ycoordShortRound)<-(monocenNames2)

    if(is.na(addMissingOTUAfter[1])){
      if(karSepar){
        for (s in 1:(length(ymCopy)-1) ) {

          diffnext<-abs(min(ymCopy[[s+1]] ) - max(ymCopy[[s]]) )
          ymCopy[[s+1]] <- ymCopy[[s+1]]-diffnext
          ifelse(amoSepar>diffnext,amoSepar2<-diffnext,amoSepar2<-amoSepar)
          ymCopy[[s+1]] <- ymCopy[[s+1]]+amoSepar2

          nameYm<-names(ymCopy)[[s+1]]

          if(length(which(names(ycoordLongRound)==nameYm))>0 ){
          ycoordLongRound[[which(names(ycoordLongRound)==nameYm)]] <- ycoordLongRound[[which(names(ycoordLongRound)==nameYm)]] -diffnext
          ycoordLongRound[[which(names(ycoordLongRound)==nameYm)]] <- ycoordLongRound[[which(names(ycoordLongRound)==nameYm)]] + amoSepar2

          ycoordShortRound[[which(names(ycoordShortRound)==nameYm)]]<- ycoordShortRound[[which(names(ycoordShortRound)==nameYm)]]-diffnext
          ycoordShortRound[[which(names(ycoordShortRound)==nameYm)]]<- ycoordShortRound[[which(names(ycoordShortRound)==nameYm)]]+ amoSepar2

          }
        }
      }
    }

    ycoordLongRound  <-ycoordLongRound[!is.na(ycoordLongRound)]
    ycoordShortRound <-ycoordShortRound[!is.na(ycoordShortRound)]

    ########################################################################################
    #
    #                                      add rulers
    #
    ########################################################################################

    ###############
    # short arm ruler labels
    ###############
    opar<-graphics::par(no.readonly = TRUE)
    on.exit(suppressWarnings(par(opar) ))
    graphics::par(mgp=c(3,rulerNumberPos,0))

    rulerPlot(ycoordShortRound,listOfdfChromSize,listOfdfChromSizeMonocen,fromZerotoMaxShort,rulerNumberSize,rulerPos,ruler.tck,
              lwd.chr,moveKarHor2,mkhValue,useMinorTicks,miniTickFactor)

    ################

    # long arm ruler labels

    ################

    rulerPlot(ycoordLongRound,listOfdfChromSize,listOfdfChromSizenoNA,fromZerotoMaxLong,rulerNumberSize,rulerPos,ruler.tck,
              lwd.chr,moveKarHor2,mkhValue,useMinorTicks,miniTickFactor)



    } # monocen

    ######################################################################################### holocen ruler


    if (length( which(names(listOfdfChromSize) %in% holocenNames) ) > 0 ) {

    maxChrRound<-lapply(1:length(listOfdfChromSizeHolocen),
                        function(x) {
                          return(tryCatch(
                        customCeiling(max(listOfdfChromSizeHolocen[[x]]$chrSize),
                        ceilingFactor),
                        error=function(e) NA )
                         )}
                       ) # l

    fromZerotoMaxChr<-list()

    for (i in 1:length(maxChrRound)) {

      corr_index<-which(names(listOfdfChromSize) %in% names(listOfdfChromSizeHolocen)[[i]] )
      divisor2<-as.numeric(attr(listOfdfChromSize[[corr_index]],"divisor"))

      if ( attr(listOfdfChromSizeHolocen[[i]], "ytitle" )=="cM" ) {
        if (!missing(rulerIntervalcM)) {
          if (rulerIntervalcM > divisor2/11 ) {
            rulerInterval2<-rulerIntervalcM/divisor2
          } else {
            message(crayon::red("rulerIntervalCM too small, using default"))
            rulerInterval2<-1
          }
        } else {
          rulerInterval2<-1
        }
      } else if( attr(listOfdfChromSizeHolocen[[i]], "ytitle" )=="Mb" ) {
        if (!missing(rulerIntervalMb)) {
          if (rulerIntervalMb > divisor2/11 ) {
            rulerInterval2<-rulerIntervalMb/(divisor2)
          } else {
            message(crayon::red("rulerIntervalMb too small, using default"))
            rulerInterval2<-1
          }
        } else {
          rulerInterval2<-1
        }
      } else if(attr(listOfdfChromSizeHolocen[[i]], "ytitle" )=="notMb" ) {
        # rulerInterval2<-rulerInterval2/divisor2
        rulerInterval2 <- ifelse(missing(rulerInterval),
                                 1,
                                 rulerInterval/divisor2
        )
      }

      if(rulerInterval2 > maxChrRound[[i]]){
        message(crayon::red(paste0("rulerInterval too big. Use smaller rulerInterval; or use rulerIntervalMb or rulerIntervalcM
                                   if you have Mb data or specialOTUNames, respectively" ) )
        )
      }


      fromZerotoMaxChr[[i]]<- tryCatch(seq(
        from = 0,
        to = (maxChrRound[[i]] + (rulerInterval2 * ifelse(maxChrRound[[i]] %% rulerInterval2>0,1,0) ) # ifelseinloop
        - maxChrRound[[i]] %% rulerInterval2),
        by = rulerInterval2), error=function(e) NA
                                       ) # try
      remove(rulerInterval2)
    } # for maxchrround

    ycoordChrRound  <- lapply(1:length(fromZerotoMaxChr), function(x) {
      pos<-as.numeric(attr(listOfdfChromSizeHolocen[[x]],"position") )
      unlist(
        lapply(1:length(fromZerotoMaxChr[[x]]), function(y)
          (karHeight/2+(fromZerotoMaxChr[[x]][y]*normalizeToOne))+(1*karHeiSpace*(pos-1)) )         # 0+(from
            ) # u
      }
    ) # l

    names(ycoordChrRound)<-holocenNames2

      if(is.na(addMissingOTUAfter[1])){
        if(karSepar){
          for (s in 1:(length(ymCopy2)-1) ) {

            diffnext<-abs(min(ymCopy2[[s+1]] ) - max(ymCopy2[[s]]) )
            ymCopy2[[s+1]]=ymCopy2[[s+1]]-diffnext
            ifelse(amoSepar>diffnext,amoSepar2<-diffnext,amoSepar2<-amoSepar)
            ymCopy2[[s+1]]=ymCopy2[[s+1]]+amoSepar2

            nameYm<-names(ymCopy2)[[s+1]]

            if(length(which(names(ycoordChrRound)==nameYm))>0 ){
            ycoordChrRound[[which(names(ycoordChrRound)==nameYm)]] <- ycoordChrRound[[which(names(ycoordChrRound)==nameYm)]] - diffnext
            ycoordChrRound[[which(names(ycoordChrRound)==nameYm)]] <- ycoordChrRound[[which(names(ycoordChrRound)==nameYm)]] + amoSepar2
            } # if
          } # for
        } # redu if
      } # if
    # )

    ycoordChrRound<-ycoordChrRound[!is.na(ycoordChrRound)]

    ####################
    #
    #   add rulers holocen
    #
    ####################

    opar<-graphics::par(no.readonly = TRUE)
    on.exit(suppressWarnings(par(opar)) )
    graphics::par(mgp=c(3,rulerNumberPos,0) )

    rulerPlot(ycoordChrRound,listOfdfChromSize,listOfdfChromSizeHolocen,fromZerotoMaxChr,rulerNumberSize,rulerPos,ruler.tck,
              lwd.chr,moveKarHor2,mkhValue,useMinorTicks,miniTickFactor)

    #
    #   title of ruler for HOLOCEN
    #

    # par(las=1)
    # rulerTitle(ycoordChrRound,listOfdfChromSizeHolocen,MbUnit,specialyTitle,yTitle,xPosRulerTitle,rulerTitleSize)

    # rulerTitle(xmnoNA,ymnoNA,chrSpacing,yPosRulerTitle,listOfdfChromSizeHolocen,MbUnit,specialyTitle,yTitle,xPosRulerTitle,rulerTitleSize)

  } # end holocen

    ###################################################3
    #   ADD TITLE OF RULER
    ###################################################

    par(las=1)

    # rulerTitle(ycoordShortRound,listOfdfChromSizeMonocen,MbUnit,specialyTitle,yTitle,xPosRulerTitle,rulerTitleSize)
    rulerTitle(xmnoNA,ymnoNA,chrSpacing,yPosRulerTitle,listOfdfChromSizenoNA,MbUnit,specialyTitle,yTitle,xPosRulerTitle,rulerTitleSize)


 }   # end rulers if
} # END     not    circular ruler

  ############################################################
  #
  #   groups line
  #
  ###########################################################

if(circularPlot==FALSE) {

    if("group" %in% colnames(dfChrSizeInternal) ) {

      # groupSegmentDistance <- ifelse(groupUp, 1, 2)
      groupSegmentDistance <- 1 #ifelse(groupUp, 1, 2)

      # chrIdCount  <- ifelse(groupUp==FALSE & chrId=="",0,1) # warn
      chrIdCount  <- ifelse(groupUp | chrId=="" ,0, 1 ) # warn

#      listOfdfChromSizenoNA2906<<-listOfdfChromSizenoNA
    for (s in 1:length(xmnoNA)) {
      if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
        chrWidth2  <-specialChrWidth
      } else {
        chrWidth2<-chrWidth
      }

      # allowing not organized groups
      if("group" %in% colnames(listOfdfChromSizenoNA[[s]] ) ) {
      lens <- rle(listOfdfChromSizenoNA[[s]]$group)$lengths
      names(lens)<-rle(listOfdfChromSizenoNA[[s]]$group)$values

      ngroup<-length(lens)
      attr(listOfdfChromSizenoNA[[s]],"groupPresence") <- ngroup

      for (g in 1: ngroup) {

        if(!is.na(names(lens)[g] ) ) {
        x0 <- xmnoNA[[s]][,3][ifelse(length(cumsum(lens)[g-1] )==0, # ifelseinloop
                               1,
                               # cumsum(table(listOfdfChromSizenoNA[[s]]$group) )[g-1]+1
                               cumsum(lens )[g-1]+1
        ) ]
        x0 <- x0[!is.na(names(lens)[g])]
        x1 <- xmnoNA[[s]][,3][cumsum(lens )[g] ] + chrWidth2
        x1 <- x1[!is.na(names(lens)[g] ) ]

        y01 <- min(ymnoNA[[s]]) - ( (distTextChr/3) * (groupSegmentDistance+chrIdCount) )
        y01 <- y01[!is.na(names(lens)[g] ) ]

        segments(x0=x0,
                 y0=y01,
                 x1=x1,
                 y1=y01
        ) # seg

      ########################
      #     group name
      ########################

        if(groupName){

          ytext <- min(ymnoNA[[s]]) - ( (distTextChr/3) * (groupSegmentDistance+chrIdCount+1 ) )

          text( c( xmnoNA[[s]][,3][1]-(chrWidth/2) * nameChrIndexPos   , ((x0+x1)/2) ),
                ytext,
                labels = c(classGroupName , names( lens[g] ) ),
               cex=indexIdTextSize
          ) # text end
        } # groupName
        } # if not NA
      } # for group

      } # group col if
    } # for sp

  } # if group column
} # circular FALSE


  ###################################################################################################################
  #                                                    chromosome names non-Circular
  ###################################################################################################################

  chrNameDistance <-1

  # original
  if(circularPlot==FALSE){

    if(chrId=="original" | chrId=="simple"){

      for (s in 1:length(xmnoNA)) {

        if(attr(xmnoNA[[s]],"cenType")=="monocen") {armFactor<-2} else {armFactor<-1}

        if(chrId=="original"){
          mylabels<-listOfdfChromSizenoNA[[s]][,"chrName"]

          if(!missing(chrIdPatternRem)){
            mylabels<- sub(chrIdPatternRem,"",mylabels )
          }

        } else if (chrId=="simple"){
          mylabels <- 1:(nrow(xmnoNA[[s]])/armFactor)
        }

        if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
          chrWidth2  <-specialChrWidth
        } else {
          chrWidth2<-chrWidth
        }

        if(as.numeric(attr(listOfdfChromSizenoNA[[s]],"groupPresence") ) > 0 & groupUp ) {
          # groupCount = 2
          ifelse(groupName,groupCount<-2,groupCount<-1)
        } else {
          groupCount=0
        }

        #########################################
        # y pos.
        #########################################

        ybasic <- min(ymnoNA[[s]] ) - ( (distTextChr/3) * ( chrNameDistance + groupCount) )

        #
        # chr. word
        #

        if( attr(listOfdfChromSizenoNA[[s]], "ytitle" )=="Mb" ){
          unit<- classMbName
        } else if ( attr(listOfdfChromSizenoNA[[s]], "ytitle" )=="cM" ) {
          unit<- classcMName
        } else {
          unit<- classChrName
        }

        graphics::text(xmnoNA[[s]][,3][1] - (chrWidth/2) * nameChrIndexPos
                       ,ybasic
                       ,labels = unit
                       ,cex=indexIdTextSize
        ) # end graphics::text

        for( lab in 1:length(mylabels) ) {

          uncommon<- mylabels[lab][which( mylabels[lab] %in% grep("FL0|FL+", mylabels[lab], value=TRUE) )]

          if(length(uncommon)==1) {

          splUncommon<-strsplit(uncommon,"")

            splUncommon1<-splUncommon[[1]]
            first<-splUncommon1[1]
            sec  <-splUncommon1[2]
            third<-splUncommon1[3]

            graphics::text(xmnoNA[[s]][,3][lab] + chrWidth2/2 ,
                           ybasic  ,
                           labels = bquote(paste(
                             .(first)[.(sec)]^.(third) )
                           ),
                           cex=indexIdTextSize
            ) # text

          } else {
            # text(0.2, 0.6, cex = 1.5, mytext)
            graphics::text(xmnoNA[[s]][,3][lab] + chrWidth2/2,
                           ybasic ,
                           labels = mylabels[lab],
                           cex=indexIdTextSize
            )
          }
        } # for mylabels[lab]

      } # for

    } # if chrId

} # end names CIRCULAR FALSE


  #################################
  # horizontal chromosome index
  #################################

    if(circularPlot==FALSE){

    chrIdCount  <- ifelse(chrId=="",0,1) # warn

    chrSizeShow <- ifelse(chrSize==TRUE,1,0)

    chrSizeMbpShow <- ifelse(chrSizeMbp==TRUE,1,0)

    perPresence <- ifelse(markPer != "",1,0)

    posPresence <- ifelse(showMarkPos,1,0)

    morphoCount<-ifelse(morpho=="Guerra" | morpho=="Levan", 1,
                        ifelse(morpho=="both",2,0
                        )
    ) #mC

    indexCount<-ifelse(chrIndex=="CI" | chrIndex == "AR", 1,
                        ifelse(chrIndex == "both",2,0
                        )
    ) # mC

  if(chrIndex=="both"){bothAddI=1} else {bothAddI=0}

  #
  #   add Chr name up
  #

#   listOfdfChromSizenoNA3238<<-listOfdfChromSizenoNA

  if(chrNameUp) {
    for (s in 1:length(listOfdfChromSizenoNA) ) {
      if("chrNameUp" %in% (colnames(listOfdfChromSizenoNA[[s]] ) ) ) {

        if(attr(xmnoNA[[s]],"cenType")=="monocen") {armFactor<-2} else {armFactor<-1}

        if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
          chrWidth2  <-specialChrWidth
        } else {
          chrWidth2<-chrWidth
        }

        chrNamesUp<- listOfdfChromSizenoNA[[s]][,"chrNameUp"]

        if(!missing(chrIdPatternRem)){
          chrNamesUp <- sub(chrIdPatternRem,"",chrNamesUp  )
        }

        graphics::text(c(xmnoNA[[s]][,3][1]-(chrWidth/2) * nameChrIndexPos , xmnoNA[[s]][,3][1:(nrow(xmnoNA[[s]])/armFactor)]+(chrWidth2/2) ),
                       max(ymnoNA[[s]] ) + (distTextChr/3)   ,
                       labels = tryCatch( c(classChrNameUp,chrNamesUp), error=function(e){NA} )
                       ,cex=indexIdTextSize
        ) # end graphics::text
      }
    } # FOR

  }
    #
    #   add Chr Size
    #

  if(chrSize==TRUE ) {

    for (s in 1:length(listOfdfChromSizenoNA) ) {

      if(attr(xmnoNA[[s]],"cenType")=="monocen") {armFactor<-2} else {armFactor<-1}

      if(as.numeric(attr(listOfdfChromSizenoNA[[s]],"groupPresence") ) > 0 ) {
        ifelse(groupName,groupCount<-2,groupCount<-1)
      } else {
        groupCount=0
      } # end ifelse

      if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
        chrWidth2 <- specialChrWidth
      } else {
        chrWidth2 <- chrWidth
      }

      if( attr(listOfdfChromSizenoNA[[s]], "ytitle" )=="Mb" ){
        unit<-MbUnit
      } else if ( attr(listOfdfChromSizenoNA[[s]], "ytitle" )=="cM" ) {
        unit<-specialyTitle
      } else {
        unit<-yTitle
      }

      divisor2<-as.numeric(attr(listOfdfChromSizenoNA[[s]],"divisor"))

      if ( attr(listOfdfChromSizenoNA[[s]], "ytitle" )=="cM" ) {
        labels<-listOfdfChromSizenoNA[[s]][,"chrSize"]*divisor2
      } else if ( attr(listOfdfChromSizenoNA[[s]], "ytitle" )=="Mb" ) {
        labels<-listOfdfChromSizenoNA[[s]][,"chrSize"]*divisor2/1e6
      } else  { # ytitle notmb
        labels<-listOfdfChromSizenoNA[[s]][,"chrSize"]*divisor2
      }

      graphics::text(c(xmnoNA[[s]][,3][1] - (chrWidth/2) * nameChrIndexPos, xmnoNA[[s]][,3][1:(nrow(xmnoNA[[s]])/armFactor)]+(chrWidth2/2) ),
                       min(ymnoNA[[s]]) - ( (distTextChr/3) * (chrIdCount + chrSizeShow + groupCount ) )  ,
                       labels = tryCatch(
                         c(paste0("S (",unit,")"), format(round(labels,nsmall),nsmall=nsmall) ),
                         error=function(e){NA}
                         )
                       ,cex=indexIdTextSize
      ) # end graphics::text
    } # FOR
  } # chrSize

    #
    #   add Chr Size column Mbp
    #

    if(chrSizeMbp==TRUE ) {

      for (s in 1:length(listOfdfChromSizenoNA) ) {

        if("Mbp" %in% (colnames(listOfdfChromSizenoNA[[s]] ) ) ) {

        if(attr(xmnoNA[[s]],"cenType")=="monocen") {armFactor<-2} else {armFactor<-1}

        if(as.numeric(attr(listOfdfChromSizenoNA[[s]],"groupPresence") ) > 0 ) {
          ifelse(groupName,groupCount<-2,groupCount<-1)
        } else {
          groupCount=0
        } # end ifelse

        if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
          chrWidth2  <-specialChrWidth
        } else {
          chrWidth2<-chrWidth
        }

        mbpLabel<-  format(round(listOfdfChromSizenoNA[[s]][,"Mbp"],nsmall),nsmall=nsmall )

        graphics::text(c(xmnoNA[[s]][,3][1]-(chrWidth/2) * nameChrIndexPos, xmnoNA[[s]][,3][1:(nrow(xmnoNA[[s]])/armFactor)]+(chrWidth2/2) ),
                       min(ymnoNA[[s]]) - ( ( (distTextChr/3) * (chrIdCount + chrSizeShow + chrSizeMbpShow + groupCount ) ) )  ,
                       labels = tryCatch( c("S (Mbp)", mbpLabel), error=function(e){NA} )
                       ,cex=indexIdTextSize
        ) # end graphics::text
        }
      } # FOR


    } # chrSize

  #
  #   add CI
  #

  if(chrIndex=="both" | chrIndex == "CI" ) {

      for (s in 1:length(listOfdfChromSizenoNA) ) {

        if(as.numeric(attr(listOfdfChromSizenoNA[[s]],"groupPresence") ) > 0 ) {
          ifelse(groupName,groupCount<-2,groupCount<-1)
        } else {
          groupCount=0
        } # end ifelse
        if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
          chrWidth2  <-specialChrWidth
        } else {
          chrWidth2<-chrWidth
        }

        if(attr(listOfdfChromSizenoNA[[s]],"indexStatus")=="success") {

          graphics::text(c(xmnoNA[[s]][,3][1]-(chrWidth/2)*nameChrIndexPos, xmnoNA[[s]][,3][1:(nrow(xmnoNA[[s]])/2)]+(chrWidth2/2)),
                     min(ymnoNA[[s]]) - ( ( (distTextChr/3) * (chrIdCount+chrSizeShow+chrSizeMbpShow+groupCount+indexCount-bothAddI ) ))  ,
                     labels = tryCatch(c("CI",listOfdfChromSizenoNA[[s]][,"CI"] ),error=function(e){NA})
                     ,cex=indexIdTextSize
                    ) # end graphics::text
       } # success
      } # FOR
  } # BORH OR CI

  #
  #   add AR (radius)
  #

  if(chrIndex=="both" | chrIndex == "AR" ){

    for (s in 1:length(listOfdfChromSizenoNA) ) {
      if(as.numeric(attr(listOfdfChromSizenoNA[[s]],"groupPresence") ) > 0 ) {ifelse(groupName,groupCount<-2,groupCount<-1)   } else{groupCount=0}
      if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
        chrWidth2  <-specialChrWidth
      } else {
        chrWidth2<-chrWidth
      }
      if(attr(listOfdfChromSizenoNA[[s]],"indexStatus")=="success") {

      ARLabel<-  format(round( listOfdfChromSizenoNA[[s]][,"AR"],nsmall),nsmall=nsmall )

      graphics::text(c(xmnoNA[[s]][,3][1]-(chrWidth/2)*nameChrIndexPos, xmnoNA[[s]][,3][1:(nrow(xmnoNA[[s]])/2)]+chrWidth2/2),
                     rep( min(ymnoNA[[s]]) - ( ( (distTextChr/3) * (chrIdCount+chrSizeShow+chrSizeMbpShow+groupCount+indexCount) ) ) ,(nrow(xmnoNA[[s]])/2)+1 ),
                     labels = tryCatch(c("r", ARLabel ),error=function(e){NA})
                     ,cex=indexIdTextSize
      ) # end graphics::text
      } # success
    } # FOR
  } # fi BOTH OR AR


  #################################
  # horizontal chromosome morphology categories
  #################################

  #
  #   add Guerra and Levan
  #

  if(morpho=="both"){bothAdd=1} else {bothAdd=0}

  #
  #   add Guerra
  #

  if(morpho=="both" | morpho == "Guerra" ) {
    for (s in 1:length(listOfdfChromSizenoNA) ) {
      if(as.numeric(attr(listOfdfChromSizenoNA[[s]],"groupPresence") ) > 0 ) {ifelse(groupName,groupCount<-2,groupCount<-1)   } else{groupCount=0}
      if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
        chrWidth2  <-specialChrWidth
      } else {
        chrWidth2<-chrWidth
      }

      if(attr(listOfdfChromSizenoNA[[s]],"indexStatus")=="success") {

      graphics::text(c(xmnoNA[[s]][,3][1]-(chrWidth/2)*nameChrIndexPos, xmnoNA[[s]][,3][1:(nrow(xmnoNA[[s]])/2)]+chrWidth2/2),
                     min(ymnoNA[[s]]) - ( (distTextChr/3)*(chrIdCount+chrSizeShow+chrSizeMbpShow+morphoCount+indexCount-bothAdd+groupCount) ),
                     labels = tryCatch(c("Guerra",listOfdfChromSizenoNA[[s]][,"Guerra"]),error=function(e){NA})
                     ,cex=indexIdTextSize
      ) # end graphics::text
      } # if success
    } # for
  } # if guerra

  #
  #   add Levan
  #

  if(morpho=="both" | morpho == "Levan" ) {
      for (s in 1:length(listOfdfChromSizenoNA) ) {
        if(as.numeric(attr(listOfdfChromSizenoNA[[s]],"groupPresence") ) > 0 ) {ifelse(groupName,groupCount<-2,groupCount<-1)   } else {groupCount=0}
        if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
          chrWidth2  <-specialChrWidth
        } else {
          chrWidth2<-chrWidth
        }
        if(attr(listOfdfChromSizenoNA[[s]],"indexStatus")=="success") {

          graphics::text(c(xmnoNA[[s]][,3][1]-(chrWidth/2)*nameChrIndexPos, xmnoNA[[s]][,3][1:(nrow(xmnoNA[[s]])/2)]+chrWidth2/2),
                         min(ymnoNA[[s]]) - ( ( (distTextChr/3)* (chrIdCount+chrSizeShow+chrSizeMbpShow+morphoCount+indexCount+groupCount) ) ) ,
                         #distVectorGue[decVector]
                         labels = tryCatch(c("Levan",listOfdfChromSizenoNA[[s]][,"Levan"]),error=function(e){NA})
                         ,cex=indexIdTextSize
          ) # end graphics::text
        } # if success
      } # for
  } # fi


      #
      #   add % Het
      #

      if(markPer!="") {
        for (s in 1:length(listOfdfChromSizenoNA) ) {

          if(attr(xmnoNA[[s]],"cenType")=="monocen") {armFactor<-2} else {armFactor<-1}

          if(as.numeric(attr(listOfdfChromSizenoNA[[s]],"groupPresence") ) > 0 ) {ifelse(groupName,groupCount<-2,groupCount<-1)   } else {groupCount=0}

          if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
            chrWidth2 <-specialChrWidth
          } else {
            chrWidth2 <- chrWidth
          }
          if(attr(listOfdfChromSizenoNA[[s]],"indexStatus")=="success") {
            yForPer<-min(ymnoNA[[s]]) - ( ( (distTextChr/3)* (chrIdCount+chrSizeShow+chrSizeMbpShow+morphoCount+indexCount+groupCount+perPresence) ) )
          } else {
            yForPer<-min(ymnoNA[[s]]) - ( ( (distTextChr/3)* (chrIdCount+chrSizeShow+chrSizeMbpShow+groupCount+perPresence+1) ) )
          }

          # fraction
          perValue <- t(perList[[s]])[,paste0(markPer,"_per")]

          if (perAsFraction==FALSE){
            perValue <- perValue*100
          }
            graphics::text(c(xmnoNA[[s]][,3][1]-(chrWidth/2)*nameChrIndexPos, xmnoNA[[s]][,3][1:(nrow(xmnoNA[[s]])/armFactor)]+chrWidth2/2 )
                            ,yForPer
                            ,labels = tryCatch(c(paste0("% ",markPer), format(round(perValue ,nsmall ),nsmall=nsmall) )
                                               ,error=function(e){NA}
                                               )
                            ,cex=indexIdTextSize
            ) # end graphics::text

        } # for
      } # fi


    #
    #   add marks' pos
    #

    if(showMarkPos & exists("dfMarkPosInternal") ) {
      for (s in 1:length(listOfdfChromSizenoNA) ) {

        if(attr(xmnoNA[[s]],"cenType")=="monocen") {armFactor<-2} else {armFactor<-1}

        if(as.numeric(attr(listOfdfChromSizenoNA[[s]],"groupPresence") ) > 0 ) {ifelse(groupName,groupCount<-2,groupCount<-1)   } else {groupCount=0}

        if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
          chrWidth2 <-specialChrWidth
        } else {
          chrWidth2 <- chrWidth
        }

        if(attr(listOfdfChromSizenoNA[[s]],"indexStatus")=="success") {
          yForPos<-min(ymnoNA[[s]]) - ( ( (distTextChr/3)* (chrIdCount+chrSizeShow+chrSizeMbpShow+morphoCount+indexCount+groupCount+perPresence+posPresence ) ) )
        } else {
          yForPos<-min(ymnoNA[[s]]) - ( ( (distTextChr/3)* (chrIdCount+chrSizeShow+chrSizeMbpShow+groupCount+perPresence+posPresence) ) )
        }

        # fraction
        posValue <- lapply(t(posTib[[s]])[,1], sort)

        posValue <- unlist(lapply(posValue, function(x) tryCatch( (paste0(format(round(x , 2),nsmall=2), collapse="/") ), error=function(e){NA} ) ) )

        graphics::text(c(xmnoNA[[s]][,3][1]-(chrWidth/2)*nameChrIndexPos, xmnoNA[[s]][,3][1:(nrow(xmnoNA[[s]])/armFactor)]+chrWidth2/2 )
                       ,yForPos
                       ,labels = tryCatch(c(paste0("pos.")
                                            # , format(round(perValue ,nsmall ),nsmall=nsmall)
                                            ,posValue
                                            )
                                            ,error=function(e){NA}
                       )
                       ,cex=indexIdTextSize
        ) # end graphics::text

      } # for
    } # fi

} # circular FALSE


      #########################################################################
      # add species names otu names
      #########################################################################

      if (circularPlot==FALSE){

        if(OTUasNote){
          addOTUName<-FALSE
          if(!missing(notes)){
            message(crayon::blurred("Error: OTUasNote is TRUE, other notes will be removed"))
          }
          notes<-data.frame(OTU=unique(dfChrSizeInternal$OTU), note=unique(dfChrSizeInternal$OTU) )
        }

        if(addOTUName){
          # message(crayon::green(paste0("OTU section start" ) ) )

          for (s in 1:length(xmnoNA) ) {
            if(as.numeric(attr(listOfdfChromSizenoNA[[s]],"groupPresence") ) > 0 ) {
              ifelse(groupName,groupCount<-2,groupCount<-1)
            } else {
              groupCount=0
            } # end ifelse

            if(attr(listOfdfChromSizenoNA[[s]],"indexStatus")=="failure") {
              indexCount2<-0
              morphoCount2<0
            } else {
              indexCount2 <- indexCount
              morphoCount2<- morphoCount
            }

            # lapply(1:length(xmnoNA), function(s) {
            if(attr(xmnoNA[[s]],"cenType")=="holocen") {
              holocenDisCount <- morphoCount2 + indexCount2 #newDistVector #+bothAdd
            } else {
              holocenDisCount <- 0
            } # ifelse holocen

            OTUcurrent<-names(listOfdfChromSizenoNA)[[s]]
            hasQuotes<-grepl("'.*'",OTUcurrent)

            if(!missing(OTUfont) ) {

              if(OTUfont==3 & hasQuotes){
                begin <- sub("\\'.*", "", OTUcurrent)
                varNames <- unlist(regmatches(OTUcurrent,gregexpr("'(.*?)'",OTUcurrent) ) )
                varConcat<-paste(varNames, collapse=" ")
                nameWithVar<- bquote(paste(italic(.(begin)),.(varConcat)  ) )
              } else {
                nameWithVar<- OTUcurrent
              }
            } else {
              nameWithVar<- OTUcurrent
            }

            graphics::text( min(xmnoNA[[s]] ) # xlimLeftMod
                            # c( (xmnoNA[[s]][1,3] - ((karIndexPos/2) ) ) )
                            ,ydistance <- min(ymnoNA[[s]]) -
                                             ( (distTextChr/3) *
                                                 (chrIdCount + chrSizeShow+ chrSizeMbpShow + morphoCount2 +
                                                    indexCount2 + groupCount + perPresence + posPresence + 2 - holocenDisCount)
                                               )
                            ,labels = nameWithVar
                            # labels = paste("",names(listOfdfChromSizenoNA)[[s]] ),
                            ,cex=OTUTextSize
                            ,adj=c(0) # justif 0 =left
                            ,font=   ifelse( !missing(OTUfont),   OTUfont,   1)
                            ,family= ifelse( !missing(OTUfamily), OTUfamily, defaultFontFamily2)
            ) # end graphics::text
          } # for

        } # fi add OTU name
      } # CIRCULAR FALSE

#################################
# karyotype index (left side)
#################################

if(circularPlot==FALSE){
  if(karIndex){
    # message(crayon::green(paste0("karyotype indices section start" ) ) )

    for (i in 1:length(listOfdfChromSizenoNA) ) { # for each OTU

      if(attr(listOfdfChromSizenoNA[[i]],"cenType")=="monocen"){
            if(is.character(names(listOfdfChromSizenoNA)[[i]]  ) ){
              message(crayon::green(paste0(names(listOfdfChromSizenoNA)[[i]],":" ) ) # otu name:  Calc. (asymmetry)
              ) # mess
            }

        ind<-asymmetry(listOfdfChromSizenoNA[[i]])
        if(is.null(ind)){
          message(crayon::red("Fix short/long measures or use karIndex=FALSE"))
        }

        if(!is.null(ind)){
            graphics::text( c( (xmnoNA[[i]][1,3]-(1*(karIndexPos/2) ) ) ),
                            # c( (xmnoNA[[i]][1,3]-(xlimLeftMod*(karIndexPos/2) ) ) ),

                            rep( min(ymnoNA[[i]][,1]), 1 ), #2
                            labels = paste("A ",ind$A ),
                            cex=indexIdTextSize,
                            adj=c(1) # justif
            ) # end graphics::text
            graphics::text(c( (xmnoNA[[i]][1,3]-(1*(karIndexPos/2) ) ) ),
                           rep( (min(ymnoNA[[i]][,1])-(distTextChr/3) ) , 1 ), # avoid overlap
                           labels = paste("A2",ind$A2 ),
                           cex=indexIdTextSize,
                           adj=c(1) # 0.5 centered
            ) # end graphics::text
        } # null

     }  else if(attr(listOfdfChromSizenoNA[[i]],"cenType")=="holocen"){ #  if monocen
        if(is.character(names(listOfdfChromSizenoNA)[[i]]  ) ){
          message(crayon::green(paste0(names(listOfdfChromSizenoNA)[[i]],":" ) )
          ) # mess
        }
        ind<-asymmetryA2(listOfdfChromSizenoNA[[i]])
        if(!is.null(ind)){
        graphics::text(c( (xmnoNA[[i]][1,3]-(1*(karIndexPos/2) ) ) ), # xlimLeftMod
                       (max(ymnoNA[[i]]) + min(ymnoNA[[i]]) ) /2 #(distTextChr/3)   #[,1]  was /3
                       ,labels = ifelse(ind$A2=="NA","",paste("A2",ind$A2 ) ),
                       cex=indexIdTextSize,
                       adj= c(1) # 0.5 centered
        ) # end graphics::text
        } # null
      } # holocen
    } # for
  } # fi
} # CIRCULAR FALSE

                                                ##########################
                                                #       MARKS            #
                                                ##########################

      ##########################################################################################################3
      #
      #                           painting Marks monocen              square marks
      #
      ############################################################################################################

      if (exists("parlistOfdfMarkPosMonocen") & exists("dfMarkColorInternal") ){
        # message(crayon::green(paste0("monocen. marks section start" ) ) )


        xMarkSq<-yMarkSq<-listOfdfMarkPosSq<-list()

        j<-1
        for (k in 1:length(parlistOfdfMarkPosMonocen)) {
          currName<-names(parlistOfdfMarkPosMonocen)[[k]]
          if(nrow(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style %in% c("square","squareLeft") ) , ] ) >0 ){
            listOfdfMarkPosSq<-c(listOfdfMarkPosSq,list(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style %in% c("square","squareLeft") ),]))
            names(listOfdfMarkPosSq)[[j]]<-currName
            j<-j+1
          }
        }

        if(length(listOfdfMarkPosSq)>0){
#          ym3213<<-ym
#         # listOfdfMarkPosSq3208<<-listOfdfMarkPosSq
          for (sm in 1:length(listOfdfMarkPosSq)) {
            yMark1<-NULL
            xMark1<-NULL
            # which(names(listOfdfChromSize) %in% names(listOfdfMarkPosSq)[[sm]] )

            corr_index<-which(names(listOfdfChromSize) %in% names(listOfdfMarkPosSq)[[sm]] )

            for (m in 1:nrow(listOfdfMarkPosSq[[sm]]) ) {

              ifelse(listOfdfMarkPosSq[[sm]][m,"chrRegion"]=="q",longORshort<- 0,longORshort<- 1) # ifelseinloop
              ifelse(listOfdfMarkPosSq[[sm]][m,"chrRegion"]=="q",column<- 1, column<- 2)
              ifelse(listOfdfMarkPosSq[[sm]][m,"chrRegion"]=="q",endColumn<- 2, endColumn<- 1)
              ifelse(listOfdfMarkPosSq[[sm]][m,"chrRegion"]=="q",whichArm<- "long", whichArm<- "short")
              ifelse(listOfdfMarkPosSq[[sm]][m,"style"]=="square",squareSide<- "right", squareSide<- "left")

              ifelse(listOfdfMarkPosSq[[sm]][m,"chrRegion"]=="q",mySign<- -1,mySign<- 1)
              rowIndex <- nrow(listOfdfChromSize[[corr_index]] ) * longORshort + (listOfdfMarkPosSq[[sm]][,"neworder"][m] )

              armStart<-ym[[corr_index]][rowIndex,column]
              armEnd  <-ym[[corr_index]][rowIndex,endColumn]
#              armStart3228<<-armStart
#              armEnd3229<<-armEnd
              armLength <- max(armStart,armEnd) - min(armStart,armEnd)
#              armLength3231<<-armLength
              # detect if whole arm
              mSize <- listOfdfMarkPosSq[[sm]][m,"markSize"]*normalizeToOne

              if(is.na(mSize)) {
                message(crayon::blue(paste0("you let square marks without size in a monocen. while using markDistCen (not NA)\n, maybe you want to mark all arm, use NA in markDistCen and use p|q|w in chrRegion. Now 0 will be used" ) ) )
                # listOfdfMarkPosSq[[sm]][m,"markSize"]<-0
                mSize<-0
              }

              # mSize <- listOfdfMarkPosSq[[sm]][m,"markSize"]*normalizeToOne

#              mSize3234<<-mSize

              if(abs(sum(armLength, - mSize) ) < efZero ) {
                wholeArm <- 'true'
              } else {
                wholeArm <- 'false'
              }

              yprox <- armStart +
                (listOfdfMarkPosSq[[sm]][m,"markDistCen"]                                                             *normalizeToOne*mySign)

              yter <- armStart +
                ( sum( listOfdfMarkPosSq[[sm]][m,"markDistCen"] , listOfdfMarkPosSq[[sm]][m,"markSize"], na.rm=TRUE ) *normalizeToOne*mySign )

              yMark1[[m]] <- if(longORshort==0) {c(yprox,yter,yter,yprox)} else {c(yter,yprox,yprox,yter)}

              attr(yMark1[[m]],"arm") <- listOfdfMarkPosSq[[sm]][m,"chrRegion"]

              # attr(yMark1[[m]],"armStart")<-armStart
              attr(yMark1[[m]],"rowIndex")<-rowIndex
              attr(yMark1[[m]],"wholeArm")<-wholeArm
              attr(yMark1[[m]],"whichArm")<-whichArm
              attr(yMark1[[m]],"squareSide")<-squareSide

              xMark1[[m]]<-xm[[corr_index]][listOfdfMarkPosSq[[sm]][,"neworder"][m],]
              attr(xMark1[[m]],"rowIndex")<-rowIndex

            }
            yMarkSq[[sm]]<-yMark1
            attr(yMarkSq[[sm]], "spname")<-names(listOfdfMarkPosSq)[[sm]]
            xMarkSq[[sm]]<-xMark1
            attr(xMarkSq[[sm]], "spname")<-names(listOfdfMarkPosSq)[[sm]]
          } # end for

#          yMarkSq3278<<-yMarkSq
#          xMarkSq3278<<-xMarkSq
          # markList<-addChrNameAttrMark(xMarkSq,yMarkSq,x)

          # xMarkSq<-markList$xMark
          # yMarkSq<-markList$yMark

          ########################
          #                      #
          #   add marks to plot monocen #
          #                      #
          ########################
          if(chromatids==FALSE | circularPlot==TRUE) {

            # if(circularPlot==FALSE){
            roundPlotMark(bannedMarkName,squareness, xMarkSq, yMarkSq,
                          dfMarkColorInternal,
                          listOfdfMarkPosSq,
                          chrWidth, #use for calc r2
                          specialChrWidth,
                          yfactor,
                          n,
                          lwd.chr,
                          listOfdfChromSize,
                          circularPlot,
                          y,
                          markLabelSize,
                          pattern,
                          separFactor,
                          labelSpacing,circleCenter,circleCenterY,radius,
                          legend,ylistTransChrSimple,rotation=rotation,labelOutwards)

          } else if (chromatids & circularPlot==FALSE) {

            chrtSqMark(squareness,yMarkSq,xMarkSq,xModifierMono,r2,dfMarkColorInternal,lwd.chr,listOfdfMarkPosSq,n)

          } # chromatids



        } else { #     if(length(listOfdfMarkPosSq)>0)
          remove(listOfdfMarkPosSq)
        }

        # square labels not centrom. (monocen.)
        booleanForsquareInlineLabel<- legend=="inline" & exists("dfMarkColorInternal") & exists("listOfdfMarkPosSq") & circularPlot==FALSE

        if(booleanForsquareInlineLabel) {
          textLabel(xMarkSq,yMarkSq,listOfdfChromSize,listOfdfMarkPosSq,specialChrSpacing,chrSpacing,markLabelSize,pattern,bannedMarkName)
        }

      } # if presence end painting marks

      ##################################################################################################
      #
      #                                                 painting Marks square holocen
      #
      ##################################################################################################

      if (exists("parlistOfdfMarkPosHolocen") & exists("dfMarkColorInternal") ) {
        # message(crayon::green(paste0("holocen. marks section start" ) ) )
        xMarkSq<-yMarkSq<-listOfdfMarkPosSq<-list()

        j<-1
        for (k in 1:length(parlistOfdfMarkPosHolocen)) {
          currName<-names(parlistOfdfMarkPosHolocen)[[k]]
          if(nrow(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style %in% c("square","squareLeft") ),])>0){
            listOfdfMarkPosSq<-c(listOfdfMarkPosSq,list(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style %in% c("square","squareLeft")  ),]) )
            names(listOfdfMarkPosSq)[[j]]<-currName
            j<-j+1
          }
        }

        if(length(listOfdfMarkPosSq)>0){
          for (sm in 1:length(listOfdfMarkPosSq)) {

            yMark1<-NULL
            xMark1<-NULL
            corr_index<-which(names(ym) %in% names(listOfdfMarkPosSq)[[sm]] )

            for (m in 1:nrow(listOfdfMarkPosSq[[sm]])){
              rowIndex<-(listOfdfMarkPosSq[[sm]][,"neworder"][m])
              chrStart<-ym[[corr_index]][rowIndex ,2]

              if(!"markSize" %in% colnames(listOfdfMarkPosSq[[sm]])){
                listOfdfMarkPosSq[[sm]]$markSize<-NA
              }
              if(!"markPos" %in% colnames(listOfdfMarkPosSq[[sm]])){
                listOfdfMarkPosSq[[sm]]$markPos<-NA
              }
              mSize<-mPos<-NULL
              mSize <- listOfdfMarkPosSq[[sm]][m,"markSize"]*normalizeToOne
              mPos <- listOfdfMarkPosSq[[sm]][m,"markPos"]

              if(is.na(mSize) ){
                message(crayon::blue(paste0("mark without size, unexpected results possible")))
              }
              if(is.na(mSize) & is.na(mPos ) ) {
                message(crayon::blue(paste0("\nyou let square marks without size nor pos. in holocen. Will use 0 as size\nand pos. maybe you want to add column chrRegion with w for whole chr. mark" ) )
                        )
                listOfdfMarkPosSq[[sm]][m,"markPos"] <-0
                listOfdfMarkPosSq[[sm]][m,"markSize"]<-0
              }

              yinf <- chrStart +                        # was ysup
                (listOfdfMarkPosSq[[sm]][m,"markPos"]                                                          *normalizeToOne)

              ysup <- chrStart +
                (  sum(listOfdfMarkPosSq[[sm]][m,"markPos"],listOfdfMarkPosSq[[sm]][m,"markSize"],na.rm=TRUE ) *normalizeToOne )

              yMark1[[m]]<-c(ysup,yinf,yinf,ysup)
              attr(yMark1[[m]],"rowIndex")<-rowIndex
              attr(yMark1[[m]],"wholeArm")<-'false' # this is to plot as mark not as arm see monocen.

              xMark1[[m]]<-xm[[corr_index]][listOfdfMarkPosSq[[sm]][,"neworder"][m],]
              attr(xMark1[[m]],"rowIndex")<-rowIndex

            }
            yMarkSq[[sm]]<-yMark1
            attr(yMarkSq[[sm]], "spname")<-names(listOfdfMarkPosSq)[[sm]]
            xMarkSq[[sm]]<-xMark1
            attr(xMarkSq[[sm]], "spname")<-names(listOfdfMarkPosSq)[[sm]]
          } # end for

          # markList<-addChrNameAttrMark(xMarkSq,yMarkSq,x)

          # xMarkSq<-markList$xMark
          # yMarkSq<-markList$yMark

          #####################
          #   add sq marks to plot holocen
          #####################
          if(chromatids==FALSE | holocenNotAsChromatids | circularPlot==TRUE) {

            roundPlotMark(bannedMarkName,squareness, xMarkSq, yMarkSq,
                          dfMarkColorInternal,
                          listOfdfMarkPosSq,
                          chrWidth, #use for calc r2
                          specialChrWidth,
                          yfactor,
                          n,
                          lwd.chr,
                          listOfdfChromSize,
                          circularPlot,
                          y,
                          markLabelSize,
                          pattern,
                          separFactor,
                          labelSpacing,circleCenter,circleCenterY,radius,
                          legend,ylistTransChrSimple,rotation=rotation,labelOutwards) #

          } else if (chromatids & holocenNotAsChromatids==FALSE & circularPlot==FALSE) {

            chrtSqMark(squareness,yMarkSq,xMarkSq,xModifierHolo,r2,dfMarkColorInternal,lwd.chr,listOfdfMarkPosSq,n)

          } # chromatids


        } #     if(length(listOfdfMarkPosSq)>0){

        else {remove(listOfdfMarkPosSq)}

        #
        #   inline legend holocen
        #

        booleanForsquareInlineLabel<- legend=="inline" & exists("dfMarkColorInternal") & exists("listOfdfMarkPosSq") & circularPlot==FALSE

        if(booleanForsquareInlineLabel) {
          textLabel(xMarkSq,yMarkSq,listOfdfChromSize,listOfdfMarkPosSq,specialChrSpacing,chrSpacing,markLabelSize,pattern,bannedMarkName)
        }

      } #   if (exists("parlistOfdfMarkPosHolocen") & exists("dfMarkColorInternal") )

      # end square marks  #

      ##########################################################################################################3
      #
      #                                               painting Marks monocen              cenStyle marks
      #
      ############################################################################################################

      if (exists("parlistOfdfMarkPosMonocen") & exists("dfMarkColorInternal") ){
        # message(crayon::green(paste0("monocen. marks section start" ) ) )

        yMarkLine<-xMarkRightLine<-xMarkLeftLine<-xMarkCenStyle<-yMarkCenStyle<-listOfdfMarkPosCenStyle<-list()

        j<-1

        for (k in 1:length(parlistOfdfMarkPosMonocen)) {
          currName<-names(parlistOfdfMarkPosMonocen)[[k]]
          if(nrow(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="cenStyle"),])>0){
            listOfdfMarkPosCenStyle<-c(listOfdfMarkPosCenStyle,list(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="cenStyle"),]))
            names(listOfdfMarkPosCenStyle)[[j]]<-currName
            j<-j+1
          }
        }

        if(length(listOfdfMarkPosCenStyle)>0) {

          for (sm in 1:length(listOfdfMarkPosCenStyle)) {
            yMark1<-yMark2<-NULL
            xMark1<-xMark2Min<-xMark2Max<-NULL
            # which(names(listOfdfChromSize) %in% names(listOfdfMarkPosCenStyle)[[sm]] )
#            #        listOfdfChromSize4918<<-listOfdfChromSize
            corr_index<-which(names(listOfdfChromSize) %in% names(listOfdfMarkPosCenStyle)[[sm]] )

            for (m in 1:nrow(listOfdfMarkPosCenStyle[[sm]]) ){
              ifelse(listOfdfMarkPosCenStyle[[sm]][m,"chrRegion"]=="q",longORshort<- 0,longORshort<- 1) # ifelseinloop
              ifelse(listOfdfMarkPosCenStyle[[sm]][m,"chrRegion"]=="q",column<- 1, column<- 2)
              ifelse(listOfdfMarkPosCenStyle[[sm]][m,"chrRegion"]=="q",mySign<- -1,mySign<- 1)
              rowIndex<- ( nrow(listOfdfChromSize[[corr_index]] ) * longORshort ) + listOfdfMarkPosCenStyle[[sm]][,"neworder"][m]
              armStart<-ym[[corr_index]][rowIndex,column]
              yprox <- armStart +
                (listOfdfMarkPosCenStyle[[sm]][m,"markDistCen"]                                                             *normalizeToOne*mySign)

              yter <- armStart +
                ( sum( listOfdfMarkPosCenStyle[[sm]][m,"markDistCen"] , listOfdfMarkPosCenStyle[[sm]][m,"markSize"], na.rm=TRUE ) *normalizeToOne*mySign )

              # yMark1[[m]]<-if(longORshort==0) {c(yprox,yter,yter,yprox)} else {c(yter,yprox,yprox,yter)}
              yMark1[[m]]<-if(longORshort==0) {c(yprox,yter,yprox,yter)} else {c(yter,yprox,yter,yprox)}
              attr(yMark1[[m]],"arm")<-listOfdfMarkPosCenStyle[[sm]][m,"chrRegion"]
              attr(yMark1[[m]],"rowIndex")<-rowIndex

              ##
              yMark2[[m]]<-if(longORshort==0) {c(yprox,yter)} else {c(yter,yprox)}
              attr(yMark2[[m]],"arm")<-listOfdfMarkPosCenStyle[[sm]][m,"chrRegion"]
              attr(yMark2[[m]],"rowIndex")<-rowIndex
              ##

              # attr(yMark1[[m]],"armStart")<-armStart

              xMark1[[m]]<-xm[[corr_index]][listOfdfMarkPosCenStyle[[sm]][,"neworder"][m],]
              xMark2Min[[m]]<-rep(min(xMark1[[m]]),2)
              xMark2Max[[m]]<-rep(max(xMark1[[m]]),2)

              attr(xMark1[[m]],"rowIndex")<-rowIndex
              attr(xMark2Min[[m]],"rowIndex")<-rowIndex
              attr(xMark2Max[[m]],"rowIndex")<-rowIndex

            }
            yMarkCenStyle[[sm]]<-yMark1
            attr(yMarkCenStyle[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]

            yMarkLine[[sm]]<- yMark2
            attr(yMarkLine[[sm]], "spname")    <-names(listOfdfMarkPosCenStyle)[[sm]]

            ##
            xMarkCenStyle[[sm]]<-xMark1
            attr(xMarkCenStyle[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]

            xMarkRightLine[[sm]]<-xMark2Max
            xMarkLeftLine[[sm]] <-xMark2Min

            attr(xMarkLeftLine[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]
            attr(xMarkRightLine[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]

            ##

          } # end for

          if(roundedCen){
            pts<- seq(-pi/2, pi*1.5, length.out = n*4)
            ptsl<-split(pts, sort(rep(1:4, each=length(pts)/4, len=length(pts))) )

            newLongyCenMarksStyle <- newLongxCenMarksStyle<-list()

            for (s in 1:length(yMarkCenStyle) ) {

              xyCoordsCenMarksStyle <- mapXYCen(1 , (length(yMarkCenStyle[[s]]) ) ,
                                                yMarkCenStyle[[s]],
                                                xMarkCenStyle[[s]] ,
                                                ptsl[[1]],ptsl[[2]],ptsl[[3]],ptsl[[4]],
                                                mimic=TRUE
              )

              newLongxCenMarksStyle[[s]] <- xyCoordsCenMarksStyle$roundedX
              newLongyCenMarksStyle[[s]] <- xyCoordsCenMarksStyle$roundedY

            } # for s

            for(s in 1: length(newLongyCenMarksStyle)) {
              attr(newLongyCenMarksStyle[[s]], "spname")<- attr(yMarkCenStyle[[s]], "spname")
              attr(newLongxCenMarksStyle[[s]], "spname")<- attr(xMarkCenStyle[[s]], "spname")

              for (a in 1: length(newLongyCenMarksStyle[[s]])){
                attr(newLongyCenMarksStyle[[s]][[a]],"rowIndex")<- attr(yMarkCenStyle[[s]][[a]],"rowIndex")
                attr(newLongxCenMarksStyle[[s]][[a]],"rowIndex")<- attr(xMarkCenStyle[[s]][[a]],"rowIndex")

              }
            }

          } # rC


          ########################
          #                      #
          #   add marks to plot monocen #
          #                      #
          ########################
#          #      amHere<<-5011
#          #      listOfdfMarkPosCenStyle5012<<-listOfdfMarkPosCenStyle
          if(roundedCen==FALSE){
            mimicCenPlotMark(squareness, xMarkCenStyle, yMarkCenStyle,
                             defCenStyleCol,
                             listOfdfMarkPosCenStyle,
                             chrWidth, #use for calc r2
                             specialChrWidth,
                             yfactor,
                             n,
                             lwd.mimicCen2,
                             listOfdfChromSize,
                             circularPlot,
                             y,
                             markLabelSize,
                             separFactor,
                             labelSpacing,circleCenter,circleCenterY,radius,
                             ylistTransChrSimple,
                             rotation=rotation,labelOutwards,
                             yMarkLine,xMarkRightLine,xMarkLeftLine,
                             x)
          } else {
            mimicCenPlotMark(squareness, newLongxCenMarksStyle, newLongyCenMarksStyle,
                             defCenStyleCol,
                             listOfdfMarkPosCenStyle,
                             chrWidth, #use for calc r2
                             specialChrWidth,
                             yfactor,
                             n,
                             lwd.mimicCen2,
                             listOfdfChromSize,
                             circularPlot,
                             y,
                             markLabelSize,
                             separFactor,
                             labelSpacing,circleCenter,circleCenterY,radius,
                             ylistTransChrSimple,
                             rotation=rotation,labelOutwards,
                             yMarkLine,xMarkRightLine,xMarkLeftLine,
                             x
                             ,roundedCen)
          } # rC

          ######## new ######################################## inside cenStyle
          xMarkCenStyleBody<-yMarkCenStyleBody<-list()
#          #      amHere<<-5057
          # if(chromatids & circularPlot==FALSE) {
#            #        amHere<<-5059
#            #        listOfdfMarkPosCenStyle5059<<-listOfdfMarkPosCenStyle
            for (sm in 1:length(listOfdfMarkPosCenStyle) ) {

              yMark1body <- NULL
              xMark1body <- NULL

#              #          listOfdfChromSize5064 <<- listOfdfChromSize
              corr_index<-which(names(listOfdfChromSize) %in% names(listOfdfMarkPosCenStyle)[[sm]] )
#              #          corr_index5061<<-corr_index

              for (m in 1:nrow(listOfdfMarkPosCenStyle[[sm]]) ){

                #
                ifelse(listOfdfMarkPosCenStyle[[sm]][m,"chrRegion"]=="q",longORshort<- 0,longORshort<- 1) # ifelseinloop
                ifelse(listOfdfMarkPosCenStyle[[sm]][m,"chrRegion"]=="q",column<- 1, column<- 2)
                ifelse(listOfdfMarkPosCenStyle[[sm]][m,"chrRegion"]=="q",mySign<- -1,mySign<- 1)
                rowIndex<- ( nrow(listOfdfChromSize[[corr_index]] ) * longORshort ) + listOfdfMarkPosCenStyle[[sm]][,"neworder"][m]
#                #            rowIndex5070<<-rowIndex

                armStart<-ym[[corr_index]][rowIndex,column]
                yprox <- armStart +
                  (listOfdfMarkPosCenStyle[[sm]][m,"markDistCen"]                                                             *normalizeToOne*mySign)


                yter <- armStart +
                  ( sum( listOfdfMarkPosCenStyle[[sm]][m,"markDistCen"] , listOfdfMarkPosCenStyle[[sm]][m,"markSize"], na.rm=TRUE ) *normalizeToOne*mySign )

                yMark1body[[m]] <-if(longORshort==0) {c(yter,yter,yprox,yprox)} else {c(yprox,yprox,yter,yter)}

                attr(yMark1body[[m]],"arm")<-listOfdfMarkPosCenStyle[[sm]][m,"chrRegion"]
                attr(yMark1body[[m]],"rowIndex")<-rowIndex
                #
                fourX<-xm[[corr_index]][listOfdfMarkPosCenStyle[[sm]][,"neworder"][m],]
                xMark1body[[m]] <- rep(c(min(fourX),max(fourX)),2)
                attr(xMark1body[[m]],"rowIndex")<-rowIndex

                # xMarkCen1[[i]] <- xcoordCents[[corr_index]][ rowIndex ,]
                # attr(xMarkCen1[[i]],"rowIndex")<-rowIndex

              } # each mark

              yMarkCenStyleBody[[sm]]<-yMark1body
#              #          yMark1body5090<<-yMark1body
              attr(yMarkCenStyleBody[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]
              xMarkCenStyleBody[[sm]]<-xMark1body
#              #          xMark1body5093<<-xMark1body

              attr(xMarkCenStyleBody[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]

            } # for each df

            if (roundedCen) {
              pts<- seq(-pi/2, pi*1.5, length.out = n*4)
              ptsl<-split(pts, sort(rep(1:4, each=length(pts)/4, len=length(pts))) )


              newLongyCenMarksBodyStyle <- newLongxCenMarksBodyStyle<-list()

              for (s in 1:length(yMarkCenStyleBody) ) {

                xyCoordsCenMarksStyle <- mapXYCen(1 , (length(yMarkCenStyleBody[[s]]) ) ,
                                                  yMarkCenStyleBody[[s]],
                                                  xMarkCenStyleBody[[s]] ,
                                                  ptsl[[1]],ptsl[[2]],ptsl[[3]],ptsl[[4]]

                )

                newLongxCenMarksBodyStyle[[s]] <- xyCoordsCenMarksStyle$roundedX
                newLongyCenMarksBodyStyle[[s]] <- xyCoordsCenMarksStyle$roundedY

              } # for s

              for(s in 1: length(newLongyCenMarksBodyStyle)) {
                attr(newLongyCenMarksBodyStyle[[s]], "spname")<- attr(yMarkCenStyleBody[[s]], "spname")
                attr(newLongxCenMarksBodyStyle[[s]], "spname")<- attr(xMarkCenStyleBody[[s]], "spname")

                for (a in 1: length(newLongyCenMarksBodyStyle[[s]])){
                  attr(newLongyCenMarksBodyStyle[[s]][[a]],"rowIndex")<- attr(yMarkCenStyleBody[[s]][[a]],"rowIndex")
                  attr(newLongxCenMarksBodyStyle[[s]][[a]],"rowIndex")<- attr(xMarkCenStyleBody[[s]][[a]],"rowIndex")

                }
              } # f

            } # rC

          if(roundedCen==FALSE){
            mimicCenPlotMarkInside(pattern,bannedMarkName,squareness, xMarkCenStyleBody, yMarkCenStyleBody ,
                                   defCenStyleCol,
                                   dfMarkColorInternal,
                                   listOfdfMarkPosCenStyle,
                                   chrWidth, #use for calc r2
                                   specialChrWidth,
                                   yfactor,
                                   n,
                                   lwd.mimicCen2,
                                   listOfdfChromSize,
                                   circularPlot,
                                   y,
                                   markLabelSize,
                                   separFactor,
                                   labelSpacing,circleCenter,circleCenterY,radius,
                                   ylistTransChrSimple,
                                   rotation=rotation,labelOutwards,
                                   yMarkLine,xMarkRightLine,xMarkLeftLine,
                                   x,lwd.chr,legend)
          } else {
            mimicCenPlotMarkInside(pattern,bannedMarkName,squareness, newLongxCenMarksBodyStyle, newLongyCenMarksBodyStyle,
                                   defCenStyleCol,
                                   dfMarkColorInternal,
                                   listOfdfMarkPosCenStyle,
                                   chrWidth, #use for calc r2
                                   specialChrWidth,
                                   yfactor,
                                   n,
                                   lwd.mimicCen2,
                                   listOfdfChromSize,
                                   circularPlot,
                                   y,
                                   markLabelSize,
                                   separFactor,
                                   labelSpacing,circleCenter,circleCenterY,radius,
                                   ylistTransChrSimple,
                                   rotation=rotation,labelOutwards,
                                   yMarkLine,xMarkRightLine,xMarkLeftLine,
                                   x,
                                   lwd.chr,
                                   legend
                                   ,roundedCen)
          } # rC

          # } # chromatids
          ########


        } else { remove(listOfdfMarkPosCenStyle) } #     if(length(listOfdfMarkPosCenStyle)>0)

      } # if exists monocen end painting marks

      ###########################################
      #
      #   plotting labels inline mimicCENTR. monocen
      #
      ###########################################

      # cen labels
      booleanColorInternalMimicMarkCen<- legend=="inline" & exists("dfMarkColorInternal") & exists("listOfdfMarkPosCenStyle") & circularPlot==FALSE

      if(booleanColorInternalMimicMarkCen)  {
#        bannedMarkName3699<<-bannedMarkName
        textLabelCen(xMarkCenStyleBody,yMarkCenStyleBody,listOfdfChromSize,listOfdfMarkPosCenStyle,specialChrSpacing,chrSpacing,markLabelSize,pattern,
                     bannedMarkName)
      } # if

      ##################################################################################################
      #
      #                                        painting Marks cenStyle holocen
      #
      ##################################################################################################

      if (exists("parlistOfdfMarkPosHolocen") & exists("dfMarkColorInternal") ) {
        # message(crayon::green(paste0("holocen. marks section start" ) ) )
        yMarkLine<-xMarkRightLine<-xMarkLeftLine<-xMarkCenStyle<-yMarkCenStyle<-listOfdfMarkPosCenStyle<-list()

        j<-1
        for (k in 1:length(parlistOfdfMarkPosHolocen)) {
          currName<-names(parlistOfdfMarkPosHolocen)[[k]]
          if(nrow(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="cenStyle"),])>0){
            listOfdfMarkPosCenStyle<-c(listOfdfMarkPosCenStyle,list(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="cenStyle"),]) )
            names(listOfdfMarkPosCenStyle)[[j]]<-currName
            j<-j+1
          }
        }

        if(length(listOfdfMarkPosCenStyle)>0){
          for (sm in 1:length(listOfdfMarkPosCenStyle)) {

            yMark1<-yMark2<-NULL
            xMark1<-xMark2Min<-xMark2Max<-NULL
            corr_index<-which(names(ym) %in% names(listOfdfMarkPosCenStyle)[[sm]] )

            for (m in 1:nrow(listOfdfMarkPosCenStyle[[sm]])){
              rowIndex<-(listOfdfMarkPosCenStyle[[sm]][,"neworder"][m])
              chrStart<-ym[[corr_index]][rowIndex ,2]
              yinf <- chrStart +                        # was ysup
                (listOfdfMarkPosCenStyle[[sm]][m,"markPos"]                                                          *normalizeToOne)

              ysup <- chrStart +
                (  sum(listOfdfMarkPosCenStyle[[sm]][m,"markPos"],listOfdfMarkPosCenStyle[[sm]][m,"markSize"],na.rm=TRUE ) *normalizeToOne )

              yMark1[[m]]<-c(ysup,yinf,ysup,yinf)
              yMark2[[m]]<-c(yinf,ysup)

              attr(yMark1[[m]],"rowIndex")<-rowIndex
              attr(yMark2[[m]],"rowIndex")<-rowIndex

              xMark1[[m]]<-xm[[corr_index]][listOfdfMarkPosCenStyle[[sm]][,"neworder"][m],]
              xMark2Min[[m]]<-rep(min(xMark1[[m]]),2)
              xMark2Max[[m]]<-rep(max(xMark1[[m]]),2)

              attr(xMark1[[m]],"rowIndex")<-rowIndex
              attr(xMark2Min[[m]],"rowIndex")<-rowIndex
              attr(xMark2Max[[m]],"rowIndex")<-rowIndex

            }

            yMarkCenStyle[[sm]]<-yMark1
            attr(yMarkCenStyle[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]

            yMarkLine[[sm]]<- yMark2
            attr(yMarkLine[[sm]], "spname")    <-names(listOfdfMarkPosCenStyle)[[sm]]

            #

            xMarkCenStyle[[sm]]<-xMark1
            xMarkRightLine[[sm]]<-xMark2Max
            xMarkLeftLine[[sm]] <-xMark2Min
            attr(xMarkCenStyle[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]
            attr(xMarkLeftLine[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]
            attr(xMarkRightLine[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]

          } # end for

          # markList<-addChrNameAttrMark(xMarkCenStyle,yMarkCenStyle,x)

          # xMarkCenStyle<-markList$xMark
          # yMarkCenStyle<-markList$yMark

          if(roundedCen){
            pts<- seq(-pi/2, pi*1.5, length.out = n*4)
            ptsl<-split(pts, sort(rep(1:4, each=length(pts)/4, len=length(pts))) )

            newLongyCenMarksStyle <- newLongxCenMarksStyle<-list()

            for (s in 1:length(yMarkCenStyle) ) {

              xyCoordsCenMarksStyle <- mapXYCen(1 , (length(yMarkCenStyle[[s]]) ) ,
                                                yMarkCenStyle[[s]],
                                                xMarkCenStyle[[s]] ,
                                                ptsl[[1]],ptsl[[2]],ptsl[[3]],ptsl[[4]],
                                                mimic=TRUE
              )

              newLongxCenMarksStyle[[s]] <- xyCoordsCenMarksStyle$roundedX
              newLongyCenMarksStyle[[s]] <- xyCoordsCenMarksStyle$roundedY

            } # for s

            # newLongyCenMarksStyle<-newLongyCenMarksStyle[!is.na(newLongyCenMarksStyle)]
            # newLongxCenMarksStyle<-newLongxCenMarksStyle[!is.na(newLongxCenMarksStyle)]

            for(s in 1: length(newLongyCenMarksStyle)) {
              attr(newLongyCenMarksStyle[[s]], "spname")<- attr(yMarkCenStyle[[s]], "spname")
              attr(newLongxCenMarksStyle[[s]], "spname")<- attr(xMarkCenStyle[[s]], "spname")

              for (a in 1: length(newLongyCenMarksStyle[[s]])){
                attr(newLongyCenMarksStyle[[s]][[a]],"rowIndex")<- attr(yMarkCenStyle[[s]][[a]],"rowIndex")
                attr(newLongxCenMarksStyle[[s]][[a]],"rowIndex")<- attr(xMarkCenStyle[[s]][[a]],"rowIndex")
              }
            }
          } # rC

          #####################
          #   add marks to plot holocen
          #####################

          if(roundedCen){
            mimicCenPlotMark(squareness, newLongxCenMarksStyle, newLongyCenMarksStyle,
                             defCenStyleCol,
                             listOfdfMarkPosCenStyle,
                             chrWidth, #use for calc r2
                             specialChrWidth,
                             yfactor,
                             n,
                             lwd.mimicCen2,
                             listOfdfChromSize,
                             circularPlot,
                             y,
                             markLabelSize,
                             separFactor,
                             labelSpacing,circleCenter,circleCenterY,radius,
                             ylistTransChrSimple,
                             rotation=rotation,labelOutwards,
                             yMarkLine,xMarkRightLine,xMarkLeftLine,
                             x
                             ,roundedCen) #

          } else {
            mimicCenPlotMark(squareness, xMarkCenStyle, yMarkCenStyle,
                             defCenStyleCol,
                             listOfdfMarkPosCenStyle,
                             chrWidth, #use for calc r2
                             specialChrWidth,
                             yfactor,
                             n,
                             lwd.mimicCen2,
                             listOfdfChromSize,
                             circularPlot,
                             y,
                             markLabelSize,
                             separFactor,
                             labelSpacing,circleCenter,circleCenterY,radius,
                             ylistTransChrSimple,
                             rotation=rotation,labelOutwards,
                             yMarkLine,xMarkRightLine,xMarkLeftLine,
                             x) #
          }

          ######## new ######################################## inside cenStyle holocen

          xMarkCenStyleBody<-yMarkCenStyleBody<-list()
#          #      amHere<<-5057
    #      if(chromatids & holocenNotAsChromatids==FALSE & circularPlot==FALSE) {
#            #        amHere<<-5059
#            #        listOfdfMarkPosCenStyle5059<<-listOfdfMarkPosCenStyle
            for (sm in 1:length(listOfdfMarkPosCenStyle) ) {

              yMark1body <- NULL
              xMark1body <- NULL

#              #          listOfdfChromSize5064 <<- listOfdfChromSize
              corr_index<-which(names(listOfdfChromSize) %in% names(listOfdfMarkPosCenStyle)[[sm]] )
#              #          corr_index5061<<-corr_index

              for (m in 1:nrow(listOfdfMarkPosCenStyle[[sm]]) ) {
                rowIndex<-(listOfdfMarkPosCenStyle[[sm]][,"neworder"][m])
                chrStart<-ym[[corr_index]][rowIndex ,2]
                yinf <- chrStart +                        # was ysup
                  (listOfdfMarkPosCenStyle[[sm]][m,"markPos"]                                                          *normalizeToOne)

                ysup <- chrStart +
                  (  sum(listOfdfMarkPosCenStyle[[sm]][m,"markPos"],listOfdfMarkPosCenStyle[[sm]][m,"markSize"],na.rm=TRUE ) *normalizeToOne )

                yMark1body[[m]]<-c(yinf,yinf,ysup,ysup)

                attr(yMark1body[[m]],"rowIndex")<-rowIndex

                fourX <-     xm[[corr_index]][listOfdfMarkPosCenStyle[[sm]][,"neworder"][m],]
                xMark1body[[m]] <- rep(c(min(fourX),max(fourX)),2)

                attr(xMark1body[[m]],"rowIndex")<-rowIndex
              } # each mark

              yMarkCenStyleBody[[sm]]<-yMark1body
#              #          yMark1body5090<<-yMark1body
              attr(yMarkCenStyleBody[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]
              xMarkCenStyleBody[[sm]]<-xMark1body
#              #          xMark1body5093<<-xMark1body

              attr(xMarkCenStyleBody[[sm]], "spname")<-names(listOfdfMarkPosCenStyle)[[sm]]

            } # for each df

            if(roundedCen) {
              pts<- seq(-pi/2, pi*1.5, length.out = n*4)
              ptsl<-split(pts, sort(rep(1:4, each=length(pts)/4, len=length(pts))) )

              newLongyCenMarksBodyStyle <- newLongxCenMarksBodyStyle<-list()

              for (s in 1:length(yMarkCenStyleBody) ) {

                xyCoordsCenMarksStyle <- mapXYCen(1 , (length(yMarkCenStyleBody[[s]]) ) ,
                                                  yMarkCenStyleBody[[s]],
                                                  xMarkCenStyleBody[[s]] ,
                                                  ptsl[[1]],ptsl[[2]],ptsl[[3]],ptsl[[4]]

                )

                newLongxCenMarksBodyStyle[[s]] <- xyCoordsCenMarksStyle$roundedX
                newLongyCenMarksBodyStyle[[s]] <- xyCoordsCenMarksStyle$roundedY

              } # for s

              for(s in 1: length(newLongyCenMarksBodyStyle)) {
                attr(newLongyCenMarksBodyStyle[[s]], "spname")<- attr(yMarkCenStyleBody[[s]], "spname")
                attr(newLongxCenMarksBodyStyle[[s]], "spname")<- attr(xMarkCenStyleBody[[s]], "spname")

                for (a in 1: length(newLongyCenMarksBodyStyle[[s]])){
                  attr(newLongyCenMarksBodyStyle[[s]][[a]],"rowIndex")<- attr(yMarkCenStyleBody[[s]][[a]],"rowIndex")
                  attr(newLongxCenMarksBodyStyle[[s]][[a]],"rowIndex")<- attr(xMarkCenStyleBody[[s]][[a]],"rowIndex")

                }
              }
            } # rC


            # } # chromatids

          if(roundedCen==FALSE){
            mimicCenPlotMarkInside(pattern,bannedMarkName,squareness, xMarkCenStyleBody, yMarkCenStyleBody ,
                                   defCenStyleCol,
                                   dfMarkColorInternal,
                                   listOfdfMarkPosCenStyle,
                                   chrWidth, #use for calc r2
                                   specialChrWidth,
                                   yfactor,
                                   n,
                                   lwd.mimicCen2,
                                   listOfdfChromSize,
                                   circularPlot,
                                   y,
                                   markLabelSize,
                                   separFactor,
                                   labelSpacing,circleCenter,circleCenterY,radius,
                                   ylistTransChrSimple,
                                   rotation=rotation,labelOutwards,
                                   yMarkLine,xMarkRightLine,xMarkLeftLine,
                                   x,lwd.chr,legend)
          } else {
#            newLongxCenMarksBodyStyle4054<<-newLongxCenMarksBodyStyle
#            newLongyCenMarksBodyStyle4055<<-newLongyCenMarksBodyStyle

            mimicCenPlotMarkInside(pattern,bannedMarkName,squareness, newLongxCenMarksBodyStyle, newLongyCenMarksBodyStyle,
                                   defCenStyleCol,
                                   dfMarkColorInternal,
                                   listOfdfMarkPosCenStyle,
                                   chrWidth, #use for calc r2
                                   specialChrWidth,
                                   yfactor,
                                   n,
                                   lwd.mimicCen2,
                                   listOfdfChromSize,
                                   circularPlot,
                                   y,
                                   markLabelSize,
                                   separFactor,
                                   labelSpacing,circleCenter,circleCenterY,radius,
                                   ylistTransChrSimple,
                                   rotation=rotation,labelOutwards,
                                   yMarkLine,xMarkRightLine,xMarkLeftLine,
                                   x,
                                   lwd.chr
                                   ,legend
                                   ,roundedCen)
          } # rC

          ########

        } else {  #     if(length(listOfdfMarkPosCenStyle)>0){
          remove(listOfdfMarkPosCenStyle)
        }

        ###########################################
        #
        #   plotting labels inline mimicCENTR. holocen
        #
        ###########################################

        # cen labels
        booleanColorInternalMimicMarkCen<- legend=="inline" & exists("dfMarkColorInternal") & exists("listOfdfMarkPosCenStyle") & circularPlot==FALSE

        if(booleanColorInternalMimicMarkCen)  {

          textLabelCen(xMarkCenStyleBody,yMarkCenStyleBody,listOfdfChromSize,listOfdfMarkPosCenStyle,
                    specialChrSpacing,chrSpacing,markLabelSize,pattern,bannedMarkName)
        } # if

      } #   if (exists("parlistOfdfMarkPosHolocen") & ("dfMarkColorInternal") )
      # end cenStyle marks

      ## end cenStyle



    ## up arrows

    ##########################################################################################################3
    #
    #                           painting Marks monocen              upArrow marks
    #
    ############################################################################################################

    arrowhead2<-1-arrowhead
    if (exists("parlistOfdfMarkPosMonocen") & exists("dfMarkColorInternal") ){
      yMarkUpAr<-xMarkUpAr<-listOfdfMarkPosUpAr<-list()
      xMarkUpAr_2nd<-list()

      j<-1

      for (k in 1:length(parlistOfdfMarkPosMonocen)) {
        currName<-names(parlistOfdfMarkPosMonocen)[[k]]
        if(nrow(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="upArrow"),])>0){
          listOfdfMarkPosUpAr<-c(listOfdfMarkPosUpAr,list(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="upArrow"),]))
          names(listOfdfMarkPosUpAr)[[j]]<-currName
          j<-j+1
        }
      }

      if(length(listOfdfMarkPosUpAr)>0){
        for (sm in 1:length(listOfdfMarkPosUpAr)) {
          yMark1<-NULL
          xMark1<-NULL
          xMark1_2nd<-NULL

          # which(names(listOfdfChromSize) %in% names(listOfdfMarkPosUpAr)[[sm]] )

          corr_index<-which(names(listOfdfChromSize) %in% names(listOfdfMarkPosUpAr)[[sm]] )

          for (m in 1:nrow(listOfdfMarkPosUpAr[[sm]]) ){
            ifelse(listOfdfMarkPosUpAr[[sm]][m,"chrRegion"]=="q",longORshort<- 0,longORshort<- 1) # ifelseinloop
            ifelse(listOfdfMarkPosUpAr[[sm]][m,"chrRegion"]=="q",column<- 1, column<- 2)
            ifelse(listOfdfMarkPosUpAr[[sm]][m,"chrRegion"]=="q",mySign<- -1,mySign<- 1)
            rowIndex<- nrow(listOfdfChromSize[[corr_index]] ) * longORshort + (listOfdfMarkPosUpAr[[sm]][,"neworder"][m])
            armStart<-ym[[corr_index]][rowIndex,column]
            yprox <- armStart +
              (listOfdfMarkPosUpAr[[sm]][m,"markDistCen"]                                                             *normalizeToOne*mySign)

            yter <- armStart +
              ( sum( listOfdfMarkPosUpAr[[sm]][m,"markDistCen"] , listOfdfMarkPosUpAr[[sm]][m,"markSize"], na.rm=TRUE ) *normalizeToOne*mySign )

            if(longORshort==0) {
                ybase<-yter
                ysize<-abs(yprox-yter)
            } else {
                ybase<-yprox
                ysize<-abs(yter-yprox)
            }

            yArrow<-c(ysize,arrowhead2*ysize,
                 arrowhead2*ysize,0,
                 0,arrowhead2*ysize,
                 arrowhead2*ysize,ysize
            )

            yMark1[[m]]<- yArrow + ybase

            attr(yMark1[[m]],"arm")<-listOfdfMarkPosUpAr[[sm]][m,"chrRegion"]
            attr(yMark1[[m]],"rowIndex")<-rowIndex

            fourX <- xm[[corr_index]][listOfdfMarkPosUpAr[[sm]][,"neworder"][m],]
            xsize<-max(fourX)-min(fourX)
            xbase<-min(fourX)

            if(arrowsToSide){
              xsize<-xsize/2
              xbase <- xbase + xsize

              if(chromatids & circularPlot==FALSE){
                xsize<-xsize-xModifierMono
                xbase <- xbase + (xModifierMono)
              }

            }

            xArrow<-c(.5*xsize, xsize-(arrowheadWidthShrink*xsize),
                 (1-shrinkArrow)*xsize,(1-shrinkArrow)*xsize,
                 shrinkArrow*xsize,shrinkArrow*xsize,
                 0+(arrowheadWidthShrink*xsize),.5*xsize
            )
            # +1 represents base (min)
            xMark1[[m]] <- xArrow + xbase
            attr(xMark1[[m]],"rowIndex")<-rowIndex

            # For chromatids, arrows in both chromatids

            if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE){
              xbase<-min(fourX)
              xsize<-max(fourX)-min(fourX)

              xsize<-xsize/2
              xsize<-xsize-xModifierMono

              xArrow<-c(.5*xsize, xsize-(arrowheadWidthShrink*xsize),
                        (1-shrinkArrow)*xsize,(1-shrinkArrow)*xsize,
                        shrinkArrow*xsize,shrinkArrow*xsize,
                        0+(arrowheadWidthShrink*xsize),.5*xsize
              )

              xMark1_2nd[[m]] <- xArrow + xbase
              attr(xMark1_2nd[[m]],"rowIndex")<-rowIndex
            } # SECOND arrow if

          } # for each mark

          yMarkUpAr[[sm]]<-yMark1
          attr(yMarkUpAr[[sm]], "spname")<-names(listOfdfMarkPosUpAr)[[sm]]
          xMarkUpAr[[sm]]<-xMark1
          attr(xMarkUpAr[[sm]], "spname")<-names(listOfdfMarkPosUpAr)[[sm]]

          if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE) {
            xMarkUpAr_2nd[[sm]]<-xMark1_2nd
            attr(xMarkUpAr_2nd[[sm]], "spname")<-names(listOfdfMarkPosUpAr)[[sm]]
          }

        } # end for

        ########################
        #                      #
        #   add marks to plot monocen #
        #                      #
        ########################
        # if(circularPlot==FALSE){
        arrowPlotMark(squareness, xMarkUpAr, yMarkUpAr,
                      dfMarkColorInternal,
                      listOfdfMarkPosUpAr,
                      chrWidth, #use for calc r2
                      n,
                      lwd.chr,
                      circularPlot,
                      y,
                      x,
                      markLabelSize,
                      separFactor,
                      labelSpacing,circleCenter,circleCenterY,radius,
                      ylistTransChrSimple,rotation=rotation,
                      arrowheadWidthShrink)
        # plot second arrow when chrt
        if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE) {
          arrowPlotMark(squareness, xMarkUpAr_2nd, yMarkUpAr,
                        dfMarkColorInternal,
                        listOfdfMarkPosUpAr,
                        chrWidth, #use for calc r2
                        n,
                        lwd.chr,
                        circularPlot,
                        y,
                        x,
                        markLabelSize,
                        separFactor,
                        labelSpacing,circleCenter,circleCenterY,radius,
                        ylistTransChrSimple,rotation=rotation,
                        arrowheadWidthShrink)
        }


      } #     if(length(listOfdfMarkPosUpAr)>0)

      else {remove(listOfdfMarkPosUpAr)}

    } # if presence end painting marks

    ##################################################################################################
    #
    #                                                 painting Marks upArrow holocen
    #
    ##################################################################################################

    if (exists("parlistOfdfMarkPosHolocen") & exists("dfMarkColorInternal") ) {
      # message(crayon::green(paste0("holocen. marks section start" ) ) )
      xMarkUpAr<-yMarkUpAr<-listOfdfMarkPosUpAr<-list()
      j<-1
      for (k in 1:length(parlistOfdfMarkPosHolocen)) {
        currName<-names(parlistOfdfMarkPosHolocen)[[k]]
        if(nrow(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="upArrow"),])>0){
          listOfdfMarkPosUpAr<-c(listOfdfMarkPosUpAr,list(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="upArrow"),]) )
          names(listOfdfMarkPosUpAr)[[j]]<-currName
          j<-j+1
        }
      }

      if(length(listOfdfMarkPosUpAr)>0){
        for (sm in 1:length(listOfdfMarkPosUpAr)) {

          yMark1<-xMark1<-NULL
          xMark1_2nd<-NULL

          corr_index<-which(names(ym) %in% names(listOfdfMarkPosUpAr)[[sm]] )

          for (m in 1:nrow(listOfdfMarkPosUpAr[[sm]])) {
            rowIndex<-(listOfdfMarkPosUpAr[[sm]][,"neworder"][m])
            chrStart<-ym[[corr_index]][rowIndex ,2]
            yinf <- chrStart +                        # was ysup
              (listOfdfMarkPosUpAr[[sm]][m,"markPos"]                                                          *normalizeToOne)

            ysup <- chrStart +
              (  sum(listOfdfMarkPosUpAr[[sm]][m,"markPos"],listOfdfMarkPosUpAr[[sm]][m,"markSize"],na.rm=TRUE ) *normalizeToOne )

            ysize <- abs(ysup-yinf)

            yArrow <- c(ysize,arrowhead2*ysize,
                 arrowhead2*ysize,0,
                 0,arrowhead2*ysize,
                 arrowhead2*ysize,ysize
            )

            yMark1[[m]] <- yArrow + yinf

            attr(yMark1[[m]],"rowIndex")<-rowIndex

            fourX <- xm[[corr_index]][listOfdfMarkPosUpAr[[sm]][,"neworder"][m],]

            xsize <- max(fourX) - min(fourX)
            xbase <- min(fourX)

            if(holocenNotAsChromatids){
              excHoloFrArrToSide<-TRUE
            }

            if(arrowsToSide & excHoloFrArrToSide==FALSE){
              xsize<-xsize/2
              xbase <- xbase + xsize
              if(chromatids & holocenNotAsChromatids==FALSE & circularPlot==FALSE){
                xsize <- xsize - xModifierHolo
                xbase <- xbase + xModifierHolo
              }
            }

            # conflict holocenNotaschr TRUE
            #          excHoloFrArrToSide FALSE
            # HOLOCENNOTASCHROMATIDS TRUE REQUIRES EXCHOLOFRARRTOSIDE TRUE

            xArrow<-c(.5*xsize, xsize-(arrowheadWidthShrink*xsize),
                      (1-shrinkArrow)*xsize,(1-shrinkArrow)*xsize,
                 shrinkArrow*xsize,shrinkArrow*xsize,
                 0+(arrowheadWidthShrink*xsize),.5*xsize
            )

            # +1 represents base (min)
            xMark1[[m]] <- xArrow + xbase
            attr(xMark1[[m]],"rowIndex")<-rowIndex

            # For chromatids, arrows in both chromatids

            if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE & excHoloFrArrToSide==FALSE  ){
              xbase<-min(fourX)
              xsize<-max(fourX)-min(fourX)

              xsize<-xsize/2
              xsize<-xsize-xModifierHolo

              xArrow<-c(.5*xsize, xsize-(arrowheadWidthShrink*xsize),
                        (1-shrinkArrow)*xsize,(1-shrinkArrow)*xsize,
                        shrinkArrow*xsize,shrinkArrow*xsize,
                        0+(arrowheadWidthShrink*xsize),.5*xsize
              )

              xMark1_2nd[[m]] <- xArrow + xbase
              attr(xMark1_2nd[[m]],"rowIndex")<-rowIndex
            } # SECOND arrow if

          } # for mark

          yMarkUpAr[[sm]]<-yMark1
          attr(yMarkUpAr[[sm]], "spname")<-names(listOfdfMarkPosUpAr)[[sm]]
          xMarkUpAr[[sm]]<-xMark1
          attr(xMarkUpAr[[sm]], "spname")<-names(listOfdfMarkPosUpAr)[[sm]]

          if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE & excHoloFrArrToSide==FALSE ) {
            xMarkUpAr_2nd[[sm]]<-xMark1_2nd
            attr(xMarkUpAr_2nd[[sm]], "spname")<-names(listOfdfMarkPosUpAr)[[sm]]
          }

        } # end for

        #####################
        #   add UpAr marks to plot holocen
        #####################

        arrowPlotMark(squareness, xMarkUpAr, yMarkUpAr,
                      dfMarkColorInternal,
                      listOfdfMarkPosUpAr,
                      chrWidth, #use for calc r2
                      n,
                      lwd.chr,
                      circularPlot,
                      y,
                      x,
                      markLabelSize,
                      separFactor,
                      labelSpacing,circleCenter,circleCenterY,radius,
                      ylistTransChrSimple,rotation=rotation,
                      arrowheadWidthShrink) #

        if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE & excHoloFrArrToSide==FALSE) {
          arrowPlotMark(squareness, xMarkUpAr_2nd, yMarkUpAr,
                        dfMarkColorInternal,
                        listOfdfMarkPosUpAr,
                        chrWidth, #use for calc r2
                        n,
                        lwd.chr,
                        circularPlot,
                        y,
                        x,
                        markLabelSize,
                        separFactor,
                        labelSpacing,circleCenter,circleCenterY,radius,
                        ylistTransChrSimple,rotation=rotation,
                        arrowheadWidthShrink) #
        }

      } #     if(length(listOfdfMarkPosUpAr)>0){

      else {remove(listOfdfMarkPosUpAr)}

      #
      #   inline legend holocen
      #

    } #   if (exists("parlistOfdfMarkPosHolocen") & exists("dfMarkColorInternal") )

    # end upArrow marks  #


    ## down arrows

    ##########################################################################################################3
    #
    #                           painting Marks monocen              downArrow marks
    #
    ############################################################################################################

    arrowhead2<-1-arrowhead
    if (exists("parlistOfdfMarkPosMonocen") & exists("dfMarkColorInternal") ){

      xMarkDwAr<-yMarkDwAr<-listOfdfMarkPosDwAr<-list()
      xMarkDwAr_2nd<-list()
      j<-1

      for (k in 1:length(parlistOfdfMarkPosMonocen)) {
        currName<-names(parlistOfdfMarkPosMonocen)[[k]]
        if(nrow(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="downArrow"),])>0){
          listOfdfMarkPosDwAr<-c(listOfdfMarkPosDwAr,list(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="downArrow"),]))
          names(listOfdfMarkPosDwAr)[[j]]<-currName
          j<-j+1
        }
      }

      if(length(listOfdfMarkPosDwAr)>0){
        for (sm in 1:length(listOfdfMarkPosDwAr)) {
          yMark1<-xMark1<-NULL
          yMark1_2nd<-xMark1_2nd<-NULL

          corr_index<-which(names(listOfdfChromSize) %in% names(listOfdfMarkPosDwAr)[[sm]] )

          for (m in 1:nrow(listOfdfMarkPosDwAr[[sm]]) ){
            ifelse(listOfdfMarkPosDwAr[[sm]][m,"chrRegion"]=="q",longORshort<- 0,longORshort<- 1) # ifelseinloop
            ifelse(listOfdfMarkPosDwAr[[sm]][m,"chrRegion"]=="q",column<- 1, column<- 2)
            ifelse(listOfdfMarkPosDwAr[[sm]][m,"chrRegion"]=="q",mySign<- -1,mySign<- 1)
            rowIndex<- nrow(listOfdfChromSize[[corr_index]] ) * longORshort + (listOfdfMarkPosDwAr[[sm]][,"neworder"][m])
            armStart<-ym[[corr_index]][rowIndex,column]
            yprox <- armStart +
              (listOfdfMarkPosDwAr[[sm]][m,"markDistCen"]                                                             *normalizeToOne*mySign)

            yter <- armStart +
              ( sum( listOfdfMarkPosDwAr[[sm]][m,"markDistCen"] , listOfdfMarkPosDwAr[[sm]][m,"markSize"], na.rm=TRUE ) *normalizeToOne*mySign )

            if(longORshort==0) {
              ybase<-yter
              ysize<-abs(yprox-yter)
            } else {
              ybase<-yprox
              ysize<-abs(yter-yprox)
            }

            yArrow<-c(ysize,arrowhead*ysize,
                      arrowhead*ysize,0,
                      arrowhead*ysize,arrowhead*ysize,
                      ysize , ysize
            )

            yMark1[[m]]<- yArrow + ybase

            attr(yMark1[[m]],"arm")<-listOfdfMarkPosDwAr[[sm]][m,"chrRegion"]
            # attr(yMark1[[m]],"armStart")<-armStart
            attr(yMark1[[m]],"rowIndex")<-rowIndex

            # xMark1[[m]] <- xm[[corr_index]][listOfdfMarkPosDwAr[[sm]][,"neworder"][m],]
            fourX <- xm[[corr_index]][listOfdfMarkPosDwAr[[sm]][,"neworder"][m],]

            xsize <- max(fourX)-min(fourX)

            xbase<-min(fourX)

            if(arrowsToSide){
              xsize<-xsize/2

              if(chromatids  & circularPlot==FALSE){
                xsize<-xsize-xModifierMono
              }
            }

            xArrow<-c((1-shrinkArrow)*xsize , (1-shrinkArrow)*xsize,
                      xsize-(arrowheadWidthShrink*xsize),.5*xsize,
                      0+(arrowheadWidthShrink*xsize),shrinkArrow*xsize,
                      shrinkArrow*xsize,(1-shrinkArrow)*xsize
            )

            # +1 represents base (min)
            xMark1[[m]] <- xArrow + xbase
            attr(xMark1[[m]],"rowIndex")<-rowIndex

            # 2nd arrow

            if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE) {
              xsize <- max(fourX) - min(fourX)
              xbase <- min(fourX)

              xsize<-xsize/2
              xbase <- xbase + xsize

              xsize<-xsize-xModifierMono
              xbase <- xbase + (xModifierMono)



            xArrow<-c((1-shrinkArrow)*xsize , (1-shrinkArrow)*xsize,
                      xsize-(arrowheadWidthShrink*xsize),.5*xsize,
                      0+(arrowheadWidthShrink*xsize),shrinkArrow*xsize,
                      shrinkArrow*xsize,(1-shrinkArrow)*xsize
            )

            xMark1_2nd[[m]] <- xArrow + xbase
            attr(xMark1_2nd[[m]],"rowIndex")<-rowIndex

            }

          } # for mark

          yMarkDwAr[[sm]]<-yMark1
          attr(yMarkDwAr[[sm]], "spname")<-names(listOfdfMarkPosDwAr)[[sm]]
          xMarkDwAr[[sm]]<-xMark1
          attr(xMarkDwAr[[sm]], "spname")<-names(listOfdfMarkPosDwAr)[[sm]]

          if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE) {
            xMarkDwAr_2nd[[sm]]<-xMark1_2nd
            attr(xMarkDwAr_2nd[[sm]], "spname")<-names(listOfdfMarkPosDwAr)[[sm]]
          }



        } # end for

        # markList<-addChrNameAttrMark(xMarkDwAr,yMarkDwAr,x)
        #
        # xMarkDwAr<-markList$xMark
        # yMarkDwAr<-markList$yMark
        ########################
        #                      #
        #   add marks to plot monocen #
        #                      #
        ########################
        # if(circularPlot==FALSE){
        arrowPlotMark(squareness, xMarkDwAr, yMarkDwAr,
                      dfMarkColorInternal,
                      listOfdfMarkPosDwAr,
                      chrWidth, #use for calc r2
                      n,
                      lwd.chr,
                      circularPlot,
                      y,
                      x,
                      markLabelSize,
                      separFactor,
                      labelSpacing,circleCenter,circleCenterY,radius,
                      ylistTransChrSimple,rotation=rotation,
                      arrowheadWidthShrink)

        if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE) {
          arrowPlotMark(squareness, xMarkDwAr_2nd, yMarkDwAr,
                        dfMarkColorInternal,
                        listOfdfMarkPosDwAr,
                        chrWidth, #use for calc r2
                        n,
                        lwd.chr,
                        circularPlot,
                        y,
                        x,
                        markLabelSize,
                        separFactor,
                        labelSpacing,circleCenter,circleCenterY,radius,
                        ylistTransChrSimple,rotation=rotation,
                        arrowheadWidthShrink)
        }

      } #     if(length(listOfdfMarkPosDwAr)>0)

      else {remove(listOfdfMarkPosDwAr)}

    } # if presence end painting marks

    ##################################################################################################
    #
    #                                                 painting Marks downArrow holocen
    #
    ##################################################################################################

    if (exists("parlistOfdfMarkPosHolocen") & exists("dfMarkColorInternal") ) {
      # message(crayon::green(paste0("holocen. marks section start" ) ) )
      xMarkDwAr<-yMarkDwAr<-listOfdfMarkPosDwAr<-list()

      j<-1
      for (k in 1:length(parlistOfdfMarkPosHolocen)) {
        currName<-names(parlistOfdfMarkPosHolocen)[[k]]
        if(nrow(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="downArrow"),])>0){
          listOfdfMarkPosDwAr<-c(listOfdfMarkPosDwAr,list(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="downArrow"),]) )
          names(listOfdfMarkPosDwAr)[[j]]<-currName
          j<-j+1
        }
      }

      if(length(listOfdfMarkPosDwAr)>0){
        for (sm in 1:length(listOfdfMarkPosDwAr)) {

          yMark1<-NULL
          xMark1<-NULL
          corr_index<-which(names(ym) %in% names(listOfdfMarkPosDwAr)[[sm]] )

          for (m in 1:nrow(listOfdfMarkPosDwAr[[sm]])){
            rowIndex<-(listOfdfMarkPosDwAr[[sm]][,"neworder"][m])
            chrStart<-ym[[corr_index]][rowIndex ,2]
            yinf <- chrStart +                        # was ysup
              (listOfdfMarkPosDwAr[[sm]][m,"markPos"]                                                          *normalizeToOne)

            ysup <- chrStart +
              (  sum(listOfdfMarkPosDwAr[[sm]][m,"markPos"],listOfdfMarkPosDwAr[[sm]][m,"markSize"],na.rm=TRUE ) *normalizeToOne )

            ysize <- abs(ysup-yinf)

            yArrow<-c(ysize,arrowhead*ysize,
                      arrowhead*ysize,0,
                      arrowhead*ysize,arrowhead*ysize,
                      ysize , ysize
            )

            yMark1[[m]] <- yArrow + yinf

            attr(yMark1[[m]],"rowIndex")<-rowIndex

            fourX <- xm[[corr_index]][listOfdfMarkPosDwAr[[sm]][,"neworder"][m],]

            xsize <- max(fourX) - min(fourX)
            xbase <- min(fourX)

            if(holocenNotAsChromatids){
              excHoloFrArrToSide<-TRUE
            }

            if(arrowsToSide & excHoloFrArrToSide==FALSE ){
              xsize<-xsize/2
              if(chromatids & holocenNotAsChromatids==FALSE & circularPlot==FALSE){
                xsize<-xsize-xModifierHolo
              }
            }

            xArrow<-c((1-shrinkArrow)*xsize , (1-shrinkArrow)*xsize,
                      xsize-(arrowheadWidthShrink*xsize),.5*xsize,
                      0+(arrowheadWidthShrink*xsize),shrinkArrow*xsize,
                      shrinkArrow*xsize,(1-shrinkArrow)*xsize
            )
            # +1 represents base (min)
            xMark1[[m]] <- xArrow + xbase
            attr(xMark1[[m]],"rowIndex") <- rowIndex

            # 2nd arrow

            if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE & excHoloFrArrToSide==FALSE) {
              xsize <- max(fourX) - min(fourX)
              xbase <- min(fourX)

              xsize<-xsize/2
              xbase <- xbase + xsize

              xsize <- xsize - xModifierHolo
              xbase <- xbase + xModifierHolo

              xArrow<-c((1-shrinkArrow)*xsize , (1-shrinkArrow)*xsize,
                        xsize-(arrowheadWidthShrink*xsize),.5*xsize,
                        0+(arrowheadWidthShrink*xsize),shrinkArrow*xsize,
                        shrinkArrow*xsize,(1-shrinkArrow)*xsize
              )

              xMark1_2nd[[m]] <- xArrow + xbase
              attr(xMark1_2nd[[m]],"rowIndex")<-rowIndex
            }

          } # for mark
          yMarkDwAr[[sm]]<-yMark1
          attr(yMarkDwAr[[sm]], "spname")<-names(listOfdfMarkPosDwAr)[[sm]]
          xMarkDwAr[[sm]]<-xMark1
          attr(xMarkDwAr[[sm]], "spname")<-names(listOfdfMarkPosDwAr)[[sm]]

          if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE & excHoloFrArrToSide==FALSE ) {
            xMarkDwAr_2nd[[sm]]<-xMark1_2nd
            attr(xMarkDwAr_2nd[[sm]], "spname")<-names(listOfdfMarkPosDwAr)[[sm]]
          }

        } # end for

        # markList<-addChrNameAttrMark(xMarkDwAr,yMarkDwAr,x)

        # xMarkDwAr<-markList$xMark
        # yMarkDwAr<-markList$yMark

        #####################
        #   add DwAr marks to plot holocen
        #####################

        arrowPlotMark(squareness, xMarkDwAr, yMarkDwAr,
                      dfMarkColorInternal,
                      listOfdfMarkPosDwAr,
                      chrWidth, #use for calc r2
                      n,
                      lwd.chr,
                      circularPlot,
                      y,
                      x,
                      markLabelSize,
                      separFactor,
                      labelSpacing,circleCenter,circleCenterY,radius,
                      ylistTransChrSimple,rotation=rotation,
                      arrowheadWidthShrink)

        if(chromatids & arrowsToSide & arrowsBothChrt & circularPlot==FALSE & excHoloFrArrToSide==FALSE) {
          arrowPlotMark(squareness, xMarkDwAr_2nd, yMarkDwAr,
                        dfMarkColorInternal,
                        listOfdfMarkPosDwAr,
                        chrWidth, #use for calc r2
                        n,
                        lwd.chr,
                        circularPlot,
                        y,
                        x,
                        markLabelSize,
                        separFactor,
                        labelSpacing,circleCenter,circleCenterY,radius,
                        ylistTransChrSimple,rotation=rotation,
                        arrowheadWidthShrink
                        )
        }

      } #     if(length(listOfdfMarkPosDwAr)>0){

      else {remove(listOfdfMarkPosDwAr)}

      #
      #   inline legend holocen
      #

    } #   if (exists("parlistOfdfMarkPosHolocen") & exists("dfMarkColorInternal") )

    # end downArrow marks  #

    ##
  ##########################################################################################################################

                                          ##########################################
                                          #
                                          #   painting Marks monocen cM style
                                          #
                                          ##########################################

if (exists("parlistOfdfMarkPosMonocen") & exists("dfMarkColorInternal") ){
    # message(crayon::green(paste0("monocen. marks section start" ) ) )

    yMarkcM<- xMarkcM<- listOfdfMarkPoscM<-list()

    j<-1

    for (k in 1:length(parlistOfdfMarkPosMonocen)) {
      currName<-names(parlistOfdfMarkPosMonocen)[[k]]
      if(nrow(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="cM"),])>0){
        listOfdfMarkPoscM<-c(listOfdfMarkPoscM,list(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="cM"),]))
        names(listOfdfMarkPoscM)[[j]]<-currName
        j<-j+1
      }
    }
#    listOfdfMarkPoscM3790<<-listOfdfMarkPoscM

    if(length(listOfdfMarkPoscM)>0){
      for (sm in 1:length(listOfdfMarkPoscM)) {
        yMark1<-NULL
        xMark1<-NULL
        for (m in 1:nrow(listOfdfMarkPoscM[[sm]]) ){

          if("protruding" %in% colnames(listOfdfMarkPoscM[[sm]]) ){
            ifelse(is.na(listOfdfMarkPoscM[[sm]][m,"protruding"] ) ,
                   protruding2<-protruding,
                   protruding2<-listOfdfMarkPoscM[[sm]][m,"protruding"]
                   )
          } else {
            protruding2<-protruding
          }
          ifelse(listOfdfMarkPoscM[[sm]][m,"chrRegion"]=="q",longORshort<-0,longORshort<-1)
          ifelse(listOfdfMarkPoscM[[sm]][m,"chrRegion"]=="q",column<-1,column<-2)
          ifelse(listOfdfMarkPoscM[[sm]][m,"chrRegion"]=="q",mySign<--1,mySign<-1)

          corr_index<-which(names(ym) %in% names(listOfdfMarkPoscM)[[sm]] )
          rowIndex <- nrow(listOfdfChromSize[[corr_index]])*longORshort+(listOfdfMarkPoscM[[sm]][,"neworder"][m])
          ypos<-ym[[corr_index]][ rowIndex  ,column]+
            (listOfdfMarkPoscM[[sm]][m,"markDistCen"]                                      *normalizeToOne*mySign)

          yMark1[[m]]<-c(ypos,ypos)
          attr(yMark1[[m]],"rowIndex")<-rowIndex

          xMark1[[m]]<-xm[[corr_index]][listOfdfMarkPoscM[[sm]][,"neworder"][m],][c(1,3)]

          initOrig2<-xMark1[[m]][2]

          if(cMBeginCenter){
            xMark1[[m]][2] <- xMark1[[m]][2] +   ( (xMark1[[m]][1] - xMark1[[m]][2])  / 2)
          }

          if(chromatids & circularPlot==FALSE) {
            xMark1[[m]][2] <- xMark1[[m]][2] +   ( (xMark1[[m]][1] - xMark1[[m]][2])  / 2) + xModifierMono
          }

          xMark1[[m]][1] <- xMark1[[m]][1] +   ( (xMark1[[m]][1] - initOrig2  )  * protruding2) # 1st protruding

          attr(xMark1[[m]],"rowIndex")<-rowIndex
        }
        yMarkcM[[sm]]<-yMark1
        attr(yMarkcM[[sm]], "spname")<-names(listOfdfMarkPoscM)[[sm]] # added 1.14
        xMarkcM[[sm]]<-xMark1
        attr(xMarkcM[[sm]], "spname")<-names(listOfdfMarkPoscM)[[sm]]
      } # end for

      #####################
      #   add marks to plot
      #####################
      # cMPlotMark(xMarkcM, yMarkcM, dfMarkColorInternal,listOfdfMarkPoscM, lwd.cM2,circularPlot)
      cMPlotMark(bannedMarkName,xMarkcM, yMarkcM,y, x, dfMarkColorInternal,listOfdfMarkPoscM, lwd.cM2,circularPlot,
        radius,circleCenter,circleCenterY,separFactor,markLabelSize,pattern,n,labelSpacing,chrWidth,
        ylistTransChrSimple,rotation=rotation,labelOutwards)

    } #     if(length(listOfdfMarkPoscM)>0){
      else {
      remove(listOfdfMarkPoscM)
      }

  #
  #   labels cM inline monocen.
  #

booleanColorInternalMarkcM<- exists ("dfMarkColorInternal") & exists ("listOfdfMarkPoscM") & circularPlot==FALSE

if(booleanColorInternalMarkcM)  {

      textLabel(xMarkcM,yMarkcM,listOfdfChromSize,listOfdfMarkPoscM,specialChrSpacing,chrSpacing,markLabelSize,pattern,bannedMarkName)
}

} # if presence end painting marks

                                  ########################################
                                  #
                                  #   painting Marks cM holocen
                                  #
                                  ########################################

if (exists("parlistOfdfMarkPosHolocen") & exists("dfMarkColorInternal") ) {
    # message(crayon::green(paste0("holocen. marks section start" ) ) )
  xMarkcM<-yMarkcM<-listOfdfMarkPoscM<-list()

    j<-1
    for (k in 1:length(parlistOfdfMarkPosHolocen)) {
      currName<-names(parlistOfdfMarkPosHolocen)[[k]]
      if(nrow(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="cM"),])>0){
        listOfdfMarkPoscM<-c(listOfdfMarkPoscM,list(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="cM"),]) )
        names(listOfdfMarkPoscM)[[j]]<-currName
        j<-j+1
      }
    }

    if(length(listOfdfMarkPoscM)>0) {
      for (sm in 1:length(listOfdfMarkPoscM)) {

        yMark1<-NULL
        xMark1<-NULL

        for (m in 1:nrow(listOfdfMarkPoscM[[sm]])){

          if("protruding" %in% colnames(listOfdfMarkPoscM[[sm]]) ){
            ifelse(is.na(listOfdfMarkPoscM[[sm]][m,"protruding"] ) ,
                   protruding2<-protruding,
                   protruding2<-listOfdfMarkPoscM[[sm]][m,"protruding"]
            )
          } else {
            protruding2<-protruding
          }

          corr_index<-which(names(ym) %in% names(listOfdfMarkPoscM)[[sm]] )
          rowIndex<-(listOfdfMarkPoscM[[sm]][,"neworder"][m])
          ypos<-  ym[[corr_index]][ rowIndex , 2]+
            (listOfdfMarkPoscM[[sm]][m,"markPos"]                                      *normalizeToOne*1)

          yMark1[[m]]<-c(ypos,ypos)
          attr(yMark1[[m]],"rowIndex")<-rowIndex

          xMark1[[m]]<-xm[[corr_index]][ rowIndex , ][c(1,3)]

          initOrig2<-xMark1[[m]][2]

          if(cMBeginCenter){
            xMark1[[m]][2] <- xMark1[[m]][2] +   ( (xMark1[[m]][1] - xMark1[[m]][2])  / 2)
          }

          if(chromatids & holocenNotAsChromatids==FALSE & circularPlot==FALSE) {
            xMark1[[m]][2] <- xMark1[[m]][2] +   ( (xMark1[[m]][1] - xMark1[[m]][2])  / 2) + xModifierHolo
          }

          xMark1[[m]][1]<-xMark1[[m]][1]+((xMark1[[m]][1] - initOrig2) *protruding2)

          attr(xMark1[[m]],"rowIndex")<-rowIndex

        }
        yMarkcM[[sm]]<-yMark1
        attr(yMarkcM[[sm]], "spname")<-names(listOfdfMarkPoscM)[[sm]] # added 1.14
        xMarkcM[[sm]]<-xMark1
        attr(xMarkcM[[sm]], "spname")<-names(listOfdfMarkPoscM)[[sm]]
      } # end for

      # markList<-addChrNameAttrMark(xMarkcM,yMarkcM,x) # 1.14

      # xMarkcM<-markList$xMark
      # yMarkcM<-markList$yMark


      #####################
      #   add marks to plot
      #####################

      # cMPlotMark(xMarkcM, yMarkcM,    dfMarkColorInternal,listOfdfMarkPoscM, lwd.cM2,circularPlot)
      cMPlotMark(bannedMarkName,xMarkcM, yMarkcM, y, x, dfMarkColorInternal,listOfdfMarkPoscM, lwd.cM2,circularPlot,
                 radius,circleCenter,circleCenterY,separFactor,markLabelSize,pattern,n,labelSpacing,chrWidth,
                 ylistTransChrSimple,rotation=rotation,labelOutwards)

    } #     if(length(listOfdfMarkPoscM)>0){

    else { # length = 0
      remove(listOfdfMarkPoscM)
    }
    # message(crayon::green(paste0("holocen. marks section end" ) ) )

  #
  #   inline legend cM holocen
  #

  booleanColorInternalMarkcM<- exists ("dfMarkColorInternal") & exists ("listOfdfMarkPoscM") & circularPlot==FALSE

  if(booleanColorInternalMarkcM)  {

    textLabel(xMarkcM,yMarkcM,listOfdfChromSize,listOfdfMarkPoscM,specialChrSpacing,chrSpacing,markLabelSize,pattern,bannedMarkName)
  } # if
} # if parlistOfdfMarkPosHolocen


    ## cMLeft ######################################################################################################################

    ##########################################
    #
    #   painting Marks monocen cMLeft style
    #
    ##########################################

    if (exists("parlistOfdfMarkPosMonocen") & exists("dfMarkColorInternal") ){
      # message(crayon::green(paste0("monocen. marks section start" ) ) )

      xMarkcMLeft <- yMarkcMLeft<-listOfdfMarkPoscMLeft<-list()
      j<-1

      for (k in 1:length(parlistOfdfMarkPosMonocen)) {
        currName<-names(parlistOfdfMarkPosMonocen)[[k]]
        if(nrow(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="cMLeft"),])>0){
          listOfdfMarkPoscMLeft<-c(listOfdfMarkPoscMLeft,list(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="cMLeft"),]))
          names(listOfdfMarkPoscMLeft)[[j]]<-currName
          j<-j+1
        }
      }

#      listOfdfMarkPoscMLeft3987<<-listOfdfMarkPoscMLeft

      if(length(listOfdfMarkPoscMLeft)>0){
        for (sm in 1:length(listOfdfMarkPoscMLeft)) {
          yMark1<-NULL
          xMark1<-NULL
          for (m in 1:nrow(listOfdfMarkPoscMLeft[[sm]]) ){

            if("protruding" %in% colnames(listOfdfMarkPoscMLeft[[sm]]) ){
              ifelse(is.na(listOfdfMarkPoscMLeft[[sm]][m,"protruding"] ) ,
                     protruding2<-protruding,
                     protruding2<-listOfdfMarkPoscMLeft[[sm]][m,"protruding"]
              )
            } else {
              protruding2<-protruding
            }

            ifelse(listOfdfMarkPoscMLeft[[sm]][m,"chrRegion"]=="q",longORshort<-0,longORshort<-1)
            ifelse(listOfdfMarkPoscMLeft[[sm]][m,"chrRegion"]=="q",column<-1,column<-2)
            ifelse(listOfdfMarkPoscMLeft[[sm]][m,"chrRegion"]=="q",mySign<--1,mySign<-1)

            corr_index<-which(names(ym) %in% names(listOfdfMarkPoscMLeft)[[sm]] )
            rowIndex <- nrow(listOfdfChromSize[[corr_index]])*longORshort+(listOfdfMarkPoscMLeft[[sm]][,"neworder"][m])
            ypos<-ym[[corr_index]][ rowIndex  ,column]+
              (listOfdfMarkPoscMLeft[[sm]][m,"markDistCen"]                                      *normalizeToOne*mySign)

            yMark1[[m]]<-c(ypos,ypos)
            attr(yMark1[[m]],"rowIndex")<-rowIndex

            xMark1[[m]]<-xm[[corr_index]][listOfdfMarkPoscMLeft[[sm]][,"neworder"][m],][c(1,3)]

            initOrig1<-xMark1[[m]][1]

            if(cMBeginCenter){
              xMark1[[m]][1] <- xMark1[[m]][1] -   ( (xMark1[[m]][1] - xMark1[[m]][2])  / 2)
            }

            if(chromatids & circularPlot==FALSE) {
              xMark1[[m]][1] <- xMark1[[m]][1] -   ( (xMark1[[m]][1] - xMark1[[m]][2])  / 2 ) - xModifierMono
            }

            xMark1[[m]][2] <- xMark1[[m]][2] -   ( (initOrig1 - xMark1[[m]][2])  * protruding2) # left
            attr(xMark1[[m]],"rowIndex")<-rowIndex
          }
          yMarkcMLeft[[sm]]<-yMark1
          attr(yMarkcMLeft[[sm]], "spname")<-names(listOfdfMarkPoscMLeft)[[sm]] # added 1.14
          xMarkcMLeft[[sm]]<-xMark1
          attr(xMarkcMLeft[[sm]], "spname")<-names(listOfdfMarkPoscMLeft)[[sm]]
        } # end for

        #####################
        #   add marks to plot
        #####################

        cMLeftPlotMark(bannedMarkName,xMarkcMLeft, yMarkcMLeft,y, x, dfMarkColorInternal,listOfdfMarkPoscMLeft, lwd.cM2,circularPlot,
                   radius,circleCenter,circleCenterY,separFactor,markLabelSize,pattern,n,labelSpacing,chrWidth,
                   ylistTransChrSimple,rotation=rotation,labelOutwards)

      } #     if(length(listOfdfMarkPoscMLeft)>0){
      else {
        remove(listOfdfMarkPoscMLeft)
      }

      #
      #   labels cMLeft inline monocen.
      #

      booleanColorInternalMarkcMLeft<- exists ("dfMarkColorInternal") & exists ("listOfdfMarkPoscMLeft") & circularPlot==FALSE

      if(booleanColorInternalMarkcMLeft)  {

        textLabelLeft(xMarkcMLeft,yMarkcMLeft,listOfdfChromSize,listOfdfMarkPoscMLeft,specialChrSpacing,chrSpacing,markLabelSize,pattern,bannedMarkName)
      }

    } # if presence end painting marks

    ########################################
    #
    #   painting Marks cMLeft holocen
    #
    ########################################

    if (exists("parlistOfdfMarkPosHolocen") & exists("dfMarkColorInternal") ) {
      # message(crayon::green(paste0("holocen. marks section start" ) ) )
      xMarkcMLeft<- yMarkcMLeft<-listOfdfMarkPoscMLeft<-list()
      j<-1
      for (k in 1:length(parlistOfdfMarkPosHolocen)) {
        currName<-names(parlistOfdfMarkPosHolocen)[[k]]
        if(nrow(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="cMLeft"),])>0){
          listOfdfMarkPoscMLeft<-c(listOfdfMarkPoscMLeft,list(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="cMLeft"),]) )
          names(listOfdfMarkPoscMLeft)[[j]]<-currName
          j<-j+1
        }
      }

      if(length(listOfdfMarkPoscMLeft)>0) {
        for (sm in 1:length(listOfdfMarkPoscMLeft)) {

          yMark1<-NULL
          xMark1<-NULL

          for (m in 1:nrow(listOfdfMarkPoscMLeft[[sm]])){

            if("protruding" %in% colnames(listOfdfMarkPoscMLeft[[sm]]) ){
              ifelse(is.na(listOfdfMarkPoscMLeft[[sm]][m,"protruding"] ) ,
                     protruding2<-protruding,
                     protruding2<-listOfdfMarkPoscMLeft[[sm]][m,"protruding"]
              )
            } else {
              protruding2<-protruding
            }

            corr_index<-which(names(ym) %in% names(listOfdfMarkPoscMLeft)[[sm]] )
            rowIndex<-(listOfdfMarkPoscMLeft[[sm]][,"neworder"][m])
            ypos<-  ym[[corr_index]][ rowIndex , 2]+
              (listOfdfMarkPoscMLeft[[sm]][m,"markPos"]                                      *normalizeToOne*1)

            yMark1[[m]]<-c(ypos,ypos)
            attr(yMark1[[m]],"rowIndex")<-rowIndex

            xMark1[[m]]<-xm[[corr_index]][ rowIndex , ][c(1,3)]

            initOrig1<-xMark1[[m]][1]

            if(cMBeginCenter){
              xMark1[[m]][1] <- xMark1[[m]][1] -   ( (xMark1[[m]][1] - xMark1[[m]][2])  / 2)
            }

            if(chromatids & holocenNotAsChromatids==FALSE & circularPlot==FALSE) {
              xMark1[[m]][1] <- xMark1[[m]][1] -   ( (xMark1[[m]][1] - xMark1[[m]][2])  / 2 ) - xModifierHolo
            }

            xMark1[[m]][2] <- xMark1[[m]][2] -   ( (initOrig1 - xMark1[[m]][2])  * protruding2) # left
            attr(xMark1[[m]],"rowIndex")<-rowIndex

          }
          yMarkcMLeft[[sm]]<-yMark1
          attr(yMarkcMLeft[[sm]], "spname")<-names(listOfdfMarkPoscMLeft)[[sm]] # added 1.14
          xMarkcMLeft[[sm]]<-xMark1
          attr(xMarkcMLeft[[sm]], "spname")<-names(listOfdfMarkPoscMLeft)[[sm]]
        } # end for

        # markList<-addChrNameAttrMark(xMarkcMLeft,yMarkcMLeft,x) # 1.14

        # xMarkcMLeft<-markList$xMark
        # yMarkcMLeft<-markList$yMark


        #####################
        #   add marks to plot
        #####################

        # cMLeftPlotMark(xMarkcMLeft, yMarkcMLeft,    dfMarkColorInternal,listOfdfMarkPoscMLeft, lwd.cMLeft2,circularPlot)
        cMLeftPlotMark(bannedMarkName,xMarkcMLeft, yMarkcMLeft, y, x, dfMarkColorInternal,listOfdfMarkPoscMLeft, lwd.cM2,circularPlot,
                   radius,circleCenter,circleCenterY,separFactor,markLabelSize,pattern,n,labelSpacing,chrWidth,
                   ylistTransChrSimple,rotation=rotation,labelOutwards)

      } #     if(length(listOfdfMarkPoscMLeft)>0){

      else { # length = 0
        remove(listOfdfMarkPoscMLeft)
      }
      # message(crayon::green(paste0("holocen. marks section end" ) ) )

      #
      #   inline legend cMLeft holocen
      #

      booleanColorInternalMarkcMLeft<- exists ("dfMarkColorInternal") & exists ("listOfdfMarkPoscMLeft") & circularPlot==FALSE

      if(booleanColorInternalMarkcMLeft)  {

        textLabelLeft(xMarkcMLeft,yMarkcMLeft,listOfdfChromSize,listOfdfMarkPoscMLeft,specialChrSpacing,chrSpacing,markLabelSize,pattern,bannedMarkName)
      } # if
    } # if parlistOfdfMarkPosHolocen

                                                            #
                                                            #         DOTS
                                                            #

  ##########################################################################################################################
  #
  # dot style of marks                        monocen dots
  #
  ##########################################################################################################################

  if (exists("parlistOfdfMarkPosMonocen") & exists("dfMarkColorInternal") ) {

    listOfdfMarkPosCr<-list()

    yMarkCr<-xMarkCr<-rad<-  colBorderCr<-colCr<-list()

    j<-1

    for (k in 1:length(parlistOfdfMarkPosMonocen)) {
      currName<-names(parlistOfdfMarkPosMonocen)[[k]]
      if(nrow(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="dots"),])>0){
      listOfdfMarkPosCr<-c(listOfdfMarkPosCr,list(parlistOfdfMarkPosMonocen[[k]][which(parlistOfdfMarkPosMonocen[[k]]$style=="dots"),]) )
      names(listOfdfMarkPosCr)[[j]]<-currName
      j<-j+1
      }
    }

    if(length(listOfdfMarkPosCr)>0) {

    for (k in 1:length(listOfdfMarkPosCr) ) {

      colBorderCr1<-colCr1<-rad1<-yMarkCr1<-xMarkCr1<-NULL

      for (i in 1:nrow(listOfdfMarkPosCr[[k]])){
        # i=1
        ifelse(listOfdfMarkPosCr[[k]][i,"chrRegion"]=="q",longORshort<-0,longORshort<-1) #ifelseinloop
        ifelse(listOfdfMarkPosCr[[k]][i,"chrRegion"]=="q",column<-1,column<-2)
        ifelse(listOfdfMarkPosCr[[k]][i,"chrRegion"]=="q",mySign<--1,mySign<-1)
        corr_index<-which(names(ym) %in% names(listOfdfMarkPosCr)[[k]] )

        rowIndex <- nrow(listOfdfChromSize[[corr_index]])*longORshort+(listOfdfMarkPosCr[[k]][,"neworder"][i])
        armStart <- ym[[corr_index]][ rowIndex,column]
        ysupCr<- armStart +
          (listOfdfMarkPosCr[[k]][i,"markDistCen"]                                                      *normalizeToOne*mySign)

        yinfCr <- armStart +
          ( sum(listOfdfMarkPosCr[[k]][i,"markDistCen"],listOfdfMarkPosCr[[k]][i,"markSize"],na.rm=TRUE)*normalizeToOne*mySign)

        yMarkCr1[[i]]<-rep(list(mean(c(yinfCr,ysupCr))),2)
        attr(yMarkCr1[[i]], "rowIndex")<- rowIndex

        xBoundaries<-xm[[corr_index]][listOfdfMarkPosCr[[k]][,"neworder"][i],3:2]
        xBoundariesQuar<-(xBoundaries[2]-xBoundaries[1])/4

        if(chromatids & circularPlot==FALSE){
          xMarkCr1[[i]]<-c(list(xBoundaries[1]+xBoundariesQuar - (xModifierMono/2) ), list(xBoundaries[1]+ 3*xBoundariesQuar + (xModifierMono/2) ) )
        } else {
          xMarkCr1[[i]]<-c(list(xBoundaries[1]+xBoundariesQuar),list(xBoundaries[1]+3*xBoundariesQuar) ) # default
        }

        attr(xMarkCr1[[i]], "rowIndex")<- rowIndex

        rad1[[i]]<-rep(list(abs(ysupCr-yinfCr)/2 ),2)
        colCr1[[i]] <- rep(list(dfMarkColorInternal$markColor[match(listOfdfMarkPosCr[[k]]$markName[i], dfMarkColorInternal$markName)]),2)
        colBorderCr1[[i]] <- rep(list(dfMarkColorInternal$markBorderColor[match(listOfdfMarkPosCr[[k]]$markName[i], dfMarkColorInternal$markName)]),2)

      }
      yMarkCr[[k]]<-yMarkCr1
      attr(yMarkCr[[k]], "spname")<-names(listOfdfMarkPosCr)[[k]] # added 1.14
      xMarkCr[[k]]<-xMarkCr1
      attr(xMarkCr[[k]], "spname")<-names(listOfdfMarkPosCr)[[k]] # added 1.14

      rad[[k]]<-rad1
      colCr[[k]]<-colCr1
      colBorderCr[[k]]<-colBorderCr1

    } # end for k OTU

      # markListCr<-addChrNameAttrMarkDots(xMarkCr,yMarkCr,x) # 1.14
      #
      # xMarkCr <- markListCr$xMark
      # yMarkCr <- markListCr$yMark

    #####################
    #   add to plot MarkCrs DOTS monocen
    #####################

    # plotDotMarks(xMarkCr,yMarkCr,rad,colCr,n,xfactor,colBorderCr)
      plotDotMarks(bannedMarkName,xMarkCr,yMarkCr, rad, colCr,n,xfactor,colBorderCr,circularPlot, y,x,radius,circleCenter,circleCenterY,separFactor,
                   chrWidth,listOfdfMarkPosCr,markLabelSize,pattern,labelSpacing,useOneDot,legend,ylistTransChrSimple,rotation=rotation,
                   labelOutwards)


    } #     if(length(listOfdfMarkPosCr)>0){

   else {
     remove(listOfdfMarkPosCr)
   }
  } # end painting MarkCrs

  #
  #         WRITE INLINE LEGENDS WHEN DOTS
  #

  booleanDotsLabels<- legend=="inline" & exists("dfMarkColorInternal") & exists("listOfdfMarkPosCr") & circularPlot==FALSE

  if (booleanDotsLabels){
    textLabelDots(xMarkCr,yMarkCr,listOfdfChromSize,listOfdfMarkPosCr,specialChrSpacing,
                  chrSpacing,markLabelSize,pattern,bannedMarkName,xBoundariesQuar)
  }

  ##############################################################################################################################
  #
  #                                            dot style of marks holocen
  #
  ########################################

  if (exists("parlistOfdfMarkPosHolocen") & exists("dfMarkColorInternal") ){
    # message(crayon::green(paste0("holocen. dot marks section start" ) ) )
    listOfdfMarkPosCr<-list()

    colBorderCr<-colCr<-rad<-xMarkCr<-yMarkCr<-list()
    j<-1

    for (k in 1:length(parlistOfdfMarkPosHolocen)) {
      currName<-names(parlistOfdfMarkPosHolocen)[[k]]

      if(nrow(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="dots"),])>0){
        listOfdfMarkPosCr<-c(listOfdfMarkPosCr,list(parlistOfdfMarkPosHolocen[[k]][which(parlistOfdfMarkPosHolocen[[k]]$style=="dots"),]) )
        names(listOfdfMarkPosCr)[[j]]<-currName
        j<-j+1
      }
    }

    if(length(listOfdfMarkPosCr)>0){

    for (k in 1:length(listOfdfMarkPosCr) ) {

      yMarkCr1<-NULL
      xMarkCr1<-NULL
      rad1<-NULL
      colCr1<-NULL
      colBorderCr1<-NULL

      for (i in 1:nrow(listOfdfMarkPosCr[[k]])){
        corr_index<-which(names(ym) %in% names(listOfdfMarkPosCr)[[k]] )
        rowIndex <- (listOfdfMarkPosCr[[k]][,"neworder"][i])
        chrStart <- ym[[corr_index]][ rowIndex , 2]
        ysupCr<- chrStart +
          (listOfdfMarkPosCr[[k]][i,"markPos"]                                      *normalizeToOne*1)
        yinfCr<- chrStart +
          ( sum(listOfdfMarkPosCr[[k]][i,"markPos"], listOfdfMarkPosCr[[k]][i,"markSize"], na.rm=TRUE )*normalizeToOne)
        yMarkCr1[[i]]<-rep(list(mean(c(yinfCr,ysupCr))),2)
        attr(yMarkCr1[[i]], "rowIndex")<- rowIndex

        xBoundaries<-xm[[corr_index]][listOfdfMarkPosCr[[k]][,"neworder"][i],3:2]
        xBoundariesQuar<-(xBoundaries[2]-xBoundaries[1])/4
         #xMarkCr1[[i]]<-c(list(xBoundaries[1]+xBoundariesQuar),list(xBoundaries[1]+3*xBoundariesQuar) )

        if(chromatids & holocenNotAsChromatids==FALSE & circularPlot==FALSE){
          xMarkCr1[[i]]<-c(list(xBoundaries[1]+xBoundariesQuar - (xModifierHolo/2) ), list(xBoundaries[1]+ 3*xBoundariesQuar + (xModifierHolo/2) ) )
        } else {
          xMarkCr1[[i]]<-c(list(xBoundaries[1]+xBoundariesQuar),list(xBoundaries[1]+3*xBoundariesQuar) ) # default
        }

        attr(xMarkCr1[[i]], "rowIndex")<- rowIndex

        rad1[[i]]<-rep(list(abs(ysupCr-yinfCr)/2 ),2)
        colCr1[[i]] <- rep(list(dfMarkColorInternal$markColor[match(listOfdfMarkPosCr[[k]]$markName[i], dfMarkColorInternal$markName)]),2)
        colBorderCr1[[i]] <- rep(list(dfMarkColorInternal$markBorderColor[match(listOfdfMarkPosCr[[k]]$markName[i], dfMarkColorInternal$markName)]),2)
      }

      yMarkCr[[k]]<-yMarkCr1
      attr(yMarkCr[[k]], "spname")<-names(listOfdfMarkPosCr)[[k]] # added 1.14
      xMarkCr[[k]]<-xMarkCr1
      attr(xMarkCr[[k]], "spname")<-names(listOfdfMarkPosCr)[[k]] # added 1.14
      rad[[k]]<-rad1
      colCr[[k]]<-colCr1
      colBorderCr[[k]]<-colBorderCr1

    } # end for

    markListCr<-addChrNameAttrMarkDots(xMarkCr,yMarkCr,x) # 1.14

    xMarkCr <- markListCr$xMark
    yMarkCr <- markListCr$yMark

#    listOfdfMarkPosCrInternal<<-listOfdfMarkPosCr

    #####################
    #   add to plot MarkCrs DOTS holocen
    #####################

    plotDotMarks(bannedMarkName,xMarkCr,yMarkCr, rad, colCr,n,xfactor,colBorderCr,circularPlot, y,x,radius,circleCenter,circleCenterY,separFactor,
    chrWidth,listOfdfMarkPosCr,markLabelSize,pattern,labelSpacing,useOneDot,legend,ylistTransChrSimple, rotation=rotation,
    labelOutwards)

    } # if(length(listOfdfMarkPosCr)>0){

  else {remove(listOfdfMarkPosCr)}
  } # end painting MarkCrs

  #
  #     write inline legend dots holocen
  #

  booleanDotsLabels<- legend=="inline" & exists("dfMarkColorInternal") & exists("listOfdfMarkPosCr") & circularPlot==FALSE

  if (booleanDotsLabels){
    textLabelDots(xMarkCr,yMarkCr,listOfdfChromSize,listOfdfMarkPosCr,specialChrSpacing,
                            chrSpacing,markLabelSize,pattern,bannedMarkName,xBoundariesQuar)
  }

  ################################
  #
  #   centromeres calculate
  #
  ################################

  if(exists("listOfdfChromSizeMonocen") ) {

#   listOfdfChromSizeMonocenI<<-listOfdfChromSizeMonocen
#   listOfdfChromSizeI<<-listOfdfChromSize
#   centromereSize2I<<-centromereSize2
#   normalizeToOneI<<-normalizeToOne

    #
    # cannot be based on listofmonocen because of addmissingotu
    #

    CentsList<-lapply(1:length(listOfdfChromSize), function(x) tryCatch(rep(0, nrow(listOfdfChromSize[[x]]) ), error=function(e) NA )
                      ) # l
    names(CentsList)<-names(listOfdfChromSize)

    ycoordCents <-list()

    for (i in 1:length(CentsList)){
      corr_index <- which(names(listOfdfChromSize) %in% names(CentsList)[[i]] )
      centromereSize2 <- tryCatch(as.numeric(attr(listOfdfChromSize[[corr_index]],"centromere") ) , error=function(e) NA )

      ycoordCents[[i]] <- t(replicate(length(CentsList[[i]])*1, (c(rep(  karHeight+  (karHeiSpace*(i-1)), 2  ),
                                                                   rep(  karHeight+(centromereSize2*normalizeToOne)+(karHeiSpace*(i-1)),2  )
      )      )      ) # radius
      ) #t
    } # for
    names(ycoordCents)<-names(CentsList)

# ycoordCents[areNA]<-NA

# suppressWarnings(
if(is.na(addMissingOTUAfter[1])) {
    if(karSepar){
      for (s in 1:(length(ymCopyC)-1) ) {
        diffnext<-abs(min(ymCopyC[[s+1]] ) - max(ymCopyC[[s]]) )
        ymCopyC[[s+1]]=ymCopyC[[s+1]]-diffnext

        ycoordCents[[s+1]] <- ycoordCents[[s+1]] - diffnext

        ifelse(amoSepar>diffnext,amoSepar2<-diffnext,amoSepar2<-amoSepar)

        ymCopyC[[s+1]]=ymCopyC[[s+1]]+amoSepar2
        ycoordCents[[s+1]] <- ycoordCents[[s+1]] + amoSepar2
      } # for
    } # redu
}
    ycoordCentsS<- lapply(1:length(ycoordCents), function(j)
      tryCatch(base::split(ycoordCents[[j]], row(ycoordCents[[j]]) ),
               error=function(e) NA
               )
      )

    names(ycoordCentsS)<-names(listOfdfChromSize)

    #
    #   remove holocen
    #

    ycoordCentsS<-tryCatch(ycoordCentsS[which(names(ycoordCentsS) %in% monocenNames)] , error=function(e) NA )
    ycoordCents <-tryCatch(ycoordCents[which(names(ycoordCents) %in% monocenNames)] , error=function(e) NA )

    ycoordCentsS<-ycoordCentsS[!is.na(ycoordCentsS)]
    ycoordCents<-ycoordCents[!is.na(ycoordCents)]

    # modify cen. position to account for squareness of chr.

    if (squareness<=20){
      r3 <- chrWidth2/(squareness*3)
    } else {
      r3 <- 0
    }

    xcoordCents<-list()

    for (s in 1:length(xmnoNA)){
      if(names(listOfdfChromSizenoNA[s]) %in% monocenNames ){
        if(attr(listOfdfChromSizenoNA[[s]],"ytitle")=="cM"){
          chrWidth2  <-specialChrWidth
        } else {
          chrWidth2<-chrWidth
        }

        xmMatrix<-xmnoNA[[s]][1:(nrow(xmnoNA[[s]])/2),2:3]
        if(!inherits(xmMatrix, "matrix")) {
          xmMatrix<-  t(as.matrix(xmMatrix) )
        }
        xcoordCents[[s]] <-cbind(sweep(xmMatrix, 2, c(-r3,r3), "+"),
                                 sweep(xmMatrix, 2, c(-r3,r3), "+")
        ) #bind

        names(xcoordCents)[s] <- names(listOfdfChromSizenoNA[s])
        } # monocen
    } # for

    xcoordCents<-Filter(function(x) {length(x) >= 1}, xcoordCents)

    xcoordCentsS<- lapply(1:length(xcoordCents), function(j) base::split(xcoordCents[[j]], row(xcoordCents[[j]]) ) )
    names(xcoordCentsS)<-names(xcoordCents)

    # xcoordCentsS<-tryCatch(xcoordCentsS[which(names(xcoordCentsS) %in% monocenNames)], error= function(e) NA )
    xcoordCentsS<-xcoordCentsS[!is.na(xcoordCentsS)]

    if(!missing(chrBorderColor)){
      cenBorder<-chrBorderColor
    } else {
      cenBorder<-cenColor2
    }
    # this protects for weak colors, adds strong border

    if(!missing(fixCenBorder)){
      if (fixCenBorder){
            if (chrColor != cenColor2){
              cenBorder<-chrColor
            } else {
              cenBorder<-cenColor2
            }
      }
    }

    #
    #     plot centromeres
    #

#    xcoordCentsS4249 <<- xcoordCentsS
#    ycoordCentsS4250 <<- ycoordCentsS
#    xcoordCents4251  <<- xcoordCents
#    ycoordCents4252  <<- ycoordCents

   ### generate newLongyCen for roundedCen
if(roundedCen){
   pts <- seq(-pi/2, pi*1.5, length.out = n*4)
   ptsl <- split(pts, sort(rep(1:4, each=length(pts)/4, len=length(pts))) )

   newLongyCen <- newLongxCen<-list()

   for (s in 1:length(ycoordCentsS) ) {

     xyCoordsCen<-mapXYCen(1 , (length(ycoordCentsS[[s]]) ) ,
                        ycoordCentsS[[s]],
                        xcoordCentsS[[s]] ,
                        ptsl[[1]],ptsl[[2]],ptsl[[3]],ptsl[[4]]
                        )

     newLongxCen[[s]] <- xyCoordsCen$roundedX
     newLongyCen[[s]] <- xyCoordsCen$roundedY

     } # for s

   # newLongyCen<-newLongyCen[!is.na(newLongyCen)]
   # newLongxCen<-newLongxCen[!is.na(newLongxCen)]

   for(s in 1: length(newLongyCen)) {
     # attr(newLongyCen[[s]], "positionnoNA")<- attr(listOfdfChromSizenoNA[[s]],"positionnoNA")
     for (a in 1: length(newLongyCen[[s]])){
       # attr(newLongyCen[[s]][[a]],"chrName1")<- attr(y[[s]][[a]],"chrName1")
       names(newLongyCen[[s]])[a]<- names(ycoordCentsS[[s]][a])
       names(newLongxCen[[s]])[a]<- names(xcoordCentsS[[s]][a])
     }
   }

   names(newLongyCen) <- names(ycoordCentsS)
   names(newLongxCen) <- names(ycoordCentsS)

#   newLongxCen91 <<- newLongxCen
#   newLongyCen92 <<- newLongyCen

} # roundedCen
   ###

#    cenBorderI<<-cenBorder

  if(circularPlot) {

        spnamexcoord<- names(xcoordCentsS)[1]
        corrSp<- which(names(x) %in% spnamexcoord)
        diffXRounded <- max(x[[corrSp]][[1]]) - max(xcoordCentsS[[1]][[1]]) # needed for all cen marks

        #
        #   y
        #

        yInterSimple<-intercalate(y,monocenNames)
        names(yInterSimple)<-names(y)

        ylistNewChrSimple<-yVertoHor(yInterSimple, monocenNames = monocenNames)
        names(ylistNewChrSimple) <- names(y)

        #
        #   x horizontal to vertical Y for cent. is same for rounded or squared
        #

        ylistTransChrSimple <- transYList(ylistNewChrSimple,shrinkFactor,monocenNames)
        names(ylistTransChrSimple) <- names(ylistNewChrSimple)


        if(roundedCen){

          ### new way of making cen.

          xlistNew <- xHortoVerRoundCen(newLongxCen,diffXRounded)
          names(xlistNew) <- names(newLongxCen)
#          xlistNew4332 <<- xlistNew

          yMarkPer <- markMapPerCen(newLongyCen,y)
          names(yMarkPer) <- names(newLongyCen)
#          yMarkPer4335<<-yMarkPer
# transyListMark corrindex error

#          ylistTransChrSimple4371<<-ylistTransChrSimple
          ylistTransMark<-transyListCen(yMarkPer,ylistTransChrSimple)
          names(ylistTransMark) <- names(yMarkPer)
#          ylistTransMark4338<<-ylistTransMark
          ###

        } else {  # ROUNDEDCEN FALSE



        xlistNew <- xHortoVerRoundCen(xcoordCentsS,diffXRounded)
        names(xlistNew) <- names(xcoordCentsS)
#        xlistNew4389 <<- xlistNew
# markMapPer corrIndex error
        yMarkPer <- markMapPerCen(ycoordCentsS,y)
        names(yMarkPer) <- names(ycoordCentsS)
#        yMarkPer4367<<-yMarkPer
# transyListMark corrIndex error
#        ylistTransChrSimple4371<<-ylistTransChrSimple
        ylistTransMark<-transyListCen(yMarkPer,ylistTransChrSimple)
        names(ylistTransMark) <- names(yMarkPer)
#        ylistTransMark4371<<-ylistTransMark

        ###


  } # roundedCen

        if(!is.null(cenColor2) ) {

          circleMapsCen  <- applyMapCircle(radius,circleCenter,circleCenterY,separFactor,ylistTransMark,xlistNew,n,0,chrWidth,
                                           rotation=rotation)

          drawCen(circleMapsCen,cenColor2,cenBorder,lwd.chr)

        } # cenColor NULL

      } else if(circularPlot==FALSE) { # end cP ################################ cP FALSE

      if(!is.null(cenColor2) ) {

        if(length(xcoordCentsS)>0){

          if(roundedCen==FALSE){

             lapply(1:length(xcoordCentsS), function(w) mapply(function(x,y,z) graphics::polygon(x=x, y=y,
                                                                                            col=cenColor2,
                                                                                            lwd=lwd.chr,
                                                                                            border=z),
                                                          x=xcoordCentsS[[w]],
                                                          y=ycoordCentsS[[w]],
                                                          z=cenBorder
                                                    ) #m
            ) #l
          } else {
            lapply(1:length(newLongyCen), function(s) mapply(function(x,y,z) graphics::polygon(x=x, y=y,
                                                                                                col=cenColor2,
                                                                                                lwd=lwd.chr,
                                                                                                border=z),
                                                              x=newLongxCen[[s]],
                                                              y=newLongyCen[[s]],
                                                              z=cenBorder
            ) #m
            ) #l
          }
      } # if length xcoordCentsS
    } # cencolor
    } # circular Plot else not

  } # monocen exist listOfdfChromSizeMonocen

                                                #####################################
                                                #   centromere with marks - cen marks
                                                #####################################

  if (exists("parlistOfdfMarkPosDataCen") & exists("dfMarkColorInternal") ) {
#    parlistOfdfMarkPosDataCenI <<- parlistOfdfMarkPosDataCen
#    ycoordCentsI<<-ycoordCents
    # names(xcoordCents)<-names(listOfdfChromSizenoNA) #1:length(xcoordCents)

    xMarkCen<-yMarkCen<-list()

    for (k in 1:length(parlistOfdfMarkPosDataCen)){
      yMarkCen1<-NULL
      xMarkCen1<-NULL
      for (i in 1:nrow(parlistOfdfMarkPosDataCen[[k]])){
        corr_index<-which(names(xcoordCents) %in% names(parlistOfdfMarkPosDataCen)[[k]] )
        rowIndex<-(parlistOfdfMarkPosDataCen[[k]][,"neworder"][i])
        ysup<-ycoordCents[[corr_index]][rowIndex,3]
        yinf<-ycoordCents[[corr_index]][rowIndex,1]

        yMarkCen1[[i]]<-c(yinf,yinf,ysup,ysup)
        attr(yMarkCen1[[i]],"rowIndex")<-rowIndex

        xMarkCen1[[i]]<-xcoordCents[[corr_index]][ rowIndex ,]
        attr(xMarkCen1[[i]],"rowIndex")<-rowIndex

      } # each mark

      yMarkCen[[k]]<-yMarkCen1
      xMarkCen[[k]]<-xMarkCen1
      attr(yMarkCen[[k]], "spname") <- names(parlistOfdfMarkPosDataCen)[[k]] # added 1.14
      attr(xMarkCen[[k]], "spname") <- names(parlistOfdfMarkPosDataCen)[[k]] # added 1.14
    } # each df

    if(!missing(fixCenBorder)){
      if (fixCenBorder){
        fixCenBorder2<-TRUE
        } else {
        fixCenBorder2<-FALSE
        }
    } else {
      fixCenBorder2<-FALSE
    }

    if(roundedCen){
      # pts<- seq(-pi/2, pi*1.5, length.out = n*4)
      # ptsl<-split(pts, sort(rep(1:4, each=length(pts)/4, len=length(pts))) )

      newLongyCenMarks <- newLongxCenMarks<-list()

      for (s in 1:length(yMarkCen) ) {

        xyCoordsCenMarks <- mapXYCen(1 , (length(yMarkCen[[s]]) ) ,
                              yMarkCen[[s]],
                              xMarkCen[[s]] ,
                              ptsl[[1]],ptsl[[2]],ptsl[[3]],ptsl[[4]]
        )

        newLongxCenMarks[[s]] <- xyCoordsCenMarks$roundedX
        newLongyCenMarks[[s]] <- xyCoordsCenMarks$roundedY

      } # for s

      # newLongyCenMarks<-newLongyCenMarks[!is.na(newLongyCenMarks)]
      # newLongxCenMarks<-newLongxCenMarks[!is.na(newLongxCenMarks)]

      for(s in 1: length(newLongyCenMarks)) {
        attr(newLongyCenMarks[[s]], "spname")<- attr(yMarkCen[[s]], "spname")
        attr(newLongxCenMarks[[s]], "spname")<- attr(xMarkCen[[s]], "spname")

        for (a in 1: length(newLongyCenMarks[[s]])){
          attr(newLongyCenMarks[[s]][[a]],"rowIndex")<- attr(yMarkCen[[s]][[a]],"rowIndex")
          attr(newLongxCenMarks[[s]][[a]],"rowIndex")<- attr(xMarkCen[[s]][[a]],"rowIndex")
          # names(newLongyCenMarks[[s]])[a]<- names(y[[s]][a])
          # names(newLongxCenMarks[[s]])[a]<- names(y[[s]][a])
        }
      }

      # names(newLongyCenMarks)<-names(yMarkCen)
      # names(newLongxCenMarks)<-names(yMarkCen)

#      newLongxCenMI <<-newLongxCenMarks
#      newLongyCenMI <<-newLongyCenMarks
    } # rC



                                                    #####################
                                                    #   marks CENTROMERE
                                                    #####################

    if(circularPlot) {

      if (roundedCen==FALSE){

        xlistNew <- xHortoVerRoundCen(xMarkCen,diffXRounded)
        names(xlistNew) <- names(xMarkCen)
#        xlistNew4544<<-xlistNew

        yMarkPer <- markMapPer(yMarkCen,y)
#        yMarkPer4542<<-yMarkPer

        ylistTransMark<-transyListMark(yMarkPer,ylistTransChrSimple)
#        ylistTransMark4545<<-ylistTransMark
      } else {

        xlistNew <- xHortoVerRoundCen(newLongxCenMarks,diffXRounded)
        names(xlistNew) <- names(newLongxCenMarks)
#        xlistNew4570<<-xlistNew


        yMarkPer<-markMapPer(newLongyCenMarks,y)
#        yMarkPer4553<<-yMarkPer

        ylistTransMark<-transyListMark(yMarkPer,ylistTransChrSimple)
#        ylistTransMark4556<<-ylistTransMark
      }

      # from mimic
      circleMapsMarksCen  <- applyMapCircle(radius,circleCenter,circleCenterY,separFactor,ylistTransMark,xlistNew,n,0,chrWidth,
                                         rotation=rotation)

      # drawPlotMark(circleMapsMarksCen,dfMarkColorInternal,parlistOfdfMarkPosDataCen,lwd.chr*3) # works
      drawCenMarks(circleMapsMarksCen,dfMarkColorInternal,parlistOfdfMarkPosDataCen,lwd.chr,fixCenBorder2,chrColor)

      if(legend=="inline"){
        circleMapsMarksLabelCen  <- applyMapCircle(radius,circleCenter,circleCenterY,separFactor,ylistTransMark,xlistNew,n,
                                                   labelSpacing ,chrWidth,rotation=rotation)
        circLabelMark(bannedMarkName,circleMapsMarksLabelCen,parlistOfdfMarkPosDataCen, markLabelSize,pattern,labelOutwards,circleCenter,circleCenterY)
      }

    } else { # no circular

      if (roundedCen==FALSE){
        lapply(1:length(xMarkCen), function(m) mapply(function(x,y,z)
          graphics::polygon(
            x=x,
            y=y,
            col = dfMarkColorInternal$markColor[match(z, dfMarkColorInternal$markName)] ,
            lwd=lwd.chr,
            border=ifelse(fixCenBorder2,
                        chrColor,
                        dfMarkColorInternal$markBorderColor[match(z, dfMarkColorInternal$markName)] # z outside
                        ) # ifelse
          ), # p
           x=xMarkCen[[m]],
           y=yMarkCen[[m]],
           z=parlistOfdfMarkPosDataCen[[m]]$markName # ifelse here gives error

        ) # mapply
        ) #l
      } else {
        lapply(1:length(newLongyCenMarks), function(m) mapply(function(x,y,z)
          graphics::polygon(
            x=x,
            y=y,
            col = dfMarkColorInternal$markColor[match(z, dfMarkColorInternal$markName)] ,
            lwd=lwd.chr,
            border=ifelse(fixCenBorder2,
                          chrColor,
                          dfMarkColorInternal$markBorderColor[match(z, dfMarkColorInternal$markName)] # z outside
            ) # ifelse
          ), # p
          x=newLongxCenMarks[[m]],
          y=newLongyCenMarks[[m]],
          z=parlistOfdfMarkPosDataCen[[m]]$markName # ifelse here gives error
        ) # mapply
        ) #l
      } # rC

    } # circular bool

  } #                                                    end centromeres with marks

  ###########################################
  #
  #   plotting labels inline CENTR. marks
  #
  ###########################################

  # cen labels
  booleanColorInternalMarkCen<- legend=="inline" & exists("dfMarkColorInternal") & exists("parlistOfdfMarkPosDataCen") & circularPlot==FALSE

    if(booleanColorInternalMarkCen)  {
      # textLabel(xMarkCen,yMarkCen,listOfdfChromSize,parlistOfdfMarkPosDataCen,specialChrSpacing,chrSpacing,markLabelSize,pattern,TRUE)
      textLabelCen(xMarkCen,yMarkCen,listOfdfChromSize,parlistOfdfMarkPosDataCen,specialChrSpacing,chrSpacing,markLabelSize,pattern,bannedMarkName)
    } # if

  ##############################################
  #
  #   labels to the right
  #
  ##############################################

  xNoNA <- removeNAFromList(x,areNA)
  yNoNA <- removeNAFromList(y,areNA)

  if(legend=="aside" & exists("dfMarkColorInternal") ){

    dfMarkColorInternalNocM <- dfMarkColorInternal[which(!dfMarkColorInternal$style %in% c("cM","cMLeft") ),]

    # remove bannedMarkNames with cenStyle style
    if(exists("bannedMarkName")) {
#      # bannedMarkName5624<<-bannedMarkName
#      # dfMarkColorInternalNocM5625<<-dfMarkColorInternalNocM
    # dfMarkColorInternalNocM <- dfMarkColorInternalNocM[which(! (dfMarkColorInternalNocM$style %in% "cenStyle" &
                                                                  # dfMarkColorInternalNocM$markName %in% bannedMarkName) ) ,]
    dfMarkColorInternalNocM <- dfMarkColorInternalNocM[which(! (dfMarkColorInternalNocM$markName %in% bannedMarkName) ) ,]
    }
    if(length(dfMarkColorInternalNocM)==0){
      remove(dfMarkColorInternalNocM)
    }

    #
    # allow cM if cen. because they are not shown as cM , they must be aside if desired
    #

    if(exists("cenMarkNames") ) {
      dfMarkColorInternalcMnoCen <- dfMarkColorInternal[which(dfMarkColorInternal$style    %in% c("cM","cMLeft") &
                                                              dfMarkColorInternal$markName %in% cenMarkNames ) , ]
      if( exists("dfMarkColorInternalNocM") ) {
        dfMarkColorInternalNocM<-dplyr::bind_rows(dfMarkColorInternalNocM,dfMarkColorInternalcMnoCen)
      } else {
        dfMarkColorInternalNocM<-dfMarkColorInternalcMnoCen
      }
    }

    if(remSimiMarkLeg){
      dfMarkColorInternalNocM$markName <- sub(pattern,"",dfMarkColorInternalNocM$markName )
      dfMarkColorInternalNocM<-dfMarkColorInternalNocM[!duplicated(dfMarkColorInternalNocM$markName),]
    }

    if(nrow(dfMarkColorInternalNocM)>0 )  {
      if(!exists("allMarkMaxSize") ){
        message(crayon::green(paste0("Seems you didn't provide markSize (NA), if you want cM style, set it in the dfMarkColor data.frame" ) ) )

        allMarkMaxSize<-1
      }


      if(circularPlot){
        maxx<-max(unlist(circleMaps), na.rm=TRUE )
      } else {
        maxx<-(max(unlist(xNoNA)) )
      }


    plotlabelsright(maxx,yNoNA, markLabelSpacer,chrWidth,dfMarkColorInternalNocM,allMarkMaxSize,normalizeToOne,
                              markLabelSize,xfactor,legendWidth, legendHeight, n*4, pattern,legendYcoord)
    } # is df
  } # if aside


  #################################
  #
  # add notes (to the right)
  #
  #################################

if(circularPlot==FALSE) {

  if(!missing(notes)){
    if(!inherits(notes, "data.frame") ) {
      message(crayon::blurred("Notes are not in a data.frame, ignoring notes"))
    } else {
      for (i in 1:length(listOfdfChromSizenoNA) ) {

        if(attr(listOfdfChromSizenoNA[[i]],"cenType")=="monocen"){
            yposNote<-min(ymnoNA[[i]][,1])
        } else {
            yposNote<-(max(ymnoNA[[i]]) + min(ymnoNA[[i]]) ) /2
        }

        xposNote<-max(xmnoNA[[i]]) + (notesPos/2)

        note <- notes[which(notes$OTU %in% names(listOfdfChromSizenoNA)[i] ), ]$note

        if(!is.null(note) & length(note)>0 ) {

            spl<-unlist(strsplit(note,"\\s\\+\\s") )

            lastpart<-unlist(strsplit(spl[length(spl)],"[0-9]{0,9}?FL[\\+0]?\\s+"   ) )
            lastpart<-lastpart[2][!is.na(lastpart[2])]
            lastpart<-ifelse(length(lastpart),lastpart,"")

            common  <-spl[which(!spl %in% grep("FL0|FL+", spl, value=TRUE) )]

            uncommon<-spl[which(spl %in% grep("FL0|FL+", spl, value=TRUE) )]

            splUncommon <- strsplit(uncommon,"")

            if(length(splUncommon)==1) {
              splUncommon1<-splUncommon[[1]]
              first<-splUncommon1[1]
              sec  <-splUncommon1[2]
              third<-splUncommon1[3]
              forth<-splUncommon1[4]
              forth<-ifelse(forth==" "|is.na(forth),"",forth)

              graphics::text(
                xposNote# xlimLeftMod
                ,yposNote + notesPosY/2
                ,labels = bquote(paste(.(paste(common,"+", collapse=" ") ),
                                       " ",
                                       .(first),
                                       .(sec)[.(third)]^.(forth),
                                       " ",
                                       .(paste(lastpart) ) )
                ),
                cex=notesTextSize,
                adj=0
                ,font =  ifelse(OTUasNote,ifelse( !missing(OTUfont),    OTUfont,   1),1),
                family = ifelse(OTUasNote,ifelse( !missing(OTUfamily),  OTUfamily, defaultFontFamily2),defaultFontFamily2) # justif
              ) # end graphics::text


            } else if (length(splUncommon)==2) {
              splUncommon1<-splUncommon[[1]]
              first<-splUncommon1[1]
              sec  <-splUncommon1[2]
              third<-splUncommon1[3]
              forth<-splUncommon1[4]
              forth<-ifelse(forth==" "|is.na(forth),"",forth)

              splUncommon2<-splUncommon[[2]]
              first2<-splUncommon2[1]
              sec2  <-splUncommon2[2]
              third2<-splUncommon2[3]
              forth2<-splUncommon2[4]
              forth2<-ifelse(forth2==" "|is.na(forth2),"",forth2)

              graphics::text(
                xposNote
                ,yposNote + notesPosY/2
                ,labels = bquote(paste(.(paste(common,"+", collapse=" ") )," ",
                                       .(first),.(sec)[.(third)]^.(forth)  ,
                                       " + ",
                                       .(first2),.(sec2)[.(third2)]^.(forth2),
                                       " ",
                                       .(paste(lastpart) ) )
                )
                ,cex=notesTextSize,
                adj=0
                ,font =  ifelse(OTUasNote,ifelse( !missing(OTUfont),    OTUfont,   1),1),
                family = ifelse(OTUasNote,ifelse( !missing(OTUfamily),  OTUfamily, defaultFontFamily2),defaultFontFamily2) # justif
              ) # end graphics::text

            } else {

              hasQuotes<-grepl("'.*'",note)

              if(!missing(OTUfont) & OTUasNote ) {

                if(OTUfont==3 & hasQuotes){
                  begin <- sub("\\'.*", "", note)
                  varNames <- unlist(regmatches(note,gregexpr("'(.*?)'",note) ) )
                  varConcat<-paste(varNames, collapse=" ")
                  nameWithVar<- bquote(paste(italic(.(begin)),.(varConcat)  ) )
                } else {
                  nameWithVar<- note
                }
              } else {
                nameWithVar<- note
              }

              # text(0.2, 0.6, cex = 1.5, mytext)
              graphics::text(
                # max(xmnoNA[[i]]) + (1*(notesPos/2) )
                xposNote
                ,yposNote + notesPosY/2
                ,labels = nameWithVar,
                cex=notesTextSize,
                adj=0
                ,font=   ifelse(OTUasNote,ifelse( !missing(OTUfont),    OTUfont,   1),1),
                family = ifelse(OTUasNote,ifelse( !missing(OTUfamily),  OTUfamily, defaultFontFamily2),defaultFontFamily2) # justif
              ) # end graphics::text
            }
            #

          } # null

      } # for

      # message(crayon::green(paste0("notes section end" ) ) )
    } # exists d.f.
  } # fi notes

  #
  #     leftNotes ##################################
  #

  if(!missing(leftNotes)){
    if(!inherits(leftNotes, "data.frame") ) {
      message(crayon::blurred("leftNotes are not in a data.frame, ignoring leftNotes"))
    } else {
      for (i in 1:length(listOfdfChromSizenoNA) ) {

          if(as.numeric(attr(listOfdfChromSizenoNA[[i]],"groupPresence") ) > 0 ) {
            ifelse(groupName,groupCount<-2,groupCount<-1)
          } else {
            groupCount=0
          } # end ifelse

          if(attr(listOfdfChromSizenoNA[[i]],"indexStatus")=="failure") {indexCount2=indexCount*0} else {indexCount2=indexCount}
          if(attr(listOfdfChromSizenoNA[[i]],"indexStatus")=="failure") {morphoCount2=morphoCount*0} else {morphoCount2=morphoCount}

          # lapply(1:length(xmnoNA), function(i) {
          if(attr(xmnoNA[[i]],"cenType")=="holocen") {
            # decVector<-5
            holocenDisCount <- morphoCount2 + indexCount2 #newDistVector #+bothAdd
          } else {
            holocenDisCount <- 0
          } # ifelse holocen

        yposNote <- min(ymnoNA[[i]]) - ((distTextChr/3) * (chrIdCount + morphoCount2 + indexCount2 + groupCount + 3 - holocenDisCount) )

        # if(attr(listOfdfChromSizenoNA[[i]],"cenType")=="monocen"){
        #   yposNote<-min(ymnoNA[[i]][,1])
        # } else {
        #   yposNote<-(max(ymnoNA[[i]]) + min(ymnoNA[[i]]) ) /2
        # }

        # if(notesLeft){
          xposNote<-min(xmnoNA[[i]]) + ((leftNotesPos/2) )
        # } else {
          # xposNote<-max(xmnoNA[[i]]) + (1*(notesPos/2) )
        # }

        note <- leftNotes[which(leftNotes$OTU %in% names(listOfdfChromSizenoNA)[i] ), ]$note

        if(!is.null(note) & length(note)>0 ) {

          spl<-unlist(strsplit(note,"\\s\\+\\s") )

          lastpart<-unlist(strsplit(spl[length(spl)],"[0-9]{0,9}?FL[\\+0]?\\s+"   ) )
          lastpart<-lastpart[2][!is.na(lastpart[2])]
          lastpart<-ifelse(length(lastpart),lastpart,"")

          common  <-spl[which(!spl %in% grep("FL0|FL+", spl, value=TRUE) )]

          uncommon<-spl[which(spl %in% grep("FL0|FL+", spl, value=TRUE) )]

          splUncommon <- strsplit(uncommon,"")

          if(length(splUncommon)==1){
            splUncommon1<-splUncommon[[1]]
            first<-splUncommon1[1]
            sec  <-splUncommon1[2]
            third<-splUncommon1[3]
            forth<-splUncommon1[4]
            forth<-ifelse(forth==" "|is.na(forth),"",forth)

            graphics::text(
              xposNote# xlimLeftMod
              ,yposNote + leftNotesPosY/2
              ,labels = bquote(paste(.(paste(common,"+", collapse=" ") ),
                                     " ",
                                     .(first),
                                     .(sec)[.(third)]^.(forth),
                                     " ",
                                     .(paste(lastpart) ) )
              ),
              cex=notesTextSize,
              adj=0
              ,font =  1#ifelse(OTUasNote,ifelse( !missing(OTUfont),    OTUfont,   1),1),
              ,family = defaultFontFamily2#ifelse(OTUasNote,ifelse( !missing(OTUfamily),  OTUfamily, defaultFontFamily2),defaultFontFamily2) # justif
            ) # end graphics::text


          } else if (length(splUncommon)==2) {
            splUncommon1<-splUncommon[[1]]
            first<-splUncommon1[1]
            sec  <-splUncommon1[2]
            third<-splUncommon1[3]
            forth<-splUncommon1[4]
            forth<-ifelse(forth==" "|is.na(forth),"",forth)

            splUncommon2<-splUncommon[[2]]
            first2<-splUncommon2[1]
            sec2  <-splUncommon2[2]
            third2<-splUncommon2[3]
            forth2<-splUncommon2[4]
            forth2<-ifelse(forth2==" "|is.na(forth2),"",forth2)

            graphics::text(
              # max(xmnoNA[[i]]) + (1*(notesPos/2) )
              xposNote
              ,yposNote + leftNotesPosY/2
              ,labels = bquote(paste(.(paste(common,"+", collapse=" ") )," ",
                                     .(first),.(sec)[.(third)]^.(forth)  ,
                                     " + ",
                                     .(first2),.(sec2)[.(third2)]^.(forth2),
                                     " ",
                                     .(paste(lastpart) ) )
              )
              ,cex=notesTextSize,
              adj=0
              ,font =  1#ifelse(OTUasNote, ifelse( !missing(OTUfont),    OTUfont,   1),1),
              ,family = defaultFontFamily2 #ifelse(OTUasNote,ifelse( !missing(OTUfamily),  OTUfamily, defaultFontFamily2),defaultFontFamily2) # justif
            ) # end graphics::text

          } else {
            # text(0.2, 0.6, cex = 1.5, mytext)
            graphics::text(
              # max(xmnoNA[[i]]) + (1*(notesPos/2) )
              xposNote
              ,yposNote + leftNotesPosY/2
              ,labels = paste(note ),
              cex=notesTextSize,
              adj=0
              ,font=  1#ifelse(OTUasNote,ifelse( !missing(OTUfont),    OTUfont,   1),1),
              ,family = defaultFontFamily2#ifelse(OTUasNote,ifelse( !missing(OTUfamily),  OTUfamily, defaultFontFamily2),defaultFontFamily2) # justif
            ) # end graphics::text
          }
          #

        } # null

      } # for

      # message(crayon::green(paste0("notes section end" ) ) )
    } # exists d.f.
  } # fi notes

} # CIRC false

}
