## ----"setup", include=FALSE---------------------------------------------------
library(inlpubs)
svg_orcid <- "<svg style='height:1em;position:relative;margin-left:.25em;vertical-align:middle;' viewBox='0 0 512 512' aria-describedby='orcid' role='img'><desc id='orcid'>ORCiD</desc><path fill='#A6CE39' d='M294.75 188.19h-45.92V342h47.47c67.62 0 83.12-51.34 83.12-76.91 0-41.64-26.54-76.9-84.67-76.9zM256 8C119 8 8 119 8 256s111 248 248 248 248-111 248-248S393 8 256 8zm-80.79 360.76h-29.84v-207.5h29.84zm-14.92-231.14a19.57 19.57 0 1 1 19.57-19.57 19.64 19.64 0 0 1-19.57 19.57zM300 369h-81V161.26h80.6c76.73 0 110.44 54.83 110.44 103.85C410 318.39 368.38 369 300 369z'></path></svg>"
svg_email <- "<svg style='height:1em;position:relative;margin-left:.25em;vertical-align:middle;' viewBox='0 0 512 512' aria-describedby='email' role='img'><desc id='email'>Email</desc><path fill='#4682B4' d='M502.3 190.8c3.9-3.1 9.7-.2 9.7 4.7V400c0 26.5-21.5 48-48 48H48c-26.5 0-48-21.5-48-48V195.6c0-5 5.7-7.8 9.7-4.7 22.4 17.4 52.1 39.5 154.1 113.6 21.1 15.4 56.7 47.8 92.2 47.6 35.7.3 72-32.8 92.3-47.6 102-74.1 131.6-96.3 154-113.7zM256 320c23.2.4 56.6-29.2 73.4-41.4 132.7-96.3 142.8-104.7 173.4-128.7 5.8-4.5 9.2-11.5 9.2-18.9v-19c0-26.5-21.5-48-48-48H48C21.5 64 0 85.5 0 112v19c0 7.4 3.4 14.3 9.2 18.9 30.6 23.9 40.7 32.4 173.4 128.7 16.8 12.2 50.2 41.8 73.4 41.4z'/></svg>"
cit <- inlpubs::pubs$citation

## ----"bib_dat", include=FALSE-------------------------------------------------
key <- rownames(pubs)

cite <- apply(pubs, 1, function(x) {
  cit <- format(x$citation$author, "family")
  if (length(cit) > 2) {
    cit <- sprintf("%s and others", cit[1])
  } else if (length(cit) == 2) {
    cit <- paste(cit, collapse = " and ")
  }
  cit
})

badge <- unlist(apply(pubs, 1, function(x) {
  doi <- x$citation$doi
  fmt <- "<a class='pubs-tag %s' href='%s' target='_blank'>DOI&nbsp;%s</a>"
  if (is.null(doi)) {
    sprintf(fmt, "other", "https://www.doi.org/", "not available")
  } else {
    sprintf(fmt, "doi", sprintf("https://doi.org/%s", doi), doi)
  }
}))

title <- apply(pubs, 1, function(x) x$citation$title)

author <- apply(pubs, 1, function(x) {
  aut <- format(x$citation$author, c("given", "family"))
  for (i in seq_along(aut)) {
    orcid <- x$citation$author[i]$comment["ORCID"]
    if (!is.null(orcid)) {
      href <- sprintf("https://orcid.org/%s", orcid)
      icon <- sprintf("<a href='%s' target='_blank'>%s</a>", href, svg_orcid)
      aut[i] <- paste0(aut[i], icon)
    }
    email <- x$citation$author[i]$email
    if (!is.null(email)) {
      href <- sprintf("mailto: %s", email)
      icon <- sprintf("<a href='%s'>%s</a>", href, svg_email)
      aut[i] <- paste0(aut[i], icon)
    }
  }
  n <- length(aut)
  if (n > 2) aut[-n] <- paste0(aut[-n], ",")
  if (n > 1) aut[n] <- paste("and", aut[n])
  paste(aut, collapse = " ")
})

reference <- vapply(cit, function(x) attr(unclass(x)[[1]], "textVersion"), character(1))

bib <- lapply(cit, function(x) {
  b <- utils::toBibtex(x)
  idx <- c(1, length(b))
  out <- stringi::stri_wrap(b[-idx], width = 60, simplify = FALSE, indent = 2, exdent = 4)
  b[-idx] <- vapply(out, function(y) {
    stringi::stri_c(y, collapse = "\n", ignore_null = TRUE)
  }, character(1))
  b
})

## ----"bib_lst", echo=FALSE, results="asis"------------------------------------
for (yr in seq(max(pubs$year), min(pubs$year))) {
  cat(sprintf("## %s", yr), "\n\n")
  idx <- which(pubs$year %in% yr)
  if (length(idx) == 0) cat("<p>None</p>", "\n\n")
  for (i in idx) {
    cat(sprintf("### %s {#%s}", cite[i], key[i]), "\n\n")
    if (!is.null(badge[i])) {
      cat("<div style='paddding:10px 0'>", "\n")
      cat(badge[i], "\n")
      cat("</div>", "\n\n")
    }
    cat("#### Title", "\n\n")
    cat(title[i], "\n\n")
    cat("#### Authors", "\n\n")
    cat(author[i], "\n\n")
    cat("#### Suggested citation", "\n\n")
    cat(reference[i], "\n\n")

    cat("<details>", "\n")
    cat("<summary>BibTeX citation</summary>", "\n")
    cat("```", "\n")
    cat(bib[[i]], sep = "\n")
    cat("```", "\n")
    cat("</details>", "\n")
    x <- pubs[i, "abstract"]
    if (!is.na(x) && nzchar(x)) {
      cat("<details>", "\n\n")
      cat("<summary>Abstract</summary>", "\n")
      cat("\n")
      cat(stringi::stri_unescape_unicode(x), sep = "\n")
      cat("\n")
      cat("</details>", "\n\n")
    }
    x <- pubs[i, "annotation"]
    if (!is.na(x) && nzchar(x)) {
      cat("<details>", "\n")
      cat("<summary>Annotation</summary>", "\n")
      cat("\n")
      cat(stringi::stri_unescape_unicode(x), sep = "\n")
      cat("\n")
      cat("</details>", "\n\n")
    }
  }
}

