% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/islasso.path.R
\name{islasso.path}
\alias{islasso.path}
\alias{print.islasso.path}
\alias{islasso.path.fit}
\alias{coef.islasso.path}
\alias{deviance.islasso.path}
\alias{fitted.islasso.path}
\alias{residuals.islasso.path}
\alias{logLik.islasso.path}
\alias{print.logLik.islasso.path}
\alias{model.matrix.islasso.path}
\title{Induced Smoothed Lasso Regularization Path}
\usage{
islasso.path(
  formula,
  family = gaussian(),
  lambda = NULL,
  nlambda = 100,
  lambda.min.ratio = ifelse(nobs < nvars, 0.01, 1e-04),
  alpha = 1,
  data,
  weights,
  subset,
  offset,
  contrasts = NULL,
  unpenalized,
  control = is.control()
)
}
\arguments{
\item{formula}{Model formula of type \code{response ~ predictors}.}

\item{family}{Response distribution. Supported families: \code{gaussian()}, \code{binomial()}, \code{poisson()}, \code{Gamma()}.}

\item{lambda}{Optional numeric vector of lambda values. If not provided, a sequence is automatically generated.}

\item{nlambda}{Integer. Number of lambda values to generate if \code{lambda} is missing. Default is \code{100}.}

\item{lambda.min.ratio}{Smallest lambda as a fraction of \code{lambda.max}. Default: \code{1e-2} if \code{nobs < nvars}, else \code{1e-3}.}

\item{alpha}{Elastic-net mixing parameter: \code{alpha = 1} is lasso, \code{alpha = 0} is ridge.}

\item{data}{Data frame containing model variables.}

\item{weights}{Optional observation weights.}

\item{subset}{Optional logical or numeric vector to subset observations.}

\item{offset}{Optional vector of prior known components for the linear predictor.}

\item{contrasts}{Optional contrast settings for factor variables.}

\item{unpenalized}{Optional vector of variable names or indices excluded from penalization.}

\item{control}{A list of control parameters via \code{\link{is.control}}.}
}
\value{
A list with components:
\item{call}{Matched function call.}
\item{Info}{Matrix with diagnostics: lambda, deviance, degrees of freedom, dispersion, iterations, convergence status.}
\item{GoF}{Model goodness-of-fit metrics: AIC, BIC, AICc, GCV, GIC, eBIC.}
\item{Coef}{Matrix of coefficients across lambda values.}
\item{SE}{Matrix of standard errors.}
\item{Weights}{Matrix of mixing weights for the smoothed penalty.}
\item{Gradient}{Matrix of gradients for the smoothed penalty.}
\item{Linear.predictors, Fitted.values, Residuals}{Matrices of fitted quantities across the path.}
\item{Input}{List of input arguments and design matrix.}
\item{control, formula, model, terms, data, xlevels, contrasts}{Standard model components.}
}
\description{
Fits a sequence of penalized regression models using the Induced Smoothing Lasso approach over a grid of lambda values.
Supports elastic-net penalties and generalized linear models: Gaussian, Binomial, Poisson, and Gamma.
}
\details{
This function fits a regularization path of models using the induced smoothing paradigm, replacing the non-smooth L1 penalty with a differentiable surrogate.
Standard errors are returned for all lambda points, allowing for Wald-based hypothesis testing. The regularization path spans a range of lambda values,
either user-defined or automatically computed.
}
\examples{
n <- 100; p <- 30; p1 <- 10  # number of nonzero coefficients

beta.veri <- sort(round(c(seq(.5, 3, length.out = p1/2),
                         seq(-1, -2, length.out = p1/2)), 2))
beta <- c(beta.veri, rep(0, p - p1))
sim1 <- simulXy(n = n, p = p, beta = beta, seed = 1, family = gaussian())
o <- islasso.path(y ~ ., data = sim1$data,
                  family = gaussian(), nlambda = 30L)
o

summary(o, lambda = 10, pval = 0.05)
coef(o, lambda = 10)
fitted(o, lambda = 10)
predict(o, type = "response", lambda = 10)
plot(o, yvar = "coef")
residuals(o, lambda = 10)
deviance(o, lambda = 10)
logLik(o, lambda = 10)
GoF.islasso.path(o)

\dontrun{
##### binomial ######
beta <- c(1, 1, 1, rep(0, p - 3))
sim2 <- simulXy(n = n, p = p, beta = beta, interc = 1, seed = 1,
                size = 100, family = binomial())
o2 <- islasso.path(cbind(y.success, y.failure) ~ ., data = sim2$data,
                   family = binomial(), lambda = seq(0.1, 100, l = 50L))
temp <- GoF.islasso.path(o2)
summary(o2, pval = 0.05, lambda = temp$lambda.min["BIC"])

##### poisson ######
beta <- c(1, 1, 1, rep(0, p - 3))
sim3 <- simulXy(n = n, p = p, beta = beta, interc = 1, seed = 1,
                family = poisson())
o3 <- islasso.path(y ~ ., data = sim3$data, family = poisson(), nlambda = 30L)
temp <- GoF.islasso.path(o3)
summary(o3, pval = 0.05, lambda = temp$lambda.min["BIC"])

##### Gamma ######
beta <- c(1, 1, 1, rep(0, p - 3))
sim4 <- simulXy(n = n, p = p, beta = beta, interc = -1, seed = 1,
                family = Gamma(link = "log"))
o4 <- islasso.path(y ~ ., data = sim4$data, family = Gamma(link = "log"),
                   nlambda = 30L)
temp <- GoF.islasso.path(o4)
summary(o4, pval = .05, lambda = temp$lambda.min["BIC"])
}

}
\references{
Cilluffo G., Sottile G., La Grutta S., Muggeo V.M.R. (2019).
\emph{The Induced Smoothed Lasso: A practical framework for hypothesis testing in high dimensional regression}.
Statistical Methods in Medical Research. DOI: 10.1177/0962280219842890

Sottile G., Cilluffo G., Muggeo V.M.R. (2019).
\emph{The R package islasso: estimation and hypothesis testing in lasso regression}.
Technical Report. DOI: 10.13140/RG.2.2.16360.11521
}
\seealso{
\code{\link{plot.islasso.path}}, \code{\link{predict.islasso.path}}, \code{\link{summary.islasso.path}},
         \code{\link{is.control}}, \code{\link{GoF.islasso.path}}, \code{\link{islasso}}, \code{\link{simulXy}}
}
\author{
Gianluca Sottile \email{gianluca.sottile@unipa.it}
}
