% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summ.R
\name{summ.merMod}
\alias{summ.merMod}
\alias{j_summ.merMod}
\title{Mixed effects regression summaries with options}
\usage{
\method{summ}{merMod}(model, scale = FALSE, confint = FALSE,
  ci.width = 0.95, digits = getOption("jtools-digits", default = 2),
  model.info = TRUE, model.fit = TRUE, r.squared = FALSE, pvals = NULL,
  n.sd = 1, center = FALSE, scale.response = FALSE, odds.ratio = FALSE,
  t.df = NULL, ...)
}
\arguments{
\item{model}{A \code{\link[lme4]{merMod}} object.}

\item{scale}{If \code{TRUE}, reports standardized regression
coefficients. Default is \code{FALSE}.}

\item{confint}{Show confidence intervals instead of standard errors? Default
is \code{FALSE}.}

\item{ci.width}{A number between 0 and 1 that signifies the width of the
desired confidence interval. Default is \code{.95}, which corresponds
to a 95\% confidence interval. Ignored if \code{confint = FALSE}.}

\item{digits}{An integer specifying the number of digits past the decimal to
report in the output. Default is 2. You can change the default number of
digits for all jtools functions with
\code{options("jtools-digits" = digits)} where digits is the desired
number.}

\item{model.info}{Toggles printing of basic information on sample size,
name of DV, and number of predictors.}

\item{model.fit}{Toggles printing of R-squared and adjusted R-squared.}

\item{r.squared}{Calculate an r-squared model fit statistic? Default is
\code{FALSE} because it seems to have convergence problems too often.}

\item{pvals}{Show p values and significance stars? If \code{FALSE}, these
are not printed. Default is \code{TRUE}, except for merMod objects (see
details).}

\item{n.sd}{If \code{scale = TRUE}, how many standard deviations should
predictors be divided by? Default is 1, though some suggest 2.}

\item{center}{If you want coefficients for mean-centered variables but don't
want to standardize, set this to \code{TRUE}.}

\item{scale.response}{Should standardization apply to response variable?
Default is \code{FALSE}.}

\item{odds.ratio}{If \code{TRUE}, reports exponentiated coefficients with
confidence intervals for exponential models like logit and Poisson models.
This quantity is known as an odds ratio for binary outcomes and incidence
rate ratio for count models.}

\item{t.df}{For \code{lmerMod} models only. User may set the degrees of
freedom used in conducting t-tests. See details for options.}

\item{...}{This just captures extra arguments that may only work for other
types of models.}
}
\value{
If saved, users can access most of the items that are returned in
the output (and without rounding).

\item{coeftable}{The outputted table of variables and coefficients}
\item{model}{The model for which statistics are displayed. This would be
most useful in cases in which \code{scale = TRUE}.}

Much other information can be accessed as attributes.
}
\description{
\code{summ} prints output for a regression model in a fashion similar to
\code{summary}, but formatted differently with more options.
}
\details{
By default, this function will print the following items to the
console:
\itemize{
\item The sample size
\item The name of the outcome variable
\item The (Pseudo-)R-squared value and AIC/BIC.
\item A table with regression coefficients, standard errors, and t-values.
}

The \code{scale} and \code{center} options are performed via refitting
the model with \code{\link{scale_lm}} and \code{\link{center_lm}},
respectively. Each of those in turn uses \code{\link{gscale}} for the
mean-centering and scaling.

\code{merMod} models are a bit different than the others. The \code{lme4}
package developers have, for instance, made a decision not to report or
compute p values for \code{lmer} models. There are good reasons for this,
most notably that the t-values produced are not "accurate" in the sense of
the Type I error rate. For certain large, balanced samples with many
groups, this is no big deal. What's
a "big" or "small" sample? How much balance is necessary? What type of
random effects structure is okay? Good luck getting a statistician to
give you any clear guidelines on this.
Some simulation studies have been done on fewer than 100 observations, so
for sure if your sample is around 100 or fewer you should not interpret
the t-values. A large number of groups is also crucial for avoiding bias
using t-values. If groups are nested or crossed in a linear model,
it is best to just get the \pkg{pbkrtest} package.

By default, this function follows \code{lme4}'s lead and does not report
the p values for \code{lmer} models. If the user has \pkg{pbkrtest}
installed, however, p values are reported using the Kenward-Roger
d.f. approximation unless \code{pvals = FALSE} or \code{t.df} is
set to something other than \code{NULL}.

See \code{\link[lme4]{pvalues}} from the \pkg{lme4} for more details.
If you're looking for a simple test with no extra packages installed,
it is better to use the confidence
intervals and check to see if they exclude zero than use the t-test.
For users of \code{glmer}, see some of the advice there as well. While
\code{lme4} and by association \code{summ} does as well, they are
still imperfect.

You have some options to customize the output in this regard with the
\code{t.df} argument. If \code{NULL}, the default, the
degrees of freedom used depends on whether the user has \pkg{pbkrtest}
installed. If installed, the Kenward-Roger approximation is used. If not,
and user sets \code{pvals = TRUE}, then the residual degrees of freedom
is used. If \code{t.df = "residual"}, then the residual d.f. is used
without a message. If the user prefers to use some other method to
determine the d.f., then any number provided as the argument will be
used.
}
\examples{

library(lme4, quietly = TRUE)
data(sleepstudy)
mv <- lmer(Reaction ~ Days + (Days | Subject), sleepstudy)

summ(mv) # Note lack of p values if you don't have pbkrtest

# Without pbkrtest, you'll get message about Type 1 errors
summ(mv, pvals = TRUE)

# To suppress message, manually specify t.df argument
summ(mv, t.df = "residual")

\dontrun{
 # Confidence intervals may be better alternative in absence of pbkrtest
 summ(mv, confint = TRUE)
}

}
\references{
Johnson, P. C. D. (2014). Extension of Nakagawa & Schielzeth’s
$R^{2}_{GLMM}$ to random slopes models. \emph{Methods in Ecology and
Evolution}, \emph{5}, 944–946. https://doi.org/10.1111/2041-210X.12225

Kenward, M. G., & Roger, J. H. (1997). Small sample inference for fixed
effects from restricted maximum likelihood. \emph{Biometrics},
\emph{53}, 983.
https://doi.org/10.2307/2533558

Luke, S. G. (2017). Evaluating significance in linear mixed-effects models
in R. \emph{Behavior Research Methods}, \emph{49}, 1494–1502.
https://doi.org/10.3758/s13428-016-0809-y

Nakagawa, S., & Schielzeth, H. (2013). A general and simple method for
obtaining $R^2$ from generalized linear mixed-effects models.
\emph{Methods in Ecology and Evolution}, \emph{4}, 133–142.
https://doi.org/10.1111/j.2041-210x.2012.00261.x
}
\seealso{
\code{\link{scale_lm}} can simply perform the standardization if
preferred.

\code{\link{gscale}} does the heavy lifting for mean-centering and scaling
behind the scenes.

\code{\link[pbkrtest]{get_ddf_Lb}} gets the Kenward-Roger degrees of
freedom if you have \pkg{pbkrtest} installed.

A tweaked version of \code{\link[MuMIn]{r.squaredGLMM}} is used to
generate the pseudo-R-squared estimates for linear models.
}
\author{
Jacob Long <\email{long.1377@osu.edu}>
}
