% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/svytools.R
\name{wgttest}
\alias{wgttest}
\title{Test whether sampling weights are needed}
\usage{
wgttest(model, weights, data = NULL, model_output = FALSE,
  test = NULL, digits = getOption("jtools-digits", default = 3))
}
\arguments{
\item{model}{The unweighted linear model (must be \code{lm},
\code{glm}, see details for other types) you want to check.}

\item{weights}{The name of the weights column in \code{model}'s data frame
or a vector of weights equal in length to the number of observations
included in \code{model}.}

\item{data}{The data frame with the data fed to the fitted model and the
weights}

\item{model_output}{Should a summary of the model with weights as predictor
be printed? Default is FALSE since the output can be very long for
complex models.}

\item{test}{Which type of test should be used in the ANOVA? The default,
\code{NULL}, chooses based on the model type ("F" for linear models).
This argument is passed to \code{anova}.}

\item{digits}{An integer specifying the number of digits past the decimal to
report in the output. Default is 3. You can change the default number of
digits for all jtools functions with
\code{options("jtools-digits" = digits)} where digits is the desired number.}
}
\description{
Use the DuMouchel-Duncan (1983) test to assess the need for sampling weights
in your linear regression analysis.
}
\details{
This is designed to be similar to the \code{wgttest} macro for Stata
(\url{http://fmwww.bc.edu/repec/bocode/w/wgttest.html}). This method,
advocated for by DuMouchel and Duncan (1983), is fairly straightforward. To
decide whether weights are needed, the weights are added to the linear model
as a predictor and interaction with each other predictor. Then, an omnibus
test of significance is performed to compare the weights-added model to the
original; if insignificant, weights are not significantly related to the
result and you can use the more efficient estimation from unweighted OLS.

It can be helpful to look at the created model using
\code{model_output = TRUE}
to see which variables might be the ones affected by inclusion of weights.

This test can support most GLMs in addition to LMs, a use validated by
Nordberg (1989). This, to my knowledge, is different from the Stata macro.
It does not work for mixed models (e.g., \code{lmer} or \code{lme}) though
it could plausibly be
implemented. However, there is no scholarly consensus how to properly
incorporate weights into mixed models. There are other types of models that
may work, but have not been tested. The function is designed to be
compatible with as many model types as possible, but the user should be
careful to make sure s/he understands whether this type of test is
appropriate for the model being considered. DuMouchel and Duncan (1983) were
only thinking about linear regression when the test was conceived.
Nordberg (1989) validated its use with generalized linear models, but to
this author's knowledge it has not been tested with other model types.
}
\examples{
# First, let's create some fake sampling weights
wts <- runif(50, 0, 5)
# Create model
fit <- lm(Income ~ Frost + Illiteracy + Murder,
          data = as.data.frame(state.x77))
# See if the weights change the model
wgttest(fit, weights = wts)

# With a GLM
wts <- runif(100, 0, 2)
x <- rnorm(100)
y <- rbinom(100, 1, .5)
fit <- glm(y ~ x, family = binomial)
wgttest(fit, wts)
## Can specify test manually
wgttest(fit, weights = wts, test = "Rao")

# Quasi family is treated differently than likelihood-based
## Dobson (1990) Page 93: Randomized Controlled Trial (plus some extra values):
counts <- c(18,17,15,20,10,20,25,13,12,18,17,15,20,10,20,25,13,12)
outcome <- gl(3,1,18)
treatment <- gl(3,6)
glm.D93 <- glm(counts ~ outcome + treatment, family = quasipoisson)
wts <- runif(18, 0, 3)
wgttest(glm.D93, weights = wts)

}
\references{
DuMouchel, W. H. & Duncan, D.J. (1983). Using sample survey weights in
multiple regression analyses of stratified samples. \emph{Journal of the
American Statistical Association}, \emph{78}. 535-543.

Nordberg, L. (1989). Generalized linear modeling of sample survey data.
\emph{Journal of Official Statistics; Stockholm}, \emph{5}, 223–239.

Winship, C. & Radbill, L. (1994). Sampling weights and regression
analysis. \emph{Sociological Methods and Research}, \emph{23}, 230-257.
}
\seealso{
Other survey tools: \code{\link{pf_sv_test}},
  \code{\link{svycor}}, \code{\link{svysd}},
  \code{\link{weights_tests}}
}
\concept{survey tools}
