% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Metric.R
\name{Metric}
\alias{Metric}
\title{Subclass the base \code{Metric} class}
\usage{
Metric(
  classname,
  initialize = NULL,
  update_state = NULL,
  result = NULL,
  ...,
  public = list(),
  private = list(),
  inherit = NULL,
  parent_env = parent.frame()
)
}
\arguments{
\item{classname}{String, the name of the custom class. (Conventionally, CamelCase).}

\item{initialize, update_state, result}{Recommended methods to implement. See description section.}

\item{..., public}{Additional methods or public members of the custom class.}

\item{private}{Named list of R objects (typically, functions) to include in
instance private environments. \code{private} methods will have all the same
symbols in scope as public methods (See section "Symbols in Scope"). Each
instance will have it's own \code{private} environment. Any objects
in \code{private} will be invisible from the Keras framework and the Python
runtime.}

\item{inherit}{What the custom class will subclass. By default, the base keras class.}

\item{parent_env}{The R environment that all class methods will have as a grandparent.}
}
\value{
A function that returns \code{Metric} instances, similar to the
builtin metric functions.
}
\description{
A \code{Metric} object encapsulates metric logic and state that can be used to
track model performance during training. It is what is returned by the family
of metric functions that start with prefix \verb{metric_*}, as well as what is
returned by custom metrics defined with \code{Metric()}.
}
\section{Examples}{
\subsection{Usage with \code{compile()}:}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{model |> compile(
  optimizer = 'sgd',
  loss = 'mse',
  metrics = c(metric_SOME_METRIC(), metric_SOME_OTHER_METRIC())
)
}\if{html}{\out{</div>}}
}

\subsection{Standalone usage:}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{m <- metric_SOME_METRIC()
for (e in seq(epochs)) \{
  for (i in seq(train_steps)) \{
    c(y_true, y_pred, sample_weight = NULL) \%<-\% ...
    m$update_state(y_true, y_pred, sample_weight)
  \}
  cat('Final epoch result: ', as.numeric(m$result()), "\\n")
  m$reset_state()
\}
}\if{html}{\out{</div>}}
}
}

\section{Full Examples}{
\subsection{Usage with \code{compile()}:}{

\if{html}{\out{<div class="sourceCode r">}}\preformatted{model <- keras_model_sequential()
model |>
  layer_dense(64, activation = "relu") |>
  layer_dense(64, activation = "relu") |>
  layer_dense(10, activation = "softmax")
model |>
  compile(optimizer = optimizer_rmsprop(0.01),
          loss = loss_categorical_crossentropy(),
          metrics = metric_categorical_accuracy())

data <- random_uniform(c(1000, 32))
labels <- random_uniform(c(1000, 10))

model |> fit(data, labels, verbose = 0)
}\if{html}{\out{</div>}}

To be implemented by subclasses (custom metrics):
\itemize{
\item \code{initialize()}: All state variables should be created in this method by
calling \code{self$add_variable()} like: \code{self$var <- self$add_variable(...)}.
\item \code{update_state()}: Updates all the state variables like:
\code{self$var$assign(...)}.
\item \code{result()}: Computes and returns a scalar value or a named list of scalar values
for the metric from the state variables.
}

Example subclass implementation:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{metric_binary_true_positives <- Metric(
  classname = "BinaryTruePositives",

  initialize = function(name = 'binary_true_positives', ...) \{
    super$initialize(name = name, ...)
    self$true_positives <-
      self$add_weight(shape = shape(),
                      initializer = 'zeros',
                      name = 'true_positives')
  \},

  update_state = function(y_true, y_pred, sample_weight = NULL) \{
    y_true <- op_cast(y_true, "bool")
    y_pred <- op_cast(y_pred, "bool")

    values <- y_true & y_pred # `&` calls op_logical_and()
    values <- op_cast(values, self$dtype)
    if (!is.null(sample_weight)) \{
      sample_weight <- op_cast(sample_weight, self$dtype)
      sample_weight <- op_broadcast_to(sample_weight, shape(values))
      values <- values * sample_weight # `*` calls op_multiply()
    \}
    self$true_positives$assign(self$true_positives + op_sum(values))
  \},

  result = function() \{
    self$true_positives
  \}
)
model <- keras_model_sequential(input_shape = 32) |> layer_dense(10)
model |> compile(loss = loss_binary_crossentropy(),
                 metrics = list(metric_binary_true_positives()))
model |> fit(data, labels, verbose = 0)
}\if{html}{\out{</div>}}
}
}

\section{Methods defined by the base \code{Metric} class:}{
\itemize{
\item

\if{html}{\out{<div class="sourceCode">}}\preformatted{__call__(...)
}\if{html}{\out{</div>}}

Calling a metric instance self like \code{m(...)} is equivalent to calling:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{function(...) \{
  m$update_state(...)
  m$result()
\}
}\if{html}{\out{</div>}}
\item

\if{html}{\out{<div class="sourceCode r">}}\preformatted{initialize(dtype=NULL, name=NULL)
}\if{html}{\out{</div>}}

Initialize self.

Args:
\itemize{
\item \code{name}: Optional name for the metric instance.
\item \code{dtype}: The dtype of the metric's computations. Defaults to \code{NULL}, which
means using \code{\link[=config_floatx]{config_floatx()}}. \code{config_floatx()} is a
\code{"float32"} unless set to different value
(via \code{config_set_floatx()}). If a \code{keras$DTypePolicy} is
provided, then the \code{compute_dtype} will be utilized.
}
\item

\if{html}{\out{<div class="sourceCode r">}}\preformatted{add_variable(shape, initializer, dtype=NULL, aggregation = 'sum', name=NULL)
}\if{html}{\out{</div>}}
\item

\if{html}{\out{<div class="sourceCode r">}}\preformatted{add_weight(shape=shape(), initializer=NULL, dtype=NULL, name=NULL)
}\if{html}{\out{</div>}}
\item

\if{html}{\out{<div class="sourceCode r">}}\preformatted{get_config()
}\if{html}{\out{</div>}}

Return the serializable config of the metric.
\item

\if{html}{\out{<div class="sourceCode r">}}\preformatted{reset_state()
}\if{html}{\out{</div>}}

Reset all of the metric state variables.

This function is called between epochs/steps,
when a metric is evaluated during training.
\item

\if{html}{\out{<div class="sourceCode r">}}\preformatted{result()
}\if{html}{\out{</div>}}

Compute the current metric value.

Returns:
A scalar tensor, or a named list of scalar tensors.
\item

\if{html}{\out{<div class="sourceCode r">}}\preformatted{stateless_result(metric_variables)
}\if{html}{\out{</div>}}
\item

\if{html}{\out{<div class="sourceCode r">}}\preformatted{stateless_reset_state()
}\if{html}{\out{</div>}}
\item

\if{html}{\out{<div class="sourceCode r">}}\preformatted{stateless_update_state(metric_variables, ...)
}\if{html}{\out{</div>}}
\item

\if{html}{\out{<div class="sourceCode r">}}\preformatted{update_state(...)
}\if{html}{\out{</div>}}

Accumulate statistics for the metric.
}
}

\section{Readonly properties}{
\itemize{
\item \code{dtype}
\item \code{variables}
}
}

\section{Symbols in scope}{
All R function custom methods (public and private) will have the
following symbols in scope:
\itemize{
\item \code{self}: The custom class instance.
\item \code{super}: The custom class superclass.
\item \code{private}: An R environment specific to the class instance.
Any objects assigned here are invisible to the Keras framework.
\item \verb{__class__} and \code{as.symbol(classname)}: the custom class type object.
}
}

\seealso{
Other metrics: \cr
\code{\link{custom_metric}()} \cr
\code{\link{metric_auc}()} \cr
\code{\link{metric_binary_accuracy}()} \cr
\code{\link{metric_binary_crossentropy}()} \cr
\code{\link{metric_binary_focal_crossentropy}()} \cr
\code{\link{metric_binary_iou}()} \cr
\code{\link{metric_categorical_accuracy}()} \cr
\code{\link{metric_categorical_crossentropy}()} \cr
\code{\link{metric_categorical_focal_crossentropy}()} \cr
\code{\link{metric_categorical_hinge}()} \cr
\code{\link{metric_concordance_correlation}()} \cr
\code{\link{metric_cosine_similarity}()} \cr
\code{\link{metric_f1_score}()} \cr
\code{\link{metric_false_negatives}()} \cr
\code{\link{metric_false_positives}()} \cr
\code{\link{metric_fbeta_score}()} \cr
\code{\link{metric_hinge}()} \cr
\code{\link{metric_huber}()} \cr
\code{\link{metric_iou}()} \cr
\code{\link{metric_kl_divergence}()} \cr
\code{\link{metric_log_cosh}()} \cr
\code{\link{metric_log_cosh_error}()} \cr
\code{\link{metric_mean}()} \cr
\code{\link{metric_mean_absolute_error}()} \cr
\code{\link{metric_mean_absolute_percentage_error}()} \cr
\code{\link{metric_mean_iou}()} \cr
\code{\link{metric_mean_squared_error}()} \cr
\code{\link{metric_mean_squared_logarithmic_error}()} \cr
\code{\link{metric_mean_wrapper}()} \cr
\code{\link{metric_one_hot_iou}()} \cr
\code{\link{metric_one_hot_mean_iou}()} \cr
\code{\link{metric_pearson_correlation}()} \cr
\code{\link{metric_poisson}()} \cr
\code{\link{metric_precision}()} \cr
\code{\link{metric_precision_at_recall}()} \cr
\code{\link{metric_r2_score}()} \cr
\code{\link{metric_recall}()} \cr
\code{\link{metric_recall_at_precision}()} \cr
\code{\link{metric_root_mean_squared_error}()} \cr
\code{\link{metric_sensitivity_at_specificity}()} \cr
\code{\link{metric_sparse_categorical_accuracy}()} \cr
\code{\link{metric_sparse_categorical_crossentropy}()} \cr
\code{\link{metric_sparse_top_k_categorical_accuracy}()} \cr
\code{\link{metric_specificity_at_sensitivity}()} \cr
\code{\link{metric_squared_hinge}()} \cr
\code{\link{metric_sum}()} \cr
\code{\link{metric_top_k_categorical_accuracy}()} \cr
\code{\link{metric_true_negatives}()} \cr
\code{\link{metric_true_positives}()} \cr
}
\concept{metrics}
