#' @title TensorBoard
#'
#' @description Enable visualizations for TensorBoard.
#'
#' @details TensorBoard is a visualization tool provided with TensorFlow. This callback logs events for TensorBoard, including:
#' * Metrics summary plots
#' * Training graph visualization
#' * Activation histograms
#' * Sampled profiling If you have installed TensorFlow with pip, you should be able
#' to launch TensorBoard from the command line: ```sh
#' tensorboard --logdir=path_to_your_logs
#' ``` You can find more information about TensorBoard
#' [here](https://www.tensorflow.org/get_started/summaries_and_tensorboard).
#'
#' @param log_dir the path of the directory where to save the log files to be parsed by TensorBoard.
#' @param histogram_freq frequency (in epochs) at which to compute activation and weight histograms 
#' for the layers of the model. If set to 0, histograms won't be computed. Validation data (or split) 
#' must be specified for histogram visualizations.
#' @param write_graph whether to visualize the graph in TensorBoard. The log file can become quite 
#' large when write_graph is set to TRUE.
#' @param write_images whether to write model weights to visualize as image in TensorBoard.
#' @param update_freq `'batch'` or `'epoch'` or integer. When using `'batch'`, writes the losses and 
#' metrics to TensorBoard after each batch. The same applies for `'epoch'`. If using an integer, let's 
#' say `1000`, the callback will write the metrics and losses to TensorBoard every 1000 samples. 
#' Note that writing too frequently to TensorBoard can slow down your training.
#' @param profile_batch Profile the batch to sample compute characteristics. By default, it will 
#' profile the second batch. Set profile_batch=0 to disable profiling. Must run in 
#' TensorFlow eager mode.
#' @param embeddings_freq frequency (in epochs) at which embedding layers will be visualized. 
#' If set to 0, embeddings won't be visualized.
#' @param embeddings_metadata a dictionary which maps layer name to a file name in which metadata 
#' for this embedding layer is saved. 
#' See the [details]( https://www.tensorflow.org/how_tos/embedding_viz/#metadata_optional) about 
#' metadata files format. In case if the same metadata file is used for all embedding layers, 
#' string can be passed.
#'
#' @section Raises:
#' ValueError: If histogram_freq is set and no validation data is provided.
#' @return None
#' @export
TensorBoard <- function(log_dir = "logs", histogram_freq = 0, write_graph = TRUE, write_images = FALSE, 
                        update_freq = "epoch", profile_batch = 2, embeddings_freq = 0, embeddings_metadata = NULL) {
  
  args <- list(
    log_dir = log_dir,
    histogram_freq = as.integer(histogram_freq),
    write_graph = write_graph,
    write_images = write_images,
    update_freq = update_freq,
    profile_batch = as.integer(profile_batch),
    embeddings_freq = as.integer(embeddings_freq),
    embeddings_metadata = embeddings_metadata
  )
  
  if(is.null(embeddings_metadata))
    args$embeddings_metadata <- NULL
  
  do.call(tf$keras$callbacks$TensorBoard, args)
  
}






