# export `lavaan' lav model description to third-party software
# 

lavExport <- function(object, target="lavaan", prefix="sem", 
                      dir.name="lavExport", export=TRUE) {

    stopifnot(class(object) == "lavaan")
    target <- tolower(target)

    ngroups <- object@Data@ngroups
    if(ngroups > 1L) {
        group.label2 <- paste(".", object@Data@group.label, sep="")
    } else {
        group.label2 <- ""
    }
    data.file <- paste(prefix, group.label2, ".", target, ".raw", sep="")

    # 2. create syntax file
    if(target == "lavaan") {
        header <- ""
        syntax <- lav2lavaan(object)
        footer <- ""
        out <- paste(header, syntax, footer, sep="")
    } else if(target == "mplus") {
        header <- mplusHeader(data.file=data.file, 
                              group.label=object@Data@group.label,
                              ov.names=vnames(object@ParTable, "ov"),
                              ov.ord.names=vnames(object@ParTable, "ov.ord"),
                              estimator=mplusEstimator(object))
        syntax <- lav2mplus(object, group.label=object@Data@group.label)
        footer <- paste("OUTPUT:\n  sampstat standardized;\n")
        out <- paste(header, syntax, footer, sep="")
    } else if(target == "lisrel") {
        syntax <- lav2lisrel(object)
    } else if(target == "eqs") {
        syntax <- lav2eqs(object)
    } else if(target == "sem") {
        syntax <- lav2sem(object)
    } else if(target == "openmx") {
        syntax <- lav2openmx(object)
    } else {
        stop("lavaan ERROR: target", target, "has not been implemented yet")
    }
    
    # export to file?
    if(export) {
        dir.create(path=dir.name)
        input.file <- paste(dir.name, "/", prefix, ".", target, ".in", sep="")
        cat(out, file=input.file, sep="")

        # write data (if available)
        if(identical(object@Data@data.type, "full")) {
            for(g in 1:ngroups) {
                write.table(object@Data@X[[g]],
                            file=paste(dir.name, "/", data.file[g], sep=""),
                            na="-999999",
                            col.names=FALSE, row.names=FALSE, quote=FALSE)
            }
        } else {
            warning("raw data not available")
        }
        return(invisible(out))
    } else {
        # just return the syntax file for inspection
        class(out) <- c("lavaan.character", "character")
    }

    out
}


lav2check <- function(lav) {
    if("lavaan" %in% class(lav)) {
        lav <- lav@ParTable
    } else if(is.list(lav)) {
        # nothing to do
    } else {
        stop("lavaan ERROR: lav must be of class `lavaan' or a parTable")
    }

    # check syntax
    if(is.null(lav$ustart)) lav$ustart <- lav$est

    # check if free is missing
    if(is.null(lav$free)) lav$free <- rep(0L, length(lav$ustart))

    # check if label is missing
    if(is.null(lav$label)) lav$label <- rep("", length(lav$ustart))

    # check if group is missing
    if(is.null(lav$group)) lav$group <- rep(1L, length(lav$ustart))

    # if eq.id not all zero, create labels instead
    if(!is.null(lav$eq.id) && !all(lav$eq.id == 0L)) {
        lav$label <- paste("p",as.character(lav$eq.id), sep="")
        lav$label[lav$label == "p0"] <- ""
    }
 
    lav
}

## FIXME: this is completely UNFINISHED (just  used to quickly get something)
lav2lavaan <- lav2lav <- function(lav) {
    lav <- lav2check(lav)
    header <- "# this model syntax is autogenerated by lavExport\n"
    footer <- "\n"

    # intercepts
    int.idx <- which(lav$op == "~1")
    lav$op[int.idx] <- "~"
    lav$rhs[int.idx] <- "1"

    # spacing around operator
    lav$op <- paste(" ",lav$op, " ", sep="")

    lav2 <- ifelse(lav$free != 0L,
                   ifelse(lav$label == "",
                          paste(lav$lhs, lav$op, lav$rhs, sep=""),
                          paste(lav$lhs, lav$op, lav$label, "*", lav$rhs, 
                                sep="")
                   ),
                   ifelse(lav$label == "",
                          paste(lav$lhs, lav$op, lav$ustart, "*", lav$rhs, 
                                sep=""),
                          paste(lav$lhs, lav$op, lav$ustart, "*", lav$rhs, 
                                "+", lav$label, "*", lav$rhs, sep="")
                   )
            )

    body <- paste(lav2, collapse="\n")
    out <- paste(header, body, footer, sep="")
    class(out) <- c("lavaan.character", "character")
    out
}

lav2mplus <- function(lav, group.label=NULL) {
    lav <- lav2check(lav)
    header <- "  ! this model syntax is autogenerated by lavExport\n"
    footer <- "\n"

    lav <- as.data.frame(lav, stringsAsFactors=FALSE)
    ngroups <- max(lav$group)

    lav_one_group <- function(lav) {
 
        # extract contraints (:=, <, >, ==)
        con.idx <- which(lav$op %in% c(":=", "<",">","=="))
        if(length(con.idx) > 0L) {
            warnings("lavaan WARNING: :=, <, >, == operators not converted")
            lav.con <- lav[con.idx,]
            lav <- lav[-con.idx,]
        }

        # end of line
        lav$eol <- rep(";", length(lav$lhs))
        lav$ustart <- ifelse(is.na(lav$ustart), "", lav$ustart)
        lav$rhs2 <- ifelse(lav$free == 0L, 
                           paste("@",lav$ustart,sep=""),
                           paste("*",lav$ustart,sep=""))
        lav$label <- ifelse(lav$label == "", lav$label,
                            paste(" (",lav$label,")",sep=""))

        # remove variances for ordered variables
        ov.names.ord <- vnames(lav, type="ov.ord")
        ord.idx <- which(lav$lhs %in% ov.names.ord &
                         lav$op == "~~" &
                         lav$free == 0L &
                         lav$lhs == lav$rhs)
        lav$lhs[ord.idx] <- paste("! ", lav$lhs[ord.idx], sep="")
        lav$op[ord.idx] <- ""
        lav$rhs[ord.idx] <- ""

        # variances
        var.idx <- which(lav$op == "~~" & lav$rhs == lav$lhs)
        lav$op[var.idx] <- ""
        lav$rhs[var.idx] <- ""

        # intercepts
        int.idx <- which(lav$op == "~1")
        lav$op[int.idx] <- ""
        lav$rhs2[int.idx] <- paste(lav$rhs2[int.idx],"]",sep="")
        lav$lhs[int.idx] <- paste("[", lav$lhs[int.idx],sep="")

        # thresholds
        th.idx <- which(lav$op == "|")
        lav$op[th.idx] <- "$"
        lav$rhs[th.idx] <- gsub("t", "", x=lav$rhs[th.idx])
        lav$rhs2[th.idx] <- paste(lav$rhs2[th.idx],"]",sep="")
        lav$lhs[th.idx] <- paste("[", lav$lhs[th.idx],sep="")

        # replace binary operators
        lav$op <- ifelse(lav$op == "=~", " BY ", lav$op)
        lav$op <- ifelse(lav$op == "~", " ON ", lav$op)
        lav$op <- ifelse(lav$op == "~~", " WITH ", lav$op)


        lav2 <- paste(lav$lhs, lav$op, lav$rhs, lav$rhs2,
                      lav$label, lav$eol, sep="")
                      
        body <- paste(" ", lav2, collapse="\n")

        body
    }

    if(ngroups == 1L) {
        body <- lav_one_group(lav)
    } else {
        # group 1
        body <- lav_one_group(lav[lav$group == 1,])

        if(is.null(group.label)) {
            group.label <- paste(1:ngroups)
        }

        for(g in 2:ngroups) {
            body <- paste(body,
                          paste("\nMODEL ", group.label[g], ":\n", sep=""),
                          lav_one_group(lav[lav$group == g,]),
                          sep="")
        }
    }
  
    out <- paste(header, body, footer, sep="")
    class(out) <- c("lavaan.character", "character")
    out
}

lav2lisrel <- function(lav) {
    lav <- lav2check(lav)
    stop("this function needs revision")
}

lav2eqs <- function(lav) {
    lav <- lav2check(lav)
    stop("this function needs revision")
}

lav2sem <- function(lav) {
    lav <- lav2check(lav)
    stop("this function needs revision")
}

lav2openmx <- function(lav) {
    lav <- lav2check(lav)
    stop("this function needs revision")
}

mplusEstimator <- function(object) {

    estimator <- object@Options$estimator
    if(estimator == "DWLS") {
        estimator <- "WLS"
    }

    if(estimator == "ML") {
        if(object@Options$test == "yuan.bentler") {
            estimator <- "MLR"
        } else if(object@Options$test == "satorra.bentler") {
            estimator <- "MLM"
        } else if(object@Options$test == "scaled.shifted") {
            estimator <- "MLMV"
        } else if(object@Options$se == "first.order") {
            estimator <- "MLF"
        }
    } else if(estimator %in% c("ULS","WLS")) {
        if(object@Options$test == "satorra.bentler") {
            estimator <- paste(estimator, "M", sep="")
        } else if(object@Options$test == "scaled.shifted") {
            estimator <- paste(estimator, "MV", sep="")
        }
    }

    estimator
}

mplusHeader <- function(data.file=NULL, group.label="", ov.names="",
                        ov.ord.names="", estimator="ML") {

    ### FIXME!!
    ### this is old code from lavaan 0.3-1
    ### surely, this can be done better...

    # TITLE command
    c.TITLE <- "TITLE:\n"
    c.TITLE <- paste(c.TITLE, 
                     "  [This syntax is autogenerated by lavExport]\n")

    # DATA command
    c.DATA <- "DATA:\n"
    ngroups <- length(data.file)    
    if(ngroups == 1L) {
        c.DATA  <- paste(c.DATA,
                         "  file is ", data.file, ";\n", sep="")
    } else {
        for(g in 1:ngroups) {
            c.DATA <- paste(c.DATA,
                            "  file (", group.label[g] ,") is ",
                            data.file[g], ";\n", sep="")
        }
    }
    
    # VARIABLE command
    c.VARIABLE <- "VARIABLE:\n"
    c.VARIABLE <- paste(c.VARIABLE, "  names are", sep="")
    nvar <- length(ov.names); tmp <- 0
    for(i in 1:nvar) {
        if(tmp%%6 == 0) { c.VARIABLE <- paste(c.VARIABLE,"\n    ", sep="") }
        c.VARIABLE <- paste(c.VARIABLE, ov.names[i], sep=" ")
        tmp <- tmp+1
    }
    c.VARIABLE <- paste(c.VARIABLE,
                        ";\n  missing are all (-999999);\n",sep="")
    # categorical?
    if(length(ov.ord.names)) {
        c.VARIABLE <- paste(c.VARIABLE, "  categorical are", sep="")
        nvar <- length(ov.ord.names); tmp <- 0
        for(i in 1:nvar) {
            if(tmp%%6 == 0) { c.VARIABLE <- paste(c.VARIABLE,"\n    ", sep="") }
            c.VARIABLE <- paste(c.VARIABLE, ov.ord.names[i])
            tmp <- tmp+1
        }
        c.VARIABLE <- paste(c.VARIABLE,";\n",sep="")
    }

    # ANALYSIS command
    c.ANALYSIS <- paste("ANALYSIS:\n  type = general;\n", sep="")
    c.ANALYSIS <- paste(c.ANALYSIS, "  estimator = ", toupper(estimator),
                        ";\n", sep="")

    # MODEL command
    c.MODEL <- paste("MODEL:\n")

    # assemble pre-model header
    out <- paste(c.TITLE, c.DATA, c.VARIABLE, c.ANALYSIS, c.MODEL, sep="")

    out
}

