% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grid_metrics.r
\name{grid_metrics}
\alias{grid_metrics}
\title{Area-Based Approach}
\usage{
grid_metrics(las, func, res = 20, start = c(0, 0), filter = NULL)
}
\arguments{
\item{las}{An object of class \link[lidR:LAS-class]{LAS} or \link[lidR:LAScatalog-class]{LAScatalog}.}

\item{func}{formula. An expression to be applied to each cell (see section "Parameter func").}

\item{res}{numeric. The resolution of the output \code{Raster}. Can optionally be a \code{RasterLayer}.
In that case the \code{RasterLayer} is used as the layout.}

\item{start}{vector of x and y coordinates for the reference raster. Default is (0,0) meaning that the
grid aligns on (0,0).}

\item{filter}{formula of logical predicates. Enables the function to run only on points of interest
in an optimized way. See examples.}
}
\value{
A \code{RasterLayer} or a \code{RasterBrick} containing a numeric value in each cell. If the
\code{RasterLayer}s are written on disk when running the function with a \code{LAScatalog}, a
virtual raster mosaic is returned (see \link[gdalUtils:gdalbuildvrt]{gdalbuildvrt})
}
\description{
Computes a series of user-defined descriptive statistics for a LiDAR dataset within
each pixel of a raster (area-based approach). The grid cell coordinates are pre-determined for a
given resolution, so the algorithm will always provide the same coordinates independently of the
dataset. When start = (0,0) and res = 20 grid_metrics will produce the following cell centers:
(10,10), (10,30), (30,10) etc. aligning the corner of a cell on (0,0). When start = (-10, -10) and
res = 20 grid_metrics will produce the following cell centers: (0,0), (0,20), (20,0) etc. aligning
the corner of a cell on (-10, -10).
}
\section{Parameter \code{func}}{

The function to be applied to each cell is a classical function (see examples) that
returns a labeled list of metrics. For example, the following function \code{f} is correctly formed.
\preformatted{
f = function(x) {list(mean = mean(x), max = max(x))}
}
And could be applied either on the \code{Z} coordinates or on the intensities. These two
statements are valid:
\preformatted{
grid_metrics(las, ~f(Z), res = 20)
grid_metrics(las, ~f(Intensity), res = 20)
}
The following existing functions allow the user to
compute some predefined metrics:
\itemize{
\item{\link[lidR:stdmetrics]{stdmetrics}}
\item{\link[lidR:entropy]{entropy}}
\item{\link[lidR:VCI]{VCI}}
\item{\link[lidR:LAD]{LAD}}
}
But usually users must write their own functions to create metrics. \code{grid_metrics} will
dispatch the point cloud in the user's function.
}

\section{Working with a \code{LAScatalog}}{

This section appears in each function that supports a LAScatalog as input.\cr

In \code{lidR} when the input of a function is a \link[lidR:LAScatalog-class]{LAScatalog} the
function uses the LAScatalog processing engine. The user can modify the engine options using
the \link[lidR:catalog_options_tools]{available options}. A careful reading of the
\link[lidR:LAScatalog-class]{engine documentation} is recommended before processing \code{LAScatalogs}. Each
\code{lidR} function should come with a section that documents the supported engine options.\cr

The \code{LAScatalog} engine supports \code{.lax} files that \emph{significantly} improve the computation
speed of spatial queries using a spatial index. Users should really take advantage a \code{.lax} files,
but this is not mandatory.\cr
}

\section{Supported processing options}{

Supported processing options for a \code{LAScatalog} in \code{grid_*} functions (in bold). For
more details see the \link[lidR:LAScatalog-class]{LAScatalog engine documentation}:
\itemize{
\item \strong{tiling_size}: How much data is loaded at once. The tiling size may be slightly modified
internally to ensure a strict continuous wall-to-wall output even when tiling size is equal to 0 (processing
by file).
\item buffer: This function guarantees a strict continuous wall-to-wall output. The \code{buffer} option
is not considered.
\item \strong{alignment}: Align the processed clusters. The alignment may be slightly modified.
\item \strong{chunk size}: How much data is loaded at once. The chunk size may be slightly modified
internally to ensure a strict continuous wall-to-wall output even when chunk size is equal to 0 (processing
by file).
\item chunk buffer: This function guarantees a strict continuous wall-to-wall output. The \code{buffer}
option is not considered.
\item \strong{chunk alignment}: Align the processed chunks. The alignment may be slightly modified
internally to ensure a strict continuous wall-to-wall output.
\item \strong{progress}: Displays a progress estimate.
\item \strong{output_files}: Return the output in R or write each cluster's output in a file. Supported
templates are \code{XLEFT}, \code{XRIGHT}, \code{YBOTTOM}, \code{YTOP}, \code{XCENTER}, \code{YCENTER}
\code{ID} and, if chunk size is equal to 0 (processing by file), \code{ORIGINALFILENAME}.
\item \strong{\emph{select}}: The \code{grid_*} functions usually 'know' what should be loaded
and this option is not considered. In \link{grid_metrics} this option is respected.
\item \strong{filter}: Read only the points of interest.
}
}

\examples{
LASfile <- system.file("extdata", "Megaplot.laz", package="lidR")
las = readLAS(LASfile)
col = height.colors(50)

# === Using all points ===

# Canopy surface model with 4 m^2 cells
metrics = grid_metrics(las, ~max(Z), 2)
plot(metrics, col = col)

# Mean height with 400 m^2 cells
metrics = grid_metrics(las, ~mean(Z), 20)
plot(metrics, col = col)

# Define your own new metrics
myMetrics = function(z, i) {
  metrics = list(
     zwimean = sum(z*i)/sum(i), # Mean elevation weighted by intensities
     zimean  = mean(z*i),       # Mean products of z by intensity
     zsqmean = sqrt(mean(z^2))) # Quadratic mean

   return(metrics)
}

metrics = grid_metrics(las, ~myMetrics(Z, Intensity))

plot(metrics, col = col)
plot(metrics, "zwimean", col = col)
plot(metrics, "zimean", col = col)

# === With point filters ===

# Compute using only some points: basic
first = lasfilter(las, ReturnNumber == 1)
metrics = grid_metrics(first, ~mean(Z), 20)

# Compute using only some points: optimized
# faster and uses less memory. No intermediate object
metrics = grid_metrics(las, ~mean(Z), 20, filter = ~ReturnNumber == 1)

# Compute using only some points: best
# ~50\% faster and uses ~10x less memory
las = readLAS(LASfile, filter = "-keep_first")
metrics = grid_metrics(las, ~mean(Z), 20)
}
