% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_model.R
\name{getModelFit}
\alias{getModelFit}
\title{Fit linear models for rhythmicity in one or more conditions}
\usage{
getModelFit(
  y,
  metadata,
  period = 24,
  nKnots = 4L,
  timeColname = "time",
  condColname = NULL,
  covarColnames = NULL,
  sampleColname = "sample",
  nShifts = 3L,
  method = c("trend", "voom", "deseq2"),
  lmFitArgs = list(),
  eBayesArgs = if (method == "trend") list(trend = TRUE) else list(),
  DESeqArgs = list(),
  keepLmFits = FALSE
)
}
\arguments{
\item{y}{Matrix-like object of measurements, with rows corresponding to
features and columns to samples.}

\item{metadata}{data.frame containing experimental design information for
each sample. Rows of \code{metadata} must correspond to columns of \code{y}. Row
names are ignored.}

\item{period}{Number specifying the period for the time variable, in the same
units as the values in the \code{timeColname} column.}

\item{nKnots}{Number of internal knots for the periodic spline for the time
variable. Use \code{NULL} to fit a cosinor-based model instead of a spline-based
model.}

\item{timeColname}{String indicating the column in \code{metadata} containing the
time at which each sample was acquired.}

\item{condColname}{String indicating the column in \code{metadata} containing the
condition in which each sample was acquired. \code{NULL} indicates all samples
came from the same condition. If not \code{NULL}, the model will include main
effects and interactions with the terms for time.}

\item{covarColnames}{Character vector indicating the columns in \code{metadata}
containing covariates to include in the model. \code{NULL} indicates no
covariates.}

\item{sampleColname}{String indicating the column in \code{metadata} containing
the name of each sample, which must correspond to the column names of \code{y}.}

\item{nShifts}{Number of shifted models to fit. Only used for periodic
splines, not for cosinor. Do not change from the default unless you know
what you're doing.}

\item{method}{String indicating method to estimate model coefficients. For
microarray data, use 'trend'. For RNA-seq count data, use 'voom' or
'deseq2'.}

\item{lmFitArgs}{List of arguments passed to \code{\link[limma:lmFit]{limma::lmFit()}}.}

\item{eBayesArgs}{List of arguments passed to \code{\link[limma:ebayes]{limma::eBayes()}}.}

\item{DESeqArgs}{List of arguments passed to \code{\link[DESeq2:DESeq]{DESeq2::DESeq()}}.}

\item{keepLmFits}{Logical indicating whether to keep the complete fit objects
from \code{limma} or \code{DESeq2}. Not needed by any functions in \code{limorhyde2}.}
}
\value{
A \code{limorhyde2} object with elements:
\itemize{
\item \code{metadata}: As supplied above, converted to a \code{data.table}.
\item \code{timeColname}: As supplied above.
\item \code{condColname}: As supplied above.
\item \code{covarColnames}: As supplied above.
\item \code{coefficients}: Matrix with rows corresponding to features and columns to
model terms, including all shifted models.
\item \code{shifts}: Numeric vector indicating amount by which timepoints were shifted
for each shifted model.
\item \code{period}: As supplied above.
\item \code{conds}: If \code{condColname} is not \code{NULL}, a vector of unique values of
the condition variable.
\item \code{nKnots}: Number of knots, where 2 indicates a cosinor-based model.
\item \code{nConds}: Number of conditions.
\item \code{nCovs}: Number of covariates.
\item \code{lmFits}: If \code{keepLmFits} is \code{TRUE}, a list of objects from \code{limma} or
\code{DESeq2}, with length equal to length of the \code{shifts} element.
}
}
\description{
This is the first step in an analysis using \code{limorhyde2}, the second is to
moderate the fits using \code{\link[=getPosteriorFit]{getPosteriorFit()}}.
}
\examples{
library('data.table')

# rhythmicity in one condition
y = GSE54650$y
metadata = GSE54650$metadata

fit = getModelFit(y, metadata)
fit = getPosteriorFit(fit)
rhyStats = getRhythmStats(fit, features = c('13170', '13869'))

# rhythmicity and differential rhythmicity in multiple conditions
y = GSE34018$y
metadata = GSE34018$metadata

fit = getModelFit(y, metadata, nKnots = 3L, condColname = 'cond')
fit = getPosteriorFit(fit)
rhyStats = getRhythmStats(fit, features = c('13170', '12686'))
diffRhyStats = getDiffRhythmStats(fit, rhyStats)
}
\seealso{
\code{\link[=getPosteriorFit]{getPosteriorFit()}}
}
