#' Convenience function for computing relative efficiencies
#'
#' \code{relative_eff} computes the the MCMC effective sample size divided by
#' the total sample size.
#'
#' @export
#' @param x A vector, matrix, 3-D array, or function. See the \strong{Methods
#'   (by class)} section below for details on the shape of \code{x}. For use
#'   with the \code{loo} function, the values in \code{x} (or generated by
#'   \code{x} if \code{x} is a function) should be \strong{likelihood} values
#'   (i.e., \code{exp(log_lik)}, not on the log scale). For generic \code{use}
#'   with \code{\link{psis}}, the values in \code{x} should be the reciprocal of
#'   the importance ratios (i.e., \code{exp(-log_ratios)}).
#' @param chain_id A vector of length \code{NROW(x)} containing MCMC chain
#'   indexes for each each row of \code{x} (if a matrix) or each value in
#'   \code{x} (if a vector). No \code{chain_id} is needed if \code{x} is a 3-D
#'   array. If there are \code{C} chains then valid chain indexes are values
#'   in \code{1:C}.
#' @param cores The number of cores to use for parallelization.
#' @return A vector of relative effective sample sizes.
#'
#' @examples
#' LLarr <- example_loglik_array()
#' LLmat <- example_loglik_matrix()
#' dim(LLarr)
#' dim(LLmat)
#'
#' rel_n_eff_1 <- relative_eff(exp(LLarr))
#' rel_n_eff_2 <- relative_eff(exp(LLmat), chain_id = rep(1:2, each = 500))
#' all.equal(rel_n_eff_1, rel_n_eff_2)
#'
relative_eff <- function(x, ...) {
  UseMethod("relative_eff")
}

#' @export
#' @templateVar fn relative_eff
#' @template vector
#'
relative_eff.default <- function(x, chain_id, ...) {
  dim(x) <- c(length(x), 1)
  class(x) <- "matrix"
  relative_eff.matrix(x, chain_id)
}

#' @export
#' @templateVar fn relative_eff
#' @template matrix
#'
relative_eff.matrix <- function(x, chain_id, ..., cores = getOption("mc.cores", 1)) {
  x <- llmatrix_to_array(x, chain_id)
  relative_eff.array(x, cores = cores)
}

#' @export
#' @templateVar fn relative_eff
#' @template array
#'
relative_eff.array <- function(x, ..., cores = getOption("mc.cores", 1)) {
  stopifnot(length(dim(x)) == 3)
  S <- prod(dim(x)[1:2]) # posterior sample size = iter * chains

  if (cores == 1) {
    n_eff_vec <- apply(x, 3, mcmc_n_eff)
  } else {
    if (.Platform$OS.type != "windows") {
      n_eff_list <-
        parallel::mclapply(
          mc.cores = cores,
          X = seq_len(dim(x)[3]),
          FUN = function(i) mcmc_n_eff(x[, , i])
        )
    } else {
      cl <- parallel::makePSOCKcluster(cores)
      on.exit(parallel::stopCluster(cl))
      n_eff_list <-
        parallel::parLapply(
          cl = cl,
          X = seq_len(dim(x)[3]),
          fun = function(i) mcmc_n_eff(x[, , i])
        )
    }
    n_eff_vec <- unlist(n_eff_list, use.names = FALSE)
  }

  return(n_eff_vec / S)
}

#' @export
#' @templateVar fn relative_eff
#' @template function
#' @param data,draws,... Same as for the \code{\link{loo}} function method.
#'
relative_eff.function <-
  function(x,
           chain_id,
           ...,
           cores = getOption("mc.cores", 1),
           data = NULL,
           draws = NULL) {

    f_i <- validate_llfun(x) # not really an llfun, should return exp(ll) or exp(-ll)
    N <- dim(data)[1]

    if (cores == 1) {
      n_eff_list <-
        lapply(
          X = seq_len(N),
          FUN = function(i) {
            val_i <- f_i(data_i = data[i, , drop = FALSE], draws = draws, ...)
            relative_eff.default(as.vector(val_i), chain_id = chain_id)
          }
        )
    } else {
      if (.Platform$OS.type != "windows") {
        n_eff_list <-
          parallel::mclapply(
            X = seq_len(N),
            FUN = function(i) {
              val_i <- f_i(data_i = data[i, , drop = FALSE], draws = draws, ...)
              relative_eff.default(as.vector(val_i), chain_id = chain_id)
            },
            mc.cores = cores
          )
      } else {
        cl <- parallel::makePSOCKcluster(cores)
        on.exit(parallel::stopCluster(cl))
        n_eff_list <-
          parallel::parLapply(
            cl = cl,
            X = seq_len(N),
            fun = function(i) {
              val_i <- f_i(data_i = data[i, , drop = FALSE], draws = draws, ...)
              relative_eff.default(as.vector(val_i), chain_id = chain_id)
            }
          )
      }
    }

    n_eff_vec <- unlist(n_eff_list, use.names = FALSE)
    return(n_eff_vec)
  }


# internal ----------------------------------------------------------------


#' Effective sample size for PSIS
#'
#' @noRd
#' @param w A vector or matrix (one column per observation) of normalized Pareto
#'   smoothed weights (not log weights).
#' @param r_eff Relative effective sample size of \code{exp(log_lik)} or
#'   \code{exp(-log_ratios)}. \cope{r_eff} should be a scalar if \code{w} is a
#'   vector and a vector of length \code{ncol(w)} if \code{w} is a matrix.
#' @return A scalar if \code{w} is a vector. A vector of length \code{ncol(w)}
#'   if \code{w} is matrix.
#'
psis_n_eff <- function(w, ...) {
  UseMethod("psis_n_eff")
}
psis_n_eff.default <- function(w, r_eff = NULL, ...) {
  ss <- sum(w^2)
  if (is.null(r_eff)) {
    warning("PSIS n_eff not adjusted based on MCMC n_eff.", call. = FALSE)
    return(1 / ss)
  }
  stopifnot(length(r_eff) == 1)
  1 / ss * r_eff
}
psis_n_eff.matrix <- function(w, r_eff = NULL, ...) {
  ss <- colSums(w^2)
  if (is.null(r_eff)) {
    warning("PSIS n_eff not adjusted based on MCMC n_eff.", call. = FALSE)
    return(1 / ss)
  }
  if (length(r_eff) != length(ss))
    stop("r_eff must have length ncol(w).", call. = FALSE)
  1 / ss * r_eff
}

#' MCMC effective sample size calculation
#'
#' @noRd
#' @param x An iterations by chains matrix of draws for a single parameter. In
#'   the case of the loo package, this will be the _exponentiated_ log-likelihood
#'   values for the ith observation.
#' @return MCMC effective sample size based on rstan's calculation.
#'
mcmc_n_eff <- function(x) {
  if (!is.matrix(x)) {
    x <- as.matrix(x)
  }
  n_chain <- ncol(x)
  n_iter <- nrow(x)

  acov <- apply(x, 2, .acov, lag_max = n_iter - 1)
  chain_means <- colMeans(x)
  mean_var <- mean(acov[1, ]) * n_iter / (n_iter - 1)
  var_plus <- mean_var * (n_iter - 1) / n_iter
  if (n_chain > 1) {
    var_plus <- var_plus + var(chain_means)
  }

  rho_hat_sum <- 0
  for (t in 2:nrow(acov)) {
    rho_hat <- 1 - (mean_var - mean(acov[t, ])) / var_plus
    if (is.nan(rho_hat))
      rho_hat <- 0
    if (rho_hat < 0)
      break
    rho_hat_sum <- rho_hat_sum + rho_hat
  }

  n_eff <- n_chain * n_iter
  if (rho_hat_sum > 0)
    n_eff <- n_eff / (1 + 2 * rho_hat_sum)

  return(n_eff)
}


# wrapper around stats::acf that returns only the info we need in mcmc_n_eff
# @param x,lag_max Vector and integer passed to stats::acf
.acov <- function(x, lag_max) {
  cov <-
    stats::acf(x,
               lag.max = lag_max,
               plot = FALSE,
               type = "covariance")

  return(cov$acf[, , 1])
}
