% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/l_copyStates.R
\name{l_copyStates}
\alias{l_copyStates}
\title{A generic function to transfer the values of the states of one `loon` structure to another.}
\usage{
l_copyStates(
  source,
  target,
  states = NULL,
  exclude = NULL,
  excludeBasicStates = TRUE,
  returnNames = FALSE
)
}
\arguments{
\item{source}{the `loon` object providing the values of the states.}

\item{target}{the `loon` object whose states are assigned the
values of the `sources` states of the same name.}

\item{states}{a character vector of the states to be copied. If `NULL` (the default),
then all states in common (excluding those identified by exclusion parameters)
are copied from the `source` to the `target`.}

\item{exclude}{a character vector naming those common states to be excluded
from copying.  Default is NULL.}

\item{excludeBasicStates}{a logical indicating whether certain basic states
are to be excluded from the copy (if `TRUE`, the default).
These states include those derived from data variables (like
"x", "xTemp", "zoomX", "panX", "deltaX", "xlabel", and the "y" counterparts)
since these values determine coordinates in the plot and so are typically not to be copied.
Similarly "swapAxes" is one of these basic states because in \code{l_compound} plots such
as \code{l_pairs()} swapping axes can wreak havoc if unintended.
Finally, an important pair of basic states to exclude are
"linkingKey" and "linkingGroup" since such changes  require proper
synchronization.

Setting `excludeBasicStates = TRUE` is a simple way to avoid copying the values of these
basic states.
Setting `excludeBasicStates = FALSE` will allow these to be copied as well.}

\item{returnNames}{a logical to indicate whether to return the names
of all states successfully copied for all plots.  Default is `FALSE`}
}
\value{
a character vector of the names of the states successfully copied
(for each plot whose states were affected), or
NULL if none were copied or `returnNames == FALSE`.
}
\description{
\code{l_copyStates} reads the values of the states of the `source` and
assigns them to the states of the same name on the `target`.
}
\examples{
if(interactive()){
# Source and target are `l_plots`
   p <- with(iris,
         l_plot(x = Sepal.Width, y = Petal.Width,
                color = Species, glyph = "ccircle",
                size = 10, showGuides = TRUE,
                title = "Edgar Anderson's Iris data"
               )
           )

   p2 <- with(iris,
          l_plot(x = Sepal.Length, y = Petal.Length,
                 title = "Fisher's Iris data"
                 )
              )
# Copy the states of p to p2
# First just the size and title
   l_copyStates(source = p, target = p2,
                states = c("size", "title")
                )
# Copy all but those associated with the variables
   l_copyStates(source = p, target = p2)

# Suppose p had a linkingGroup, say "Edgar"
   l_configure(p, linkingGroup = "Edgar", sync = "push")

# To force this linkingGroup to be copied to a new plot
   p3 <- with(iris,
          l_plot(x = Sepal.Length, y = Petal.Length,
                 title = "Fisher's Iris data"
                 )
              )
   l_copyStates(source = p, target = p3,
                states = c("linkingGroup"),
                # To allow this to happen:
                excludeBasicStates = FALSE
                )

   h <- with(iris,
             l_hist((Petal.Width * Petal.Length),
                    showStackedColors = TRUE,
                    yshows = "density")
                    )
   l_copyStates(source = p, target = h)

   sa <- l_serialaxes(iris, axes = "parallel")
   l_copyStates(p, sa)

   pp <- l_pairs(iris, showHistograms = TRUE)
   l_copyStates(p, pp)  # makes no copy (not one to one)

   pp2 <- l_pairs(iris,
                  color = iris$Species,
                  showGuides = TRUE,
                  title ="Iris data",
                  glyph = "ctriangle")
   l_copyStates(pp2, pp)
   l_copyStates(pp2, p)
}

}
\seealso{
\code{\link{l_saveStates}} \code{\link{l_info_states}} \code{\link{saveRDS}}
}
